﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_FileStorage.h>
#include <nn/fssystem/utilTool/fs_BinaryMatchPrivate.h>
#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

namespace nn { namespace fssystem { namespace utilTool {

class BinaryMatchTest
{
public:
    static const int BlockSize = 16;
    static const int RegionSize = BinaryMatch::RegionSizeMin;

public:
    typedef BinaryMatch::HashQueue HashQueue;
    typedef BinaryMatch::HashQueueArray HashQueueArray;

public:
    BinaryMatchTest(char* oldBuffer, int oldBufferSize, char* newBuffer, int newBufferSize) NN_NOEXCEPT
        : m_OldStorage(oldBuffer, oldBufferSize)
        , m_OldStorageSize(oldBufferSize)
        , m_NewStorage(newBuffer, newBufferSize)
        , m_NewStorageSize(newBufferSize)
        , m_Match(BlockSize, RegionSize, 0)
    {
    }

    BinaryMatchTest(char* oldBuffer, int oldBufferSize, char* newBuffer, int newBufferSize, size_t matchSize) NN_NOEXCEPT
        : m_OldStorage(oldBuffer, oldBufferSize)
        , m_OldStorageSize(oldBufferSize)
        , m_NewStorage(newBuffer, newBufferSize)
        , m_NewStorageSize(newBufferSize)
        , m_Match(BlockSize, RegionSize, matchSize)
    {
    }

    nn::Result Run() NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Match.Run(
            nn::fs::SubStorage(&m_OldStorage, 0, m_OldStorageSize),
            nn::fs::SubStorage(&m_NewStorage, 0, m_NewStorageSize),
            BlockSize
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result Run(size_t shiftSize) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Match.Run(
            nn::fs::SubStorage(&m_OldStorage, 0, m_OldStorageSize),
            nn::fs::SubStorage(&m_NewStorage, 0, m_NewStorageSize),
            shiftSize
        ));
        NN_RESULT_SUCCESS;
    }

    template< int Length >
    nn::Result Run(const BinaryMatch::Hint(&hints)[Length]) NN_NOEXCEPT
    {
        m_Match.SetHint(hints, Length);

        NN_RESULT_DO(m_Match.Run(
            nn::fs::SubStorage(&m_OldStorage, 0, m_OldStorageSize),
            nn::fs::SubStorage(&m_NewStorage, 0, m_NewStorageSize),
            BlockSize
        ));
        NN_RESULT_SUCCESS;
    }

    template< int Length >
    nn::Result RunHintOnly(const BinaryMatch::Hint(&hints)[Length]) NN_NOEXCEPT
    {
        const auto workBufferSize = RegionSize * 4;

        std::unique_ptr<char[]> buffer(new char[workBufferSize]);
        NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultAllocationMemoryFailedNew());

        m_Match.SetHint(hints, Length);
        m_Match.m_OldStorage = nn::fs::SubStorage(&m_OldStorage, 0, m_OldStorageSize);
        m_Match.m_OldStorageSize = m_OldStorageSize;
        m_Match.m_NewStorage = nn::fs::SubStorage(&m_NewStorage, 0, m_NewStorageSize);
        m_Match.m_NewStorageSize = m_NewStorageSize;

        NN_RESULT_DO(m_Match.MakeRegionHash(buffer.get()));
        NN_RESULT_DO(m_Match.CompareDataWithHint(buffer.get()));

        if( !m_Match.m_Result.empty() )
        {
            BinaryMatch::Result::Merge(&m_Match.m_Result);
        }

        NN_RESULT_SUCCESS;
    }

    std::vector<BinaryMatch::Result> GetResult() NN_NOEXCEPT
    {
        return m_Match.GetResult();
    }

private:
    nn::fs::MemoryStorage m_OldStorage;
    const int64_t m_OldStorageSize;
    nn::fs::MemoryStorage m_NewStorage;
    const int64_t m_NewStorageSize;
    BinaryMatch m_Match;
};

}}}

namespace {

typedef nn::fssystem::utilTool::BinaryMatch BinaryMatch;
typedef nn::fssystem::utilTool::BinaryMatchPhase BinaryMatchPhase;
typedef nn::fssystem::utilTool::BinaryMatchProgress BinaryMatchProgress;
typedef nn::fssystem::utilTool::BinaryMatchTest BinaryMatchTest;
typedef BinaryMatch::Result BinaryMatchResult;
typedef BinaryMatch::Hint BinaryMatchHint;

const auto BlockSize = BinaryMatchTest::BlockSize;
const auto RegionSize = BinaryMatchTest::RegionSize;
const auto InvalidOffset = BinaryMatchResult::InvalidOffset;

}

#if !defined(NN_SDK_BUILD_RELEASE)
/**
 *  @brief  BinaryMatch の事前条件をテストします。
 */
TEST(BinaryMatchDeathTest, Precondition)
{
    // 4 引数のコンストラクタをチェック
    EXPECT_DEATH_IF_SUPPORTED(BinaryMatch(0, RegionSize, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(BinaryMatch(1, RegionSize - 1, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(BinaryMatch(2, RegionSize / 2, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(BinaryMatch(2, RegionSize + 1, 0), "");

    // SetWindowSize() のチェック
    EXPECT_DEATH_IF_SUPPORTED(BinaryMatch(1, RegionSize, 4).SetWindowSize(0), "");

    // AddExcludeRangeForOldStorage() のチェック
    {
        BinaryMatch::ExcludeRange range = { -1, 1 };
        EXPECT_DEATH_IF_SUPPORTED(
            BinaryMatch(16, RegionSize, 0).AddExcludeRangeForOldStorage(range), "");
    }
    {
        BinaryMatch::ExcludeRange range = { 15, 1 };
        EXPECT_DEATH_IF_SUPPORTED(
            BinaryMatch(16, RegionSize, 0).AddExcludeRangeForOldStorage(range), "");
    }
    {
        BinaryMatch::ExcludeRange range = { 0, 0 };
        EXPECT_DEATH_IF_SUPPORTED(
            BinaryMatch(16, RegionSize, 0).AddExcludeRangeForOldStorage(range), "");
    }
    {
        BinaryMatch::ExcludeRange range = { 0, 15 };
        EXPECT_DEATH_IF_SUPPORTED(
            BinaryMatch(16, RegionSize, 0).AddExcludeRangeForOldStorage(range), "");
    }

    // AddExcludeRangeForNewStorage() のチェック
    {
        BinaryMatch::ExcludeRange range = { -1, 1 };
        EXPECT_DEATH_IF_SUPPORTED(
            BinaryMatch(16, RegionSize, 0).AddExcludeRangeForNewStorage(range), "");
    }
    {
        BinaryMatch::ExcludeRange range = { 15, 1 };
        EXPECT_DEATH_IF_SUPPORTED(
            BinaryMatch(16, RegionSize, 0).AddExcludeRangeForNewStorage(range), "");
    }
    {
        BinaryMatch::ExcludeRange range = { 0, 0 };
        EXPECT_DEATH_IF_SUPPORTED(
            BinaryMatch(16, RegionSize, 0).AddExcludeRangeForNewStorage(range), "");
    }
    {
        BinaryMatch::ExcludeRange range = { 0, 15 };
        EXPECT_DEATH_IF_SUPPORTED(
            BinaryMatch(16, RegionSize, 0).AddExcludeRangeForNewStorage(range), "");
    }

    // SetHint のチェック
    {
        EXPECT_DEATH_IF_SUPPORTED(
            BinaryMatch(16, RegionSize, 0).SetHint(nullptr, -1), "");

        EXPECT_DEATH_IF_SUPPORTED(
            BinaryMatch(16, RegionSize, 0).SetHint(nullptr, 1), "");

        BinaryMatchHint hint[] = { 0, 32, 0, 32 };

        EXPECT_DEATH_IF_SUPPORTED(
            BinaryMatch(16, RegionSize, 0).SetHint(hint, -1), "");
    }
}

/**
 *  @brief  BinaryMatchProgress の事前条件をテストします。
 */
TEST(BinaryMatchProgressDeathTest, Precondition)
{
    // SetPhase() のチェック
    EXPECT_DEATH_IF_SUPPORTED(BinaryMatchProgress().SetPhase(BinaryMatchPhase(-1), 0), "");
    EXPECT_DEATH_IF_SUPPORTED(BinaryMatchProgress().SetPhase(BinaryMatchPhase::Count, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(BinaryMatchProgress().SetPhase(BinaryMatchPhase::MakeRegionHash, -1), "");

    // SetValue() のチェック
    EXPECT_DEATH_IF_SUPPORTED(BinaryMatchProgress().SetValue(-1), "");
}
#endif

/**
 *  @brief  BinaryMatch::HashQueue をテストします。
 */
TEST(BinaryMatchTest, HashQueue)
{
    static const int BufferSize = RegionSize * 3;

    std::unique_ptr<char[]> buffer(new char[BufferSize + BlockSize]);

    nnt::fs::util::FillBufferWithRandomValue(buffer.get(), BufferSize + BlockSize);

    BinaryMatchTest::HashQueueArray hashQueueArray(BlockSize + 1, RegionSize / BlockSize);
    ASSERT_TRUE(hashQueueArray.IsValid());

    for( int i = 0; i < BlockSize; ++i )
    {
        hashQueueArray[i].Reset(buffer.get() + i, BlockSize);
    }

    auto& resetHashQueue = hashQueueArray[BlockSize];

    // 毎回リセットしているハッシュと更新しているハッシュが同値か判定
    for( int offset = 0; offset + RegionSize < BufferSize; ++offset )
    {
        hashQueueArray[BlockSize].Reset(buffer.get() + offset, BlockSize);

        auto& updateHashQueue = hashQueueArray[offset % BlockSize];
        ASSERT_EQ(updateHashQueue.GetRegionHash(), resetHashQueue.GetRegionHash());

        updateHashQueue.Push(buffer.get() + RegionSize + offset, BlockSize);
    }
}

/**
 *  @brief  AddExcludeRange 系をテストします。
 */
TEST(BinaryMatchTest, AddExcludeRange)
{
    BinaryMatch match(16, RegionSize, 0);

    // AddExcludeRangeForOldStorage() のチェック
    {
        BinaryMatch::ExcludeRange range1 = { 16 * 4, 16 * 2 };
        NNT_ASSERT_RESULT_SUCCESS(match.AddExcludeRangeForOldStorage(range1));

        BinaryMatch::ExcludeRange range2 = { 16 * 8, 16 * 2 };
        NNT_ASSERT_RESULT_SUCCESS(match.AddExcludeRangeForOldStorage(range2));
    }
    {
        // range1 と range2 に接するが成功
        BinaryMatch::ExcludeRange range = { 16 * 6, 16 * 2 };
        NNT_ASSERT_RESULT_SUCCESS(match.AddExcludeRangeForOldStorage(range));
    }
    {
        // range1 と同値で失敗
        BinaryMatch::ExcludeRange range = { 16 * 4, 16 * 2 };
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset, match.AddExcludeRangeForOldStorage(range));
    }
    {
        // range1 と後方が被って失敗
        BinaryMatch::ExcludeRange range = { 16 * 3, 16 * 2 };
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset, match.AddExcludeRangeForOldStorage(range));
    }
    {
        // range1 と前方が被って失敗
        BinaryMatch::ExcludeRange range = { 16 * 5, 16 * 2 };
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset, match.AddExcludeRangeForOldStorage(range));
    }
    {
        // 4 GB を超える領域
        BinaryMatch::ExcludeRange range1 = { static_cast<int64_t>(4) * 1024 * 1024 * 1024 + 16 * 4, 16 * 2 };
        NNT_ASSERT_RESULT_SUCCESS(match.AddExcludeRangeForOldStorage(range1));

        BinaryMatch::ExcludeRange range2 = { static_cast<int64_t>(4) * 1024 * 1024 * 1024 + 16 * 8, 16 * 2 };
        NNT_ASSERT_RESULT_SUCCESS(match.AddExcludeRangeForOldStorage(range2));
    }

    // AddExcludeRangeForNewStorage() のチェック
    {
        BinaryMatch::ExcludeRange range1 = { 16 * 4, 16 * 2 };
        NNT_ASSERT_RESULT_SUCCESS(match.AddExcludeRangeForNewStorage(range1));

        BinaryMatch::ExcludeRange range2 = { 16 * 8, 16 * 2 };
        NNT_ASSERT_RESULT_SUCCESS(match.AddExcludeRangeForNewStorage(range2));
    }
    {
        // range1 と range2 に接するが成功
        BinaryMatch::ExcludeRange range = { 16 * 6, 16 * 2 };
        NNT_ASSERT_RESULT_SUCCESS(match.AddExcludeRangeForNewStorage(range));
    }
    {
        // range1 と同値で失敗
        BinaryMatch::ExcludeRange range = { 16 * 4, 16 * 2 };
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset, match.AddExcludeRangeForNewStorage(range));
    }
    {
        // range1 と後方が被って失敗
        BinaryMatch::ExcludeRange range = { 16 * 3, 16 * 2 };
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset, match.AddExcludeRangeForNewStorage(range));
    }
    {
        // range1 と前方が被って失敗
        BinaryMatch::ExcludeRange range = { 16 * 5, 16 * 2 };
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidOffset, match.AddExcludeRangeForNewStorage(range));
    }
    {
        // 4 GB を超える領域
        BinaryMatch::ExcludeRange range1 = { static_cast<int64_t>(4) * 1024 * 1024 * 1024 + 16 * 4, 16 * 2 };
        NNT_ASSERT_RESULT_SUCCESS(match.AddExcludeRangeForNewStorage(range1));

        BinaryMatch::ExcludeRange range2 = { static_cast<int64_t>(4) * 1024 * 1024 * 1024 + 16 * 8, 16 * 2 };
        NNT_ASSERT_RESULT_SUCCESS(match.AddExcludeRangeForNewStorage(range2));
    }
}

/**
*  @brief  データ末尾のリージョンサイズに満たない領域（断片）が正しく処理されるかテストします。
*/
TEST(BinaryMatchTest, FragmentData)
{
    // old 側の断片
    {
        const auto OldStorageSize = RegionSize + RegionSize / 2;
        const auto NewStorageSize = RegionSize * 2;
        std::unique_ptr<char[]> buffer1(new char[OldStorageSize]);
        std::unique_ptr<char[]> buffer2(new char[NewStorageSize]);

        // 断片単体ではマッチしない
        {
            nnt::fs::util::FillBufferWithRandomValue(buffer1.get(), OldStorageSize);
            nnt::fs::util::FillBufferWithRandomValue(buffer2.get(), NewStorageSize);
            std::memcpy(buffer2.get() + OldStorageSize, buffer1.get() + RegionSize, RegionSize / 2);

            BinaryMatchTest match(buffer1.get(), OldStorageSize, buffer2.get(), NewStorageSize);

            NNT_ASSERT_RESULT_SUCCESS(match.Run());

            auto results = match.GetResult();
            ASSERT_EQ(1, results.size());
            EXPECT_EQ(InvalidOffset, results[0].oldOffset);
            EXPECT_EQ(0, results[0].newOffset);
            EXPECT_EQ(RegionSize * 2, results[0].size);
        }

        // 断片の直前がマッチすると断片もマッチする
        {
            nnt::fs::util::FillBufferWithRandomValue(buffer1.get(), OldStorageSize);
            nnt::fs::util::FillBufferWithRandomValue(buffer2.get(), RegionSize / 2);
            std::memcpy(buffer2.get() + RegionSize / 2, buffer1.get(), RegionSize + RegionSize / 2);

            BinaryMatchTest match(buffer1.get(), OldStorageSize, buffer2.get(), NewStorageSize);

            NNT_ASSERT_RESULT_SUCCESS(match.Run());

            auto results = match.GetResult();
            ASSERT_EQ(2, results.size());
            EXPECT_EQ(InvalidOffset, results[0].oldOffset);
            EXPECT_EQ(0, results[0].newOffset);
            EXPECT_EQ(RegionSize / 2, results[0].size);
            EXPECT_EQ(0, results[1].oldOffset);
            EXPECT_EQ(RegionSize / 2, results[1].newOffset);
            EXPECT_EQ(RegionSize + RegionSize / 2, results[1].size);
        }
    }

    // new 側の断片
    {
        const auto OldStorageSize = RegionSize * 2;
        const auto NewStorageSize = RegionSize + RegionSize / 2;
        std::unique_ptr<char[]> buffer1(new char[OldStorageSize]);
        std::unique_ptr<char[]> buffer2(new char[NewStorageSize]);

        nnt::fs::util::FillBufferWithRandomValue(buffer1.get(), OldStorageSize);
        std::memcpy(buffer2.get(), buffer1.get() + RegionSize / 2, RegionSize + RegionSize / 2);

        BinaryMatchTest match(buffer1.get(), OldStorageSize, buffer2.get(), NewStorageSize);

        NNT_ASSERT_RESULT_SUCCESS(match.Run());

        auto results = match.GetResult();
        ASSERT_EQ(1, results.size());
        EXPECT_EQ(RegionSize / 2, results[0].oldOffset);
        EXPECT_EQ(0, results[0].newOffset);
        EXPECT_EQ(RegionSize + RegionSize / 2, results[0].size);
    }
}

/**
*  @brief  リージョンハッシュをまとめる処理が正しく動作するかテストします。
*/
TEST(BinaryMatchTest, CollectRegionHash)
{
    static const int BufferSize = RegionSize * 8;

    std::unique_ptr<char[]> buffer1(new char[BufferSize]);
    std::unique_ptr<char[]> buffer2(new char[BufferSize]);

    // まとめた領域と同じ長さで比較する
    {
        nnt::fs::util::FillBufferWithRandomValue(buffer1.get(), RegionSize);
        std::memset(buffer1.get() + RegionSize, 0, RegionSize * 3);
        nnt::fs::util::FillBufferWithRandomValue(buffer1.get() + RegionSize * 4, RegionSize);

        nnt::fs::util::FillBufferWithRandomValue(buffer2.get(), RegionSize);
        std::memset(buffer2.get() + RegionSize, 0, RegionSize * 3);
        nnt::fs::util::FillBufferWithRandomValue(buffer2.get() + RegionSize * 4, RegionSize);

        const auto OldStorageSize = RegionSize * 5;
        const auto NewStorageSize = RegionSize * 5;
        BinaryMatchTest match(buffer1.get(), OldStorageSize, buffer2.get(), NewStorageSize);

        NNT_ASSERT_RESULT_SUCCESS(match.Run());

        auto results = match.GetResult();
        ASSERT_EQ(             3, results.size());
        EXPECT_EQ( InvalidOffset, results[0].oldOffset);
        EXPECT_EQ(             0, results[0].newOffset);
        EXPECT_EQ(    RegionSize, results[0].size);
        EXPECT_EQ(    RegionSize, results[1].oldOffset);
        EXPECT_EQ(    RegionSize, results[1].newOffset);
        EXPECT_EQ(RegionSize * 3, results[1].size);
        EXPECT_EQ( InvalidOffset, results[2].oldOffset);
        EXPECT_EQ(RegionSize * 4, results[2].newOffset);
        EXPECT_EQ(    RegionSize, results[2].size);
    }

    // まとめた領域より短い長さで比較する
    {
        nnt::fs::util::FillBufferWithRandomValue(buffer1.get(), RegionSize);
        std::memset(buffer1.get() + RegionSize, 0, RegionSize * 3);
        nnt::fs::util::FillBufferWithRandomValue(buffer1.get() + RegionSize * 4, RegionSize);

        nnt::fs::util::FillBufferWithRandomValue(buffer2.get(), RegionSize);
        std::memset(buffer2.get() + RegionSize, 0, RegionSize * 2);
        nnt::fs::util::FillBufferWithRandomValue(buffer2.get() + RegionSize * 3, RegionSize);

        const auto OldStorageSize = RegionSize * 5;
        const auto NewStorageSize = RegionSize * 4;
        BinaryMatchTest match(buffer1.get(), OldStorageSize, buffer2.get(), NewStorageSize);

        NNT_ASSERT_RESULT_SUCCESS(match.Run());

        auto results = match.GetResult();
        ASSERT_EQ(             3, results.size());
        EXPECT_EQ( InvalidOffset, results[0].oldOffset);
        EXPECT_EQ(             0, results[0].newOffset);
        EXPECT_EQ(    RegionSize, results[0].size);
        EXPECT_EQ(    RegionSize, results[1].oldOffset);
        EXPECT_EQ(    RegionSize, results[1].newOffset);
        EXPECT_EQ(RegionSize * 2, results[1].size);
        EXPECT_EQ( InvalidOffset, results[2].oldOffset);
        EXPECT_EQ(RegionSize * 3, results[2].newOffset);
        EXPECT_EQ(    RegionSize, results[2].size);
    }

    // まとめた領域より長い長さで比較する
    {
        nnt::fs::util::FillBufferWithRandomValue(buffer1.get(), RegionSize);
        std::memset(buffer1.get() + RegionSize, 0, RegionSize * 3);
        nnt::fs::util::FillBufferWithRandomValue(buffer1.get() + RegionSize * 4, RegionSize);

        nnt::fs::util::FillBufferWithRandomValue(buffer2.get(), RegionSize);
        std::memset(buffer2.get() + RegionSize, 0, RegionSize * 4);
        nnt::fs::util::FillBufferWithRandomValue(buffer2.get() + RegionSize * 5, RegionSize);

        const auto OldStorageSize = RegionSize * 5;
        const auto NewStorageSize = RegionSize * 6;
        BinaryMatchTest match(buffer1.get(), OldStorageSize, buffer2.get(), NewStorageSize);

        NNT_ASSERT_RESULT_SUCCESS(match.Run());

        auto results = match.GetResult();
        ASSERT_EQ(             4, results.size());
        EXPECT_EQ( InvalidOffset, results[0].oldOffset);
        EXPECT_EQ(             0, results[0].newOffset);
        EXPECT_EQ(    RegionSize, results[0].size);
        EXPECT_EQ(    RegionSize, results[1].oldOffset);
        EXPECT_EQ(    RegionSize, results[1].newOffset);
        EXPECT_EQ(RegionSize * 3, results[1].size);
        EXPECT_EQ(    RegionSize, results[2].oldOffset);
        EXPECT_EQ(RegionSize * 4, results[2].newOffset);
        EXPECT_EQ(    RegionSize, results[2].size);
        EXPECT_EQ( InvalidOffset, results[3].oldOffset);
        EXPECT_EQ(RegionSize * 5, results[3].newOffset);
        EXPECT_EQ(    RegionSize, results[3].size);
    }

    // まとめた領域より短い長さで後方の領域が一致する
    {
        nnt::fs::util::FillBufferWithRandomValue(buffer1.get(), RegionSize);
        std::memset(buffer1.get() + RegionSize, 0, RegionSize * 3);
        nnt::fs::util::FillBufferWithRandomValue(buffer1.get() + RegionSize * 4, RegionSize);

        nnt::fs::util::FillBufferWithRandomValue(buffer2.get(), RegionSize);
        std::memset(buffer2.get() + RegionSize, 0, RegionSize * 2);
        std::memcpy(buffer2.get() + RegionSize * 3, buffer1.get() + RegionSize * 4, RegionSize);

        const auto OldStorageSize = RegionSize * 5;
        const auto NewStorageSize = RegionSize * 4;
        BinaryMatchTest match(buffer1.get(), OldStorageSize, buffer2.get(), NewStorageSize);

        NNT_ASSERT_RESULT_SUCCESS(match.Run());

        auto results = match.GetResult();
        ASSERT_EQ(             2, results.size());
        EXPECT_EQ( InvalidOffset, results[0].oldOffset);
        EXPECT_EQ(             0, results[0].newOffset);
        EXPECT_EQ(    RegionSize, results[0].size);
        EXPECT_EQ(RegionSize * 2, results[1].oldOffset);
        EXPECT_EQ(    RegionSize, results[1].newOffset);
        EXPECT_EQ(RegionSize * 3, results[1].size);
    }
} // NOLINT(impl/function_size)

/**
*  @brief  ヒントを使用した比較が正しく動作するかテストします。
*/
TEST(BinaryMatchTest, CompareWithHint)
{
    static const int BufferSize = RegionSize * 20;

    std::unique_ptr<char[]> buffer1(new char[BufferSize]);
    std::unique_ptr<char[]> buffer2(new char[BufferSize]);

    nnt::fs::util::FillBufferWithRandomValue(buffer1.get(), BufferSize);
    nnt::fs::util::FillBufferWithRandomValue(buffer2.get(), BufferSize);

    // old [0, 1) -> new [1, 2)
    std::memcpy(buffer1.get(), buffer2.get() + RegionSize, RegionSize);
    // old [3, 6) -> new [4.5, 7.5)
    std::memcpy(buffer1.get() + RegionSize * 3, buffer2.get() + RegionSize * 4 + RegionSize / 2, RegionSize * 3);
    // old [9, 9.5) -> new [10, 10.5)
    std::memcpy(buffer1.get() + RegionSize * 9, buffer2.get() + RegionSize * 10, RegionSize / 2);
    // old [12, 14) -> new [13.x, 15.x)
    std::memcpy(buffer1.get() + RegionSize * 12, buffer2.get() + RegionSize * 14 - BlockSize * 5, RegionSize * 2);

    BinaryMatchTest match(buffer1.get(), BufferSize, buffer2.get(), BufferSize);

    const BinaryMatchHint Hints[] =
    {
        {              0,  RegionSize,     RegionSize,                      RegionSize },
        { RegionSize * 3,  RegionSize * 3, RegionSize * 4 + RegionSize / 2, RegionSize * 3 },
        { RegionSize * 9,  RegionSize / 2, RegionSize * 10,                 RegionSize / 2 },
        { RegionSize * 12, RegionSize * 2, RegionSize * 14 - BlockSize * 5, RegionSize * 2 },
        { RegionSize * 16, RegionSize,     RegionSize * 17,                 RegionSize },
    };
    NNT_ASSERT_RESULT_SUCCESS(match.RunHintOnly(Hints));

    auto results = match.GetResult();

    ASSERT_EQ(results.size(), 3);
    EXPECT_EQ(         0,                      results[0].oldOffset);
    EXPECT_EQ(RegionSize,                      results[0].newOffset);
    EXPECT_EQ(RegionSize,                      results[0].size);
    EXPECT_EQ(RegionSize * 3,                  results[1].oldOffset);
    EXPECT_EQ(RegionSize * 4 + RegionSize / 2, results[1].newOffset);
    EXPECT_EQ(RegionSize * 3,                  results[1].size);
    EXPECT_EQ(RegionSize * 12,                 results[2].oldOffset);
    EXPECT_EQ(RegionSize * 14 - BlockSize * 5, results[2].newOffset);
    EXPECT_EQ(RegionSize * 2,                  results[2].size);
}

/**
*  @brief  ヒントの使用で繰り返しデータを正しく比較できるかテストします。
*/
TEST(BinaryMatchTest, ComparePatternWithHint)
{
    static const int BufferSize = RegionSize * 8;

    std::unique_ptr<char[]> buffer1(new char[BufferSize]);
    std::unique_ptr<char[]> buffer2(new char[BufferSize]);

    nnt::fs::util::FillBufferWithRandomValue(buffer1.get(), BufferSize);
    nnt::fs::util::FillBufferWithRandomValue(buffer2.get(), BufferSize);

    char pattern[BlockSize];
    nnt::fs::util::FillBufferWithRandomValue(pattern, sizeof(pattern));

    int offset = RegionSize * 2 - BlockSize * 2;
    for( ; offset < RegionSize * 4 + BlockSize * 2; offset += BlockSize )
    {
        std::memcpy(buffer1.get() + offset, pattern, BlockSize);
        std::memcpy(buffer2.get() + offset, pattern, BlockSize);
    }

    // ヒント無しだと意図した通りにマッチしてくれない
    {
        BinaryMatchTest match(buffer1.get(), BufferSize, buffer2.get(), BufferSize);

        NNT_ASSERT_RESULT_SUCCESS(match.Run());

        auto results = match.GetResult();

        ASSERT_EQ(results.size(), 3);
        EXPECT_EQ(InvalidOffset,                  results[0].oldOffset);
        EXPECT_EQ(0,                              results[0].newOffset);
        EXPECT_EQ(RegionSize * 2 - BlockSize * 2, results[0].size);
        EXPECT_EQ(RegionSize * 2,                 results[1].oldOffset);
        EXPECT_EQ(RegionSize * 2 - BlockSize * 2, results[1].newOffset);
        EXPECT_EQ(RegionSize * 2 + BlockSize * 2, results[1].size);
        EXPECT_EQ(InvalidOffset,                  results[2].oldOffset);
        EXPECT_EQ(RegionSize * 4,                 results[2].newOffset);
        EXPECT_EQ(RegionSize * 4,                 results[2].size);
    }

    // ヒントありだと意図した通りにマッチしてくれる
    {
        BinaryMatchTest match(buffer1.get(), BufferSize, buffer2.get(), BufferSize);

        const BinaryMatchHint Hints[] =
        {
            {
                RegionSize * 2 - BlockSize * 2,
                RegionSize * 2 + BlockSize * 2,
                RegionSize * 2 - BlockSize * 2,
                RegionSize * 2 + BlockSize * 2,
            },
        };
        NNT_ASSERT_RESULT_SUCCESS(match.Run(Hints));

        auto results = match.GetResult();

        ASSERT_EQ(results.size(), 3);
        EXPECT_EQ(InvalidOffset,                  results[0].oldOffset);
        EXPECT_EQ(0,                              results[0].newOffset);
        EXPECT_EQ(RegionSize * 2 - BlockSize * 2, results[0].size);
        EXPECT_EQ(RegionSize * 2 - BlockSize * 2, results[1].oldOffset);
        EXPECT_EQ(RegionSize * 2 - BlockSize * 2, results[1].newOffset);
        EXPECT_EQ(RegionSize * 2 + BlockSize * 4, results[1].size);
        EXPECT_EQ(InvalidOffset,                  results[2].oldOffset);
        EXPECT_EQ(RegionSize * 4 + BlockSize * 2, results[2].newOffset);
        EXPECT_EQ(RegionSize * 4 - BlockSize * 2, results[2].size);
    }
}

/**
*  @brief  BlockSize 以下の shiftSize で正しく比較できるかテストします。
*/
TEST(BinaryMatchTest, CompareSmallAlignment)
{
    static const int BufferSize = RegionSize * 4;

    std::unique_ptr<char[]> buffer1(new char[BufferSize]);
    std::unique_ptr<char[]> buffer2(new char[BufferSize]);

    auto CompareResult = [](const BinaryMatchResult& lhs, const BinaryMatchResult& rhs) NN_NOEXCEPT -> bool
    {
        return lhs.newOffset == rhs.newOffset &&
               lhs.oldOffset == rhs.oldOffset &&
               lhs.size == rhs.size;
    };

    const BinaryMatchResult LargeAlignmentResult[]
    {
        BinaryMatchResult::MakeUnknown(0, BufferSize),
    };
    BinaryMatchResult smallAlignmentResult[3];

    for( int stride = 1; stride <= BlockSize; stride <<= 1 )
    {
        nnt::fs::util::FillBufferWithRandomValue(buffer1.get(), BufferSize);
        nnt::fs::util::FillBufferWithRandomValue(buffer2.get(), BufferSize);

        const int copyOffset = BlockSize * 16 + BlockSize * 4 + stride;

        std::memcpy(buffer2.get() + copyOffset, buffer1.get() + RegionSize, RegionSize * 2);

        smallAlignmentResult[0] = BinaryMatchResult::MakeUnknown(0, copyOffset);
        smallAlignmentResult[1] = BinaryMatchResult::MakeMatch(copyOffset, RegionSize, RegionSize * 2);
        smallAlignmentResult[2] = BinaryMatchResult::MakeUnknown(copyOffset + RegionSize * 2, RegionSize * 2 - copyOffset);

        for( int shiftSize = 1; shiftSize <= BlockSize; shiftSize <<= 1 )
        {
            BinaryMatchTest match(buffer1.get(), BufferSize, buffer2.get(), BufferSize, RegionSize);

            NNT_ASSERT_RESULT_SUCCESS(match.Run(shiftSize));

            auto results = match.GetResult();

            if( shiftSize <= stride )
            {
                ASSERT_EQ(results.size(), 3);
                EXPECT_TRUE(CompareResult(results[0], smallAlignmentResult[0]));
                EXPECT_TRUE(CompareResult(results[1], smallAlignmentResult[1]));
                EXPECT_TRUE(CompareResult(results[2], smallAlignmentResult[2]));
            }
            else
            {
                ASSERT_EQ(results.size(), 1);
                EXPECT_TRUE(CompareResult(results[0], LargeAlignmentResult[0]));
            }
        }
    }
}

/**
*  @brief  4 GB を超えるストレージのマッチングが正しく処理されるかテストします。
*/
TEST(BinaryMatchTest, FileLargeHeavy)
{
    const auto RegionSizeForLarge = 16 * 1024;

    const auto BaseStorageSize = static_cast<int64_t>(64) * 1024 * 1024 * 1024;
    const auto OldStorageSize = BaseStorageSize + RegionSizeForLarge;
    const auto NewStorageSize = BaseStorageSize + RegionSizeForLarge * 2;

    // old ストレージ作成
    nnt::fs::util::VirtualMemoryStorage oldStorage;
    oldStorage.Initialize(OldStorageSize);
    NN_UTIL_SCOPE_EXIT
    {
        oldStorage.Finalize();
    };
    nn::fs::SubStorage oldSubStorage(&oldStorage, 0, OldStorageSize);

    // new ストレージ作成
    nnt::fs::util::VirtualMemoryStorage newStorage;
    newStorage.Initialize(NewStorageSize);
    nn::fs::SubStorage newSubStorage(&newStorage, 0, NewStorageSize);

    {
        auto buffer = nnt::fs::util::AllocateBuffer(RegionSizeForLarge);

        std::memset(buffer.get(), 1, RegionSizeForLarge);
        NNT_ASSERT_RESULT_SUCCESS(oldSubStorage.Write(BaseStorageSize, buffer.get(), RegionSizeForLarge));

        std::memset(buffer.get(), 2, RegionSizeForLarge);
        NNT_ASSERT_RESULT_SUCCESS(newSubStorage.Write(BaseStorageSize, buffer.get(), RegionSizeForLarge));
        std::memset(buffer.get(), 1, RegionSizeForLarge);
        NNT_ASSERT_RESULT_SUCCESS(newSubStorage.Write(BaseStorageSize + RegionSizeForLarge, buffer.get(), RegionSizeForLarge));
    }

    // マッチング
    BinaryMatch match(BlockSize, RegionSizeForLarge, 0);
    NNT_ASSERT_RESULT_SUCCESS(match.Run(oldSubStorage, newSubStorage, BlockSize));

    auto results = match.GetResult();
    ASSERT_EQ(3, results.size());

    EXPECT_EQ(0, results[0].oldOffset);
    EXPECT_EQ(0, results[0].newOffset);
    EXPECT_EQ(BaseStorageSize, results[0].size);

    EXPECT_EQ(InvalidOffset, results[1].oldOffset);
    EXPECT_EQ(BaseStorageSize, results[1].newOffset);
    EXPECT_EQ(RegionSizeForLarge, results[1].size);

    EXPECT_EQ(BaseStorageSize, results[2].oldOffset);
    EXPECT_EQ(BaseStorageSize + RegionSizeForLarge, results[2].newOffset);
    EXPECT_EQ(RegionSizeForLarge, results[2].size);
}

/**
*   @brief  同じリージョンハッシュを持つ候補が多くても最長一致が保証されるかテストします。
*/
TEST(BinaryMatchTest, EnsureLongestMatchInManyCandidate)
{
    static const auto Buffer1Size = RegionSize * 2      // a
                                + RegionSize * 2 * 80   // b x80
                                + RegionSize * 3        // c（前半は b と同じ）
                                + RegionSize * 2;       // d

    static const auto Buffer2Size = RegionSize * 2      // A（a と同じ）
                                + RegionSize * 2 * 80   // B（適当な値）
                                + RegionSize * 3        // C（c と同じ）
                                + RegionSize * 2        // D（適当な値）
                                + RegionSize * 2;       // E（d と同じ）

    std::unique_ptr<char[]> buffer1(new char[Buffer1Size]);
    {
        char* buffer = buffer1.get();
        // a
        {
            std::memset(buffer, 0, RegionSize * 2);
            buffer += RegionSize * 2;
        }
        // b
        for( int i = 0; i < 80; ++i )
        {
            for( int j = 0; j < 2; ++j )
            {
                std::memset(buffer, j + 1, RegionSize);
                buffer += RegionSize;
            }
        }
        // c
        for( int j = 0; j < 3; ++j )
        {
            std::memset(buffer, j + 1, RegionSize);
            buffer += RegionSize;
        }
        // d
        {
            std::memset(buffer, 5, RegionSize * 2);
            buffer += RegionSize * 2;
        }
    }

    std::unique_ptr<char[]> buffer2(new char[Buffer2Size]);
    {
        char* buffer = buffer2.get();
        // A
        {
            std::memset(buffer, 0, RegionSize * 2);
            buffer += RegionSize * 2;
        }
        // B
        {
            std::memset(buffer, 10, RegionSize * 2 * 80);
            buffer += RegionSize * 2 * 80;
        }
        // C
        for( int j = 0; j < 3; ++j )
        {
            std::memset(buffer, j + 1, RegionSize);
            buffer += RegionSize;
        }
        // D
        {
            std::memset(buffer, 11, RegionSize * 2);
            buffer += RegionSize * 2;
        }
        // E
        {
            std::memset(buffer, 5, RegionSize * 2);
            buffer += RegionSize * 2;
        }
    }

    BinaryMatchTest match(buffer1.get(), Buffer1Size, buffer2.get(), Buffer2Size);

    const BinaryMatchHint Hints[] =
    {
        {
            0, RegionSize * 2,
            0, RegionSize * 2,
        },
        {
            Buffer1Size - RegionSize * 2, RegionSize * 2,
            Buffer2Size - RegionSize * 2, RegionSize * 2,
        },
    };
    NNT_ASSERT_RESULT_SUCCESS(match.Run(Hints));

    auto results = match.GetResult();

    ASSERT_EQ(results.size(), 5);

    EXPECT_EQ(0,                                        results[0].oldOffset);
    EXPECT_EQ(0,                                        results[0].newOffset);
    EXPECT_EQ(RegionSize * 2,                           results[0].size);

    EXPECT_EQ(InvalidOffset,                            results[1].oldOffset);
    EXPECT_EQ(results[0].newOffset + results[0].size,   results[1].newOffset);
    EXPECT_EQ(RegionSize * 2 * 80,                      results[1].size);

    // b,c は先頭が同じリージョンハッシュになるが最長一致で c が選ばれる
    EXPECT_EQ(RegionSize * 2
            + RegionSize * 2 * 80,                      results[2].oldOffset);
    EXPECT_EQ(results[1].newOffset + results[1].size,   results[2].newOffset);
    EXPECT_EQ(RegionSize * 3,                           results[2].size);

    EXPECT_EQ(InvalidOffset,                            results[3].oldOffset);
    EXPECT_EQ(results[2].newOffset + results[2].size,   results[3].newOffset);
    EXPECT_EQ(RegionSize * 2,                           results[3].size);

    EXPECT_EQ(RegionSize * 2
            + RegionSize * 2 * 80
            + RegionSize * 3,                           results[4].oldOffset);
    EXPECT_EQ(results[3].newOffset + results[3].size,   results[4].newOffset);
    EXPECT_EQ(RegionSize * 2,                           results[4].size);
}

/**
*   @brief  同じリージョンハッシュを持つ候補が多くても動作するかテストします。
*/
TEST(BinaryMatchTest, ManySameRegionHash)
{
    std::unique_ptr<char[]> data1(new char[RegionSize * 7]);
    ASSERT_NE(data1, nullptr);
    for( int i = 0; i < RegionSize; ++i )
    {
        for( int j = 0; j < 16; ++j )
        {
            data1[RegionSize * 0 + j + 64] = char(j + 1);
        }
        for( int j = 0; j < 16; ++j )
        {
            data1[RegionSize * 1 + j + 64 * 2] = char(j + 1);
        }
        for( int j = 0; j < 16; ++j )
        {
            data1[RegionSize * 2 + j + 64 * 4] = char(j + 1);
        }
        for( int j = 0; j < 16; ++j )
        {
            data1[RegionSize * 3 + j + 64 * 6] = char(j + 1);
        }
        for( int j = 0; j < 16; ++j )
        {
            data1[RegionSize * 4 + j + 64 * 8] = char(j + 1);
        }
        for( int j = 0; j < 16; ++j )
        {
            data1[RegionSize * 5 + j + 64 * 10] = char(j + 1);
        }
    }

    std::unique_ptr<char[]> data2(new char[RegionSize * 7]);
    ASSERT_NE(data2, nullptr);
    std::memcpy(data2.get(), data1.get(), RegionSize * 7);
    {
        std::unique_ptr<char[]> replace(new char[512]);
        ASSERT_NE(replace, nullptr);
        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        {
            const auto count = std::uniform_int_distribution<>(32, 512)(mt);
            nnt::fs::util::FillBufferWithRandomValue(replace.get(), count);

            const auto index = std::uniform_int_distribution<>(0, RegionSize * 6)(mt);
            std::memcpy(data2.get() + index, replace.get(), count);
        }
    }

    std::unique_ptr<char[]> data3(new char[RegionSize]);
    ASSERT_NE(data3, nullptr);

    const int BufferSize = 128 * 1024 * 1024;
    std::unique_ptr<char[]> buffer1(new char[BufferSize]);
    ASSERT_NE(buffer1, nullptr);
    std::unique_ptr<char[]> buffer2(new char[BufferSize]);
    ASSERT_NE(buffer2, nullptr);

    char* ptr1 = buffer1.get();
    char* ptr2 = buffer2.get();

    for( int i = 0; i < BufferSize; i += RegionSize * 8 )
    {
        nnt::fs::util::FillBufferWithRandomValue(data3.get(), RegionSize);

        std::memcpy(ptr1, data1.get(), RegionSize * 7);
        ptr1 += RegionSize * 7;
        std::memcpy(ptr1, data3.get(), RegionSize);
        ptr1 += RegionSize;

        std::memcpy(ptr2, data2.get(), RegionSize * 7);
        ptr2 += RegionSize * 7;
        std::memcpy(ptr2, data3.get(), RegionSize);
        ptr2 += RegionSize;
    }

    BinaryMatchTest match(buffer1.get(), BufferSize, buffer2.get(), BufferSize);

    const auto start = std::chrono::system_clock::now();

    // SIGLO-81360 以降では概ね 2 秒以内に終わる。それより前は 5 秒以上
    NNT_ASSERT_RESULT_SUCCESS(match.Run());

    const auto elapsed = std::chrono::duration_cast<std::chrono::milliseconds>(
        std::chrono::system_clock::now() - start).count();

    NN_LOG("BinaryMatch elapsed: %lldms\n", elapsed);

    int64_t mismatchSize = 0;
    for( const auto& result : match.GetResult() )
    {
        if( result.detail != nn::fssystem::utilTool::BinaryMatchDetail_Match )
        {
            mismatchSize += result.size;
        }
    }

    NN_LOG("BinaryMatch mismatch: %.2lf%%\n", double(mismatchSize * 100) / BufferSize);
}
