﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <numeric>
#include <nn/crypto.h>
#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nn/fssystem/fs_AesCtrStorage.h>
#include <nn/fssystem/fs_AesCtrCounterExtendedStorage.h>
#include <nn/fssystem/fs_BucketTreeBuilder.h>
#include <nn/fssystem/fs_NcaHeader.h>

namespace nn { namespace fssystem {

/**
 * @brief   AesCtrCounterExtendedStorage のテストクラスです。
 */
class AesCtrCounterExtendedStorageTest : public ::testing::Test
{
public:
    class StorageWrapperBase;
    class StorageWrapper;
    class LargeSizeStorageWrapper;

public:
    typedef nn::fssystem::AesCtrCounterExtendedStorage ImplType;
    typedef ImplType::Entry TableEntry;

public:
    static const size_t BufferSize = 16 * 1024 * 1024;
    static const size_t BlockSize = ImplType::BlockSize;
    static const size_t KeySize = ImplType::KeySize;
    static const size_t CounterSize = ImplType::CounterSize;
    static const size_t EntrySize = sizeof(ImplType::Entry);
    static const size_t NodeSize = ImplType::NodeSize;
    static const int KeyCount = 16;

public:
    //! BucketTree を取得します。
    static const BucketTree& GetTable(const ImplType& storage) NN_NOEXCEPT
    {
        return storage.m_Table;
    }
};

NN_DEFINE_STATIC_CONSTANT(const size_t AesCtrCounterExtendedStorageTest::BlockSize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesCtrCounterExtendedStorageTest::KeySize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesCtrCounterExtendedStorageTest::CounterSize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesCtrCounterExtendedStorageTest::NodeSize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesCtrCounterExtendedStorageTest::EntrySize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesCtrCounterExtendedStorageTest::BufferSize);
NN_DEFINE_STATIC_CONSTANT(const int AesCtrCounterExtendedStorageTest::KeyCount);

/**
 * @brief   AesCtrCounterExtendedStorage のラッパークラスです。
 */
class AesCtrCounterExtendedStorageTest::StorageWrapperBase
{
public:
    typedef nn::fssystem::BucketTreeBuilder TableBuilder;

    struct Entry
    {
        int64_t offset;
        size_t size;
        int keyIndex;
        int generation;
    };

public:
    //! コンストラクタです。
    explicit StorageWrapperBase(const StorageWrapperBase& storage) NN_NOEXCEPT
        : m_TableStorage()
        , m_EntryCount(0)
        , m_Impl()
    {
        std::memcpy(m_Key, storage.m_Key, KeySize);
    }

    explicit StorageWrapperBase() NN_NOEXCEPT
        : m_TableStorage()
        , m_EntryCount(0)
        , m_Impl()
    {
    }

    virtual ~StorageWrapperBase() NN_NOEXCEPT
    {
    }

    void InitializeStorage(nn::fs::IStorage* pStorage, nn::fs::IStorage* pPlainStorage, bool isInitialize) NN_NOEXCEPT
    {
        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        std::generate(std::begin(m_Key), std::end(m_Key), [&]()
        {
            return static_cast<char>(std::uniform_int_distribution<>(
                std::numeric_limits<char>::min(), std::numeric_limits<char>::max())(mt));
        });

        int64_t storageSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&storageSize));

        // エントリデータを作成
        nnt::fs::util::Vector<Entry> entries;
        MakeEntries(&entries);

        // エントリに従って平文を暗号化
        for( const auto& entry : entries )
        {
            auto buffer = nnt::fs::util::AllocateBuffer(entry.size);
            NNT_ASSERT_RESULT_SUCCESS(pPlainStorage->Read(entry.offset, buffer.get(), entry.size));
            EncryptData(buffer.get(), buffer.get(), entry);
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(entry.offset, buffer.get(), entry.size));
        }

        BuildTable(entries, storageSize);

        if( isInitialize )
        {
            Initialize(pStorage);
        }
    }

    //! テーブルを生成します。
    void BuildTable(const nnt::fs::util::Vector<Entry>& entries, int64_t endOffset) NN_NOEXCEPT
    {
        const int entryCount = static_cast<int>(entries.size());
        const auto headerStorageSize =
            TableBuilder::QueryHeaderStorageSize();
        const auto nodeStorageSize =
            TableBuilder::QueryNodeStorageSize(NodeSize, EntrySize, entryCount);
        const auto entryStorageSize =
            TableBuilder::QueryEntryStorageSize(NodeSize, EntrySize, entryCount);
        const auto headerStorageOffset = 0;
        const auto nodeStorageOffset = headerStorageSize;
        const auto entryStorageOffset = headerStorageSize + nodeStorageSize;

        m_TableStorage.Initialize(headerStorageSize + nodeStorageSize + entryStorageSize);
        m_EntryCount = entryCount;

        // テーブルデータを作成
        TableBuilder builder;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            builder.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                nn::fs::SubStorage(&m_TableStorage, headerStorageOffset, headerStorageSize),
                nn::fs::SubStorage(&m_TableStorage, nodeStorageOffset, nodeStorageSize),
                nn::fs::SubStorage(&m_TableStorage, entryStorageOffset, entryStorageSize),
                NodeSize,
                EntrySize,
                entryCount
            )
        );

        for( auto& entry : entries )
        {
            ImplType::Entry data;
            data.SetOffset(entry.offset);
            data.reserved = 0;
            data.generation = entry.generation;

            NN_ABORT_UNLESS_RESULT_SUCCESS(builder.Write(data));
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(builder.Commit(endOffset));
    }

    //! 初期化をします。
    void Initialize(nn::fs::IStorage* pStorage) NN_NOEXCEPT
    {
        int64_t storageSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(pStorage->GetSize(&storageSize));

        // ストレージを初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            m_Impl.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                m_Key,
                KeySize,
                0,
                nn::fs::SubStorage(pStorage, 0, storageSize),
                nn::fs::SubStorage(&m_TableStorage, 0, m_TableStorage.GetSize())
            )
        );
    }

    virtual void MakeEntries(nnt::fs::util::Vector<Entry>* outValue) NN_NOEXCEPT = 0;

    //! ストレージを読み込みます。
    nn::Result Read(int64_t offset, char* buffer, size_t size) NN_NOEXCEPT
    {
        return m_Impl.Read(offset, buffer, size);
    }

    //! ストレージを取得します。
    ImplType& GetImpl() NN_NOEXCEPT
    {
        return m_Impl;
    }

    //! ストレージを取得します。
    const ImplType& GetImpl() const NN_NOEXCEPT
    {
        return m_Impl;
    }

    //! テーブルのヘッダサイズを取得します。
    static int64_t QueryHeaderSize() NN_NOEXCEPT
    {
        return BucketTree::QueryHeaderStorageSize();
    }

    //! テーブルのノードサイズを取得します。
    int64_t QueryNodeSize() const NN_NOEXCEPT
    {
        return BucketTree::QueryNodeStorageSize(NodeSize, EntrySize, m_EntryCount);
    }

    //! テーブルのエントリサイズを取得します。
    int64_t QueryEntrySize() const NN_NOEXCEPT
    {
        return BucketTree::QueryEntryStorageSize(NodeSize, EntrySize, m_EntryCount);
    }

    //! テーブルストレージを取得します。
    nnt::fs::util::SafeMemoryStorage& GetTableStorage() NN_NOEXCEPT
    {
        return m_TableStorage;
    }

    //! テーブルストレージを取得します。
    const nnt::fs::util::SafeMemoryStorage& GetTableStorage() const NN_NOEXCEPT
    {
        return m_TableStorage;
    }

private:
    //! 平文を暗号化します。
    void EncryptData(void* pDst, const void* pSrc, const Entry& entry) const NN_NOEXCEPT
    {
        NcaAesCtrUpperIv upperIv;
        upperIv.part.generation = entry.generation;
        upperIv.part.secureValue = 0;

        char counter[CounterSize];
        AesCtrStorage::MakeIv(counter, CounterSize, upperIv.value, entry.offset);

        const auto dstSize = entry.size;
        const auto outSize =
            nn::crypto::EncryptAes128Ctr(
                pDst,
                dstSize,
                m_Key,
                KeySize,
                counter,
                CounterSize,
                pSrc,
                dstSize
            );
        NN_SDK_ASSERT_EQUAL(dstSize, outSize);
        NN_UNUSED(outSize);
    }

private:
    nnt::fs::util::SafeMemoryStorage m_TableStorage;
    int m_EntryCount;
    char m_Key[KeySize];
    ImplType m_Impl;
};

class AesCtrCounterExtendedStorageTest::StorageWrapper : public AesCtrCounterExtendedStorageTest::StorageWrapperBase
{
public:
    explicit StorageWrapper(const StorageWrapper& storage) NN_NOEXCEPT
        : StorageWrapperBase(storage)
    {
        m_PlainTextStorage.Initialize(BufferSize);
        std::memcpy(m_PlainTextStorage.GetBuffer(), storage.m_PlainTextStorage.GetBuffer(), BufferSize);

        m_DataStorage.Initialize(BufferSize);
        std::memcpy(m_DataStorage.GetBuffer(), storage.m_DataStorage.GetBuffer(), BufferSize);
    }

    explicit StorageWrapper(bool isInitialize) NN_NOEXCEPT
    {
        m_PlainTextStorage.Initialize(BufferSize);
        auto plainTextBuffer = reinterpret_cast<char*>(m_PlainTextStorage.GetBuffer());
        std::iota(plainTextBuffer, reinterpret_cast<char*>(m_PlainTextStorage.GetBuffer()) + BufferSize, '\x0');

        m_DataStorage.Initialize(BufferSize);

        InitializeStorage(&m_DataStorage, &m_PlainTextStorage, isInitialize);
    }

    void Initialize() NN_NOEXCEPT
    {
        StorageWrapperBase::Initialize(&m_DataStorage);
    }

    void BuildTable(const nnt::fs::util::Vector<Entry>& entries) NN_NOEXCEPT
    {
        StorageWrapperBase::BuildTable(entries, BufferSize);
    }

    virtual void MakeEntries(nnt::fs::util::Vector<Entry>* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        ASSERT_NE(nullptr, outValue);

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        int64_t storageSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(m_DataStorage.GetSize(&storageSize));

        int64_t offset = 0;
        while( offset < storageSize )
        {
            const size_t blockCountMax = static_cast<size_t>((storageSize - offset) / BlockSize);
            const auto useBlockCountMax = std::min<size_t>(blockCountMax, 16);
            Entry entry =
            {
                static_cast<int64_t>(offset),
                std::uniform_int_distribution<size_t>(1, useBlockCountMax)(mt) * BlockSize,
                std::uniform_int_distribution<>(0, KeyCount - 1)(mt),
                0
            };

            if( outValue->empty() || outValue->back().keyIndex != entry.keyIndex )
            {
                outValue->push_back(entry);
                offset += entry.size;
            }
        }
        NN_SDK_ASSERT_EQUAL(storageSize, offset);
    }

    //! データストレージを取得します。
    nnt::fs::util::SafeMemoryStorage& GetDataStorage() NN_NOEXCEPT
    {
        return m_DataStorage;
    }

    //! データストレージを取得します。
    const nnt::fs::util::SafeMemoryStorage& GetDataStorage() const NN_NOEXCEPT
    {
        return m_DataStorage;
    }

    //! バッファが正しく読み込めたか検査します。
    bool Verify(int64_t offset, const char* buffer, size_t size) const NN_NOEXCEPT
    {
        return std::equal(buffer, buffer + size, reinterpret_cast<const char*>(m_PlainTextStorage.GetBuffer()) + offset);
    }

    //! 平文を取得します。
    char* GetPlainText() NN_NOEXCEPT
    {
        return reinterpret_cast<char*>(m_PlainTextStorage.GetBuffer());
    }

    //! 平文を取得します。
    const char* GetPlainText() const NN_NOEXCEPT
    {
        return reinterpret_cast<const char*>(m_PlainTextStorage.GetBuffer());
    }

private:
    nnt::fs::util::SafeMemoryStorage m_DataStorage;
    nnt::fs::util::SafeMemoryStorage m_PlainTextStorage;
};

class AesCtrCounterExtendedStorageTest::LargeSizeStorageWrapper : public AesCtrCounterExtendedStorageTest::StorageWrapperBase
{
public:
    static const int BufferSize = 1024;
    static const int64_t FileSize = static_cast<int64_t>(64) * 1024 * 1024 * 1024 + BufferSize;

    static const int64_t OffsetList[];
    static const int OffsetListLength;

public:
    LargeSizeStorageWrapper() NN_NOEXCEPT
    {
        m_BaseStorage.Initialize(FileSize);
        m_PlainStorage.Initialize(FileSize);

        auto writeBuffer = nnt::fs::util::AllocateBuffer(BlockSize);
        for( int i = 0; i < OffsetListLength; ++i )
        {
            const auto offset = OffsetList[i];

            nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), BlockSize);

            NN_ABORT_UNLESS_RESULT_SUCCESS(m_PlainStorage.Write(offset, writeBuffer.get(), BlockSize));
        }

        InitializeStorage(&m_BaseStorage, &m_PlainStorage, true);
    }

    virtual ~LargeSizeStorageWrapper() NN_NOEXCEPT NN_OVERRIDE
    {
        m_PlainStorage.Finalize();
        m_BaseStorage.Finalize();
    }

    virtual void MakeEntries(nnt::fs::util::Vector<Entry>* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        ASSERT_NE(nullptr, outValue);

        for( int i = 0; i < OffsetListLength; ++i )
        {
            const auto offset = OffsetList[i];

            Entry entry =
            {
                static_cast<int64_t>(offset),
                BlockSize,
                0,
                0
            };

            outValue->push_back(entry);
        }
    }

    //! バッファが正しく読み込めたか検査します。
    bool Verify(int64_t offset, const char* buffer, size_t size) NN_NOEXCEPT
    {
        auto readBuffer = nnt::fs::util::AllocateBuffer(size);
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_PlainStorage.Read(offset, readBuffer.get(), size));
        return std::equal(buffer, buffer + size, readBuffer.get());
    }

private:
    nnt::fs::util::VirtualMemoryStorage m_BaseStorage;
    nnt::fs::util::VirtualMemoryStorage m_PlainStorage;
};

const int64_t AesCtrCounterExtendedStorageTest::LargeSizeStorageWrapper::OffsetList[] =
{
    0,
    static_cast<int64_t>(4) * 1024 * 1024 * 1024,
    static_cast<int64_t>(8) * 1024 * 1024 * 1024,
    static_cast<int64_t>(16) * 1024 * 1024 * 1024,
    static_cast<int64_t>(32) * 1024 * 1024 * 1024,
    static_cast<int64_t>(64) * 1024 * 1024 * 1024,
};

const int AesCtrCounterExtendedStorageTest::LargeSizeStorageWrapper::OffsetListLength = sizeof(OffsetList) / sizeof(int64_t);

}}

namespace {

typedef nn::fssystem::AesCtrCounterExtendedStorageTest
            AesCtrCounterExtendedStorageTest;

typedef nn::fssystem::AesCtrCounterExtendedStorageTest
            AesCtrCounterExtendedStorageDeathTest;

}

#if !defined(NN_SDK_BUILD_RELEASE)
/**
 * @brief   事前検証の範囲外のテストをします。
 */
TEST_F(AesCtrCounterExtendedStorageDeathTest, Precondition)
{
    nn::fs::SubStorage subStorage;
    nn::fssystem::AesCtrCounterExtendedStorage storage;
    char key[KeySize];

    static const int64_t counterOffset = 0;

    // pAllocator 不正
    {
        std::unique_ptr<nn::fssystem::AesCtrCounterExtendedStorage::IDecryptor> pDecryptor;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::AesCtrCounterExtendedStorage::CreateSoftwareDecryptor(&pDecryptor));

        EXPECT_DEATH_IF_SUPPORTED(
            storage.Initialize(
                nullptr,
                key,
                KeySize,
                0,
                counterOffset,
                subStorage,
                subStorage,
                subStorage,
                1,
                std::move(pDecryptor)),
            ""
            );
    }

    {
        std::unique_ptr<nn::fssystem::AesCtrCounterExtendedStorage::IDecryptor> pDecryptor;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::AesCtrCounterExtendedStorage::CreateSoftwareDecryptor(&pDecryptor));

        // pKey 不正
        EXPECT_DEATH_IF_SUPPORTED(
            storage.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                nullptr,
                KeySize,
                0,
                counterOffset,
                subStorage,
                subStorage,
                subStorage,
                1,
                std::move(pDecryptor)
                ),
            ""
            );
    }

    {
        std::unique_ptr<nn::fssystem::AesCtrCounterExtendedStorage::IDecryptor> pDecryptor;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::AesCtrCounterExtendedStorage::CreateSoftwareDecryptor(&pDecryptor));

        // keySize 不正
        EXPECT_DEATH_IF_SUPPORTED(
            storage.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                key,
                KeySize - 1,
                0,
                counterOffset,
                subStorage,
                subStorage,
                subStorage,
                1,
                std::move(pDecryptor)
                ),
            ""
            );
    }

    {
        std::unique_ptr<nn::fssystem::AesCtrCounterExtendedStorage::IDecryptor> pDecryptor;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::AesCtrCounterExtendedStorage::CreateSoftwareDecryptor(&pDecryptor));

        EXPECT_DEATH_IF_SUPPORTED(
            storage.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                key,
                KeySize + 1,
                0,
                counterOffset,
                subStorage,
                subStorage,
                subStorage,
                1,
                std::move(pDecryptor)
                ),
            ""
            );
    }

    // pDecryptor 不正
    EXPECT_DEATH_IF_SUPPORTED(
        storage.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            key,
            KeySize,
            0,
            counterOffset,
            subStorage,
            subStorage,
            subStorage,
            1,
            nullptr
        ),
        ""
    );

    // offset 不正
    EXPECT_DEATH_IF_SUPPORTED(storage.Read(-1, nullptr, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(storage.Read(1, nullptr, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(storage.Read(15, nullptr, 1), "");

    // buffer 不正
    EXPECT_DEATH_IF_SUPPORTED(storage.Read(0, nullptr, 1), "");

    // size 不正
    {
        char buffer[BlockSize];
        EXPECT_DEATH_IF_SUPPORTED(storage.Read(0, buffer, 1), "");
        EXPECT_DEATH_IF_SUPPORTED(storage.Read(0, buffer, 15), "");
    }

    // 再初期化
    {
        std::unique_ptr<nn::fssystem::AesCtrCounterExtendedStorage::IDecryptor> pDecryptor;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::AesCtrCounterExtendedStorage::CreateSoftwareDecryptor(&pDecryptor));

        EXPECT_DEATH_IF_SUPPORTED(
            StorageWrapper(true).GetImpl().Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                key,
                KeySize,
                0,
                counterOffset,
                subStorage,
                subStorage,
                subStorage,
                1,
                std::move(pDecryptor)
                ),
            ""
            );
    }
} // NOLINT(impl/function_size)
#endif

/**
 * @brief   想定したリザルトを返すかをテストします。
 */
TEST_F(AesCtrCounterExtendedStorageTest, ReturnResult)
{
    nn::fs::SubStorage subStorage;
    StorageWrapper storage(true);
    nnt::fs::util::Vector<char> buffer(BlockSize);

    // 0 バイト読み込み
    NNT_EXPECT_RESULT_SUCCESS(storage.Read(0, nullptr, 0));

    // ストレージ範囲内のアクセス
    NNT_EXPECT_RESULT_SUCCESS(storage.Read(0, buffer.data(), BlockSize));
    NNT_EXPECT_RESULT_SUCCESS(storage.Read(BufferSize - BlockSize, buffer.data(), BlockSize));
    NNT_EXPECT_RESULT_SUCCESS(storage.Read(BufferSize, buffer.data(), 0));

    // ストレージ範囲外のアクセス
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultOutOfRange,
        storage.Read(BufferSize, buffer.data(), BlockSize)
    );
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultOutOfRange,
        storage.Read(BufferSize - BlockSize, buffer.data(), BlockSize * 2)
    );

    // サイズ取得
    int64_t size = 0;
    NNT_EXPECT_RESULT_SUCCESS(storage.GetImpl().GetSize(&size));
}

/**
 * @brief   全てのデータを順に読み込みます。
 */
TEST_F(AesCtrCounterExtendedStorageTest, ReadAll)
{
    static const int ReadSize = 1024;

    nnt::fs::util::Vector<char> buffer(ReadSize);
    StorageWrapper storage(true);

    for( size_t offset = 0; offset < BufferSize; offset += ReadSize )
    {
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, buffer.data(), ReadSize));

        ASSERT_TRUE(storage.Verify(offset, buffer.data(), ReadSize));
    }
}

/**
 * @brief   ランダムなサイズとオフセットでストレージを読み込みます。
 */
TEST_F(AesCtrCounterExtendedStorageTest, ReadRandom)
{
    static const int ReadSize = 16 * 1024;

    nnt::fs::util::Vector<char> buffer(ReadSize);
    StorageWrapper storage(true);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( auto count = 0; count < 500; ++count )
    {
        const auto offset =
            nn::util::align_down(
                std::uniform_int_distribution<int64_t>(0, buffer.size() - 1)(mt),
                BlockSize
            );
        const auto size =
            nn::util::align_up(
                std::uniform_int_distribution<size_t>(
                    1, static_cast<size_t>(buffer.size() - offset))(mt),
                BlockSize
            );

        NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, buffer.data(), size));

        ASSERT_TRUE(storage.Verify(offset, buffer.data(), size));
    }
}

/**
 * @brief   マルチスレッドでのストレージの読み込みをテストします。
 */
TEST_F(AesCtrCounterExtendedStorageTest, MultiThread)
{
    static const int ReadSize = 16 * 1024;
    static const auto StackSize = 32 * 1024;
    static const auto ThreadCount = 8;

    NN_OS_ALIGNAS_THREAD_STACK static char s_Stack[StackSize * ThreadCount] = {};

    StorageWrapper storage(true);
    nn::os::ThreadType threads[ThreadCount];
    const auto func = [](void* pArg) NN_NOEXCEPT
    {
        const auto pStorage = reinterpret_cast<StorageWrapper*>(pArg);

        nnt::fs::util::Vector<char> buffer(ReadSize);
        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        for( auto count = 0; count < 500; ++count )
        {
        const auto offset =
            nn::util::align_down(
                std::uniform_int_distribution<int64_t>(0, buffer.size() - 1)(mt),
                BlockSize
            );
        const auto size =
            nn::util::align_up(
                std::uniform_int_distribution<size_t>(
                    1, static_cast<size_t>(buffer.size() - offset))(mt),
                BlockSize
            );

            NNT_ASSERT_RESULT_SUCCESS(
                pStorage->Read(offset, buffer.data(), size));

            ASSERT_TRUE(pStorage->Verify(offset, buffer.data(), size));
        }
    };

    // スレッド起動
    for( auto& thread : threads )
    {
        nn::os::CreateThread(
            &thread,
            func,
            &storage,
            s_Stack + StackSize * (&thread - threads),
            StackSize,
            nn::os::DefaultThreadPriority
        );
        nn::os::StartThread(&thread);
    }

    // スレッド削除
    for( auto& thread : threads )
    {
        nn::os::WaitThread(&thread);
        nn::os::DestroyThread(&thread);
    }
}

/**
 * @brief   壊れたノードデータでエラーが返ってくるかテストします。
 */
TEST_F(AesCtrCounterExtendedStorageTest, DestroyNodeData)
{
    typedef nn::fssystem::BucketTree Tree;

    nnt::fs::util::Vector<char> buffer(256);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // ノード部分を壊す
    {
        StorageWrapper storage(false);
        auto* ptr = storage.GetTableStorage().GetBuffer();

        const auto nodeOffset = storage.QueryHeaderSize();
        const auto nodeSize = storage.QueryNodeSize() - sizeof(Tree::NodeHeader);

        for( int i = 0; i < 300; ++i )
        {
            auto offset = static_cast<ptrdiff_t>(nodeOffset +
                nn::util::align_down(
                    std::uniform_int_distribution<int64_t>(
                        sizeof(int64_t), nodeSize - sizeof(int64_t))(mt),
                    sizeof(int64_t)
                ));

            // ヘッダ部分は壊さないようにする
            offset += sizeof(Tree::NodeHeader);

            auto& dst = *nn::util::BytePtr(ptr).Advance(offset).Get<int64_t>();
            dst ^= 0xFFFFFFFFFFFFFFFF;
        }

        // 壊れたデータで初期化
        storage.Initialize();

        int failure = 0;

        for( int count = 0; count < 500; ++count )
        {
            const auto offset =
                nn::util::align_down(
                    std::uniform_int_distribution<int64_t>(BlockSize, buffer.size() - 1)(mt),
                    BlockSize
                );
            const auto size =
                nn::util::align_up(
                    std::uniform_int_distribution<size_t>(
                        1, static_cast<size_t>(buffer.size() - offset))(mt),
                    BlockSize
                );

            // 例外で落ちないことを確認
            nn::Result result = storage.Read(offset, buffer.data(), size);

            if( result.IsSuccess() )
            {
                ASSERT_TRUE(storage.Verify(offset, buffer.data(), size));
            }
            else
            {
                ASSERT_TRUE(nn::fs::ResultAesCtrCounterExtendedStorageCorrupted::Includes(result) ||
                    nn::fs::ResultOutOfRange::Includes(result));

                ++failure;
            }
        }

        NN_LOG("DestroyNode: %d times failure.\n", failure);
    }
}

/**
* @brief   壊れたエントリーデータでエラーが返ってくるかテストします。
*/
TEST_F(AesCtrCounterExtendedStorageTest, DestroyEntryData)
{
    typedef nn::fssystem::BucketTree Tree;

    nnt::fs::util::Vector<char> buffer(256);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // エントリ部分を壊す
    {
        StorageWrapper storage(false);
        auto* ptr = storage.GetTableStorage().GetBuffer();

        const auto entryOffset = storage.QueryHeaderSize() + storage.QueryNodeSize();
        const auto entrySize = storage.QueryEntrySize() - sizeof(Tree::NodeHeader);

        for( int i = 0; i < 20000; ++i )
        {
            auto offset = static_cast<ptrdiff_t>(entryOffset +
                nn::util::align_down(
                    std::uniform_int_distribution<int64_t>(0, entrySize - sizeof(int))(mt),
                    sizeof(int)
                ));

            // ヘッダ部分は壊さないようにする
            offset += sizeof(Tree::NodeHeader);

            auto& dst = *nn::util::BytePtr(ptr).Advance(offset).Get<int>();
            dst ^= 0xFFFFFFFF;
        }

        // 壊れたデータで初期化
        storage.Initialize();

        int failure = 0;
        int verifyDataError = 0;

        for( int count = 0; count < 500; ++count )
        {
            const auto offset =
                nn::util::align_down(
                    std::uniform_int_distribution<int64_t>(BlockSize, buffer.size() - 1)(mt),
                    BlockSize
                );
            const auto size =
                nn::util::align_up(
                    std::uniform_int_distribution<size_t>(
                        1, static_cast<size_t>(buffer.size() - offset))(mt),
                    BlockSize
                );

            // 例外で落ちないことを確認
            nn::Result result = storage.Read(offset, buffer.data(), size);

            if( result.IsSuccess() )
            {
                if( !storage.Verify(offset, buffer.data(), size) )
                {
                    ++verifyDataError;
                }
            }
            else
            {
                ASSERT_TRUE(nn::fs::ResultInvalidArgument::Includes(result) ||
                            nn::fs::ResultAesCtrCounterExtendedStorageCorrupted::Includes(result) ||
                            nn::fs::ResultOutOfRange::Includes(result));

                ++failure;
            }
        }

        NN_LOG("DestroyEntry: %d times failure, %d times data verify error.\n",
            failure, verifyDataError);
    }
}

/**
 * @brief   暗号に利用するカウンタが正しく生成されているかテストします。
 */
TEST_F(AesCtrCounterExtendedStorageTest, CheckEncryptCounter)
{
    static const int ReadSize = 16 * 1024;

    nnt::fs::util::Vector<char> buffer(ReadSize);
    StorageWrapper storage(true);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // 旧テーブルをテストしておく
    for( auto count = 0; count < 100; ++count )
    {
        const auto offset =
            nn::util::align_down(
                std::uniform_int_distribution<int64_t>(0, buffer.size() - 1)(mt),
                BlockSize
            );
        const auto size =
            nn::util::align_up(
                std::uniform_int_distribution<size_t>(
                    1, static_cast<size_t>(buffer.size() - offset))(mt),
                BlockSize
            );

        NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, buffer.data(), size));

        ASSERT_TRUE(storage.Verify(offset, buffer.data(), size));
    }

    StorageWrapper newStorage(storage); // 平文と暗号文は旧テーブルと同じ
    nnt::fs::util::Vector<StorageWrapper::Entry> newEntries;
    {
        const auto* pSrcText = newStorage.GetPlainText();
        const auto* pOldData = reinterpret_cast<char*>(storage.GetDataStorage().GetBuffer());
        auto* pNewData = reinterpret_cast<char*>(newStorage.GetDataStorage().GetBuffer());

        nn::fssystem::BucketTree::Visitor visitor;
        NNT_ASSERT_RESULT_SUCCESS(GetTable(storage.GetImpl()).Find(&visitor, 0));

        auto entry1 = *visitor.Get<TableEntry>();
        while( visitor.MoveNext().IsSuccess() )
        {
            const auto& entry2 = *visitor.Get<TableEntry>();

            const StorageWrapper::Entry newEntry =
            {
                entry1.GetOffset(),
                static_cast<size_t>(entry2.GetOffset() - entry1.GetOffset()),
                0,
                entry1.generation
            };

            // エントリを分割する（単純に半分）
            if( nn::util::is_aligned(newEntry.size, BlockSize * 2) &&
                std::uniform_int_distribution<>(0, 10)(mt) == 0 )
            {
                // 前半のエントリ
                StorageWrapper::Entry entry = newEntry;
                entry.size /= 2;
                newEntries.push_back(entry);

                // 後半のエントリ
                entry.offset += entry.size;
                newEntries.push_back(entry);
            }
            else
            {
                newEntries.push_back(newEntry);
            }

            pSrcText += newEntry.size;
            pOldData += newEntry.size;
            pNewData += newEntry.size;

            entry1 = entry2;
        }

        StorageWrapper::Entry newEntry = { entry1.GetOffset() };
        newEntries.push_back(newEntry);
    }

    newStorage.BuildTable(newEntries);
    newStorage.Initialize();

    // 新テーブルをテスト
    for( auto count = 0; count < 500; ++count )
    {
        const auto offset =
            nn::util::align_down(
                std::uniform_int_distribution<int64_t>(0, buffer.size() - 1)(mt),
                BlockSize
            );
        const auto size =
            nn::util::align_up(
                std::uniform_int_distribution<size_t>(
                    1, static_cast<size_t>(buffer.size() - offset))(mt),
                BlockSize
            );

        NNT_ASSERT_RESULT_SUCCESS(newStorage.Read(offset, buffer.data(), size));

        ASSERT_TRUE(newStorage.Verify(offset, buffer.data(), size));
    }
}

/**
 * @brief   直接作成したテーブルを使用してテストします。
 */
TEST_F(AesCtrCounterExtendedStorageTest, RawData)
{
    typedef nn::fssystem::BucketTree Table;

    struct NodeData
    {
        int32_t index;
        int32_t count;
        int64_t end;
        int64_t begin;
    };
    NN_STATIC_ASSERT(sizeof(NodeData) == (sizeof(Table::NodeHeader) + sizeof(int64_t)));

    static const int EntryCount = 10;

    struct EntryData
    {
        Table::NodeHeader header;

        struct Entry
        {
            int64_t offset;
            int32_t reserved;
            int32_t generation;
        }
        entries[EntryCount];
    };
    NN_STATIC_ASSERT(
        sizeof(EntryData) == (sizeof(Table::NodeHeader) + sizeof(TableEntry) * EntryCount));

    static const size_t DataSize = 32;

    // 平文
    static const char PlaneText[] =
    {
        "GVE64YZB0UBH4BBCXH85LETW20JWQH8Q"
        "X5AVN9A2HX7BEK7READ8133MIH4LF2MB"
        "OIJNMAC4RMI4V27L616ON4LUGMCNOSFE"
        "IC4T8WV6EWV4QVZU2FRDZV0M0QNGXFCE"
        "CW3XL6VACM9SNFNDM338N6M3L37NVXWL"
        "SSJPZLMW84JEV4THC2RND8QAMGWASPEX"
        "ZMJNB31HKGPMAN74TPEY16UHU50HKGXM"
        "G1KWEOHSTIZHPIID8I8MY1JAF9NN7DP8"
        "C439SQYX26E4L4CSPI8T5OKGQ7GCO7N4"
        "Y7UGM20B3RCQ7I7AGOO20BOTI98QXJBY"
    };
    NN_STATIC_ASSERT(sizeof(PlaneText) == DataSize * EntryCount + 1);

    // L1 ノード
    static const NodeData NodeL1Data =
    {
        0,
        1,
        DataSize * EntryCount,
        0
    };
    // エントリデータ
    static const EntryData EntriesData =
    {
        { 0, EntryCount, DataSize * EntryCount },
        {
            { DataSize *  0, 0, 0 },
            { DataSize *  1, 0, 0 },
            { DataSize *  2, 0, 0 },
            { DataSize *  3, 0, 0 },
            { DataSize *  4, 0, 0 },
            { DataSize *  5, 0, 0 },
            { DataSize *  6, 0, 0 },
            { DataSize *  7, 0, 0 },
            { DataSize *  8, 0, 0 },
            { DataSize *  9, 0, 0 },
        }
    };

    static const int TableStorageSize = NodeSize * 2;
    static const int DataStorageSize = sizeof(PlaneText) - 1; // ヌル文字分引く

    static const int64_t counterOffset = 0;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    char key[KeySize];

    // 鍵の生成
    std::generate(std::begin(key), std::end(key), [&]()
    {
        return static_cast<char>(std::uniform_int_distribution<>(
            std::numeric_limits<char>::min(), std::numeric_limits<char>::max())(mt));
    });

    ASSERT_EQ(
        Table::QueryNodeStorageSize(NodeSize, sizeof(EntryData::Entry), EntryCount),
        NodeSize
    );
    ASSERT_EQ(
        Table::QueryEntryStorageSize(NodeSize, sizeof(EntryData::Entry), EntryCount),
        NodeSize
    );

    // テーブルの設定
    nnt::fs::util::SafeMemoryStorage tableStorage(TableStorageSize);
    ASSERT_TRUE(
        nn::util::is_aligned(
            reinterpret_cast<uintptr_t>(tableStorage.GetBuffer()),
            nn::DefaultAlignment
        )
    );
    {
        nn::util::BytePtr buffer(tableStorage.GetBuffer());
        std::fill(buffer.Get<char>(), buffer.Get<char>() + TableStorageSize, '\x0');

        // L1 部分のコピー
        std::memcpy(buffer.Get(), &NodeL1Data, sizeof(NodeL1Data));

        buffer.Advance(NodeSize);

        // エントリのコピー
        std::memcpy(buffer.Get(), &EntriesData, sizeof(EntriesData));
    }

    // データの設定
    nnt::fs::util::SafeMemoryStorage dataStorage(DataStorageSize);
    ASSERT_TRUE(
        nn::util::is_aligned(
            reinterpret_cast<uintptr_t>(dataStorage.GetBuffer()),
            nn::DefaultAlignment
        )
    );
    {
        const auto* pSrc = PlaneText;
        auto* pDst = reinterpret_cast<char*>(dataStorage.GetBuffer());

        // 各データを暗号化
        for( int i = 0; i < EntryCount; ++i )
        {
            const auto& entry = EntriesData.entries[i];

            nn::fssystem::NcaAesCtrUpperIv upperIv;
            upperIv.part.generation = entry.generation;
            upperIv.part.secureValue = 0;

            char counter[CounterSize];
            nn::fssystem::AesCtrStorage::MakeIv(
                counter, CounterSize, upperIv.value, entry.offset);

            ASSERT_EQ(
                DataSize,
                nn::crypto::EncryptAes128Ctr(
                    pDst,
                    DataSize,
                    key,
                    KeySize,
                    counter,
                    CounterSize,
                    pSrc,
                    DataSize
                )
            );

            pSrc += DataSize;
            pDst += DataSize;
        }
    }

    nn::fssystem::AesCtrCounterExtendedStorage storage;

    {
        std::unique_ptr<nn::fssystem::AesCtrCounterExtendedStorage::IDecryptor> pDecryptor;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::AesCtrCounterExtendedStorage::CreateSoftwareDecryptor(&pDecryptor));

        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                key,
                KeySize,
                0,
                counterOffset,
                nn::fs::SubStorage(&dataStorage, 0, DataStorageSize),
                nn::fs::SubStorage(&tableStorage, 0, NodeSize),
                nn::fs::SubStorage(&tableStorage, NodeSize, NodeSize),
                EntryCount,
                std::move(pDecryptor)
                )
            );
    }

    nnt::fs::util::Vector<char> buffer(DataStorageSize);

    // すべて読み込み
    NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, buffer.data(), DataStorageSize));
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(PlaneText, buffer.data(), DataStorageSize);

    // ランダムで読み込み
    for( int loop = 0; loop < 100; ++loop )
    {
        const auto offset =
            nn::util::align_down(
                std::uniform_int_distribution<int64_t>(0, buffer.size() - 1)(mt),
                BlockSize
            );
        const auto size =
            nn::util::align_up(
                std::uniform_int_distribution<size_t>(
                    1, static_cast<size_t>(buffer.size() - offset))(mt),
                BlockSize
            );

        NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, buffer.data(), size));
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(&PlaneText[offset], buffer.data(), size);
    }
} // NOLINT(impl/function_size)

/**
 * @brief   QueryRange をテストします。
 */
TEST_F(AesCtrCounterExtendedStorageTest, QueryRange)
{
    StorageWrapper storage(true);

    int64_t size = 0;
    NNT_ASSERT_RESULT_SUCCESS(storage.GetImpl().GetSize(&size));

    nn::fs::QueryRangeInfo info;

    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultNullptrArgument,
        storage.GetImpl().OperateRange(nullptr, sizeof(info), nn::fs::OperationId::QueryRange, 0, size, nullptr, 0));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize,
        storage.GetImpl().OperateRange(&info, 0, nn::fs::OperationId::QueryRange, 0, size, nullptr, 0));

    NNT_ASSERT_RESULT_SUCCESS(storage.GetImpl().OperateRange(&info, sizeof(info), nn::fs::OperationId::QueryRange, 0, size, nullptr, 0));
    EXPECT_EQ(info.aesCtrKeyTypeFlag, static_cast<int64_t>(nn::fs::AesCtrKeyTypeFlag::InternalKeyForHardwareAes));
}

TEST_F(AesCtrCounterExtendedStorageTest, ReadLargeOffset)
{
    nnt::fs::util::Vector<char> buffer(BlockSize);
    LargeSizeStorageWrapper storage;

    for( int i = 0; i < LargeSizeStorageWrapper::OffsetListLength; ++i )
    {
        const auto offset = LargeSizeStorageWrapper::OffsetList[i];

        NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, buffer.data(), BlockSize));

        ASSERT_TRUE(storage.Verify(offset, buffer.data(), BlockSize));
    }
}

