﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <array>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/nn_BitTypes.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/fs_PartitionFileSystemMeta.h>

using namespace nn;
using namespace nn::fs;
using nn::fssystem::PartitionFileSystemMeta;
using nn::fs::MemoryStorage;


// 与えられたエントリ群で構築し、読めることをテストする
template<typename T>
void ConstructAndVerify(T& entries)
{
    for(auto entry : entries)
    {
        NN_LOG("%s, %lld, %lld\n", entry.name, entry.offset, entry.size);
    }
    NN_LOG("\n");

    // バッファ上に構築
    size_t bufferSize;
    NNT_EXPECT_RESULT_SUCCESS(PartitionFileSystemMeta::QueryMetaDataSize(&bufferSize, entries.data(), static_cast<int>(entries.size())));

    // メタデータは 32 バイトアラインにパディングされる
    EXPECT_EQ(0, bufferSize & 0x1f);

    std::unique_ptr<char> pBuffer(new char[bufferSize + 1]);
    ASSERT_NE(pBuffer, nullptr);

    const char Canary = 'C';
    pBuffer.get()[bufferSize] = Canary;

    NNT_EXPECT_RESULT_SUCCESS(PartitionFileSystemMeta::ConstructMetaData(pBuffer.get(), bufferSize, entries.data(), static_cast<int>(entries.size())));

    // IStorage 化して読み込み
    MemoryStorage storage(pBuffer.get(), bufferSize);

    // ベリファイ
    {
        // メタデータサイズチェック
        size_t bufferSize2;
        NNT_EXPECT_RESULT_SUCCESS(PartitionFileSystemMeta::QueryMetaDataSize(&bufferSize2, &storage));
        EXPECT_EQ(bufferSize, bufferSize2);

        std::unique_ptr<char> pBuffer2(new char[bufferSize2]);
        ASSERT_NE(pBuffer, nullptr);

        // 初期化
        PartitionFileSystemMeta meta;
        NNT_EXPECT_RESULT_SUCCESS(meta.Initialize(&storage, pBuffer2.get(), bufferSize2));

        // エントリ数チェック
        EXPECT_EQ(entries.size(), meta.GetEntryCount());

        // 各エントリ情報チェック
        for(auto entry : entries)
        {
            int index = meta.GetEntryIndex(entry.name);
            EXPECT_GE(index, 0);

            auto pEntry = meta.GetEntry(index);
            EXPECT_NE(nullptr, pEntry);

            EXPECT_EQ(entry.offset, pEntry->offset);
            EXPECT_EQ(entry.size, pEntry->size);
            EXPECT_STREQ(entry.name, meta.GetEntryName(index));
        }
    }

    // バッファをはみ出ていないかチェック
    EXPECT_EQ(Canary, pBuffer.get()[bufferSize]);
}

TEST(PartitionFileSystemMeta, Basic)
{
    {
        std::array<PartitionFileSystemMeta::FileEntryForConstruct, 1> entries =
        {{
            {0x00, 1, "file"},
        }};
        ConstructAndVerify(entries);
    }
    {
        // 無名エントリ
        std::array<PartitionFileSystemMeta::FileEntryForConstruct, 2> entries =
        {{
            {0x20, 2, ""},
            {0x00, 1, "file"},
        }};
        ConstructAndVerify(entries);
    }
    {
        // 順によらず正しく扱えること
        std::array<PartitionFileSystemMeta::FileEntryForConstruct, 2> entries =
        {{
            {0x00, 1, "file"},
            {0x20, 2, ""},
        }};
        ConstructAndVerify(entries);
    }
    {
        // 一方が他方のプレフィクスなファイル名
        // 最大長ファイル名
        std::array<PartitionFileSystemMeta::FileEntryForConstruct, 5> entries =
        {{
            {0x00, 1, ""},
            {0x40, 3, "file_2"},
            {0x20, 2, "file"},
            {0xA0, 5, "(to be filled)"},
            {0x60, 4, "file_3"},
        }};
        memset(entries[3].name, 'A', sizeof(entries[3].name)); // 最大パス長
        entries[3].name[sizeof(entries[3].name) - 1] = '\0';
        ConstructAndVerify(entries);
    }
}

TEST(PartitionFileSystemMeta, InvalidParam)
{
    // 不正なフォーマット
    {
        const size_t bufferSize = 64;
        char buffer[bufferSize];
        memset(buffer, 0x00, bufferSize);

        MemoryStorage storage(buffer, bufferSize);

        size_t size;
        NNT_EXPECT_RESULT_FAILURE(ResultPartitionSignatureVerificationFailed, PartitionFileSystemMeta::QueryMetaDataSize(&size, &storage));

        char buffer2[bufferSize];
        PartitionFileSystemMeta meta;
        NNT_EXPECT_RESULT_FAILURE(ResultPartitionSignatureVerificationFailed, meta.Initialize(&storage, buffer2, bufferSize));
    }

    // 長すぎるパス
    {
        std::array<PartitionFileSystemMeta::FileEntryForConstruct, 2> entries =
        {{
            {0x40, 1, "invalid_file_path"},
            {0x60, 2, "file"},
        }};
        memset(entries[0].name, 'A', sizeof(entries[0].name));
        size_t bufferSize;
        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, PartitionFileSystemMeta::QueryMetaDataSize(&bufferSize, entries.data(), static_cast<int>(entries.size())));
    }
    {
        std::array<PartitionFileSystemMeta::FileEntryForConstruct, 2> entries =
        {{
            {0x40, 1, "invalid_file_path"},
            {0x60, 2, "invalid_file_path2"},
        }};
        memset(entries.data(), 'A', sizeof(entries));
        size_t bufferSize;
        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, PartitionFileSystemMeta::QueryMetaDataSize(&bufferSize, entries.data(), static_cast<int>(entries.size())));
    }

    // バッファ不足
    {
        std::array<PartitionFileSystemMeta::FileEntryForConstruct, 1> entries =
        {{
            {0x40, 1, "file"},
        }};
        size_t bufferSize;
        NNT_EXPECT_RESULT_SUCCESS(PartitionFileSystemMeta::QueryMetaDataSize(&bufferSize, entries.data(), static_cast<int>(entries.size())));

        std::unique_ptr<char> pBuffer(new char[bufferSize]);
        ASSERT_NE(pBuffer, nullptr);

        // ConstructMetaData() のバッファ不足
        {
            NNT_EXPECT_RESULT_FAILURE(ResultInvalidSize, PartitionFileSystemMeta::ConstructMetaData(pBuffer.get(), bufferSize - 1, entries.data(), static_cast<int>(entries.size())));
        }

        // Initialize() のバッファ不足
        {
            NNT_EXPECT_RESULT_SUCCESS(PartitionFileSystemMeta::ConstructMetaData(pBuffer.get(), bufferSize, entries.data(), static_cast<int>(entries.size())));
            MemoryStorage storage(pBuffer.get(), bufferSize);

            size_t bufferSize2;
            NNT_EXPECT_RESULT_SUCCESS(PartitionFileSystemMeta::QueryMetaDataSize(&bufferSize2, &storage));
            std::unique_ptr<char> pBuffer2(new char[bufferSize2]);
            ASSERT_NE(pBuffer, nullptr);

            PartitionFileSystemMeta meta;
            NNT_EXPECT_RESULT_FAILURE(ResultInvalidSize, meta.Initialize(&storage, pBuffer2.get(), bufferSize2 - 1));
        }
    }

}


// 名前テーブル長が変化しても正しくパディングされることのテスト
TEST(PartitionFileSystemMeta, PaddingSize)
{
    std::array<PartitionFileSystemMeta::FileEntryForConstruct, 1> entries =
    {{
        {0x1, 2, ""},
    }};

    for(int i = 0; i < 64; ++i)
    {
        entries[0].name[i] = 'A';
        entries[0].name[i + 1] = '\0';

        ConstructAndVerify(entries);
    }

}

