﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/fs.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

#include <nn/fs/fs_MemoryStorage.h>

#define NN_DBM_CREATE_METADATA

#include <nn/fssystem/fs_DbmHierarchicalRomFileTableTemplate.impl.h>
#include "TestBuffer.h"


#define LONGFILENAME768 \
    "12345678901234567890123456789012345678901234567890" \
    "12345678901234567890123456789012345678901234567890" \
    "12345678901234567890123456789012345678901234567890" \
    "12345678901234567890123456789012345678901234567890" \
    "12345678901234567890123456789012345678901234567890" \
    "12345678901234567890123456789012345678901234567890" \
    "12345678901234567890123456789012345678901234567890" \
    "12345678901234567890123456789012345678901234567890" \
    "12345678901234567890123456789012345678901234567890" \
    "12345678901234567890123456789012345678901234567890" \
    "12345678901234567890123456789012345678901234567890" \
    "12345678901234567890123456789012345678901234567890" \
    "12345678901234567890123456789012345678901234567890" \
    "12345678901234567890123456789012345678901234567890" \
    "12345678901234567890123456789012345678901234567890" \
    "123456789012345678" // 768

namespace nn { namespace fssystem {

//!< ディレクトリーバケット用ストレージ
class SimpleDirectoryBucketRomStorage : public nn::fs::MemoryStorage
{
public:
    explicit SimpleDirectoryBucketRomStorage(void* buffer = NULL, size_t size = 0)
        : MemoryStorage(buffer, size)
    {
    }
};

//!< ディレクトリーエントリー用ストレージ
class SimpleDirectoryEntryRomStorage : public nn::fs::MemoryStorage
{
public:
    explicit SimpleDirectoryEntryRomStorage(void* buffer = NULL, size_t size = 0)
        : MemoryStorage(buffer, size)
    {
    }
};

//!< ファイルバケット用ストレージ
class SimpleFileBucketRomStorage : public nn::fs::MemoryStorage
{
public:
    explicit SimpleFileBucketRomStorage(void* buffer = NULL, size_t size = 0)
        : MemoryStorage(buffer, size)
    {
    }
};

//!< ファイルエントリー用ストレージ
class SimpleFileEntryRomStorage : public nn::fs::MemoryStorage
{
public:
    explicit SimpleFileEntryRomStorage(void* buffer = NULL, size_t size = 0)
        : MemoryStorage(buffer, size)
    {
    }
};

typedef HierarchicalRomFileTableTemplate<
            SimpleDirectoryBucketRomStorage,
            SimpleDirectoryEntryRomStorage,
            SimpleFileBucketRomStorage,
            SimpleFileEntryRomStorage
        >
        SimpleRomFileTable;
const int RomFsRootDirectoryEntrySize = SimpleRomFileTable::QueryDirectoryEntrySize(0);

class HierarchicalRomFileTableTemplateTestSuite : public ::testing::Test
{
public:
    HierarchicalRomFileTableTemplateTestSuite(){}
    virtual ~HierarchicalRomFileTableTemplateTestSuite() {}

private:

    // テストデータ保持用クラス
    class FileTableSetup
    {
    public:
        TestBuffer m_BufDirectoryBucket;
        TestBuffer m_BufDirectoryEntry;
        TestBuffer m_BufFileBucket;
        TestBuffer m_BufFileEntry;
        SimpleDirectoryBucketRomStorage m_StorageDirectoryBucket;
        SimpleDirectoryEntryRomStorage m_StorageDirectoryEntry;
        SimpleFileBucketRomStorage m_StorageFileBucket;
        SimpleFileEntryRomStorage m_StorageFileEntry;
        SimpleRomFileTable m_FileTable;
        size_t m_CountDirectoryBucket;
        size_t m_CountDirectoryEntry;
        size_t m_CountFileBucket;
        size_t m_CountFileEntry;

    public:
        FileTableSetup(
            uint32_t countDirectoryBucket = 1023,
            uint32_t countDirectoryEntry = 50,
            uint32_t countFileBucket = 1023,
            uint32_t countFileEntry = 50,
            uint32_t countDirectoryEntryStorageExtend = 0,
            uint32_t countFileEntryStorageExtend = 0
        )
        : m_BufDirectoryBucket(SimpleRomFileTable::QueryDirectoryEntryBucketStorageSize(countDirectoryBucket)),
          m_BufDirectoryEntry(
              RomFsRootDirectoryEntrySize
              + (countDirectoryEntry + countDirectoryEntryStorageExtend)
                  * SimpleRomFileTable::QueryDirectoryEntrySize(RomPathTool::MAX_PATH_LENGTH)),
          m_BufFileBucket(SimpleRomFileTable::QueryFileEntryBucketStorageSize(countFileBucket)),
          m_BufFileEntry(
              (countFileEntry + countFileEntryStorageExtend)
                  * SimpleRomFileTable::QueryFileEntrySize(RomPathTool::MAX_PATH_LENGTH)),
          m_StorageDirectoryBucket(m_BufDirectoryBucket, m_BufDirectoryBucket.GetBufferSize()),
          m_StorageDirectoryEntry(m_BufDirectoryEntry, m_BufDirectoryEntry.GetBufferSize()),
          m_StorageFileBucket(m_BufFileBucket, m_BufFileBucket.GetBufferSize()),
          m_StorageFileEntry(m_BufFileEntry, m_BufFileEntry.GetBufferSize()),
          m_CountDirectoryBucket(countDirectoryBucket),
          m_CountDirectoryEntry(countDirectoryEntry),
          m_CountFileBucket(countFileBucket),
          m_CountFileEntry(countFileEntry)
        {
            Result result;

            // ファイルテーブル領域をフォーマットします。
            result = SimpleRomFileTable::Format(
                         &m_StorageDirectoryBucket,
                         m_BufDirectoryBucket.GetOffset(),
                         static_cast<uint32_t>(m_BufDirectoryBucket.GetSize()),
                         &m_StorageDirectoryEntry,
                         m_BufDirectoryEntry.GetOffset(),
                         static_cast<uint32_t>(m_BufDirectoryEntry.GetSize()),
                         &m_StorageFileBucket,
                         m_BufFileBucket.GetOffset(),
                         static_cast<uint32_t>(m_BufFileBucket.GetSize()),
                         &m_StorageFileEntry,
                         m_BufFileEntry.GetOffset(),
                         static_cast<uint32_t>(m_BufFileEntry.GetSize())
                    );
            NNT_EXPECT_RESULT_SUCCESS(result);

            // ファイルテーブルをマウントします。
            result = m_FileTable.Initialize(
                         &m_StorageDirectoryBucket,
                         m_BufDirectoryBucket.GetOffset(),
                         static_cast<uint32_t>(m_BufDirectoryBucket.GetSize()),
                         &m_StorageDirectoryEntry,
                         m_BufDirectoryEntry.GetOffset(),
                         static_cast<uint32_t>(m_BufDirectoryEntry.GetSize()),
                         &m_StorageFileBucket,
                         m_BufFileBucket.GetOffset(),
                         static_cast<uint32_t>(m_BufFileBucket.GetSize()),
                         &m_StorageFileEntry,
                         m_BufFileEntry.GetOffset(),
                         static_cast<uint32_t>(m_BufFileEntry.GetSize())
                     );
            NNT_EXPECT_RESULT_SUCCESS(result);
        }

        void AssertValid()
        {
            m_BufDirectoryBucket.AssertValid();
            m_BufDirectoryEntry.AssertValid();
            m_BufFileBucket.AssertValid();
            m_BufFileEntry.AssertValid();
        }
    };

protected:
    void HierarchicalRomFileTableTemplateTest1();
    void HierarchicalRomFileTableTemplateTest2();
    void HierarchicalRomFileTableTemplateTest3();
    void HierarchicalRomFileTableTemplateTest4();
};

//! プライベートクラス SimpleRomFileTable::DirectoryEntryMapTable のテスト
void HierarchicalRomFileTableTemplateTestSuite::HierarchicalRomFileTableTemplateTest1()
{
    uint32_t countDirectoryBucket = 1;
    uint32_t countDirectoryEntry = 1;

    // ストレージを生成します。
    TestBuffer bufDirectoryBucket(SimpleRomFileTable::QueryDirectoryEntryBucketStorageSize(countDirectoryBucket));
    TestBuffer bufDirectoryEntry(RomFsRootDirectoryEntrySize + countDirectoryEntry * SimpleRomFileTable::QueryDirectoryEntrySize(RomPathTool::MAX_PATH_LENGTH));
    SimpleDirectoryBucketRomStorage storageDirectoryBucket(bufDirectoryBucket, bufDirectoryBucket.GetBufferSize());
    SimpleDirectoryEntryRomStorage storageDirectoryEntry(bufDirectoryEntry, bufDirectoryEntry.GetBufferSize());

    SimpleRomFileTable::DirectoryEntryMapTable dir;
    NNT_EXPECT_RESULT_SUCCESS(
        dir.Format(
            &storageDirectoryBucket,
            bufDirectoryBucket.GetOffset(),
            countDirectoryBucket,
            &storageDirectoryEntry,
            bufDirectoryEntry.GetOffset(),
            static_cast<uint32_t>(bufDirectoryEntry.GetSize())
        )
     );
    NNT_EXPECT_RESULT_SUCCESS(
        dir.Initialize(
            &storageDirectoryBucket,
            bufDirectoryBucket.GetOffset(),
            countDirectoryBucket,
            &storageDirectoryEntry,
            bufDirectoryEntry.GetOffset(),
            static_cast<uint32_t>(bufDirectoryEntry.GetSize())
        )
     );

    static const SimpleRomFileTable::DirectoryEntryMapTable::Key2 e1 = {1, { 8, "00000001" } };
    static const SimpleRomFileTable::DirectoryEntryMapTable::Key2 e2 = {1, { 8, "00000002" } };
    static const SimpleRomFileTable::DirectoryEntryMapTable::Value v1 = {};
    static const SimpleRomFileTable::DirectoryEntryMapTable::Value v2 = {};

    SimpleRomFileTable::DirectoryEntryMapTable::Position index, index1, index2;
    SimpleRomFileTable::DirectoryEntryMapTable::Key e;
    SimpleRomFileTable::DirectoryEntryMapTable::Value v;

    NNT_EXPECT_RESULT_SUCCESS(dir.Add(&index1, e1, v1));
    NNT_EXPECT_RESULT_SUCCESS(dir.Add(&index2, e2, v2));

    // 登録した値を取得します。
    NNT_EXPECT_RESULT_SUCCESS(dir.Get(&index, &v, e1));
    EXPECT_EQ(index, index1);
    EXPECT_EQ(0, std::memcmp(&v, &v1, sizeof(v)));
    NNT_EXPECT_RESULT_SUCCESS(dir.Get(&index, &v, e2));
    EXPECT_EQ(index, index2);
    EXPECT_EQ(0, std::memcmp(&v, &v2, sizeof(v)));

    // インデックスで引きます。
    RomPathChar buf[RomPathTool::MAX_PATH_LENGTH];
    size_t size;
    NNT_EXPECT_RESULT_SUCCESS(dir.GetByPosition(&e, &v, &buf, &size, index1));
    EXPECT_EQ(e.parentDir, e1.key.parentDir);
    EXPECT_EQ(0, std::memcmp(&v, &v1, sizeof(v)));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "00000001", size / sizeof(RomPathChar)));
    NNT_EXPECT_RESULT_SUCCESS(dir.GetByPosition(&e, &v, &buf, &size, index2));
    EXPECT_EQ(e.parentDir, e2.key.parentDir);
    EXPECT_EQ(0, std::memcmp(&v, &v2, sizeof(v)));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "00000002", size / sizeof(RomPathChar)));
}

void HierarchicalRomFileTableTemplateTestSuite::HierarchicalRomFileTableTemplateTest2()
{
    Result result;

    // 関数のエラーチェック
    FileTableSetup fts(3, 4, 3, 6);
    SimpleRomFileTable &fileTable = fts.m_FileTable;

    RomDirectoryId dirID;
    RomFileId fileID;

    // 何もない状態でファイルを列挙します。
    SimpleRomFileTable::FindPosition fi = {0xCC};
    //RomPathTool::RomEntryName dirName;
    //RomPathTool::RomEntryName fileName;
    SimpleRomFileTable::DirectoryInfo dirInfo;
    SimpleRomFileTable::FileInfo fileInfo;
    RomPathChar buf[RomPathTool::MAX_PATH_LENGTH + 1];

    // 存在しないディレクトリを指定します。
    NNT_EXPECT_RESULT_FAILURE(ResultDbmDirectoryNotFound, fileTable.FindOpen(&fi, "/dir"));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmDirectoryNotFound, fileTable.FindOpen(&fi, "/dir1/"));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmDirectoryNotFound, fileTable.FindOpen(&fi, "/dir1/dir2"));

    // ルートディレクトリを生成します。
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateRootDirectory());

    // ルートディレクトリはデフォルトで生成されている必要があります。
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/"));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmFindFinished, fileTable.FindNextDirectory(buf, &fi));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmFindFinished, fileTable.FindNextFile(buf, &fi));

    // ディレクトリの再帰生成はサポートされません。
    NNT_EXPECT_RESULT_FAILURE(ResultDbmDirectoryNotFound, fileTable.CreateDirectory(&dirID, "/dir1/dir2", dirInfo));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmDirectoryNotFound, fileTable.CreateDirectory(&dirID, "/dir1/dir2/", dirInfo));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmDirectoryNotFound, fileTable.CreateDirectory(&dirID, "/dir1/dir2/dir3", dirInfo));

    // 先頭にスラッシュなしの指定は失敗します。
    NNT_EXPECT_RESULT_FAILURE(ResultDbmInvalidPathFormat, fileTable.CreateDirectory(&dirID, "dir1", dirInfo));

    // 連続した区切りを許可します。
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "//"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "///"));

    // ルートディレクトリはすでに存在しています。
    NNT_EXPECT_RESULT_FAILURE(ResultDbmAlreadyExists, fileTable.CreateDirectory(&dirID, "/", dirInfo));

    // ファイル操作に対してディレクトリが与えられました。
    NNT_EXPECT_RESULT_FAILURE(ResultDbmInvalidOperation, fileTable.CreateFile(&fileID, "/", fileInfo));

    // ディレクトリを作成します。
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateDirectory(&dirID, "/dir1", dirInfo));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateDirectory(&dirID, "/dir1/dir2", dirInfo));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateDirectory(&dirID, "/dir1/dir2/dir3", dirInfo));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateDirectory(&dirID, "/dir1/dir2/dir3/dir4dir4", dirInfo));

    nnt::fs::util::String longlongPath = "/";
    for (int i = 0; i < 1024; i++)
    {
        longlongPath += "0123456789"[i % 10];

        if (i < RomPathTool::MAX_PATH_LENGTH)
        {
            // ディレクトリ名が長すぎます。
            NNT_EXPECT_RESULT_FAILURE(ResultDbmDirectoryNotFound, fileTable.FindOpen(&fi, longlongPath.c_str()));
        }
        else
        {
            // ディレクトリ名が長すぎます。
            NNT_EXPECT_RESULT_FAILURE(ResultDbmDirectoryNameTooLong, fileTable.FindOpen(&fi, longlongPath.c_str()));
        }
    }

    // ファイル名が長すぎます。
    NNT_EXPECT_RESULT_FAILURE(ResultDbmFileNameTooLong, fileTable.CreateFile(&fileID,
        "/"
        LONGFILENAME768
        "_"
        , fileInfo));

    // ディレクトリを列挙します。
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir1", 4));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmFindFinished, fileTable.FindNextDirectory(buf, &fi));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmFindFinished, fileTable.FindNextFile(buf, &fi));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/dir1"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir2", 4));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmFindFinished, fileTable.FindNextDirectory(buf, &fi));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmFindFinished, fileTable.FindNextFile(buf, &fi));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/dir1/dir2/"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir3", 4));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmFindFinished, fileTable.FindNextDirectory(buf, &fi));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmFindFinished, fileTable.FindNextFile(buf, &fi));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/dir1/dir2/dir3/"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir4dir4", 8));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmFindFinished, fileTable.FindNextDirectory(buf, &fi));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmFindFinished, fileTable.FindNextFile(buf, &fi));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/dir1/dir2/dir3/dir4dir4"));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmFindFinished, fileTable.FindNextDirectory(buf, &fi));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmFindFinished, fileTable.FindNextFile(buf, &fi));

    // ファイルを作成します。
    // スラッシュなしの指定は失敗します。
    NNT_EXPECT_RESULT_FAILURE(ResultDbmInvalidPathFormat, fileTable.CreateFile(&fileID, "file1", fileInfo));
    // ディレクトリと同名のファイルは作れません。
    NNT_EXPECT_RESULT_FAILURE(ResultDbmAlreadyExists, fileTable.CreateFile(&fileID, "/dir1", fileInfo));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateFile(&fileID, "/dir1/dir2/dir3/dir4dir4/file1", fileInfo));
    // ファイル以下にディレクトリは作れません。
    NNT_EXPECT_RESULT_FAILURE(ResultDbmInvalidOperation, fileTable.CreateDirectory(&dirID, "/dir1/dir2/dir3/dir4dir4/file1/dir5", dirInfo));
    // ファイル以下にファイルは作れません。
    NNT_EXPECT_RESULT_FAILURE(ResultDbmInvalidOperation, fileTable.CreateFile(&fileID, "/dir1/dir2/dir3/dir4dir4/file1/file1", fileInfo));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateFile(&fileID, "/dir1/dir2/dir3/file1", fileInfo));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateFile(&fileID, "/dir1/dir2/file1", fileInfo));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateFile(&fileID, "/dir1/file1", fileInfo));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateFile(&fileID, "/file1", fileInfo));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateFile(&fileID, "/file2file2fi", fileInfo));
    // 既に存在するファイルは作れません。
    NNT_EXPECT_RESULT_FAILURE(ResultDbmAlreadyExists, fileTable.CreateFile(&fileID, "/file2file2fi", fileInfo));

    // 最大ファイル名長
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateFile(&fileID, "/" LONGFILENAME768, fileInfo));

    // ダンプします。
    NNT_EXPECT_RESULT_SUCCESS(fileTable.DumpTree());

    // ファイルをオープンします。
    // スラッシュなしの指定は失敗します。
    NNT_EXPECT_RESULT_FAILURE(ResultDbmInvalidPathFormat, fileTable.OpenFile(&fileInfo, "file1"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/dir1/dir2/dir3/dir4dir4/file1"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/dir1/dir2/dir3/file1"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/dir1/dir2/file1"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/dir1/file1"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/file1"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/file2file2fi"));

    // 追加テスト "." の扱いテスト
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateDirectory(&dirID, "/dir1/dir2/dir3/dir4", dirInfo));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/."));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir1", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/./"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir1", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/./."));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir1", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/././"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir1", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/././."));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir1", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/././dir1"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir2", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/././dir1/"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir2", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/././dir1/."));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir2", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/././dir1/./"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir2", 4));

    NNT_EXPECT_RESULT_FAILURE(ResultDbmAlreadyExists, fileTable.CreateDirectory(&dirID, "/.", dirInfo));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmAlreadyExists, fileTable.CreateDirectory(&dirID, "/dir1/dir2/.", dirInfo));

    // 追加テスト ".." の扱いテスト
    NNT_EXPECT_RESULT_FAILURE(ResultDbmInvalidOperation, fileTable.FindOpen(&fi, "/.."));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmInvalidOperation, fileTable.FindOpen(&fi, "/dir1/../.."));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmInvalidOperation, fileTable.FindOpen(&fi, "/dir1/../../.."));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmInvalidOperation, fileTable.FindOpen(&fi, "/dir1/dir2/dir3/dir4/../../../../.."));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmInvalidOperation, fileTable.FindOpen(&fi, "/dir1/dir2/dir3/../dir3/../../dir2/../../.."));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/dir1/.."));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir1", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/./dir1/../."));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir1", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/dir1/dir2/.."));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir2", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/dir1/dir2/../."));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir2", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "//dir1/../dir1/./dir2/.././//"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir2", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/dir1/dir2/dir3/dir4/../../../../"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir1", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/dir1/dir2/dir3/dir4/../../../"));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir2", 4));

    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindOpen(&fi, "/dir1/dir2/dir3/dir4/../../dir3/."));
    NNT_EXPECT_RESULT_SUCCESS(fileTable.FindNextDirectory(buf, &fi));
    EXPECT_EQ(true, RomPathTool::IsEqualPath(buf, "dir4", 4));

    NNT_EXPECT_RESULT_FAILURE(ResultDbmAlreadyExists, fileTable.CreateDirectory(&dirID, "/dir1/../dir1/dir2/dir3/../.", dirInfo));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmAlreadyExists, fileTable.CreateDirectory(&dirID, "/dir1/dir2/..", dirInfo));
    NNT_EXPECT_RESULT_FAILURE(ResultDbmAlreadyExists, fileTable.CreateDirectory(&dirID, "/dir1/dir2/../..", dirInfo));
} // NOLINT(readability/fn_size)

void HierarchicalRomFileTableTemplateTestSuite::HierarchicalRomFileTableTemplateTest3()
{
    Result result;

    // サイズ0を指定します。
    FileTableSetup fts(1, 0, 1, 0);
    SimpleRomFileTable &fileTable = fts.m_FileTable;

    RomFileId fileID;

    // ルートディレクトリを生成します。
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateRootDirectory());

    // ファイルは作れません。
    SimpleRomFileTable::FileInfo fi;
    NNT_EXPECT_RESULT_FAILURE(ResultDbmFileEntryFull, fileTable.CreateFile(&fileID, "/file", fi));

    // ルートディレクトリだけが存在するファイルツリーの内容をチェックします。
    uint32_t sizeDir, sizeFile;
    NNT_EXPECT_RESULT_SUCCESS(fileTable.QueryRomFileSystemSize(&sizeDir, &sizeFile));
    EXPECT_EQ(0, sizeFile);
}

void HierarchicalRomFileTableTemplateTestSuite::HierarchicalRomFileTableTemplateTest4()
{
    Result result;

    static const uint32_t DIRECTORY_COUNT = 24;
    static const uint32_t FILE_COUNT = 32;

    // 非常に長いパス名を指定します。
    FileTableSetup fts(DIRECTORY_COUNT, DIRECTORY_COUNT, FILE_COUNT, FILE_COUNT);
    SimpleRomFileTable &fileTable = fts.m_FileTable;
    SimpleRomFileTable::FileInfo fi;
    SimpleRomFileTable::DirectoryInfo di;
    RomFileId fileID;
    RomDirectoryId dirID;

    static int seed = 1;
    std::mt19937 mt(seed++);
    std::uniform_int_distribution<uint32_t> random(0, 0xFFFFFFFF);

    nnt::fs::util::String dirFullDirName = "/";

    // ルートディレクトリを生成します。
    NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateRootDirectory());

    for (int i = 0; i < DIRECTORY_COUNT; i++)
    {
        char dirEntry[RomPathTool::MAX_PATH_LENGTH + 1];
        std::sprintf(dirEntry, "%08X/", random(mt));
        dirFullDirName += dirEntry;
        std::memset(&di, i & 0xFF, sizeof(di));
        NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateDirectory(&dirID, dirFullDirName.c_str(), di));
    }

    for (int i = 0; i < FILE_COUNT; i++)
    {
        char fileEntry[RomPathTool::MAX_PATH_LENGTH + 1];
        std::sprintf(fileEntry, "%08X%04X", i, i);
        nnt::fs::util::String dirFullPath = dirFullDirName;
        dirFullPath += fileEntry;
        std::memset(&fi, i & 0xFF, sizeof(fi));
        NNT_EXPECT_RESULT_SUCCESS(fileTable.CreateFile(&fileID, dirFullPath.c_str(), fi));
    }
}



TEST_F(HierarchicalRomFileTableTemplateTestSuite, 1)
{
    HierarchicalRomFileTableTemplateTest1();
}
TEST_F(HierarchicalRomFileTableTemplateTestSuite, 2)
{
    HierarchicalRomFileTableTemplateTest2();
}
TEST_F(HierarchicalRomFileTableTemplateTestSuite, 3)
{
    HierarchicalRomFileTableTemplateTest3();
}
TEST_F(HierarchicalRomFileTableTemplateTestSuite, 4)
{
    HierarchicalRomFileTableTemplateTest4();
}


}}
