﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/os.h>
#include <nn/os/os_MultipleWaitApi.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fssrv/fssrv_DeferredProcessManager.h>
#include <nn/fssrv/fssrv_FileSystemProxyServerSessionResourceManager.h>
#include <nn/fssrv/fssrv_RequestHook.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_ServiceContext.h>
#include <nn/sf/sf_Result.h>
#include <nn/sf/sf_ContextControl.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

namespace {
    std::atomic_int g_ObjectCount;

    class SessionCounter
    {
    public:
        SessionCounter() NN_NOEXCEPT
            : m_MutexCount(false),
              m_RealtimeCount(0),
              m_RealtimeCountMax(0),
              m_OtherCount(0),
              m_OtherCountMax(0),
              m_BackgroundCount(0),
              m_BackgroundCountMax(0)
        {
        }

        void Reset() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(m_MutexCount);
            m_RealtimeCount = 0;
            m_RealtimeCountMax = 0;
            m_OtherCount = 0;
            m_OtherCountMax = 0;
            m_BackgroundCount = 0;
            m_BackgroundCountMax = 0;
        }

        void Increment() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(m_MutexCount);
            int* pCount;
            int* pMax;
            switch( nn::fssystem::GetServiceContext()->GetPriority() )
            {
            case nn::fs::PriorityRaw_Realtime:
                pCount = &m_RealtimeCount;
                pMax = &m_RealtimeCountMax;
                break;
            case nn::fs::PriorityRaw_Normal:
            case nn::fs::PriorityRaw_Low:
                pCount = &m_OtherCount;
                pMax = &m_OtherCountMax;
                break;
            case nn::fs::PriorityRaw_Background:
                pCount = &m_BackgroundCount;
                pMax = &m_BackgroundCountMax;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }

            ++(*pCount);
            if( *pCount > *pMax )
            {
                *pMax = *pCount;
            }
        }

        void Decrement() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(m_MutexCount);
            switch( nn::fssystem::GetServiceContext()->GetPriority() )
            {
            case nn::fs::PriorityRaw_Realtime:
                --m_RealtimeCount;
                break;
            case nn::fs::PriorityRaw_Normal:
            case nn::fs::PriorityRaw_Low:
                --m_OtherCount;
                break;
            case nn::fs::PriorityRaw_Background:
                --m_BackgroundCount;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }

        void Check() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(m_MutexCount);
            EXPECT_LE(m_RealtimeCountMax, nn::fssrv::FileSystemProxyServerSessionResourceManager::RealtimeSessionCount);
            EXPECT_LE(m_OtherCountMax, nn::fssrv::FileSystemProxyServerSessionResourceManager::OtherSessionCount
                + nn::fssrv::FileSystemProxyServerSessionResourceManager::BackgroundSessionCount);
            EXPECT_LE(m_BackgroundCountMax, nn::fssrv::FileSystemProxyServerSessionResourceManager::BackgroundSessionCount);
            DumpMax();
        }

        void CheckWhenBackgroundDeferred() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(m_MutexCount);
            EXPECT_LE(m_RealtimeCountMax, nn::fssrv::FileSystemProxyServerSessionResourceManager::RealtimeSessionCount);
            EXPECT_LE(m_OtherCountMax, nn::fssrv::FileSystemProxyServerSessionResourceManager::OtherSessionCount);
            EXPECT_LE(m_BackgroundCountMax, nn::fssrv::FileSystemProxyServerSessionResourceManager::BackgroundSessionCount);
            DumpMax();
        }

        bool IsEmpty() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(m_MutexCount);
            return m_RealtimeCount == 0
                && m_OtherCount == 0
                && m_BackgroundCount == 0;
        }

    private:
        void DumpMax() NN_NOEXCEPT
        {
            NN_LOG("%d %d %d\n", m_RealtimeCountMax, m_OtherCountMax, m_BackgroundCountMax);
        }

    private:
        nn::os::Mutex m_MutexCount;
        int m_RealtimeCount;
        int m_RealtimeCountMax;
        int m_OtherCount;
        int m_OtherCountMax;
        int m_BackgroundCount;
        int m_BackgroundCountMax;
    };

    SessionCounter g_SessionCounter;

    class Object : public nn::os::MultiWaitHolderType, public nn::fs::detail::Newable
    {
    public:
        static const uintptr_t Tag = 64;
        static const nn::Bit64 TestProcessId = 1;

    public:
        enum class Action
        {
            Sleep,
            Defer,
            Wait,
            Signal,
            InvokeDeferred
        };

    public:
        void Initialize(nn::fs::PriorityRaw priority, Action action) NN_NOEXCEPT
        {
            nn::os::InitializeEvent(&m_Event, true, nn::os::EventClearMode::EventClearMode_AutoClear);
            nn::os::InitializeMultiWaitHolder(this, &m_Event);
            nn::os::SetMultiWaitHolderUserData(this, Tag);

            m_Priority = priority;
            m_Action = action;
        }

        nn::fs::PriorityRaw GetPriority() const NN_NOEXCEPT
        {
            return m_Priority;
        }

        nn::Result Invoke() NN_NOEXCEPT
        {
            g_SessionCounter.Increment();
            NN_UTIL_SCOPE_EXIT
            {
                g_SessionCounter.Decrement();
            };

            switch( m_Action )
            {
            case Action::Sleep:
            {
                if( !g_IsStop )
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
                }

                NN_RESULT_SUCCESS;
            }

            case Action::Defer:
                nn::fssystem::GetServiceContext()->SetDeferredProcessProcessId(TestProcessId);
                NN_RESULT_THROW(nn::sf::DeferProcess());

            case Action::InvokeDeferred:
                nn::fssystem::GetServiceContext()->RequestToInvokeDeferred(TestProcessId);
                NN_RESULT_SUCCESS;

            case Action::Wait:
                g_Event.Wait();
                g_Event.Clear();
                NN_RESULT_SUCCESS;

            case Action::Signal:
                g_Event.Signal();
                NN_RESULT_SUCCESS;

            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }

        void SetAction(Action action) NN_NOEXCEPT
        {
            m_Action = action;
        }

        static void Unstop() NN_NOEXCEPT
        {
            g_IsStop = false;
        }

        static void Stop() NN_NOEXCEPT
        {
            g_IsStop = true;
        }

    private:
        nn::os::EventType m_Event;
        nn::fs::PriorityRaw m_Priority;
        Action m_Action;
        static bool g_IsStop;
        static nn::os::LightEvent g_Event;
    };

    NN_DEFINE_STATIC_CONSTANT(const nn::Bit64 Object::TestProcessId);

    bool Object::g_IsStop = false;
    nn::os::LightEvent Object::g_Event(nn::os::EventClearMode::EventClearMode_ManualClear);

    size_t GetDeferredQueueCount() NN_NOEXCEPT;

    class FileSystemProxyServerMock
    {
    public:
        FileSystemProxyServerMock() NN_NOEXCEPT
            : m_MutexDeferred(false),
              m_MutexSelect(false),
              m_DeferredQueueCountMax(0),
              m_pRequestHook(nullptr)
        {
            nn::os::InitializeMultiWait(&m_MultiWait);

            nn::os::InitializeEvent(&m_EventStop, false, nn::os::EventClearMode::EventClearMode_ManualClear);
            nn::os::InitializeMultiWaitHolder(&m_EventHolderStop, &m_EventStop);
            nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderStop);

            nn::os::InitializeEvent(&m_EventNotify, false, nn::os::EventClearMode::EventClearMode_ManualClear);
            nn::os::InitializeMultiWaitHolder(&m_EventHolderNotify, &m_EventNotify);
            nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderNotify);

            nn::os::InitializeMultiWait(&m_Deferred);
        }

    public:
        nn::os::MultiWaitHolderType* Wait() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(m_MutexSelect);

            for( ;; )
            {
                {
                    std::lock_guard<nn::os::Mutex> lockDeferred(m_MutexDeferred);
                    nn::os::MoveAllMultiWaitHolder(&m_MultiWait, &m_Deferred);
                }

                auto pObject = nn::os::WaitAny(&m_MultiWait);
                if( pObject == &m_EventHolderStop )
                {
                    return nullptr;
                }
                else if( pObject == &m_EventHolderNotify )
                {
                    nn::os::ClearEvent(&m_EventNotify);
                }
                else
                {
                    {
                        auto count = GetDeferredQueueCount();
                        if( count > m_DeferredQueueCountMax )
                        {
                            m_DeferredQueueCountMax = count;
                        }
                    }

                    nn::os::UnlinkMultiWaitHolder(pObject);
                    return pObject;
                }
            }
        }

        nn::Result ProcessInvokeRequestWithDefer(nn::os::MultiWaitHolderType* request) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(nn::os::GetMultiWaitHolderUserData(request) == Object::Tag);
            auto pObject = reinterpret_cast<Object*>(request);
            auto priority = pObject->GetPriority();
            nn::fs::SetPriorityRawOnCurrentThread(priority);

            NN_UTIL_SCOPE_EXIT
            {
                m_pRequestHook->AfterInvoke();
            };
            NN_RESULT_DO(m_pRequestHook->BeforeInvoke());

            NN_RESULT_DO(pObject->Invoke());

            // テストが new したオブジェクトを破棄する
            // テストは g_ObjectCount が 0 になるまでテストを終了しないことにより、メモリリークしないことを保証する
            // BeforeInvoke/Invoke は defer する Result を返すだけなので、いつか必ずここに到達する
            delete pObject;
            --g_ObjectCount;

            NN_RESULT_SUCCESS;
        }

        void AddUserWaitHolder(nn::os::MultiWaitHolderType* request) NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(m_MutexDeferred);
            nn::os::LinkMultiWaitHolder(&m_Deferred, request);
            nn::os::SignalEvent(&m_EventNotify);
        }

        void SetManagerHandler(nn::sf::IHipcServerSessionManagerHandler* pManagerHandler) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pManagerHandler);
            m_pRequestHook = pManagerHandler;
        }

        void Start() NN_NOEXCEPT
        {
            m_DeferredQueueCountMax = 0;
            nn::os::ClearEvent(&m_EventStop);
        }

        void Stop() NN_NOEXCEPT
        {
            nn::os::SignalEvent(&m_EventStop);
        }

        size_t GetDeferredQueueCountMax() const NN_NOEXCEPT
        {
            return m_DeferredQueueCountMax;
        }

    private:
        nn::os::Mutex m_MutexDeferred;
        nn::os::Mutex m_MutexSelect;
        nn::os::MultiWaitType m_MultiWait;
        nn::os::MultiWaitType m_Deferred;
        nn::os::EventType m_EventStop;
        nn::os::MultiWaitHolderType m_EventHolderStop;
        nn::os::EventType m_EventNotify;
        nn::os::MultiWaitHolderType m_EventHolderNotify;
        size_t m_DeferredQueueCountMax;
        nn::sf::IHipcServerSessionManagerHandler* m_pRequestHook;
    };

    FileSystemProxyServerMock g_FileSystemProxyServerManager;

    void NotifyProcessDeferred(nn::Bit64 processId) NN_NOEXCEPT
    {
        ASSERT_EQ(Object::TestProcessId, processId);
    }

    nn::fssrv::DeferredProcessManager<FileSystemProxyServerMock, NotifyProcessDeferred> g_DeferredProcessManager;
    nn::fssrv::FileSystemProxyServerSessionResourceManager g_SessionResourceManager(g_DeferredProcessManager.GetDeferredProcessQueueForPriority());
    nn::fssrv::RequestHook g_RequestHook(&g_SessionResourceManager);

    size_t GetDeferredQueueCount() NN_NOEXCEPT
    {
        return g_DeferredProcessManager.GetDeferredProcessQueueForPriority()->Count();
    }

    class ConsumerThread
    {
        NN_DISALLOW_COPY(ConsumerThread);
        NN_DISALLOW_MOVE(ConsumerThread);

    public:
        ConsumerThread() NN_NOEXCEPT
            : m_IsInitialized(false)
        {
        }

    public:
        nn::Result Initialize() NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_ALIGNED(m_StackBuffer, nn::os::ThreadStackAlignment);
            NN_ABORT_UNLESS(!m_IsInitialized);
            NN_RESULT_DO(nn::os::CreateThread(
                &m_Thread,
                OnThread,
                this,
                m_StackBuffer,
                StackSize,
                NN_SYSTEM_THREAD_PRIORITY(fs, WorkerRealTimeAccess)
            ));
            m_IsFailed = false;
            m_IsInitialized = true;
            NN_RESULT_SUCCESS;
        }

        void Start() NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(m_IsInitialized);
            nn::os::StartThread(&m_Thread);
        }

        void Wait() NN_NOEXCEPT
        {
            if( m_IsInitialized )
            {
                nn::os::WaitThread(&m_Thread);
            }
        }

        void Finalize() NN_NOEXCEPT
        {
            if( m_IsInitialized )
            {
                nn::os::DestroyThread(&m_Thread);
                m_IsInitialized = false;
            }
        }

        bool IsFailed() const NN_NOEXCEPT
        {
            return m_IsFailed;
        }

    private:
        static const size_t StackSize = 16 * 1024;

    private:
        void OnThread() NN_NOEXCEPT
        {
            nn::fssystem::RegisterServiceContext(&m_ServiceContext);

            nn::fssrv::RequestHookContext requestHookContext;
            {
                auto* pContext = nn::fssystem::GetServiceContext();
                if( pContext != nullptr )
                {
                    pContext->SetRequestHookContext(&requestHookContext);
                }
            }

            for( ; ; )
            {
                auto pObject = g_FileSystemProxyServerManager.Wait();
                if( pObject == nullptr )
                {
                    break;
                }

                auto tag = nn::os::GetMultiWaitHolderUserData(pObject);
                if( tag == Object::Tag )
                {
                    auto result = g_FileSystemProxyServerManager.ProcessInvokeRequestWithDefer(pObject);
                    if( nn::sf::ResultProcessDeferred::Includes(result) )
                    {
                        // 遅延されたオブジェクトをキューイング
                        g_DeferredProcessManager.QueueDeferredProcess(&g_FileSystemProxyServerManager, pObject);
                    }
                    else if( result.IsFailure() )
                    {
                        m_IsFailed = true;
                    }
                }
                else if( tag == g_DeferredProcessManager.InvokeTag )
                {
                    g_DeferredProcessManager.ClearInvokeDeferredProcessEvent();
                }

                // 遅延されたオブジェクトの解除要求が来ていたら同じスレッドで引き続き処理する
                g_DeferredProcessManager.InvokeDeferredProcess(&g_FileSystemProxyServerManager, &g_SessionResourceManager);
            }
        }

        static void OnThread(void* pArgument) NN_NOEXCEPT
        {
            reinterpret_cast<ConsumerThread*>(pArgument)->OnThread();
        }

    private:
        bool m_IsInitialized;
        nn::os::ThreadType m_Thread;
        char NN_OS_ALIGNAS_THREAD_STACK m_StackBuffer[StackSize];
        nn::fssystem::ServiceContext m_ServiceContext;
        bool m_IsFailed;
    };

    class ProducerThread
    {
        NN_DISALLOW_COPY(ProducerThread);
        NN_DISALLOW_MOVE(ProducerThread);

    public:
        ProducerThread() NN_NOEXCEPT
            : m_RandomNumberGenerator(nnt::fs::util::GetRandomSeed()),
              m_IsInitialized(false)
        {
        }

    public:
        nn::Result Initialize() NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_ALIGNED(m_StackBuffer, nn::os::ThreadStackAlignment);
            NN_ABORT_UNLESS(!m_IsInitialized);
            NN_RESULT_DO(nn::os::CreateThread(
                &m_Thread,
                OnThread,
                this,
                m_StackBuffer,
                StackSize,
                NN_SYSTEM_THREAD_PRIORITY(fs, WorkerRealTimeAccess)
            ));
            m_IsInitialized = true;
            NN_RESULT_SUCCESS;
        }

        void Start() NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(m_IsInitialized);
            nn::os::StartThread(&m_Thread);
        }

        void Wait() NN_NOEXCEPT
        {
            if( m_IsInitialized )
            {
                nn::os::WaitThread(&m_Thread);
            }
        }

        void Finalize() NN_NOEXCEPT
        {
            if( m_IsInitialized )
            {
                nn::os::DestroyThread(&m_Thread);
                m_IsInitialized = false;
            }
        }

        static void Unstop() NN_NOEXCEPT
        {
            g_IsStop = false;
        }

        static void Stop() NN_NOEXCEPT
        {
            g_IsStop = true;
        }

    private:
        static const size_t StackSize = 16 * 1024;

    private:
        void OnThread() NN_NOEXCEPT
        {
            while( !g_IsStop )
            {
                auto priority = GeneratePriority();
                Object* pObject = new Object();
                pObject->Initialize(priority, Object::Action::Sleep);
                ++g_ObjectCount;

                g_FileSystemProxyServerManager.AddUserWaitHolder(pObject);

                while( g_ObjectCount > 1000 && !g_IsStop )
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
                }
            }
        }

        static void OnThread(void* pArgument) NN_NOEXCEPT
        {
            reinterpret_cast<ProducerThread*>(pArgument)->OnThread();
        }

        nn::fs::PriorityRaw GeneratePriority() NN_NOEXCEPT
        {
            static const nn::fs::PriorityRaw Priorities[] =
            {
                nn::fs::PriorityRaw_Realtime,
                nn::fs::PriorityRaw_Normal,
                nn::fs::PriorityRaw_Low,
            };
            return Priorities[
                std::uniform_int_distribution<size_t>(0, sizeof(Priorities) / sizeof(Priorities[0]) - 1)(m_RandomNumberGenerator)
            ];
        }

    private:
        std::mt19937 m_RandomNumberGenerator;
        bool m_IsInitialized;
        nn::os::ThreadType m_Thread;
        char NN_OS_ALIGNAS_THREAD_STACK m_StackBuffer[StackSize];
        static bool g_IsStop;
    };

    bool ProducerThread::g_IsStop = false;

    ConsumerThread g_ConsumerThreads[nn::fssrv::FileSystemProxyServerActiveSessionCount];
    ProducerThread g_ProducerThreads[1];

    void Finalize() NN_NOEXCEPT
    {
        for( auto& thread : g_ProducerThreads )
        {
            thread.Finalize();
        }
        for( auto& thread : g_ConsumerThreads )
        {
            thread.Finalize();
        }
    }

    void Unstop() NN_NOEXCEPT
    {
        g_SessionCounter.Reset();
        g_FileSystemProxyServerManager.Start();
        ProducerThread::Unstop();
        Object::Unstop();
    }

    void WaitDeferredByDeviceError() NN_NOEXCEPT
    {
        while( g_DeferredProcessManager.GetDeferredProcessQueueForDeviceError()->IsEmpty() )
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
        }
    }

    void WaitDeferredByPriority() NN_NOEXCEPT
    {
        while( !g_DeferredProcessManager.GetDeferredProcessQueueForPriority()->HasAny(nn::fssrv::FileSystemProxyServerSessionType::Realtime)
            && !g_DeferredProcessManager.GetDeferredProcessQueueForPriority()->HasAny(nn::fssrv::FileSystemProxyServerSessionType::Other)
            && !g_DeferredProcessManager.GetDeferredProcessQueueForPriority()->HasAny(nn::fssrv::FileSystemProxyServerSessionType::Background) )
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
        }
    }

    void WaitInvoke() NN_NOEXCEPT
    {
        while( g_SessionCounter.IsEmpty() )
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
        }
    }

    void WaitNoInvoking() NN_NOEXCEPT
    {
        while( !g_SessionCounter.IsEmpty() )
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
        }
    }

    void WaitDone() NN_NOEXCEPT
    {
        while( g_ObjectCount > 0 )
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
        }
    }

    void Stop() NN_NOEXCEPT
    {
        ProducerThread::Stop();
        for( auto& thread : g_ProducerThreads )
        {
            thread.Wait();
        }

        Object::Stop();
        WaitDone();

        g_FileSystemProxyServerManager.Stop();
        for( auto& thread : g_ConsumerThreads )
        {
            thread.Wait();
        }

        for( auto& thread : g_ConsumerThreads )
        {
            ASSERT_FALSE(thread.IsFailed());
        }

        auto queueCountMax = g_FileSystemProxyServerManager.GetDeferredQueueCountMax();
        NN_LOG("queue max %d\n", queueCountMax);
        EXPECT_LT(
            queueCountMax,
            nn::fssrv::DeferredProcessQueueForPriority::WaitConditionCount
            + static_cast<size_t>(nn::fssrv::FileSystemProxyServerActiveSessionCount)
        );

        {
            int realtimeCount;
            int backgroundCount;
            int otherCount;
            g_SessionResourceManager.GetSessionCountsForDebug(&realtimeCount, &backgroundCount, &otherCount);
            EXPECT_EQ(0, realtimeCount);
            EXPECT_EQ(0, backgroundCount);
            EXPECT_EQ(0, otherCount);
        }
    }
}

TEST(DeferredProcessQueueTest, Basic)
{
    NN_UTIL_SCOPE_EXIT
    {
        Finalize();
    };

    for( auto& thread : g_ProducerThreads )
    {
        NNT_ASSERT_RESULT_SUCCESS(thread.Initialize());
    }
    for( auto& thread : g_ConsumerThreads )
    {
        NNT_ASSERT_RESULT_SUCCESS(thread.Initialize());
    }

    Unstop();

    for( auto& thread : g_ConsumerThreads )
    {
        thread.Start();
    }
    for( auto& thread : g_ProducerThreads )
    {
        thread.Start();
    }

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    NNT_FS_ASSERT_NO_FATAL_FAILURE(g_SessionCounter.Check());

    Stop();
}

TEST(DeferredProcessQueueTest, BackgroundDeferred)
{
    NN_UTIL_SCOPE_EXIT
    {
        Finalize();
    };

    for( auto& thread : g_ProducerThreads )
    {
        NNT_ASSERT_RESULT_SUCCESS(thread.Initialize());
    }
    for( auto& thread : g_ConsumerThreads )
    {
        NNT_ASSERT_RESULT_SUCCESS(thread.Initialize());
    }

    Unstop();

    for( auto& thread : g_ConsumerThreads )
    {
        thread.Start();
    }

    {
        Object* pObject = new Object();
        pObject->Initialize(nn::fs::PriorityRaw_Background, Object::Action::Wait);
        g_FileSystemProxyServerManager.AddUserWaitHolder(pObject);
        ++g_ObjectCount;
    }
    WaitInvoke();

    for( auto& thread : g_ProducerThreads )
    {
        thread.Start();
    }

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    NNT_FS_ASSERT_NO_FATAL_FAILURE(g_SessionCounter.CheckWhenBackgroundDeferred());

    {
        Object* pObject = new Object();
        pObject->Initialize(nn::fs::PriorityRaw_Normal, Object::Action::Signal);
        g_FileSystemProxyServerManager.AddUserWaitHolder(pObject);
        ++g_ObjectCount;
    }

    Stop();
}

TEST(DeferredProcessQueueTest, InvokeRequestToEmptyQueue)
{
    NN_UTIL_SCOPE_EXIT
    {
        Finalize();
    };

    static const int ThreadCount = 1;

    for( int i = 0; i < ThreadCount; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(g_ConsumerThreads[i].Initialize());
    }

    Unstop();

    for( int i = 0; i < ThreadCount; ++i )
    {
        g_ConsumerThreads[i].Start();
    }

    // コンテキストの Reset が適切でなければ以下の実行が完了しない
    {
        Object* pObject = new Object();
        pObject->Initialize(nn::fs::PriorityRaw_Normal, Object::Action::InvokeDeferred);
        g_FileSystemProxyServerManager.AddUserWaitHolder(pObject);
        ++g_ObjectCount;
    }

    Stop();
}

TEST(DeferredProcessQueueTest, DeferredByDeviceErrorAndPriority)
{
    NN_UTIL_SCOPE_EXIT
    {
        Finalize();
    };

    static const int ThreadCount = 2;

    for( int i = 0; i < ThreadCount; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(g_ConsumerThreads[i].Initialize());
    }

    Unstop();

    for( int i = 0; i < ThreadCount; ++i )
    {
        g_ConsumerThreads[i].Start();
    }

    auto addObject = [](std::function<void (Object*)> initializer) NN_NOEXCEPT
    {
        Object* pObject = new Object();
        initializer(pObject);
        g_FileSystemProxyServerManager.AddUserWaitHolder(pObject);
        ++g_ObjectCount;
        return pObject;
    };

    // シナリオ:
    //   1. Device Error で defer
    //
    //   2. Realtime で Wait
    //   3. Device Error で defer するのを invoke するが、再度 Device Error で defer
    //   4. Realtime で Wait するのを解除
    //
    //   5. Device Error で defer するのを解除
    //   6. Realtime で Wait
    //   7. Device Error キューを invoke する要求が優先度のために defer
    //   8. Realtime で Wait するのを解除
    //   9. 優先度キューの invoke により、Device Error キューを invoke する要求が完了
    //  10. Device Error キューを invoke
    //
    //  11. 全てのタスクが完了する

    Object* pObjectDefer = addObject([](Object* pObject) NN_NOEXCEPT
        {
            pObject->Initialize(nn::fs::PriorityRaw_Realtime, Object::Action::Defer);
        });
    WaitDeferredByDeviceError();

    addObject([](Object* pObject) NN_NOEXCEPT
        {
            pObject->Initialize(nn::fs::PriorityRaw_Realtime, Object::Action::Wait);
        });
    WaitInvoke();

    addObject([](Object* pObject) NN_NOEXCEPT
        {
            pObject->Initialize(nn::fs::PriorityRaw_Normal, Object::Action::InvokeDeferred);
        });
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20)); // 処理されなことを確認するために少し待つ
    WaitDeferredByDeviceError();

    // 現在 1 つ実行中。Signal の実行で 2 つ実行中になり、最終的に実行中は 0 になる。
    addObject([](Object* pObject) NN_NOEXCEPT
        {
            pObject->Initialize(nn::fs::PriorityRaw_Normal, Object::Action::Signal);
        });
    WaitNoInvoking();

    addObject([](Object* pObject) NN_NOEXCEPT
        {
            pObject->Initialize(nn::fs::PriorityRaw_Realtime, Object::Action::Wait);
        });
    WaitInvoke();

    pObjectDefer->SetAction(Object::Action::Sleep);

    addObject([](Object* pObject) NN_NOEXCEPT
        {
            pObject->Initialize(nn::fs::PriorityRaw_Realtime, Object::Action::InvokeDeferred);
        });
    WaitDeferredByPriority();

    addObject([](Object* pObject) NN_NOEXCEPT
        {
            pObject->Initialize(nn::fs::PriorityRaw_Normal, Object::Action::Signal);
        });

    // Realtime セッションが解放される
    // 優先度キューが invoke されて Device Error キューが invoke される

    Stop();
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fssystem::InitializeAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    g_DeferredProcessManager.Initialize();
    g_FileSystemProxyServerManager.SetManagerHandler(&g_RequestHook);

    auto ret = RUN_ALL_TESTS();

    g_DeferredProcessManager.Finalize();

    nnt::Exit(ret);
}
