﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>

#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.h>
#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.impl.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

namespace nnt { namespace fs {
    nn::fssystem::IBufferManager* GetBufferManager() NN_NOEXCEPT;
}}

//! ファイルテーブルクラス
typedef nn::fssystem::dbm::HierarchicalFileTableTemplate<
            nn::fssystem::dbm::DirectoryName,
            nn::fssystem::dbm::FileName,
            nn::fssystem::dbm::detail::DirectoryInfo,
            nn::fssystem::dbm::detail::FileInfo
        >
        SimpleHierarchicalFileTable;

//! 非公開の型やクラスを公開する定義をまとめたクラス
class HierarchicalFileTableTest : public testing::Test
{
public:
    typedef SimpleHierarchicalFileTable::DirectoryInfo DirectoryInfo;
    typedef SimpleHierarchicalFileTable::FileInfo FileInfo;
    typedef SimpleHierarchicalFileTable::DirectoryEntryList DirectoryEntryList;
    typedef SimpleHierarchicalFileTable::FileEntryList FileEntryList;
    typedef HierarchicalFileTableTest::DirectoryEntryList::TypeKey Key;
    typedef HierarchicalFileTableTest::DirectoryEntryList::TypeValue Value;
    typedef HierarchicalFileTableTest::DirectoryEntryList::Index Index;

    static int64_t QueryDirectoryEntryStorageSize(uint32_t countDirectoryEntry) NN_NOEXCEPT
    {
        return SimpleHierarchicalFileTable::QueryFileEntryStorageSize(countDirectoryEntry);
    }
    static int64_t QueryFileEntryStorageSize(uint32_t countDirectoryEntry) NN_NOEXCEPT
    {
        return SimpleHierarchicalFileTable::QueryFileEntryStorageSize(countDirectoryEntry);
    }
};

// テストデータ保持用クラス
class FileTableSetup
{
public:
    // コンストラクタで渡したエントリ数は少なくとも確保できます。
    // 場合によっては 1 個余分にエントリを作成できることがあります。
    FileTableSetup(uint32_t countDirectoryEntry, uint32_t countFileEntry) NN_NOEXCEPT
    : m_BufDirectoryEntry(QueryDirectoryEntryStorageSize(countDirectoryEntry)),
      m_BufFileEntry(QueryFileEntryStorageSize(countFileEntry)),
      m_CountDirectoryEntry(countDirectoryEntry),
      m_CountFileEntry(countFileEntry)
    {
        // NNT_ASSERT_RESULT_SUCCESS は関数から戻ろうとするためコンストラクタでは使えません。
        nn::Result result;

        // ディレクトリのテーブルとストレージを確保します。
        {
            const size_t directoryStorageSize
                = SimpleHierarchicalFileTable::QueryDirectoryEntryStorageSize(countDirectoryEntry);
            const int64_t sizeEntryTable
                = ((directoryStorageSize + BlockSize - 1) & ~(BlockSize - 1));
            const uint32_t blockCount = static_cast<uint32_t>(sizeEntryTable / BlockSize);
            const int64_t offsetEntryTable
                = nn::fssystem::dbm::AllocationTable::QuerySize(blockCount);
            nn::fs::SubStorage directoryTableStorage(
                &m_BufDirectoryEntry, 0, offsetEntryTable
            );
            nn::fs::SubStorage directoryEntryStorage(
                &m_BufDirectoryEntry, offsetEntryTable, sizeEntryTable
            );

            result = nn::fssystem::dbm::AllocationTable::Format(
                directoryTableStorage,
                blockCount
            );
            // NNT_ASSERT_RESULT_SUCCESS は関数から戻ろうとするためコンストラクタでは使えません。
            NN_ASSERT(result.IsSuccess());
            m_AllocationTableDirectoryEntry.Initialize(
                directoryTableStorage,
                blockCount
            );
            uint32_t index;
            result = m_AllocationTableDirectoryEntry.Allocate(&index, blockCount);
            NN_ASSERT(result.IsSuccess());
            result = m_StorageDirectoryEntry.InitializeBuffered(
                         &m_AllocationTableDirectoryEntry,
                         index,
                         BlockSize,
                         directoryEntryStorage,
                         nnt::fs::GetBufferManager()
                     );
            NN_ASSERT(result.IsSuccess());
            result = m_StorageDirectoryEntry.AllocateBlock(1);
            NN_ASSERT(nn::fs::ResultAllocationTableFull::Includes(result));
        }

        // ファイルのテーブルとストレージを確保します。
        {
            const size_t fileStorageSize
                = SimpleHierarchicalFileTable::QueryFileEntryStorageSize(countFileEntry);
            const int64_t sizeEntryTable = ((fileStorageSize + BlockSize - 1) & ~(BlockSize - 1));
            const uint32_t blockCount = static_cast<uint32_t>(sizeEntryTable / BlockSize);
            const int64_t offsetEntryTable
                = nn::fssystem::dbm::AllocationTable::QuerySize(blockCount);
            nn::fs::SubStorage fileTableStorage(&m_BufFileEntry, 0, offsetEntryTable);
            nn::fs::SubStorage fileEntryStorage(
                &m_BufFileEntry, offsetEntryTable, sizeEntryTable
            );
            result = nn::fssystem::dbm::AllocationTable::Format(
                         fileTableStorage,
                         blockCount
                     );
            NN_ASSERT(result.IsSuccess());
            m_AllocationTableFileEntry.Initialize(
                fileTableStorage,
                blockCount
            );
            uint32_t index;
            result = m_AllocationTableFileEntry.Allocate(&index, blockCount);
            NN_ASSERT(result.IsSuccess());
            result = m_StorageFileEntry.InitializeBuffered(
                         &m_AllocationTableFileEntry,
                         index,
                         BlockSize,
                         fileEntryStorage,
                         nnt::fs::GetBufferManager()
                     );
            NN_ASSERT(result.IsSuccess());

            result = m_StorageFileEntry.AllocateBlock(1);
            NN_ASSERT(nn::fs::ResultAllocationTableFull::Includes(result));
        }

        // ファイルテーブル領域をフォーマットします。
        result = SimpleHierarchicalFileTable::Format(
                    &m_StorageDirectoryEntry,
                    &m_StorageFileEntry
                );
        NN_ASSERT(result.IsSuccess());

        // ファイルテーブルをマウントします。
        m_FileTable.Initialize(
            &m_StorageDirectoryEntry,
            &m_StorageFileEntry
        );
    }

    ~FileTableSetup() NN_NOEXCEPT
    {
        m_FileTable.Finalize();
    }

public:
    SimpleHierarchicalFileTable& GetFileTable() NN_NOEXCEPT
    {
        return m_FileTable;
    }

private:
    static const int64_t BlockSize = 1 << 7;

private:
    //! ディレクトリの最大エントリー数からテーブルのストレージに必要なサイズを取得します。
    static int64_t QueryDirectoryEntryStorageSize(uint32_t countDirectoryEntry) NN_NOEXCEPT
    {
        const int64_t directoryStorageSize
            = HierarchicalFileTableTest::QueryDirectoryEntryStorageSize(countDirectoryEntry);
        const int64_t sizeEntryTable
            = ((directoryStorageSize + BlockSize - 1) & ~(BlockSize - 1));
        const int64_t sizeAllocationMeta = nn::fssystem::dbm::AllocationTable::QuerySize(
            static_cast<uint32_t>(sizeEntryTable)
        );
        return sizeAllocationMeta + sizeEntryTable;
    }

    //! ファイルの最大エントリー数からテーブルのストレージに必要なサイズを取得します。
    static int64_t QueryFileEntryStorageSize(uint32_t countFileEntry) NN_NOEXCEPT
    {
        const int64_t fileStorageSize
            = HierarchicalFileTableTest::QueryFileEntryStorageSize(countFileEntry);
        const int64_t sizeEntryTable = ((fileStorageSize + BlockSize - 1) & ~(BlockSize - 1));
        const int64_t sizeAllocationMeta = nn::fssystem::dbm::AllocationTable::QuerySize(
            static_cast<uint32_t>(sizeEntryTable)
        );
        return sizeAllocationMeta + sizeEntryTable;
    }

public:
    nnt::fs::util::SafeMemoryStorage m_BufDirectoryEntry;
    nnt::fs::util::SafeMemoryStorage m_BufFileEntry;
    nn::fssystem::dbm::AllocationTable m_AllocationTableDirectoryEntry;
    nn::fssystem::dbm::AllocationTable m_AllocationTableFileEntry;
    nn::fssystem::dbm::BufferedAllocationTableStorage m_StorageDirectoryEntry;
    nn::fssystem::dbm::BufferedAllocationTableStorage m_StorageFileEntry;
    SimpleHierarchicalFileTable m_FileTable;
    size_t m_CountDirectoryEntry;
    size_t m_CountFileEntry;
};

//! 再帰的に削除します。
static void RemoveRecursive(
                SimpleHierarchicalFileTable& fileTable,
                const char* pStrPath
            ) NN_NOEXCEPT
{
    SimpleHierarchicalFileTable::FindIndex findIndex;

    NNT_ASSERT_RESULT_SUCCESS(fileTable.FindOpen(&findIndex, pStrPath));

    for( ; ; )
    {
        char directoryName[sizeof(nn::fssystem::dbm::DirectoryName) + 1] = { 0 };
        bool isFinished = true;
        NNT_ASSERT_RESULT_SUCCESS(
            fileTable.FindNextDirectory(
                reinterpret_cast<nn::fssystem::dbm::DirectoryName*>(&directoryName),
                &isFinished,
                &findIndex
            )
        );
        if( isFinished )
        {
            // ディレクトリの探索完了です。
            break;
        }
        // 再帰的に入ります。
        nnt::fs::util::String strFullPath = pStrPath;
        if( strFullPath.size() > 1 )
        {
            strFullPath += "/";
        }
        strFullPath += directoryName;
        RemoveRecursive(fileTable, strFullPath.c_str());

        // このフォルダを削除します。
        NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteDirectory(strFullPath.c_str()));
    }

    for( ; ; )
    {
        char fileName[sizeof(nn::fssystem::dbm::FileName) + 1] = { 0 };
        int64_t sizeFile;
        bool isFinished = true;
        NNT_ASSERT_RESULT_SUCCESS(
            fileTable.FindNextFile(
                &sizeFile,
                reinterpret_cast<nn::fssystem::dbm::FileName*>(&fileName),
                &isFinished,
                &findIndex
            )
        );
        if( isFinished )
        {
            // ファイルの探索完了です。
            break;
        }
        // このファイルを削除します。
        nnt::fs::util::String strFullPath = pStrPath;
        strFullPath += "/";
        strFullPath += fileName;
        NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteFile(strFullPath.c_str()));
    }
}

//! プライベートクラス HierarchicalFileTableTemplate::DirectoryEntryList のテスト
//! ディレクトリ情報の登録と取得をテストします。
TEST(HierarchicalFileTableTest, TestDirectoryEntry)
{
    const uint32_t countDirectoryEntry = 2;

    // ストレージを生成します。
    const int64_t sizeEntryTable = SimpleHierarchicalFileTable::QueryDirectoryEntryStorageSize(
                                       countDirectoryEntry
                                   );
    const uint32_t blockSize = 512;
    const uint32_t blockCount = static_cast<uint32_t>((sizeEntryTable + blockSize - 1) / blockSize);
    const int64_t sizeAllocationTable = nn::fssystem::dbm::AllocationTable::QuerySize(blockCount);
    nnt::fs::util::SafeMemoryStorage bufDirectoryEntry(sizeAllocationTable + sizeEntryTable);
    nn::fs::SubStorage directoryTableStorage(
        &bufDirectoryEntry, 0, sizeAllocationTable + sizeEntryTable
    );
    nn::fs::SubStorage directoryEntryStorage(
        &bufDirectoryEntry, sizeAllocationTable, sizeEntryTable
    );
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::dbm::AllocationTable::Format(directoryTableStorage, blockCount)
    );
    nn::fssystem::dbm::AllocationTable allocationTableDirectoryEntry;
    allocationTableDirectoryEntry.Initialize(directoryTableStorage, blockCount);
    uint32_t indexAllocationTable;
    NNT_ASSERT_RESULT_SUCCESS(
        allocationTableDirectoryEntry.Allocate(&indexAllocationTable, blockCount)
    );
    nn::fssystem::dbm::BufferedAllocationTableStorage storageDirectoryEntry;
    NNT_ASSERT_RESULT_SUCCESS(
        storageDirectoryEntry.InitializeBuffered(
            &allocationTableDirectoryEntry,
            indexAllocationTable,
            blockSize,
            directoryEntryStorage,
            nnt::fs::GetBufferManager()
        )
    );

    // DirectoryEntryList の作成と初期化
    HierarchicalFileTableTest::DirectoryEntryList directoryEntryList;

    NNT_ASSERT_RESULT_SUCCESS(directoryEntryList.Format(&storageDirectoryEntry));
    directoryEntryList.Initialize(&storageDirectoryEntry);

    // { uint32_t, { char[64] } }
    static const HierarchicalFileTableTest::Key e1 = { 1, { "0000001" } };
    static const HierarchicalFileTableTest::Key e2 = { 1, { "0000002" } };
    // { uint32_t, uint32_t, uint32_t, { Bit8[4] } }
    static const HierarchicalFileTableTest::Value v1 = { 1, 2, 3, { { 0 } } };
    static const HierarchicalFileTableTest::Value v2 = { 4, 5, 6, { { 0 } } };

    HierarchicalFileTableTest::Index index;
    HierarchicalFileTableTest::Index index1;
    HierarchicalFileTableTest::Index index2;
    HierarchicalFileTableTest::Key e;
    HierarchicalFileTableTest::Value v;

    // ディレクトリ情報を登録します。
    NNT_ASSERT_RESULT_SUCCESS(directoryEntryList.Add(&index1, e1, v1));
    NNT_ASSERT_RESULT_SUCCESS(directoryEntryList.Add(&index2, e2, v2));

    // 登録した値を取得します。
    NNT_ASSERT_RESULT_SUCCESS(directoryEntryList.Get(&index, &v, e1));
    ASSERT_EQ(index, index1);
    ASSERT_EQ(0, std::memcmp(&v, &v1, sizeof(v)));
    NNT_ASSERT_RESULT_SUCCESS(directoryEntryList.Get(&index, &v, e2));
    ASSERT_EQ(index, index2);
    ASSERT_EQ(0, std::memcmp(&v, &v2, sizeof(v)));

    // インデックスで引きます。
    NNT_ASSERT_RESULT_SUCCESS(directoryEntryList.GetByIndex(&e, &v, index1));
    ASSERT_EQ(e, e1);
    ASSERT_EQ(0, std::memcmp(&v, &v1, sizeof(v)));
    NNT_ASSERT_RESULT_SUCCESS(directoryEntryList.GetByIndex(&e, &v, index2));
    ASSERT_EQ(e, e2);
    ASSERT_EQ(0, std::memcmp(&v, &v2, sizeof(v)));
}

//! ルートディレクトリの中身の列挙をテストします。
TEST(HierarchicalFileTableTest, TestFindOpenRoot)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    // 何もない状態でファイルを列挙します。
    SimpleHierarchicalFileTable::FindIndex findIndex = { 0xCC };
    nn::fssystem::dbm::DirectoryName directoryName;
    nn::fssystem::dbm::FileName fileName;

    // 存在しないディレクトリを指定します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileTable.FindOpen(&findIndex, "/dir")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileTable.FindOpen(&findIndex, "/dir1/")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileTable.FindOpen(&findIndex, "/dir1/dir2")
    );

    // ディレクトリ名が長すぎます。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileTable.FindOpen(&findIndex, "/longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglongd/")
    );

    // ルートディレクトリはデフォルトで生成されている必要があります。
    // ルートディレクトリの中身は空です。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.FindOpen(&findIndex, "/"));
    bool isFinished = false;
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.FindNextDirectory(
            &directoryName,
            &isFinished,
            &findIndex
        )
    );
    NN_SDK_ASSERT(isFinished);
    isFinished = false;
    int64_t sizeFile;
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.FindNextFile(
            &sizeFile,
            &fileName,
            &isFinished,
            &findIndex
        )
    );
    NN_SDK_ASSERT(isFinished);
}

//! ディレクトリの作成をテストします。
TEST(HierarchicalFileTableTest, TestCreateDirectory)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };

    // ディレクトリの再帰生成はサポートされません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileTable.CreateDirectory("/dir1/dir2", directoryInfo)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileTable.CreateDirectory("/dir1/dir2/", directoryInfo)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileTable.CreateDirectory("/dir1/dir2/dir3", directoryInfo)
    );

    // 長いエントリー名です。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileTable.CreateDirectory("/longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglongd", directoryInfo)
    );

    // 先頭にスラッシュなしの指定は失敗します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathFormat,
        fileTable.CreateDirectory("dir1", directoryInfo)
    );

    // 区切り文字が連続していますが、同一視します
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.CreateDirectory("//", directoryInfo)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.CreateDirectory("///", directoryInfo)
    );

    // ルートディレクトリはすでに存在しています。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.CreateDirectory("/", directoryInfo)
    );

    // ディレクトリを作成します。
    directoryInfo.info[0] = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1", directoryInfo));
    directoryInfo.info[0] = 2;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1//dir2", directoryInfo));
    directoryInfo.info[0] = 3;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1/dir2/dir3//", directoryInfo));
    directoryInfo.info[0] = 4;
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.CreateDirectory("/dir1/dir2/dir3///dir4dir4", directoryInfo)
    );
}

//! ディレクトリの列挙をテストします。
TEST(HierarchicalFileTableTest, TestIterateDirectory)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };

    // ディレクトリを作成します。
    directoryInfo.info[0] = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1", directoryInfo));
    directoryInfo.info[0] = 2;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1//dir2", directoryInfo));
    directoryInfo.info[0] = 3;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1/dir2/dir3//", directoryInfo));
    directoryInfo.info[0] = 4;
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.CreateDirectory("/dir1/dir2/dir3///dir4dir4", directoryInfo)
    );

    // ディレクトリを列挙します。

    // 中身を列挙するディレクトリ名と、中身のディレクトリ名
    static const int CheckCount = 5;
    static const nn::fssystem::dbm::PathChar* FindParameters[CheckCount][2] =
    {
        {"/",                        "dir1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"},
        {"/dir1",                    "dir2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"},
        {"/dir1/dir2/",              "dir3\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"},
        {"/dir1/dir2/dir3",          "dir4dir4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"},
        {"/dir1/dir2/dir3/dir4dir4", "\0" }
    };

    for( int checkIndex = 0; checkIndex < CheckCount; ++checkIndex )
    {
        SimpleHierarchicalFileTable::FindIndex findIndex = { 0xCC };
        nn::fssystem::dbm::DirectoryName directoryName;
        bool isFinished = true;

        NNT_ASSERT_RESULT_SUCCESS(fileTable.FindOpen(&findIndex, FindParameters[checkIndex][0]));

        if( FindParameters[checkIndex][1][0] != '\0' )
        {
            // 中身のディレクトリ名が一致するかテスト
            // 出力用引数が書き換わることをテストするために一度クリアします。
            isFinished = true;
            NNT_ASSERT_RESULT_SUCCESS(
                fileTable.FindNextDirectory(&directoryName, &isFinished, &findIndex)
            );
            ASSERT_EQ(
                0,
                std::memcmp(
                    &directoryName,
                    FindParameters[checkIndex][1],
                    sizeof(directoryName)
                )
            );
            ASSERT_FALSE(isFinished);
        }

        // 中にディレクトリはもうないことをテスト
        isFinished = false;
        NNT_ASSERT_RESULT_SUCCESS(
            fileTable.FindNextDirectory(&directoryName, &isFinished, &findIndex)
        );
        ASSERT_TRUE(isFinished);

        // 中にファイルはないことをテスト
        isFinished = false;
        nn::fssystem::dbm::FileName fileName;
        int64_t sizeFile;
        NNT_ASSERT_RESULT_SUCCESS(
            fileTable.FindNextFile(&sizeFile, &fileName, &isFinished, &findIndex)
        );
        ASSERT_TRUE(isFinished);
    }
}

//! ファイルの作成をテストします。
TEST(HierarchicalFileTableTest, TestCreateFile)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };
    HierarchicalFileTableTest::FileInfo fileInfo = { { 1 } };

    // あらかじめディレクトリを作成します。
    directoryInfo.info[0] = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1", directoryInfo));
    directoryInfo.info[0] = 2;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1//dir2", directoryInfo));
    directoryInfo.info[0] = 3;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1/dir2/dir3//", directoryInfo));
    directoryInfo.info[0] = 4;
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.CreateDirectory("/dir1/dir2/dir3///dir4dir4", directoryInfo)
    );

    // ファイルを作成します。
    // スラッシュなしの指定は失敗します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathFormat,
        fileTable.CreateFile("file1", fileInfo)
    );
    // ディレクトリと同名のファイルは作れません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.CreateFile("/dir1", fileInfo)
    );
    // 長いエントリー名です。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileTable.CreateFile("/longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglongfilen", fileInfo)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileTable.CreateFile(
            "/longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglonglongd/longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglongfilen", fileInfo
        )
    );

    // ファイル操作に対してディレクトリが与えられました。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.CreateFile("/", fileInfo)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.CreateFile("//", fileInfo)
    );

    // ファイルを作成します。
    fileInfo.fo.info[0] = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir1/dir2/dir3/dir4dir4/file1", fileInfo));

    // ファイル以下にディレクトリは作れません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.CreateDirectory("/dir1/dir2/dir3/dir4dir4/file1/dir5", directoryInfo)
    );
    // ファイル以下にファイルは作れません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.CreateFile("/dir1/dir2/dir3/dir4dir4/file1/file1", fileInfo)
    );

    // 更にファイルを作成します。
    fileInfo.fo.info[0] = 2;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir1/dir2/dir3/file1", fileInfo));
    fileInfo.fo.info[0] = 3;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir1/dir2/file1", fileInfo));
    fileInfo.fo.info[0] = 4;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir1/file1", fileInfo));
    fileInfo.fo.info[0] = 5;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/file1", fileInfo));
    fileInfo.fo.info[0] = 6;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/file2file2fi", fileInfo));

    // 既に存在するファイルは作れません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.CreateFile("/file2file2fi", fileInfo)
    );
}

// ファイルを開くテストをします。
TEST(HierarchicalFileTableTest, TestOpenFile)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };
    HierarchicalFileTableTest::FileInfo fileInfo = { { 1 } };

    // あらかじめディレクトリを作成します。
    directoryInfo.info[0] = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1", directoryInfo));
    directoryInfo.info[0] = 2;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1//dir2", directoryInfo));
    directoryInfo.info[0] = 3;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1/dir2/dir3//", directoryInfo));
    directoryInfo.info[0] = 4;
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.CreateDirectory("/dir1/dir2/dir3///dir4dir4", directoryInfo)
    );

    // あらかじめファイルを作成します。
    fileInfo.fo.info[0] = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir1/dir2/dir3/dir4dir4/file1", fileInfo));
    fileInfo.fo.info[0] = 2;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir1/dir2/dir3/file1", fileInfo));
    fileInfo.fo.info[0] = 3;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir1/dir2/file1", fileInfo));
    fileInfo.fo.info[0] = 4;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir1/file1", fileInfo));
    fileInfo.fo.info[0] = 5;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/file1", fileInfo));
    fileInfo.fo.info[0] = 6;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/file2file2fi", fileInfo));

    // スラッシュなしの指定は失敗します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathFormat,
        fileTable.OpenFile(&fileInfo, "file1")
    );
    // ディレクトリをファイルとして開くことは出来ません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.OpenFile(&fileInfo, "/dir1")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathFormat,
        fileTable.OpenFile(&fileInfo, "/dir1/.")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathFormat,
        fileTable.OpenFile(&fileInfo, "/dir1/dir2/..")
    );
    // 存在しないファイルは開けません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultFileNotFound,
        fileTable.OpenFile(&fileInfo, "/dir1/dir2/file9")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultFileNotFound,
        fileTable.OpenFile(&fileInfo, "/dir1/dir2/...")
    );

    // ファイルをオープンします。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/dir1/dir2/dir3/dir4dir4/file1"));
    ASSERT_EQ(1, fileInfo.fo.info[0]);
    NNT_ASSERT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/dir1/dir2/dir3/file1"));
    ASSERT_EQ(2, fileInfo.fo.info[0]);
    NNT_ASSERT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/dir1/dir2/file1"));
    ASSERT_EQ(3, fileInfo.fo.info[0]);
    NNT_ASSERT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/dir1/file1"));
    ASSERT_EQ(4, fileInfo.fo.info[0]);
    NNT_ASSERT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/file1"));
    ASSERT_EQ(5, fileInfo.fo.info[0]);
    NNT_ASSERT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/file2file2fi"));
    ASSERT_EQ(6, fileInfo.fo.info[0]);
}

//! ファイル情報の更新をテストします。
TEST(HierarchicalFileTableTest, TestUpdateFileInformation)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::FileInfo fileInfo = { { 1 } };

    // あらかじめファイルを作成します。
    fileInfo.fo.info[0] = 5;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/file1", fileInfo));
    fileInfo.fo.info[0] = 6;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/file2file2fi", fileInfo));

    // ファイル情報を更新します。
    fileInfo.fo.info[0] = 0xAA;
    fileInfo.fo.info[sizeof(fileInfo.fo.info) - 1] = 0xBB;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.UpdateFileInformation("/file2file2fi", fileInfo));

    // 出力用引数が書き換わることをテストするために一度クリアします。
    fileInfo.fo.info[0] = 0;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/file1"));
    ASSERT_EQ(5, fileInfo.fo.info[0]);

    fileInfo.fo.info[0] = 0;
    fileInfo.fo.info[sizeof(fileInfo.fo.info) - 1] = 0;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/file2file2fi"));
    ASSERT_EQ(0xAA, fileInfo.fo.info[0]);
    ASSERT_EQ(fileInfo.fo.info[sizeof(fileInfo.fo.info) - 1], 0xBB);

    // ルートディレクトリから始まっていないパスなので失敗します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathFormat,
        fileTable.UpdateFileInformation("file1", fileInfo)
    );
}

//! ファイル名の変更でディレクトリに関するエラーをテストします。
TEST(HierarchicalFileTableTest, TestRenameFileWithDirectory)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };
    HierarchicalFileTableTest::FileInfo fileInfo = { { 1 } };

    // あらかじめディレクトリを作成します。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1", directoryInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1//dir2", directoryInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1/dir2/dir3//", directoryInfo));
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.CreateDirectory("/dir1/dir2/dir3///dir4dir4", directoryInfo)
    );

    // あらかじめファイルを作成します。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/file1", fileInfo));

    bool isFile;
    bool isSameEntry;

    // ファイルをリネームします。
    // 既存のディレクトリと同じ名前にはリネームできません。
    // ファイル名にディレクトリ表記が与えられたエラーが優先されます。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.RenameFile(&isFile, &isSameEntry, "/dir1/", "/file1")
    );

    // 出力用引数が元の値に依存せず関数呼び出しで書き換わることを確認するために代入します。
    isFile = true;
    isSameEntry = true;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.RenameFile(&isFile, &isSameEntry, "/dir1", "/file1")
    );
    ASSERT_FALSE(isFile);
    ASSERT_FALSE(isSameEntry);

    // ルートディレクトリにはリネームできません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.RenameFile(&isFile, &isSameEntry, "/", "/file1")
    );

    // 既存のディレクトリと同じ名前にはリネームできません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.RenameFile(&isFile, &isSameEntry, "/dir1/", "/file1")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.RenameFile(&isFile, &isSameEntry, "/dir1/dir2/", "/file1")
    );

    isFile = true;
    isSameEntry = true;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.RenameFile(&isFile, &isSameEntry, "/dir1/dir2", "/file1")
    );
    ASSERT_FALSE(isFile);
    ASSERT_FALSE(isSameEntry);

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathFormat,
        fileTable.RenameFile(&isFile, &isSameEntry, "/dir1/dir2/..", "/file1")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathFormat,
        fileTable.RenameFile(&isFile, &isSameEntry, "/dir1/.", "/file1")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathFormat,
        fileTable.RenameFile(&isFile, &isSameEntry, "/dir1/dir2/.", "/file1")
    );

    isFile = true;
    isSameEntry = true;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.RenameFile(&isFile, &isSameEntry, "/dir1/./dir2", "/file1")
    );
    ASSERT_FALSE(isFile);
    ASSERT_FALSE(isSameEntry);

    // ルートディレクトリにはリネームできません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.RenameFile(&isFile, &isSameEntry, "/", "/file1")
    );

    // ルートディレクトリよりも上の階層にはリネームできません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnobtainable,
        fileTable.RenameFile(&isFile, &isSameEntry, "/../file1", "/file1")
    );

    // ディレクトリに対してファイル用リネームをしようとしました。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.RenameFile("/rename1", "/dir1")
    );
}

//! ファイル名の変更をテストします。
TEST(HierarchicalFileTableTest, TestRenameFile)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };
    HierarchicalFileTableTest::FileInfo fileInfo = { { 1 } };

    // あらかじめディレクトリを作成します。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1", directoryInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1//dir2", directoryInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1/dir2/dir3//", directoryInfo));
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.CreateDirectory("/dir1/dir2/dir3///dir4dir4", directoryInfo)
    );

    // あらかじめファイルを作成します。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir1/file1", fileInfo));
    fileInfo.fo.info[0] = 5;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/file1", fileInfo));
    fileInfo.fo.info[0] = 6;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/file2file2fi", fileInfo));

    bool isFile;
    bool isSameEntry;

    // ファイルをリネームします。
    // ファイル名にできない名前が与えられました。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.RenameFile(&isFile, &isSameEntry, "/file1/..", "/file2file2fi")
    );
    // パスが長すぎます。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileTable.RenameFile(
            &isFile, &isSameEntry, "/file1file1file1file1file1file1file1file1file1file1file1file1file1", "/file1"
        )
    );
    // 先頭に区切り文字がありません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathFormat,
        fileTable.RenameFile(&isFile, &isSameEntry, "/file1", "file1")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathFormat,
        fileTable.RenameFile(&isFile, &isSameEntry, "file1", "/file1")
    );
    // ファイル操作に対してディレクトリ表記が与えられました。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.RenameFile(&isFile, &isSameEntry, "/file1/", "/file1")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.RenameFile(&isFile, &isSameEntry, "/file1", "/file1/")
    );

    // 出力用引数が元の値に依存せず関数呼び出しで書き換わることを確認するために代入します。
    isFile = false;
    isSameEntry = false;
    // 同名ファイル間リネームはできません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.RenameFile(&isFile, &isSameEntry, "/file1", "/file1")
    );
    ASSERT_TRUE(isFile);
    ASSERT_TRUE(isSameEntry);

    isFile = false;
    isSameEntry = false;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.RenameFile(&isFile, &isSameEntry, "/file1", "/./file1")
    );
    ASSERT_TRUE(isFile);
    ASSERT_TRUE(isSameEntry);

    isFile = false;
    isSameEntry = false;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.RenameFile(&isFile, &isSameEntry, "/./file1", "/file1")
    );
    ASSERT_TRUE(isFile);
    ASSERT_TRUE(isSameEntry);

    // 普通のリネームです。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.RenameFile("/file2", "/file1"));

    // 出力用引数が元の値に依存せず関数呼び出しで書き換わることを確認するために代入します。
    isFile = false;
    isSameEntry = true;
    // 既に存在する名前にリネームできません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.RenameFile(&isFile, &isSameEntry, "/file2file2fi", "/file2")
    );
    ASSERT_TRUE(isFile);
    ASSERT_FALSE(isSameEntry);

    isFile = false;
    isSameEntry = true;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.RenameFile(&isFile, &isSameEntry, "/dir1/file1", "/file2")
    );
    ASSERT_TRUE(isFile);
    ASSERT_FALSE(isSameEntry);

    isFile = false;
    isSameEntry = true;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.RenameFile(&isFile, &isSameEntry, "/dir1/dir2/../file1", "/file2")
    );
    ASSERT_TRUE(isFile);
    ASSERT_FALSE(isSameEntry);

    // フォルダ越しのリネームです。
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.RenameFile(
            &isFile, &isSameEntry, "/dir1/dir2/dir3/dir4dir4/file2file2fi", "/file2file2fi"
        )
    );
    // リネーム結果をチェックします。
    // 出力用引数が書き換わることをテストするために一度クリアします。
    fileInfo.fo.info[0] = 0;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/file2"));
    ASSERT_EQ(5, fileInfo.fo.info[0]);
    fileInfo.fo.info[0] = 0;
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.OpenFile(&fileInfo, "/dir1/dir2/dir3/dir4dir4/file2file2fi")
    );
    ASSERT_EQ(6, fileInfo.fo.info[0]);
} // NOLINT(impl/function_size)

// ルートディレクトリの名前変更エラーの変更をテストします。
TEST(HierarchicalFileTableTest, TestRenameRootDirectory)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };

    // あらかじめディレクトリを作成します。
    directoryInfo.info[0] = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1", directoryInfo));
    directoryInfo.info[0] = 2;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1//dir2", directoryInfo));

    bool isFile;
    bool isSameEntry;
    bool isParentEntry;

    // ルートディレクトリはリネームできません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/dir8", "/")
    );

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/dir8", "//")
    );

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/dir8", "/.")
    );

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "dir8", "/dir1/..")
    );

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(
            &isFile, &isSameEntry, &isParentEntry,"/dir8", "/dir1/../dir1/dir2/../.."
        )
    );

    // 出力用引数が元の値に依存せず関数呼び出しで書き換わることを確認するために代入します。
    isFile = true;
    isSameEntry = true;
    isParentEntry = false;
    // ルートディレクトリにはリネームできません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(&isFile, &isSameEntry, &isParentEntry,"/", "/dir1")
    );
    ASSERT_FALSE(isFile);
    ASSERT_FALSE(isSameEntry);
    ASSERT_TRUE(isParentEntry);

    isFile = true;
    isSameEntry = true;
    isParentEntry = false;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(&isFile, &isSameEntry, &isParentEntry,"//", "/dir1")
    );
    ASSERT_FALSE(isFile);
    ASSERT_FALSE(isSameEntry);
    ASSERT_TRUE(isParentEntry);

    isFile = true;
    isSameEntry = true;
    isParentEntry = false;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(&isFile, &isSameEntry, &isParentEntry,"/dir1/..", "/dir1")
    );
    ASSERT_FALSE(isFile);
    ASSERT_FALSE(isSameEntry);
    ASSERT_TRUE(isParentEntry);

    isFile = true;
    isSameEntry = true;
    isParentEntry = false;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(
            &isFile, &isSameEntry, &isParentEntry,"/dir1/../dir1/dir2/../..", "/dir1"
        )
    );
    ASSERT_FALSE(isFile);
    ASSERT_FALSE(isSameEntry);
    ASSERT_TRUE(isParentEntry);

    // ルートディレクトリよりも上の階層はリネームできません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnobtainable,
        fileTable.RenameDirectory(&isFile, &isSameEntry, &isParentEntry,"/dir8", "/..")
    );

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnobtainable,
        fileTable.RenameDirectory(&isFile, &isSameEntry, &isParentEntry,"/dir8", "/../..")
    );

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileTable.RenameDirectory(&isFile, &isSameEntry, &isParentEntry,"/dir8", "/..../..")
    );
}

// ディレクトリ名の変更でファイルに関するエラーをテストします。
TEST(HierarchicalFileTableTest, TestRenameDirectoryWithFile)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };
    HierarchicalFileTableTest::FileInfo fileInfo = { { 1 } };

    // あらかじめディレクトリを作成します。
    directoryInfo.info[0] = 2;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir2", directoryInfo));
    directoryInfo.info[0] = 3;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir2/dir3", directoryInfo));

    // あらかじめファイルを作成します。
    fileInfo.fo.info[0] = 5;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/file2", fileInfo));

    bool isFile;
    bool isSameEntry;
    bool isParentEntry;

    // ファイル以下へのリネームはできません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.RenameDirectory("/file2/dir3", "/dir2/dir3")
    );

    // 出力用引数が元の値に依存せず関数呼び出しで書き換わることを確認するために代入します。
    isFile = false;
    isSameEntry = true;
    isParentEntry = true;
    // すでに同名のファイルが存在しています。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/file2", "/dir2/dir3")
    );
    ASSERT_TRUE(isFile);
    ASSERT_FALSE(isSameEntry);
    ASSERT_FALSE(isParentEntry);

    // ファイルに対してディレクトリリネームを呼び出しました。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileTable.RenameDirectory("/file1", "/file2")
    );
}

// 既に存在するパスへのディレクトリ名の変更エラーをテストします。
TEST(HierarchicalFileTableTest, TestRenameDirectoryAlreadyExists)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 5 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(5, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };

    // あらかじめディレクトリを作成します。
    directoryInfo.info[0] = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1", directoryInfo));
    directoryInfo.info[0] = 2;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1//dir2", directoryInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/rename2", directoryInfo));
    directoryInfo.info[0] = 3;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/rename2/dir3//", directoryInfo));
    directoryInfo.info[0] = 4;
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.CreateDirectory("/rename2/dir3///dir4dir4", directoryInfo)
    );

    bool isFile;
    bool isSameEntry;
    bool isParentEntry;

    // 同名ディレクトリ間リネームはできません。
    {
        // リネーム先のパスとリネーム元のパス
        static const nn::fssystem::dbm::PathChar* PathPairs[][2] =
        {
            {"/dir1",          "/dir1"},
            {"/dir1",          "/dir1/."},
            {"/dir1/",         "/dir1"},
            {"/dir1/dir2/..",  "/dir1"},
        };
        for( const auto pathPair : PathPairs )
        {
            // 出力用引数が元の値に依存せず関数呼び出しで書き換わることを確認するために代入します。
            isFile = true;
            isSameEntry = false;
            isParentEntry = true;
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAlreadyExists,
                fileTable.RenameDirectory(
                    &isFile,
                    &isSameEntry,
                    &isParentEntry,
                    pathPair[0],
                    pathPair[1]
                )
            );
            ASSERT_FALSE(isFile);
            ASSERT_TRUE(isSameEntry);
            ASSERT_FALSE(isParentEntry);
        }
    }

    // すでに同名のディレクトリが存在しています。
    {
        // リネーム先のパスとリネーム元のパス
        static const nn::fssystem::dbm::PathChar* PathPairs[][2] =
        {
            {"/dir1",      "/rename2/dir3"},
            {"/rename2",   "/dir1"},
            {"/dir1",      "/rename2/dir3/dir4dir4"},
            {"/dir1/.",    "/rename2/dir3/dir4dir4"},
            {"/dir1/.",    "/rename2/dir3/dir4dir4/."},
        };
        for( const auto pathPair : PathPairs )
        {
            isFile = true;
            isSameEntry = true;
            isParentEntry = true;
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAlreadyExists,
                fileTable.RenameDirectory(
                    &isFile,
                    &isSameEntry,
                    &isParentEntry,
                    pathPair[0],
                    pathPair[1]
                )
            );
            ASSERT_FALSE(isFile);
            ASSERT_FALSE(isSameEntry);
            ASSERT_FALSE(isParentEntry);
        }
    }

    // 同名へのリネームです。
    isFile = true;
    isSameEntry = false;
    isParentEntry = true;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.RenameDirectory(
            &isFile, &isSameEntry, &isParentEntry, "/dir1/../rename2/.", "/rename2/dir3/../."
        )
    );
    ASSERT_FALSE(isFile);
    ASSERT_TRUE(isSameEntry);
    ASSERT_FALSE(isParentEntry);
}

// 親ディレクトリへのディレクトリ名の変更のエラーをテストします。
TEST(HierarchicalFileTableTest, TestRenameDirectoryToParent)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };

    // あらかじめディレクトリを作成します。
    directoryInfo.info[0] = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1", directoryInfo));
    directoryInfo.info[0] = 2;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1//dir2", directoryInfo));
    directoryInfo.info[0] = 3;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1/dir2/dir3//", directoryInfo));

    bool isFile;
    bool isSameEntry;
    bool isParentEntry;

    // 出力用引数が元の値に依存せず関数呼び出しで書き換わることを確認するために代入します。
    isFile = true;
    isSameEntry = true;
    isParentEntry = false;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(
            &isFile, &isSameEntry, &isParentEntry, "/dir1/dir2", "/dir1/dir2/dir3/"
        )
    );
    ASSERT_FALSE(isFile);
    ASSERT_FALSE(isSameEntry);
    ASSERT_TRUE(isParentEntry);

    isFile = true;
    isSameEntry = true;
    isParentEntry = false;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(
            &isFile, &isSameEntry, &isParentEntry, "/dir1/dir2/..", "/dir1/dir2/dir3/"
        )
    );
    ASSERT_FALSE(isFile);
    ASSERT_FALSE(isSameEntry);
    ASSERT_TRUE(isParentEntry);

    isFile = true;
    isSameEntry = true;
    isParentEntry = false;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(
            &isFile, &isSameEntry, &isParentEntry, "/dir1/.", "/dir1/dir2/dir3/"
        )
    );
    ASSERT_FALSE(isFile);
    ASSERT_FALSE(isSameEntry);
    ASSERT_TRUE(isParentEntry);

    isFile = true;
    isSameEntry = true;
    isParentEntry = false;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(
            &isFile, &isSameEntry, &isParentEntry, "/dir1/./dir2", "/dir1/dir2/dir3/"
        )
    );
    ASSERT_FALSE(isFile);
    ASSERT_FALSE(isSameEntry);
    ASSERT_TRUE(isParentEntry);

    isFile = true;
    isSameEntry = true;
    isParentEntry = false;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/dir1/..", "/dir1/dir2/")
    );
    ASSERT_FALSE(isFile);
    ASSERT_FALSE(isSameEntry);
    ASSERT_TRUE(isParentEntry);
}

// ディレクトリ名の変更をテストします。
TEST(HierarchicalFileTableTest, TestRenameDirectory)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };
    HierarchicalFileTableTest::FileInfo fileInfo = { { 1 } };

    // あらかじめディレクトリを作成します。
    directoryInfo.info[0] = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1", directoryInfo));
    directoryInfo.info[0] = 2;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1//dir2", directoryInfo));
    directoryInfo.info[0] = 3;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1/dir2/dir3//", directoryInfo));
    directoryInfo.info[0] = 4;
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.CreateDirectory("/dir1/dir2/dir3///dir4dir4", directoryInfo)
    );

    // あらかじめファイルを作成します。
    fileInfo.fo.info[0] = 6;
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.CreateFile("/dir1/dir2/dir3/dir4dir4/file2file2fi", fileInfo)
    );

    bool isFile;
    bool isSameEntry;
    bool isParentEntry;

    // ディレクトリをリネームします。
    // 自分以下にはリネームできません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/dir1/dir1", "/dir1")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileTable.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/dir1/dir2/dir5", "/dir1")
    );

    // パスが長すぎます。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileTable.RenameDirectory(
            "/dir1dir1dir1dir1dir1dir1dir1dir1dir1dir1dir1dir1dir1dir1dir1dir1dir1dir1dir1dir", "/dir1"
        )
    );

    // 先頭に区切り文字がありません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathFormat,
        fileTable.RenameDirectory("/rename1", "dir1")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathFormat,
        fileTable.RenameDirectory("rename1", "/dir1")
    );

    // 普通のリネームです。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.RenameDirectory("/rename1", "/dir1"));

    // ディレクトリを付け替えます。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.RenameDirectory("/rename2", "/rename1/dir2"));

    // リネーム元が存在しません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileTable.RenameDirectory("/rename1/dir2/dir3", "/rename2/dir3")
    );

    // リネーム先のパスが長すぎます。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileTable.RenameDirectory(
            "/rename1/dir3dir3dir3dir3dir3dir3dir3dir3dir3dir3dir3dir3dir3dir3dir3dir3dir3dir3dir3dir3di", "/rename2/dir3"
        )
    );

    // パスの途中に存在しないディレクトリがあります。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileTable.RenameDirectory("/rename1", "/rename2/dir8/dir4")
    );

    // ファイルアクセスの確認をします。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.OpenFile(&fileInfo, "/rename2/dir3/dir4dir4/file2file2fi"));

    // 元に戻します。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.RenameDirectory("/dir1", "/rename1"));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.RenameDirectory("/dir1/dir2", "/rename2"));
}

//! ファイルの削除をテストします。
TEST(HierarchicalFileTableTest, TestDeleteFile)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };
    HierarchicalFileTableTest::FileInfo fileInfo = { { 1 } };

    // あらかじめディレクトリを作成します。
    directoryInfo.info[0] = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1", directoryInfo));
    directoryInfo.info[0] = 2;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1//dir2", directoryInfo));
    directoryInfo.info[0] = 3;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1/dir2/dir3//", directoryInfo));
    directoryInfo.info[0] = 4;
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.CreateDirectory("/dir1/dir2/dir3///dir4dir4", directoryInfo)
    );

    // あらかじめファイルを作成します。
    fileInfo.fo.info[0] = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir1/dir2/dir3/dir4dir4/file1", fileInfo));
    fileInfo.fo.info[0] = 2;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir1/dir2/dir3/file1", fileInfo));
    fileInfo.fo.info[0] = 3;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir1/dir2/file1", fileInfo));
    fileInfo.fo.info[0] = 4;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir1/file1", fileInfo));
    fileInfo.fo.info[0] = 5;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/file2", fileInfo));

    // ファイルを削除します。
    // パスが長すぎます。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileTable.DeleteFile("/dir1/dir2/dir3/dir4dir4/file2file2file2file2file2file2file2file2file2file2file2file2file2")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileTable.DeleteFile(
            "/dir1/dir2/dir3/dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4/file2file2fi"
        )
    );
    // 指定されたファイルが見つかりません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultFileNotFound,
        fileTable.DeleteFile("/dir1/dir2/dir3/dir4dir4/file2file2fx")
    );
    // 指定されたパスのディレクトリが存在しません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileTable.DeleteFile("/dir1/dir2/dir3/dir4dirx/file2file2fi")
    );
    // ファイルを削除します。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteFile("/dir1/dir2/dir3/dir4dir4/file1"));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteFile("/dir1/dir2/dir3/file1"));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteFile("/dir1/dir2/file1"));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteFile("/dir1/file1"));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteFile("/file2"));
    // 削除されたファイルは存在しません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultFileNotFound,
        fileTable.DeleteFile("/file2")
    );
}

//! ディレクトリの削除をテストします。
TEST(HierarchicalFileTableTest, TestDeleteDirectory)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };
    HierarchicalFileTableTest::FileInfo fileInfo = { { 1 } };

    // あらかじめディレクトリを作成します。
    directoryInfo.info[0] = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1", directoryInfo));
    directoryInfo.info[0] = 2;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1//dir2", directoryInfo));
    directoryInfo.info[0] = 3;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1/dir2/dir3//", directoryInfo));
    directoryInfo.info[0] = 4;
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.CreateDirectory("/dir1/dir2/dir3///dir4dir4", directoryInfo)
    );

    // あらかじめファイルを作成します。
    fileInfo.fo.info[0] = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir1/dir2/dir3/dir4dir4/file1", fileInfo));

    // ディレクトリを削除します。
    // ファイルが存在するディレクトリは削除できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotEmpty,
        fileTable.DeleteDirectory("/dir1/dir2/dir3/dir4dir4")
    );

    // ファイルを削除します。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteFile("/dir1/dir2/dir3/dir4dir4/file1"));

    // パスが長すぎます。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileTable.DeleteDirectory(
            "/dir1/dir2/dir3/dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4dir4x"
        )
    );
    // 中身のある状態では削除できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotEmpty,
        fileTable.DeleteDirectory("/dir1/dir2/dir3")
    );

    // 順番に消します。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteDirectory("/dir1/dir2/dir3/dir4dir4"));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteDirectory("/dir1/dir2/dir3/"));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteDirectory("/dir1/dir2/"));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteDirectory("/dir1"));

    // 中身のあるディレクトリを空にして削除します。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir0", directoryInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir0/file1", fileInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir0/file2", fileInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir0/file3", fileInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir0/file4", fileInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile("/dir0/file5", fileInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteFile("/dir0/file4"));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteFile("/dir0/file3"));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteFile("/dir0/file1"));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteFile("/dir0/file5"));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteFile("/dir0/file2"));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteDirectory("/dir0"));

    // ルートディレクトリは削除できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUndeletable,
        fileTable.DeleteDirectory("/")
    );
}

//! パスの "." の扱いをテストします。
TEST(HierarchicalFileTableTest, TestDotPath)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };

    SimpleHierarchicalFileTable::FindIndex findIndex = { 0xCC };
    nn::fssystem::dbm::DirectoryName directoryName;

    // あらかじめディレクトリを作ります。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1", directoryInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1/dir2", directoryInfo));

    {
        // ルートディレクトリを指すパスを開き、中にディレクトリ dir1 があることをテストします。
        static const nn::fssystem::dbm::PathChar* OpenPaths[] =
        {
            "/.", "/./", "/./.", "/././", "/././."
        };
        for( const auto openPath : OpenPaths )
        {
            NNT_ASSERT_RESULT_SUCCESS(fileTable.FindOpen(&findIndex, openPath));
            bool isFinished = true;
            NNT_ASSERT_RESULT_SUCCESS(
                fileTable.FindNextDirectory(&directoryName, &isFinished, &findIndex)
            );
            ASSERT_EQ(
                0,
                std::memcmp(
                    "dir1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
                    directoryName.name,
                    8
                )
            );
            ASSERT_FALSE(isFinished);
        }
    }

    {
        // /dir1 を指すパスを開き、中にディレクトリ dir2 があることをテストします。
        static const nn::fssystem::dbm::PathChar* OpenPaths[] =
        {
            "/././dir1", "/././dir1/", "/././dir1/.", "/././dir1/./"
        };
        for( const auto openPath : OpenPaths )
        {
            NNT_ASSERT_RESULT_SUCCESS(fileTable.FindOpen(&findIndex, openPath));
            bool isFinished = true;
            NNT_ASSERT_RESULT_SUCCESS(
                fileTable.FindNextDirectory(&directoryName, &isFinished, &findIndex)
            );
            ASSERT_EQ(
                0,
                std::memcmp(
                    "dir2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
                    directoryName.name,
                    8
                )
            );
            ASSERT_FALSE(isFinished);
        }
    }

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.CreateDirectory("/.", directoryInfo)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.CreateDirectory("/dir1/dir2/.", directoryInfo)
    );
}

//! パスの ".." の扱いをテストします。
TEST(HierarchicalFileTableTest, TestDoubleDotPath)
{
    // HierarchicalFileTableTemplate のオブジェクトを作成します。
    // 4 ディレクトリ、 6 ファイルは確保できるように初期化します。
    FileTableSetup fileTableSetup(4, 6);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 0 } };

    SimpleHierarchicalFileTable::FindIndex findIndex = { 0xCC };
    nn::fssystem::dbm::DirectoryName directoryName;

    // あらかじめディレクトリを作ります。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1", directoryInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1/dir2", directoryInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1/dir2/dir3", directoryInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory("/dir1/dir2/dir3/dir4", directoryInfo));

    // ルートディレクトリの親は開けません。
    {
        static const nn::fssystem::dbm::PathChar* OpenPaths[] =
        {
            "/..",
            "/dir1/../..",
            "/dir1/dir2/dir3/dir4/../../../../..",
            "/dir1/dir2/dir3/../dir3/../../dir2/../../.."
        };
        for( const auto openPath : OpenPaths )
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnobtainable,
                fileTable.FindOpen(&findIndex, openPath)
            );
        }
    }

    bool isFinished = true;

    {
        // ルートディレクトリを指すパスを開き、中にディレクトリ dir1 があることをテストします。
        static const nn::fssystem::dbm::PathChar* OpenPaths[] =
        {
            "/dir1/..", "/./dir1/../.", "/dir1/dir2/dir3/dir4/../../../../"
        };
        for( const auto openPath : OpenPaths )
        {
            NNT_ASSERT_RESULT_SUCCESS(fileTable.FindOpen(&findIndex, openPath));
            // 出力用引数が書き換わることをテストするために一度クリアします。
            isFinished = true;
            NNT_ASSERT_RESULT_SUCCESS(
                fileTable.FindNextDirectory(&directoryName, &isFinished, &findIndex)
            );
            ASSERT_EQ(
                0,
                std::memcmp(
                    "dir1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
                    directoryName.name,
                    8
                )
            );
            ASSERT_FALSE(isFinished);
        }
    }

    {
        // dir1 を指すパスを開き、中にディレクトリ dir2 があることをテストします。
        static const nn::fssystem::dbm::PathChar* OpenPaths[] =
        {
            "/dir1/dir2/..",
            "/dir1/dir2/../.",
            "//dir1/../dir1/./dir2/.././//",
            "/dir1/dir2/dir3/dir4/../../../"
        };
        for( const auto openPath : OpenPaths )
        {
            NNT_ASSERT_RESULT_SUCCESS(fileTable.FindOpen(&findIndex, openPath));
            isFinished = true;
            NNT_ASSERT_RESULT_SUCCESS(
                fileTable.FindNextDirectory(&directoryName, &isFinished, &findIndex)
            );
            ASSERT_EQ(
                0,
                std::memcmp(
                    "dir2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
                    directoryName.name,
                    8
                )
            );
            ASSERT_FALSE(isFinished);
        }
    }

    // dir3 を指すパスを開き、中にディレクトリ dir4 があることをテストします。
    NNT_ASSERT_RESULT_SUCCESS(fileTable.FindOpen(&findIndex, "/dir1/dir2/dir3/dir4/../../dir3/."));
    isFinished = true;
    NNT_ASSERT_RESULT_SUCCESS(fileTable.FindNextDirectory(&directoryName, &isFinished, &findIndex));
    ASSERT_EQ(
        0,
        std::memcmp(
            "dir4\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
            directoryName.name,
            8
        )
    );
    ASSERT_FALSE(isFinished);

    // 親ディレクトリを作ろうとして失敗することをテストします。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.CreateDirectory("/dir1/../dir1/dir2/dir3/../.", directoryInfo)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.CreateDirectory("/dir1/dir2/..", directoryInfo)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileTable.CreateDirectory("/dir1/dir2/..", directoryInfo)
    );
}

//! エントリーに空きがない時ファイルやディレクトリが作成できないことをテストします。
TEST(HierarchicalFileTableTest, TestFullEntry)
{
    // サイズ 0 を指定します。
    FileTableSetup fileTableSetup(0, 0);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    // ファイルは作れません。
    HierarchicalFileTableTest::FileInfo findIndex = { { 1 } };
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        fileTable.CreateFile("/file", findIndex)
    );

    // ブロックサイズが 128 バイト、エントリサイズが 96 バイト、予約エントリ数が 3 個なので
    // 確保するブロック数は 3 個(384 バイト) となり、384 - 96 * 3 = 192 となり、
    // エントリ 1 つ分の空き領域ができるため 1 つだけディレクトリが作成できる
    HierarchicalFileTableTest::DirectoryInfo directoryInfo = { { 2 } };
    NNT_ASSERT_RESULT_SUCCESS(
        fileTable.CreateDirectory("/dir0", directoryInfo)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        fileTable.CreateDirectory("/dir1", directoryInfo)
    );
}

//! ディレクトリ情報の読み書きをテストします。
TEST(HierarchicalFileTableTest, TestDirecotryInformation)
{
    static const uint32_t DirectoryCount = 10;

    // 非常に長いパス名を指定します。
    FileTableSetup fileTableSetup(DirectoryCount, 0);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();
    HierarchicalFileTableTest::DirectoryInfo directoryInfo, directoryInfo2;
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::String directoryFullPath = "/";

    // ランダムな名前のディレクトリ階層を作ります。
    for( int i = 0; i < DirectoryCount; ++i )
    {
        char directoryName[10];
        std::sprintf(
            directoryName, "%08X/", std::uniform_int_distribution<uint32_t>(0, 0xFFFFFFFF)(mt)
        );
        directoryFullPath += directoryName;
        // ディレクトリ情報は固定値を入れて後でテストします。
        std::memset(&directoryInfo, i & 0xFF, sizeof(directoryInfo));
        NNT_ASSERT_RESULT_SUCCESS(
            fileTable.CreateDirectory(
                directoryFullPath.c_str(),
                directoryInfo
            )
        );
    }

    // エントリーいっぱいまで作りました。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        fileTable.CreateDirectory("/dir", directoryInfo)
    );

    // データをベリファイします。
    for( int i = 0; i < DirectoryCount; ++i )
    {
        // 最後のディレクトリパスを途中までコピーして途中のディレクトのパスにします。
        nnt::fs::util::Vector<char> directoryPartName;
        directoryPartName.resize(1 + (i + 1) * 9 + 1);
        directoryFullPath.copy(&directoryPartName.at(0), directoryPartName.size() - 1);
        std::memset(&directoryInfo, i & 0xFF, sizeof(directoryInfo));
        SimpleHierarchicalFileTable::StorageIndex index;

        // ディレクトリ情報を取得して書き込んだ固定値かテストします。
        // 出力用引数が書き換わることをテストするために一度クリアします。
        std::memset(&directoryInfo2, 0, sizeof(directoryInfo2));
        NNT_ASSERT_RESULT_SUCCESS(
            fileTable.GetDirectoryInformation(
                &index,
                &directoryInfo2,
                &directoryPartName.at(0)
            )
        );
        ASSERT_EQ(0, std::memcmp(&directoryInfo, &directoryInfo2, sizeof(directoryInfo)));
    }

    // ディレクトリ情報をアップデートします。
    for( int i = 0; i < DirectoryCount; ++i )
    {
        nnt::fs::util::Vector<char> directoryPartName;
        directoryPartName.resize(1 + (i + 1) * 9 + 1);
        directoryFullPath.copy(&directoryPartName.at(0), directoryPartName.size() - 1);
        std::memset(&directoryInfo, (~i) & 0xFF, sizeof(directoryInfo));
        NNT_ASSERT_RESULT_SUCCESS(
            fileTable.UpdateDirectoryInformation(
                &directoryPartName.at(0),
                directoryInfo
            )
        );
    }

    // アップデートしたデータをベリファイします。
    for( int i = 0; i < DirectoryCount; ++i )
    {
        nnt::fs::util::Vector<char> directoryPartName;
        directoryPartName.resize(1 + (i + 1) * 9 + 1);
        directoryFullPath.copy(&directoryPartName.at(0), directoryPartName.size() - 1);
        std::memset(&directoryInfo, (~i) & 0xFF, sizeof(directoryInfo));
        SimpleHierarchicalFileTable::StorageIndex index;

        std::memset(&directoryInfo2, 0, sizeof(directoryInfo2));
        NNT_ASSERT_RESULT_SUCCESS(
            fileTable.GetDirectoryInformation(
                &index,
                &directoryInfo2,
                &directoryPartName.at(0)
            )
        );
        ASSERT_EQ(0, std::memcmp(&directoryInfo, &directoryInfo2, sizeof(directoryInfo)));
    }

    // 再帰的に削除します。
    RemoveRecursive(fileTable, "/");
}

//! ファイル情報の読み書きをテストします。
TEST(HierarchicalFileTableTest, TestFileInformation)
{
    static const uint32_t FileCount = 8;

    // 非常に長いパス名を指定します。
    FileTableSetup fileTableSetup(1, FileCount);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();
    HierarchicalFileTableTest::FileInfo findIndex, findIndex2;
    nnt::fs::util::String directoryFullPath = "/";

    // 連番の名前のファイルを複数作成します。
    for( int i = 0; i < FileCount; ++i )
    {
        char fileName[13];
        std::sprintf(fileName, "%08X%04X", i, i);
        nnt::fs::util::String fileFullPath = directoryFullPath;
        fileFullPath += fileName;
        // ファイル情報は固定値を入れて後でテストします。
        std::memset(&findIndex, i & 0xFF, sizeof(findIndex));
        NNT_ASSERT_RESULT_SUCCESS(
            fileTable.CreateFile(
                fileFullPath.c_str(),
                findIndex
            )
        );
    }

    // エントリーいっぱいまで作りました。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        fileTable.CreateFile("/file", findIndex)
    );

    // データをベリファイします。
    for( int i = 0; i < FileCount; ++i )
    {
        char fileName[13];
        std::sprintf(fileName, "%08X%04X", i, i);
        nnt::fs::util::String fileFullPath = directoryFullPath;
        fileFullPath += fileName;
        std::memset(&findIndex, i & 0xFF, sizeof(findIndex));
        // ファイル情報を取得して書き込んだ固定値かテストします。
        // 出力用引数が書き換わることをテストするために一度クリアします。
        std::memset(&findIndex2, 0, sizeof(findIndex2));
        NNT_ASSERT_RESULT_SUCCESS(fileTable.OpenFile(&findIndex2, fileFullPath.c_str()));
        ASSERT_EQ(0, std::memcmp(&findIndex, &findIndex2, sizeof(findIndex)));
    }

    // ファイル情報をアップデートします。
    for( int i = 0; i < FileCount; ++i )
    {
        char fileName[13];
        std::sprintf(fileName, "%08X%04X", i, i);
        nnt::fs::util::String fileFullPath = directoryFullPath;
        fileFullPath += fileName;
        std::memset(&findIndex, (~i) & 0xFF, sizeof(findIndex));
        NNT_ASSERT_RESULT_SUCCESS(
            fileTable.UpdateFileInformation(
                fileFullPath.c_str(),
                findIndex
            )
        );
    }

    // アップデートしたデータをベリファイします。
    for( int i = 0; i < FileCount; ++i )
    {
        char fileName[13];
        std::sprintf(fileName, "%08X%04X", i, i);
        nnt::fs::util::String fileFullPath = directoryFullPath;
        fileFullPath += fileName;
        std::memset(&findIndex, (~i) & 0xFF, sizeof(findIndex));
        std::memset(&findIndex2, 0, sizeof(findIndex2));
        NNT_ASSERT_RESULT_SUCCESS(fileTable.OpenFile(&findIndex2, fileFullPath.c_str()));
        ASSERT_EQ(0, std::memcmp(&findIndex, &findIndex2, sizeof(findIndex)));
    }

    // 再帰的に削除します。
    RemoveRecursive(fileTable, "/");
}

//! 複数のファイルやディレクトリの作成と削除をテストします。
TEST(HierarchicalFileTableTest, TestDelete)
{
    static const HierarchicalFileTableTest::FileInfo FindIndex = { { 1 } };
    static const HierarchicalFileTableTest::DirectoryInfo DirectoryInfo = { { 2 } };
    nn::fssystem::dbm::PathChar fullPath[64];

    FileTableSetup fileTableSetup(1000, 10000);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    static const int MaxLoop = 100;

    // ディレクトリを作成してその中にファイルを作成します。
    for( int i = 0; i < MaxLoop; ++i )
    {
        std::sprintf(fullPath, "/%u", i);
        NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory(fullPath, DirectoryInfo));

        for( int j = 0; j < 10; ++j )
        {
            std::sprintf(fullPath, "/%u/%u-%u", i, i, j);
            NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile(fullPath, FindIndex));
        }
    }

    // ディレクトリの中のファイルを削除してディレクトリを削除します。
    for( int i = 0; i < MaxLoop; ++i )
    {
        for( int j = 0; j < 10; ++j )
        {
            std::sprintf(fullPath, "/%u/%u-%u", i, i, j);
            NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteFile(fullPath));
        }

        std::sprintf(fullPath, "/%u", i);
        NNT_ASSERT_RESULT_SUCCESS(fileTable.DeleteDirectory(fullPath));
    }
}

//! リネームで存在しなくなったパスへのファイルやディレクトリ作成をテストします。
TEST(HierarchicalFileTableTest, TestCreateToRenamed)
{
    nn::fssystem::dbm::PathChar path[64];
    nn::fssystem::dbm::PathChar path2[64];
    static const HierarchicalFileTableTest::FileInfo FindIndex = { { 1 } };
    static const HierarchicalFileTableTest::DirectoryInfo DirectoryInfo = { { 2 } };

    FileTableSetup fileTableSetup(1000, 10000);
    SimpleHierarchicalFileTable& fileTable = fileTableSetup.GetFileTable();

    // ファイル、ディレクトリを作成してディレクトリリネームを行ないます。
    std::memset(path, 0, sizeof(path));
    std::sprintf(path, "/directory");
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory(path, DirectoryInfo));

    std::memset(path, 0, sizeof(path));
    std::sprintf(path, "/directory/file1");
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile(path, FindIndex));

    std::memset(path, 0, sizeof(path));
    std::sprintf(path, "/directory/file2");
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile(path, FindIndex));

    std::memset(path, 0, sizeof(path));
    std::sprintf(path, "/directory/file3");
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile(path, FindIndex));

    std::memset(path, 0, sizeof(path));
    std::sprintf(path, "/directory");
    std::memset(path2, 0, sizeof(path2));
    std::sprintf(path2, "/dir2");
    NNT_ASSERT_RESULT_SUCCESS(fileTable.RenameDirectory(path2, path));

    // リネームで空いたスペースにファイル、ディレクトリを作成できることを確認します。
    std::memset(path, 0, sizeof(path));
    std::sprintf(path, "/directory");
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateDirectory(path, DirectoryInfo));

    std::memset(path, 0, sizeof(path));
    std::sprintf(path, "/directory/file1");
    NNT_ASSERT_RESULT_SUCCESS(fileTable.CreateFile(path, FindIndex));
}
