﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssystem/dbm/fs_AllocationTable.h>

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/util/util_IntUtil.h>

#include <nnt/nnt_Compiler.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

void AssertValidTable(
        nn::fssystem::dbm::AllocationTable* pTable,
        uint32_t* pIndexes,
        int indexCount
    ) NN_NOEXCEPT;

//! 確保したデータ領域の情報を表す構造体です。
struct AllocateInfo
{
    uint32_t    top;        //! 確保したデータ領域の先頭インデックス
    uint32_t    blockCount; //! 確保したデータ領域のブロック数
};

//! AllocationTable とそれが使用するバッファを初期化、保持するクラスです。
class AllocationTableSetup
{
public:
    //! コンストラクタです。与えられたブロック数で AllocationTable を初期化します。
    explicit AllocationTableSetup(uint32_t blockCount) NN_NOEXCEPT
    : m_TableStore(nn::fssystem::dbm::AllocationTable::QuerySize(blockCount)),
      m_SubStorage(
        &m_TableStore, 0, nn::fssystem::dbm::AllocationTable::QuerySize(blockCount)
      ),
      m_Table()
    {
        Initialize(blockCount);
    }

    //! コンストラクタです。テーブルの拡張のためストレージを余分に確保します。
    explicit AllocationTableSetup(uint32_t blockCount, uint32_t maxBlockCount) NN_NOEXCEPT
    : m_TableStore(nn::fssystem::dbm::AllocationTable::QuerySize(maxBlockCount)),
      m_SubStorage(&m_TableStore, 0, nn::fssystem::dbm::AllocationTable::QuerySize(maxBlockCount)),
      m_Table()
    {
        NN_ASSERT_GREATER_EQUAL(maxBlockCount, blockCount);
        Initialize(blockCount);
    }

    ~AllocationTableSetup() NN_NOEXCEPT
    {
        // ストレージの範囲外アクセスが発生していないかテストします。
        NN_ASSERT(m_TableStore.CheckValid());

        m_Table.Finalize();
    }

public:
    nn::fs::SubStorage& GetSubStorage() NN_NOEXCEPT
    {
        return m_SubStorage;
    }

    nn::fssystem::dbm::AllocationTable& GetTable() NN_NOEXCEPT
    {
        return m_Table;
    }

private:
    void Initialize(uint32_t blockCount) NN_NOEXCEPT
    {
        // アロケーションテーブルをフォーマットします。
        NNT_ASSERT_RESULT_SUCCESS(m_Table.Format(m_SubStorage, blockCount));

        // ストレージの範囲外アクセスが発生していないかテストします。
        ASSERT_TRUE(m_TableStore.CheckValid());

        // ストレージをマウントします。
        m_Table.Initialize(m_SubStorage, blockCount);

        // 割り当てたサイズが空き領域数になっているかをチェックします。
        uint32_t freeCount = 0;
        NNT_ASSERT_RESULT_SUCCESS(m_Table.CalcFreeListLength(&freeCount));
        ASSERT_EQ(blockCount, freeCount);
    }

private:
    nnt::fs::util::SafeMemoryStorage m_TableStore;
    nn::fs::SubStorage m_SubStorage;
    nn::fssystem::dbm::AllocationTable m_Table;
};

//! AllocationTable のサイズ計算をテストします。
TEST(AllocationTableTest, TestQuerySize)
{
    static const uint32_t CountAllocationTable = 13;

    // sizeof(TableElement)
    static const auto SizeElement = 8;

    // nn::fssystem::dbm::AllocationTable::SectorReservedCount
    static const auto SectorReservedCount = 1;

    const int64_t sizeTable = nn::fssystem::dbm::AllocationTable::QuerySize(CountAllocationTable);

    // 容量チェック
    // データサイズ: アロケーションテーブルのエントリーサイズ * (ブロック数 + 予約領域)
    ASSERT_EQ(sizeTable, SizeElement * (CountAllocationTable + SectorReservedCount));
}

//! データ領域を確保していない初期状態から領域を確保するテストをします。
TEST(AllocationTableTest, TestAllocateFromInitialize)
{
    // 10 ブロック分のテーブルを初期化して準備します。
    static const uint32_t CountAllocationTable = 10;
    AllocationTableSetup tableSetup(CountAllocationTable);
    nn::fssystem::dbm::AllocationTable& table = tableSetup.GetTable();

    // 領域確保情報を保存する配列を用意します。
    AllocateInfo info[5];

    // エントリーを確保していきます
    uint32_t index;

    // 9 ブロック確保して、1 ブロックだけ空くようにします。

    // まず 1 ブロック確保します。
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&index, 1));
    info[0].top = index;
    info[0].blockCount = 1;

    // 空きブロックが 1 減っていることをテストします。
    uint32_t freeCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcFreeListLength(&freeCount));
    ASSERT_EQ(CountAllocationTable - 1, freeCount);

    // 2 ブロック確保します。（合計 3 ブロック）
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&index, 2));
    info[1].top = index;
    info[1].blockCount = 2;

    // 空きブロックが 3 減っていることをテストします。
    // 出力用引数が書き換わることをテストするために一度クリアします。
    freeCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcFreeListLength(&freeCount));
    ASSERT_EQ(CountAllocationTable - 3, freeCount);

    // 3 ブロック確保します。（合計 6 ブロック）
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&index, 3));
    info[2].top = index;
    info[2].blockCount = 3;

    // 空きブロックが 6 減っていることをテストします。
    freeCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcFreeListLength(&freeCount));
    ASSERT_EQ(CountAllocationTable - 6, freeCount);

    // 3 ブロック確保します。（合計 9 ブロック）
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&index, 3));
    info[3].top = index;
    info[3].blockCount = 3;

    // 空きブロックが 9 減っていることをテストします。
    freeCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcFreeListLength(&freeCount));
    ASSERT_EQ(CountAllocationTable - 9, freeCount);

    // 9 ブロック確保して、1 ブロックだけ空いた状態になりました。
    // ブロックのインデックスと、確保したブロックを持つ info の対応関係
    // block: | 0   | 1 2 | 3 4 5 | 6 7 8 | 9 |
    // info:  | [0] | [1] | [2]   | [3]   | - |

    // 2 ブロック確保しようとすると、空きがありません。
    // 1 ブロック不足しています。
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultAllocationTableFull, table.Allocate(&index, 2));

    // 1 ブロック確保します。（合計 10 ブロック）
    // 全てのエントリーを取りきりました
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&index, 1));
    info[4].top = index;
    info[4].blockCount = 1;

    // ブロックのインデックスと、確保したブロックを持つ info の対応関係
    // block: | 0   | 1 2 | 3 4 5 | 6 7 8 | 9   |
    // info:  | [0] | [1] | [2]   | [3]   | [4] |

    // 空きブロックが 0 になっていることをテストします。
    freeCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcFreeListLength(&freeCount));
    ASSERT_EQ(0, freeCount);

    // 1 ブロック確保しようとすると、空きがありません。
    // ブロック不足です
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultAllocationTableFull, table.Allocate(&index, 1));

    // イテレーション
    // 確保した領域は分断していません。
    for( int i = 0; i < 4; ++i )
    {
        if( info[i].blockCount > 0 )
        {
            // 先頭のデータ領域から次のデータ領域を取得します。
            uint32_t nextIndex = 0;
            uint32_t countIndex = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                table.ReadNext(
                    &nextIndex,
                    &countIndex,
                    info[i].top
                )
            );

            // 分断していないので次の連続ブロックチェインはありません。
            ASSERT_EQ(table.IndexEnd, nextIndex);

            // 分断していないので先頭の連続ブロックチェインのサイズが確保したサイズになります。
            ASSERT_EQ(info[i].blockCount, countIndex);
        }
    }

    // アロケーションテーブルの正当性をテストします。
    uint32_t indexList[5] = {0, 1, 3, 6, 9};
    AssertValidTable(&table, indexList, 5);
}

//! 分断した空き領域から連続した領域を確保するテストをします。
TEST(AllocationTableTest, TestAllocateUnitedFromPartitioned)
{
    // テーブルを初期化して準備します。
    AllocationTableSetup tableSetup(10);
    nn::fssystem::dbm::AllocationTable& table = tableSetup.GetTable();

    // 最初に確保するブロック数を代入した状態で AllocateInfo の配列を初期化します。
    AllocateInfo info[5] = {{0, 1}, {0, 2}, {0, 3}, {0, 3}, {0, 1}};

    // 分断した空き領域がある状態を作ります。
    // まずデータ領域を確保します。
    for( AllocateInfo& infoElement : info )
    {
        NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&infoElement.top, infoElement.blockCount));
    }

    // ブロックのインデックスと、確保したブロックを持つ info の対応関係
    // block: | 0   | 1 2 | 3 4 5 | 6 7 8 | 9   |
    // info:  | [0] | [1] | [2]   | [3]   | [4] |

    // 分断するように領域を解放します。
    NNT_ASSERT_RESULT_SUCCESS(table.Free(info[1].top));
    info[1].blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.Free(info[3].top));
    info[3].blockCount = 0;

    // ブロックのインデックスと、確保したブロックを持つ info の対応関係
    // block: | 0   | 1 2 | 3 4 5 | 6 7 8 | 9   |
    // info:  | [0] | -   | [2]   | -     | [4] |

    // 空き領域が正しいことをテストします。
    // 出力用引数が書き換わることをテストするために一度クリアします。
    uint32_t freeCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcFreeListLength(&freeCount));
    ASSERT_EQ(5, freeCount);

    // 分断されないように領域を確保します
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&info[1].top, 3));
    info[1].blockCount = 3;

    // ブロックのインデックスと、確保したブロックを持つ info の対応関係
    // block: | 0   | 1 2 | 3 4 5 | 6 7 8 | 9   |
    // info:  | [0] | -   | [2]   | [1]   | [4] |

    // 確保したブロック数分、空きが減っていることをテストします。
    freeCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcFreeListLength(&freeCount));
    ASSERT_EQ(2, freeCount);

    // 分断されずにひとつの連続ブロックチェインが確保されていることをテストします。
    // 先頭のデータ領域から次のデータ領域を取得します。
    uint32_t nextIndex = 0;
    uint32_t countIndex = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.ReadNext(&nextIndex, &countIndex, info[1].top));

    // 分断していないので次の連続ブロックチェインはありません。
    ASSERT_EQ(table.IndexEnd, nextIndex);

    // 分断していないので先頭の連続ブロックチェインのサイズが確保したサイズになります。
    ASSERT_EQ(info[1].blockCount, countIndex);

    // 空き領域に隣接する領域を解放します。
    // 解放した領域は連続化します。
    NNT_ASSERT_RESULT_SUCCESS(table.Free(info[0].top));
    info[0].blockCount = 0;

    // ブロックのインデックスと、確保したブロックを持つ info の対応関係
    // block: | 0 | 1 2 | 3 4 5 | 6 7 8 | 9   |
    // info:  | - | -   | [2]   | [1]   | [4] |

    // 解放したブロック数分の空きがあることをテストします。
    freeCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcFreeListLength(&freeCount));
    ASSERT_EQ(3, freeCount);

    // 空きは3ブロックなので4ブロックは確保できません。
    uint32_t index;
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultAllocationTableFull, table.Allocate(&index, 4));

    // 連続化した空き領域から確保します。
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&info[0].top, 3));
    info[0].blockCount = 3;

    // ブロックのインデックスと、確保したブロックを持つ info の対応関係
    // block: | 0 1 2 | 3 4 5 | 6 7 8 | 9   |
    // info:  | [0]   | [2]   | [1]   | [4] |

    // 分断されずにひとつの連続ブロックチェインが確保されていることをテストします。
    // 先頭の連続ブロックチェインが確保したブロック数になり、次の連続ブロックチェインは存在しません。
    nextIndex = 0;
    countIndex = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.ReadNext(&nextIndex, &countIndex, info[0].top));
    ASSERT_EQ(table.IndexEnd, nextIndex);
    ASSERT_EQ(info[0].blockCount, countIndex);

    // アロケーションテーブルの正当性をテストします。
    uint32_t indexList[4] = {0, 3, 6, 9};
    AssertValidTable(&table, indexList, 4);
}

//! 分断した空き領域から分断した領域を確保するテストをします。
TEST(AllocationTableTest, TestAllocatePartitionedFromPartitioned)
{
    // テーブルを初期化して準備します。
    AllocationTableSetup tableSetup(10);
    nn::fssystem::dbm::AllocationTable& table = tableSetup.GetTable();

    // 最初に確保するブロック数を代入した状態で AllocateInfo の配列を初期化します。
    AllocateInfo info[4] = {{0, 3}, {0, 3}, {0, 3}, {0, 1}};

    // 分断した空き領域がある状態を作ります。
    // まずデータ領域を確保します。
    for( AllocateInfo& infoElement : info )
    {
        NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&infoElement.top, infoElement.blockCount));
    }

    // ブロックのインデックスと、確保したブロックを持つ info の対応関係
    // block: | 0 1 2 | 3 4 5 | 6 7 8 | 9   |
    // info:  | [0]   | [1]   | [2]   | [3] |

    // 分断をさせて確保します
    // まず連続していない2つの領域を解放します。
    NNT_ASSERT_RESULT_SUCCESS(table.Free(info[0].top));
    info[0].blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.Free(info[2].top));
    info[2].blockCount = 0;

    // ブロックのインデックスと、確保したブロックを持つ info の対応関係
    // block: | 0 1 2 | 3 4 5 | 6 7 8 | 9   |
    // info:  | -     | [1]   | -     | [3] |

    // 解放したブロック数分の空きがあることをテストします。
    uint32_t freeCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcFreeListLength(&freeCount));
    ASSERT_EQ(6, freeCount);

    // 連続した領域だけでは確保できないサイズを確保します。
    uint32_t index;
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&index, 5));
    info[0].top = index;
    info[0].blockCount = 5;

    // ブロックのインデックスと、確保したブロックを持つ info の対応関係
    // block: | 0 1 2 | 3 4 5 | 6 7 | 8 | 9   |
    // info:  | [0]   | [1]   | [0] | - | [3] |

    // 確保したブロック数分、空きが減っていることをテストします。
    // 出力用引数が書き換わることをテストするために一度クリアします。
    freeCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcFreeListLength(&freeCount));
    ASSERT_EQ(1, freeCount);

    // 分断領域をイテレーションします
    uint32_t nextIndex = 0;
    uint32_t countIndex = 0;

    // 先頭の連続ブロックチェインは 3 ブロックで、その前の連続ブロックチェインは存在しません。
    NNT_ASSERT_RESULT_SUCCESS(table.ReadPrevious(&nextIndex, &countIndex, info[0].top));
    ASSERT_EQ(nextIndex, table.IndexEnd);
    ASSERT_EQ(3, countIndex);

    // 先頭の連続ブロックチェインは 3 ブロックで、その次の連続ブロックチェインが存在します。
    nextIndex = nn::fssystem::dbm::AllocationTable::IndexEnd;
    countIndex = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.ReadNext(&nextIndex, &countIndex, info[0].top));
    ASSERT_NE(nextIndex, table.IndexEnd);
    ASSERT_EQ(3, countIndex);

    // 2 番目の連続ブロックチェインが最後の連続ブロックチェインです。
    uint32_t tailIndex = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.LookupTailParentIndex(&tailIndex, info[0].top));
    ASSERT_EQ(tailIndex, nextIndex);

    // 2 番目の連続ブロックチェインは 2 ブロックで、その前の連続ブロックチェインは先頭です。
    uint32_t previousIndex = 0;
    countIndex = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.ReadPrevious(&previousIndex, &countIndex, nextIndex));
    ASSERT_EQ(info[0].top, previousIndex);
    ASSERT_EQ(2, countIndex);

    // 2 番目の連続ブロックチェインは 2 ブロックで、その次の連続ブロックチェインは存在しません。
    nextIndex = 0;
    countIndex = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.ReadNext(&nextIndex, &countIndex, nextIndex));
    ASSERT_EQ(table.IndexEnd, nextIndex);
    ASSERT_EQ(2, countIndex);

    // アロケーションテーブルの正当性をテストします。
    uint32_t indexList[3] = {3, 6, 9};
    AssertValidTable(&table, indexList, 3);
}

//! アロケーションテーブルに割り当てた領域全体の確保をテストします。
TEST(AllocationTableTest, TestAllocateAll)
{
    static const uint32_t CountAllocationTable = 4;

    // テーブルを初期化して準備します。
    AllocationTableSetup tableSetup(CountAllocationTable);
    nn::fssystem::dbm::AllocationTable& table = tableSetup.GetTable();

    // 割り当てた領域ぴったりのサイズ確保します。
    uint32_t index;
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&index, CountAllocationTable));
    ASSERT_EQ(0, index);

    // これ以上確保できないことを確認します。
    uint32_t index2;
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultAllocationTableFull, table.Allocate(&index2, 1));

    // アロケーションテーブルの正当性をテストします。
    uint32_t indexList[2] = {index, nn::fssystem::dbm::AllocationTable::IndexEnd};
    AssertValidTable(&table, indexList, 2);
}

//! 確保したデータ領域の分割をテストします。
TEST(AllocationTableTest, TestSplit)
{
    // インデックス配列を初期化します。
    uint32_t indexes[32];
    for( uint32_t& index : indexes )
    {
        index = nn::fssystem::dbm::AllocationTable::IndexEnd;
    }

    // 少量のストレージを用意します。
    static const uint32_t CountAllocationTable = 130;
    static const uint32_t CountEntry0 = 6;
    static const uint32_t CountEntry1 = 8;

    // テーブルを初期化して準備します。
    AllocationTableSetup tableSetup(CountAllocationTable);
    nn::fssystem::dbm::AllocationTable& table = tableSetup.GetTable();

    // エントリーを確保します。
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&indexes[0], CountEntry0));
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&indexes[1], CountEntry1));

    // ブロックのインデックスと、確保したブロックを持つ indexes の対応
    // block:   | 0 1 2 3 4 5 | 6 7 8 9 a b c d |
    // indexes: | [0]         | [1]             |

    // indexes[0] のデータを先頭から 5 個の位置で分割
    NNT_ASSERT_RESULT_SUCCESS(table.Split(&indexes[2], indexes[0], 5));

    // 分割された位置が正しいかテストします。
    ASSERT_EQ(5, indexes[2]);

    // 分割された前半部分のブロック数をテストします。
    uint32_t blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcTotalBlockCount(&blockCount, indexes[0]));
    ASSERT_EQ(5, blockCount);

    // 分割された後半部分のブロック数をテストします。
    // 出力用引数が書き換わることをテストするために一度クリアします。
    blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcTotalBlockCount(&blockCount, indexes[2]));
    ASSERT_EQ(CountEntry0 - 5, blockCount);

    // ブロックのインデックスと、確保したブロックを持つ indexes の対応
    // block:   | 0 1 2 3 4 | 5   | 6 7 8 9 a b c d |
    // indexes: | [0]       | [2] | [1]             |

    // アロケーションテーブルの正当性をテストします。
    AssertValidTable(&table, indexes, sizeof(indexes) / sizeof(indexes[0]));

    // indexes[1] のデータを分割します。
    NNT_ASSERT_RESULT_SUCCESS(table.Split(&indexes[3], indexes[1], 3));

    // 分割された位置が正しいかテストします。
    ASSERT_EQ(CountEntry0 + 3, indexes[3]);

    // 分割された前半部分のブロック数をテストします。
    blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcTotalBlockCount(&blockCount, indexes[1]));
    ASSERT_EQ(3, blockCount);

    // 分割された後半部分のブロック数をテストします。
    blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcTotalBlockCount(&blockCount, indexes[3]));
    ASSERT_EQ(CountEntry1 - 3, blockCount);

    // ブロックのインデックスと、確保したブロックを持つ indexes の対応
    // block:   | 0 1 2 3 4 | 5   | 6 7 8 | 9 a b c d |
    // indexes: | [0]       | [2] | [1]   | [3]       |

    // アロケーションテーブルの正当性をテストします。
    AssertValidTable(&table, indexes, sizeof(indexes) / sizeof(indexes[0]));

    // indexes[0] のデータを分割します。
    NNT_ASSERT_RESULT_SUCCESS(table.Split(&indexes[4], indexes[0], 1));

    // 分割された前半部分のブロック数をテストします。
    blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcTotalBlockCount(&blockCount, indexes[0]));
    ASSERT_EQ(1, blockCount);

    // 分割された後半部分のブロック数をテストします。
    blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.CalcTotalBlockCount(&blockCount, indexes[4]));
    ASSERT_EQ(4, blockCount);

    // ブロックのインデックスと、確保したブロックを持つ indexes の対応
    // block:   | 0   | 1 2 3 4 | 5   | 6 7 8 | 9 a b c d |
    // indexes: | [0] | [4]     | [2] | [1]   | [3]       |

    // アロケーションテーブルの正当性をテストします。
    AssertValidTable(&table, indexes, sizeof(indexes) / sizeof(indexes[0]));
}

//! アロケーションテーブルの拡張をテストします。
TEST(AllocationTableTest, TestExpand)
{
    // サイズ 1 のアロケーションテーブルを生成します。
    static const uint32_t CountAllocationTable = 1;
    static const uint32_t CountExpand = 3;

    // インデックス配列を初期化します。
    uint32_t indexes[64];
    for( uint32_t& index : indexes )
    {
        index = nn::fssystem::dbm::AllocationTable::IndexEnd;
    }

    // テーブルを初期化して準備します。
    AllocationTableSetup tableSetup(CountAllocationTable, CountAllocationTable + CountExpand * 3);
    nn::fssystem::dbm::AllocationTable& table = tableSetup.GetTable();
    nn::fs::SubStorage& subStorage = tableSetup.GetSubStorage();

    // 唯一のエントリーを確保します。
    int indexCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&indexes[indexCount], CountAllocationTable));
    ++indexCount;

     // ブロックのインデックスと、確保したブロックを持つ indexes の対応
    // block:   | 0   |
    // indexes: | [0] |

    // これ以上エントリーは確保できません。
    uint32_t temporaryIndex;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        table.Allocate(&temporaryIndex, 1)
    );

    // アロケーションテーブルを拡張します。
    table.Finalize();
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::dbm::AllocationTable::Expand(
            subStorage,
            CountAllocationTable,
            CountAllocationTable + CountExpand
        )
    );
    table.Initialize(subStorage, CountAllocationTable + CountExpand);

    // テーブルの正当性をテストします。
    AssertValidTable(&table, indexes, sizeof(indexes) / sizeof(indexes[0]));

    // ブロックのインデックスと、確保したブロックを持つ indexes の対応
    // block:   | 0   | 1 2 3 |
    // indexes: | [0] | -     |

    // 拡張した個数 - 1 個のエントリーを確保します。
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&indexes[indexCount], CountExpand - 1));
    ++indexCount;

    // ブロックのインデックスと、確保したブロックを持つ indexes の対応
    // block:   | 0   | 1 2 | 3 |
    // indexes: | [0] | [1] | - |

    // 拡張した個数ぴったりの個数エントリーが確保できるか確認します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        table.Allocate(&temporaryIndex, 2)
    );
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&indexes[indexCount], 1));
    ++indexCount;

    // ブロックのインデックスと、確保したブロックを持つ indexes の対応
    // block:   | 0   | 1 2 | 3   |
    // indexes: | [0] | [1] | [2] |

    // テーブルの正当性をテストします。
    AssertValidTable(&table, indexes, sizeof(indexes) / sizeof(indexes[0]));

    // 適当な箇所に空き領域がある状態で再度拡張します。
    // まず、適当な箇所を解放します。
    NNT_ASSERT_RESULT_SUCCESS(table.Free(indexes[0]));
    indexes[0] = nn::fssystem::dbm::AllocationTable::IndexEnd;

    // アロケーションテーブルを再度拡張します。
    table.Finalize();
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::dbm::AllocationTable::Expand(
            subStorage,
            CountAllocationTable + CountExpand,
            CountAllocationTable + CountExpand * 2
        )
    );
    table.Initialize(subStorage, CountAllocationTable + CountExpand * 2);

    // ブロックのインデックスと、確保したブロックを持つ indexes の対応
    // block:   | 0 | 1 2 | 3   | 4 5 6 |
    // indexes: | - | [1] | [2] | -     |

    // テーブルの正当性をテストします。
    AssertValidTable(&table, indexes, sizeof(indexes) / sizeof(indexes[0]));

    // 拡張した個数+解放した個数ぴったりの個数エントリーが確保できるか確認します。
    for( uint32_t i = 0; i < CountExpand + 1; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&indexes[indexCount], 1));
        ++indexCount;
    }
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        table.Allocate(&temporaryIndex, 1)
    );

    // ブロックのインデックスと、確保したブロックを持つ indexes の対応
    // block:   | 0   | 1 2 | 3   | 4   | 5   | 6   |
    // indexes: | [3] | [1] | [2] | [4] | [5] | [6] |

    // テーブルの正当性をテストします。
    AssertValidTable(&table, indexes, sizeof(indexes) / sizeof(indexes[0]));

    // 確保したブロックを解放します。
    for( uint32_t& index : indexes )
    {
        if( index != nn::fssystem::dbm::AllocationTable::IndexEnd )
        {
            NNT_ASSERT_RESULT_SUCCESS(table.Free(index));
            index = nn::fssystem::dbm::AllocationTable::IndexEnd;
        }
    }

    // テーブルの正当性をテストします。
    AssertValidTable(&table, indexes, sizeof(indexes) / sizeof(indexes[0]));

    // 最大領域サイズを取得します。
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&indexes[0], CountAllocationTable + CountExpand * 2));
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        table.Allocate(&temporaryIndex, 1)
    );
    NNT_ASSERT_RESULT_SUCCESS(table.Free(indexes[0]));

    // 再度拡張します。
    table.Finalize();
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::dbm::AllocationTable::Expand(
            subStorage,
            CountAllocationTable + CountExpand * 2,
            CountAllocationTable + CountExpand * 3
        )
    );
    table.Initialize(subStorage, CountAllocationTable + CountExpand * 3);

    // 領域を確保します。
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&indexes[0], CountExpand + 1));

    // アロケーションテーブルの正当性をテストします。
    AssertValidTable(&table, indexes, sizeof(indexes) / sizeof(indexes[0]));
}

//! データ領域の連結をテストします。
TEST(AllocationTableTest, TestConcat)
{
    // テーブルを初期化して準備します。
    AllocationTableSetup tableSetup(10);
    nn::fssystem::dbm::AllocationTable& table = tableSetup.GetTable();

    // 最初に確保するブロック数を代入した状態で AllocateInfo の配列を初期化します。
    AllocateInfo info[] = {{0, 1}, {0, 2}, {0, 3}, {0, 1}};

    // 分断した空き領域がある状態を作ります。
    // まずデータ領域を確保します。
    for( AllocateInfo& infoElement : info )
    {
        NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&infoElement.top, infoElement.blockCount));
    }

    // ブロックのインデックスと、確保したブロックを持つ info の対応関係
    // block: | 0   | 1 2 | 3 4 5 | 6   | 7 8 9 |
    // info:  | [0] | [1] | [2]   | [3] | -     |
    // 確保したデータ領域のブロック: {0}, {1, 2}, {3, 4, 5}, {6}

    // 確保した領域を連結します。
    NNT_ASSERT_RESULT_SUCCESS(table.Concat(info[0].top, info[2].top));

    // 連結したデータ領域: {0} → {3, 4, 5}

    // 連結したデータ領域のイテレーション
    uint32_t nextIndex;
    uint32_t blockCount;

    // 1 番目の連続ブロックチェインの情報と 2 番目の連続ブロックチェインを取得
    // 出力用引数が書き換わることをテストするために一度クリアします。
    nextIndex = 0;
    blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.ReadNext(&nextIndex, &blockCount, info[0].top));

    // 1 番目の連続ブロックチェインのブロック数をチェック
    ASSERT_EQ(info[0].blockCount, blockCount);

    // 2 番目の連続ブロックチェインは info[2] で確保していた領域
    ASSERT_EQ(info[2].top, nextIndex);

    // 2 番目の連続ブロックチェインの情報と次の連続ブロックチェインを取得
    blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.ReadNext(&nextIndex, &blockCount, nextIndex));

    // 2 番目の連続ブロックチェインのブロック数をチェック
    ASSERT_EQ(info[2].blockCount, blockCount);

    // 2 番目の連続ブロックチェインが最後
    ASSERT_EQ(table.IndexEnd, nextIndex);

    // 途中の連続ブロックチェインを連結
    NNT_ASSERT_RESULT_SUCCESS(table.Concat(info[3].top, info[2].top));

    // 連結したデータ領域: {6} → {3, 4, 5}

    // 連続ブロックチェインのつながりとブロック数をテスト
    nextIndex = 0;
    blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.ReadNext(&nextIndex, &blockCount, info[3].top));
    ASSERT_EQ(blockCount, info[3].blockCount);
    ASSERT_EQ(nextIndex, info[2].top);

    // 最初に連結したデータ領域は変化しません。
    // {0} → {3, 4, 5}
    nextIndex = 0;
    blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.ReadNext(&nextIndex, &blockCount, info[0].top));
    ASSERT_EQ(info[0].blockCount, blockCount);
    ASSERT_EQ(info[2].top, nextIndex);

    // さらに連結します。
    NNT_ASSERT_RESULT_SUCCESS(table.Concat(info[1].top, info[0].top));

    // 連結したデータ領域: {1, 2} → {0} → {3, 4, 5}

    // 連結したデータ領域のイテレーション
    // 1 番目の連続ブロックチェインの情報と 2 番目の連続ブロックチェインを取得
    nextIndex = 0;
    blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.ReadNext(&nextIndex, &blockCount, info[1].top));

    // 1 番目の連続ブロックチェインのブロック数をチェック
    ASSERT_EQ(info[1].blockCount, blockCount);

    // 2 番目の連続ブロックチェインは info[0] で確保していた領域
    ASSERT_EQ(info[0].top, nextIndex);

    // 2 番目の連続ブロックチェインの情報と 3 番目の連続ブロックチェインを取得
    blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.ReadNext(&nextIndex, &blockCount, nextIndex));

    // 2 番目の連続ブロックチェインのブロック数をチェック
    ASSERT_EQ(info[0].blockCount, blockCount);

    // 3 番目の連続ブロックチェインは info[2] で確保していた領域
    ASSERT_EQ(info[2].top, nextIndex);

    // 3 番目の連続ブロックチェインの情報とその次の連続ブロックチェインを取得
    blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.ReadNext(&nextIndex, &blockCount, nextIndex));

    // 3 番目の連続ブロックチェインのブロック数をチェック
    ASSERT_EQ(info[2].blockCount, blockCount);

    // 3 番目の連続ブロックチェインが最後
    ASSERT_EQ(table.IndexEnd, nextIndex);

    // 空き領域を連結
    uint32_t freeIndex = nn::fssystem::dbm::AllocationTable::IndexEnd;
    NNT_ASSERT_RESULT_SUCCESS(table.ReadFreeListHead(&freeIndex));
    ASSERT_NE(table.IndexEnd, freeIndex);
    NNT_ASSERT_RESULT_SUCCESS(table.Concat(freeIndex, info[1].top));

    // 連結したデータ領域: {7, 8, 9} → {1, 2} → {0} → {3, 4, 5}

    // 連続ブロックチェインのつながりとブロック数をテスト
    nextIndex = 0;
    blockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(table.ReadNext(&nextIndex, &blockCount, freeIndex));
    ASSERT_EQ(blockCount, 3);
    ASSERT_EQ(nextIndex, info[1].top);
}

//! 複数ブロックにまたがるアロケーションに失敗した時の挙動をテストします。
TEST(AllocationTableTest, TestAllocationTableFull)
{
    // テーブルを初期化して準備します。
    AllocationTableSetup tableSetup(12);
    nn::fssystem::dbm::AllocationTable& table = tableSetup.GetTable();

    uint32_t indexList[4];
    // データ領域を確保してから解放することで、連続した空き領域を作成します。
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&indexList[0], 2));
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&indexList[1], 1));
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&indexList[2], 4));
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&indexList[3], 5));

    NNT_ASSERT_RESULT_SUCCESS(table.Free(indexList[2]));
    NNT_ASSERT_RESULT_SUCCESS(table.Free(indexList[1]));
    indexList[1] = nn::fssystem::dbm::AllocationTable::IndexEnd;
    indexList[2] = nn::fssystem::dbm::AllocationTable::IndexEnd;

    // ブロックのインデックスと、確保したブロックを持つ indexList の対応関係
    // block:      | 0 1 | 2 | 3 4 5 6 | 7 8 9 a b |
    // indexList:  | [0] | - | -       | [3]       |

    // アロケーションテーブルの正当性をテストします。
    AssertValidTable(&table, indexList, 4);

    // 空き領域は 5 ブロックなので、6 ブロックは確保できない
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        table.Allocate(&indexList[1], 6)
    );

    // アロケーションテーブルの正当性をテストします。
    AssertValidTable(&table, indexList, 4);

    // 空き領域は 5 ブロックなので、5 ブロックは確保できる
    NNT_ASSERT_RESULT_SUCCESS(table.Allocate(&indexList[1], 5));

    // アロケーションテーブルの正当性をテストします。
    AssertValidTable(&table, indexList, 4);
}

NNT_DISABLE_OPTIMIZATION
/**
*  ランダムに以下の操作を行いテストします。
*   - データ領域の解放と確保を行います。
*   - 確保済みのデータ領域を解放します。
*   - データ領域の分割を行います。
*   - アロケーションテーブルを拡張します。
*/
TEST(AllocationTableTest, TestRandom)
{
    // データの確保サイズを定義します。
    static const size_t MaxBlockCount = 2048;

    uint32_t countTable = 128;

    // テーブルを初期化して準備します。
    AllocationTableSetup tableSetup(countTable, MaxBlockCount);
    nn::fssystem::dbm::AllocationTable& table = tableSetup.GetTable();
    nn::fs::SubStorage& subStorage = tableSetup.GetSubStorage();

    // インデックス配列を初期化します。
    static const int SizeIndexes = 64;
    uint32_t indexes[SizeIndexes];
    for( uint32_t& index : indexes )
    {
        index = nn::fssystem::dbm::AllocationTable::IndexEnd;
    }

    // ランダムにデータ領域への操作を行います。
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    for( int i = 0; i < 100000; ++i )
    {
        if( i % 3000 == 0 )
        {
            // アロケーションテーブルの整合性が崩れていないことをテストします。
            AssertValidTable(&table, indexes, SizeIndexes);
        }

        switch( std::uniform_int_distribution<>(0, 4)(mt) )
        {
        case 0:
            {
                // データ領域を解放して新しく確保し、領域が足りなければ拡張します。
                // 確保するデータ領域をランダムに選びます。
                const int index = std::uniform_int_distribution<int>(0, SizeIndexes - 1)(mt);

                // すでに確保済みのデータ領域を解放します。
                if( indexes[index] != nn::fssystem::dbm::AllocationTable::IndexEnd )
                {
                    NNT_ASSERT_RESULT_SUCCESS(table.Free(indexes[index]));
                    indexes[index] = nn::fssystem::dbm::AllocationTable::IndexEnd;
                }

                // データ領域を解放した位置に新しい領域を確保します。
                const int size = std::uniform_int_distribution<int>(1, 20)(mt);
                const nn::Result result = table.Allocate(&indexes[index], size);

                if( nn::fs::ResultAllocationTableFull::Includes(result) )
                {
                    // 領域が足りなくなれば拡張します。
                    table.Finalize();
                    const uint32_t prevCount = countTable;
                    // 拡張サイズをランダムに決定します。
                    countTable += std::uniform_int_distribution<uint32_t>(0, 127)(mt) + 20;
                    if( countTable > MaxBlockCount )
                    {
                        countTable = MaxBlockCount;
                    }
                    if( prevCount != countTable )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(table.Expand(subStorage, prevCount, countTable));
                    }
                    table.Initialize(subStorage, countTable);
                }
            }
            break;

        case 1:
            {
                // すでに確保済みのデータ領域を解放します。
                const int index = std::uniform_int_distribution<int>(0, SizeIndexes - 1)(mt);
                if( indexes[index] != nn::fssystem::dbm::AllocationTable::IndexEnd )
                {
                    NNT_ASSERT_RESULT_SUCCESS(table.Free(indexes[index]));
                    indexes[index] = nn::fssystem::dbm::AllocationTable::IndexEnd;
                }
            }
            break;

        case 2:
        case 3:
            {
                // データ領域を分割します。
                // 分割するデータ領域をランダムに選びます。
                const int index0 = std::uniform_int_distribution<int>(0, SizeIndexes - 1)(mt);
                const int index1 = std::uniform_int_distribution<int>(0, SizeIndexes - 1)(mt);
                const bool isIndexEnd
                    = indexes[index0] == nn::fssystem::dbm::AllocationTable::IndexEnd;
                if( (index0 != index1) && !isIndexEnd )
                {
                    // すでに確保済みのデータ領域を解放します。
                    if( indexes[index1] != nn::fssystem::dbm::AllocationTable::IndexEnd )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(table.Free(indexes[index1]));
                        indexes[index1] = nn::fssystem::dbm::AllocationTable::IndexEnd;
                    }

                    // 既存のデータ領域を分割します。
                    uint32_t blockCount = 0;
                    nn::Result result = table.CalcTotalBlockCount(&blockCount, indexes[index0]);
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    if( blockCount >= 2 )
                    {
                        // 2 つの領域がどちらも 1 以上になるように
                        // 分割後のサイズをランダムに決定します。
                        const uint32_t count1 = std::uniform_int_distribution<uint32_t>(
                                                    1, blockCount - 1
                                                )(mt);
                        const uint32_t count2 = blockCount - count1;
                        // count1、count2のブロック個数で分割します
                        NNT_ASSERT_RESULT_SUCCESS(
                            table.Split(
                                &indexes[index1],
                                indexes[index0],
                                count1
                            )
                        );
                        // 分割した領域前後のサイズを確認します
                        // 出力用引数が書き換わることをテストするために一度クリアします。
                        blockCount = 0;
                        result = table.CalcTotalBlockCount(&blockCount, indexes[index0]);
                        NNT_ASSERT_RESULT_SUCCESS(result);
                        ASSERT_EQ(count1, blockCount);
                        blockCount = 0;
                        result = table.CalcTotalBlockCount(&blockCount, indexes[index1]);
                        NNT_ASSERT_RESULT_SUCCESS(result);
                        ASSERT_EQ(count2, blockCount);
                    }
                }
            }
            break;

        case 4:
            {
                // 領域を拡張します。
                table.Finalize();
                const uint32_t prevCount = countTable;
                // 拡張する領域をランダムに決定します。
                countTable += std::uniform_int_distribution<uint32_t>(0, 9)(mt);
                if( countTable > MaxBlockCount )
                {
                    countTable = MaxBlockCount;
                }
                if( prevCount != countTable )
                {
                    NNT_ASSERT_RESULT_SUCCESS(table.Expand(subStorage, prevCount, countTable));
                }
                table.Initialize(subStorage, countTable);
            }
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    // 終了
    table.Finalize();
} // NOLINT(impl/function_size)
NNT_RESTORE_OPTIMIZATION

/**
* @brief        アロケーションテーブル内部データの正当性をテストします。
*
* @param[in]    pTable      アロケーションテーブル
* @param[in]    pIndexes    データ領域の先頭インデックスの配列
* @param[in]    indexCount  配列 pIndexes の長さ
*
* @details      テスト内容:
*   - 引数で与えられたデータ領域の連続ブロックチェインが正しくつながっているか。
*   - 引数で与えられたテーブルの空き領域の連続ブロックチェインが正しくつながっているか。
*/
void AssertValidTable(
        nn::fssystem::dbm::AllocationTable* pTable,
        uint32_t* pIndexes,
        int indexCount
    ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pTable);

    ASSERT_TRUE(nn::util::IsIntValueRepresentable<int>(pTable->GetBlockCount()));

    const int totalBlockCount = pTable->GetBlockCount();

    // 各ブロック毎の参照フラグを初期化します。
    nnt::fs::util::Vector<bool> checkExists(totalBlockCount, false);

    // 順方向にイテレーションを行い、ブロックの参照状態を更新します。
    for( int i = 0; i < indexCount; ++i )
    {
        uint32_t headIndex = pIndexes[i];
        while( headIndex != nn::fssystem::dbm::AllocationTable::IndexEnd )
        {
            ASSERT_TRUE(nn::util::IsIntValueRepresentable<int>(headIndex));
            uint32_t nextIndex = 0;
            uint32_t blockCount = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                pTable->ReadNext(&nextIndex, &blockCount, headIndex)
            );
            ASSERT_TRUE(nn::util::IsIntValueRepresentable<int>(headIndex + blockCount));
            const int indexLast = headIndex + blockCount;
            for( int index = headIndex; index < indexLast; ++index )
            {
                ASSERT_TRUE((0 <= index) && (index < totalBlockCount));

                // 同じインデックスが複数回登場していないか確認します。
                ASSERT_FALSE(checkExists[index]);

                // 参照されたことを覚えておきます。
                checkExists[index] = true;
            }
            headIndex = nextIndex;
        }
    }

    // フリーリストに対して順方向にイテレーションを行い、ブロックの参照状態を更新します。
    uint32_t freeHeadIndex = nn::fssystem::dbm::AllocationTable::IndexEnd;
    NNT_ASSERT_RESULT_SUCCESS(pTable->ReadFreeListHead(&freeHeadIndex));
    while( freeHeadIndex != nn::fssystem::dbm::AllocationTable::IndexEnd )
    {
        ASSERT_TRUE(nn::util::IsIntValueRepresentable<int>(freeHeadIndex));
        uint32_t nextIndex = nn::fssystem::dbm::AllocationTable::IndexEnd;
        uint32_t blockCount = 0;
        NNT_ASSERT_RESULT_SUCCESS(pTable->ReadNext(&nextIndex, &blockCount, freeHeadIndex));
        ASSERT_TRUE(nn::util::IsIntValueRepresentable<int>(freeHeadIndex + blockCount));
        const int indexLast = freeHeadIndex + blockCount;
        for( int index = freeHeadIndex; index < indexLast; ++index )
        {
            ASSERT_TRUE((0 <= index) && (index < totalBlockCount));

            // 同じインデックスが複数回登場していないか確認します。
            ASSERT_FALSE(checkExists[index]);

            // 参照されたことを覚えておきます。
            checkExists[index] = true;
        }
        freeHeadIndex = nextIndex;
    }

    // 参照されていないブロックが存在しないことを確認します。
    for( const auto checkExist : checkExists )
    {
        ASSERT_TRUE(checkExist);
    }

    // 各ブロック毎の参照フラグをリセットします。
    std::fill(checkExists.begin(), checkExists.end(), false);

    // 最後の連続領域から逆方向にイテレーションを行い、ブロックの参照状態を更新します。
    uint32_t tailIndex = nn::fssystem::dbm::AllocationTable::IndexEnd;
    for( int i = 0; i < indexCount; ++i )
    {
        // 最後の連続領域を取得して tailIndex に代入します。
        uint32_t headIndex = pIndexes[i];
        while( headIndex != nn::fssystem::dbm::AllocationTable::IndexEnd )
        {
            uint32_t nextIndex = nn::fssystem::dbm::AllocationTable::IndexEnd;
            uint32_t blockCount = 0;
            NNT_ASSERT_RESULT_SUCCESS(pTable->ReadNext(&nextIndex, &blockCount, headIndex));
            tailIndex = headIndex;
            headIndex = nextIndex;
        }

        // 逆方向のイテレーションを行います。
        headIndex = tailIndex;
        while( headIndex != nn::fssystem::dbm::AllocationTable::IndexEnd )
        {
            ASSERT_TRUE(nn::util::IsIntValueRepresentable<int>(headIndex));
            uint32_t previousIndex = 0;
            uint32_t blockCount = 0;
            NNT_ASSERT_RESULT_SUCCESS(pTable->ReadPrevious(&previousIndex, &blockCount, headIndex));
            ASSERT_TRUE(nn::util::IsIntValueRepresentable<int>(headIndex + blockCount));
            const int indexLast = headIndex + blockCount;
            for( int index = headIndex; index < indexLast; ++index )
            {
                ASSERT_TRUE((0 <= index) && (index < totalBlockCount));

                // 同じインデックスが複数回登場していないか確認します。
                ASSERT_FALSE(checkExists[index]);

                // 参照されたことを覚えておきます。
                checkExists[index] = true;
            }
            headIndex = previousIndex;
        }

        tailIndex = nn::fssystem::dbm::AllocationTable::IndexEnd;
    }

    // フリーリストに対して逆方向にイテレーションを行い、ブロックの参照状態を更新します。
    // 出力用引数が書き換わることをテストするために一度クリアします。
    tailIndex = nn::fssystem::dbm::AllocationTable::IndexEnd;
    freeHeadIndex = nn::fssystem::dbm::AllocationTable::IndexEnd;
    NNT_ASSERT_RESULT_SUCCESS(pTable->ReadFreeListHead(&freeHeadIndex));

    // 最後の連続領域を取得して tailIndex に代入します。
    while( freeHeadIndex != nn::fssystem::dbm::AllocationTable::IndexEnd )
    {
        uint32_t nextIndex = nn::fssystem::dbm::AllocationTable::IndexEnd;
        uint32_t blockCount = 0;
        NNT_ASSERT_RESULT_SUCCESS(pTable->ReadNext(&nextIndex, &blockCount, freeHeadIndex));
        tailIndex = freeHeadIndex;
        freeHeadIndex = nextIndex;
    }

    // 逆方向のイテレーションを行います。
    freeHeadIndex = tailIndex;
    while( freeHeadIndex != nn::fssystem::dbm::AllocationTable::IndexEnd )
    {
        ASSERT_TRUE(nn::util::IsIntValueRepresentable<int>(freeHeadIndex));
        uint32_t previousIndex = 0;
        uint32_t blockCount = 0;
        NNT_ASSERT_RESULT_SUCCESS(pTable->ReadPrevious(&previousIndex, &blockCount, freeHeadIndex));
        ASSERT_TRUE(nn::util::IsIntValueRepresentable<int>(freeHeadIndex + blockCount));
        const int indexLast = freeHeadIndex + blockCount;
        for( int index = freeHeadIndex; index < indexLast; ++index )
        {
            ASSERT_TRUE((0 <= index) && (index < totalBlockCount));

            // 同じインデックスが複数回登場していないか確認します。
            ASSERT_FALSE(checkExists[index]);

            // 参照されたことを覚えておきます。
            checkExists[index] = true;
        }
        freeHeadIndex = previousIndex;
    }

    // 参照されていないブロックが存在しないことを確認します。
    for( int i = 0; i < totalBlockCount; ++i )
    {
        ASSERT_TRUE(checkExists[i]);
    }
} // NOLINT(impl/function_size)
