﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>
#include <random>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_Stress_AccessFilesTestCase.h"

namespace nnt { namespace fs {

class WriteFilesTestCase : public AccessFilesTestCase
{
public:
    static const auto FileSize = 1 * 1024;

public:
    WriteFilesTestCase(int threadCount, int fileCount, bool append) NN_NOEXCEPT
        : AccessFilesTestCase(
            threadCount,
            fileCount,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Write
                | (append ? nn::fs::OpenMode_AllowAppend : 0))),
          m_Append(append)
    {
    }

    virtual ~WriteFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);

        ASSERT_FALSE(pTest->IsReadOnly(GetTestCaseIndex()));
        ASSERT_LE(GetEntryCount() * GetThreadCount(), FsStressTest::EntryCountMax);
        std::unique_ptr<char[]> buffer(new char[FileSize]);
        std::iota(buffer.get(), buffer.get() + FileSize, static_cast<char>(threadIndex));
        for( auto entryIndex = 0; entryIndex < GetEntryCount(); ++entryIndex )
        {
            auto succeeded = false;
            WriteFile(&succeeded, buffer.get(), FileSize, GetEntryIndex(threadIndex, entryIndex));
            ASSERT_TRUE(succeeded);
        }

        Succeed(threadIndex);
    }

protected:
    virtual void SetUpFile(bool* outSucceeded, int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        if( !m_Append )
        {
            const auto file = GetFile(entryIndex);

            NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetFileSize(file, FileSize));
        }

        *outSucceeded = true;
    }

    virtual void WriteFile(
        bool* outSucceeded,
        const char* buffer,
        size_t size,
        int entryIndex) NN_NOEXCEPT = 0;

private:
    bool m_Append;
};

NN_DEFINE_STATIC_CONSTANT(const int WriteFilesTestCase::FileSize);

class InBulkWriteFilesTestCase : public WriteFilesTestCase
{
public:
    InBulkWriteFilesTestCase(int threadCount, int fileCount) NN_NOEXCEPT
        : WriteFilesTestCase(threadCount, fileCount, false)
    {
    }

    virtual ~InBulkWriteFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 5;
    }

protected:
    virtual void WriteFile(
        bool* outSucceeded,
        const char* buffer,
        size_t size,
        int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteFile(file, 0, buffer, size, nn::fs::WriteOption()));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(file));

        *outSucceeded = true;
    }
};

class SequentialWriteFilesTestCase : public WriteFilesTestCase
{
public:
    SequentialWriteFilesTestCase(
        int threadCount,
        int fileCount,
        bool append,
        bool flush) NN_NOEXCEPT
        : WriteFilesTestCase(threadCount, fileCount, append),
          m_Append(append),
          m_Flush(flush)
    {
    }

    virtual ~SequentialWriteFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Append ? 1 : 3;
    }

protected:
    virtual void WriteFile(
        bool* outSucceeded,
        const char* buffer,
        size_t bufferSize,
        int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        for( int64_t offset = 0; offset < static_cast<int64_t>(bufferSize); ++offset )
        {
            const auto option = m_Flush && ((offset + 1) % 1024) == 0
                ? nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)
                : nn::fs::WriteOption();
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteFile(
                file,
                offset,
                buffer + offset,
                1,
                option));
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(file));

        *outSucceeded = true;
    }

private:
    bool m_Append;
    bool m_Flush;
};

class RandomWriteFilesTestCase : public WriteFilesTestCase
{
public:
    RandomWriteFilesTestCase(
        int threadCount,
        int fileCount,
        bool append,
        bool flush) NN_NOEXCEPT
        : WriteFilesTestCase(threadCount, fileCount, append),
          m_Flush(flush)
    {
    }

    virtual ~RandomWriteFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 3;
    }

protected:
    virtual void WriteFile(
        bool* outSucceeded,
        const char* buffer,
        size_t size,
        int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        std::unique_ptr<std::mt19937> rng(new std::mt19937(nnt::fs::util::GetRandomSeed()));
        for( size_t count = 0; count < size; ++count )
        {
            const auto offset = std::uniform_int_distribution<int64_t>(0, size - 1)(*rng);
            const auto option = m_Flush && ((count  + 1) % 1024) == 0
                ? nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)
                : nn::fs::WriteOption();
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteFile(
                file,
                offset,
                buffer + offset,
                1,
                option));
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(file));

        *outSucceeded = true;
    }

private:
    bool m_Flush;
};

class LimitWriteFiles : public AccessFilesTestCase
{
public:
    LimitWriteFiles() NN_NOEXCEPT
        : AccessFilesTestCase(
            FsStressTest::ThreadCountMax,
            5,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend))
    {
    }

    virtual ~LimitWriteFiles() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 5;
    }

    virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);

        ASSERT_FALSE(pTest->IsReadOnly(GetTestCaseIndex()));
        ASSERT_TRUE(pTest->IsSaveData(GetTestCaseIndex()));
        ASSERT_LE(GetEntryCount() * GetThreadCount(), FsStressTest::EntryCountMax);

        int64_t totalSize = 0;
        {
            char path[PathLength];
            nn::util::SNPrintf(path, PathLength, "%s:/", GetMountName());
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetTotalSpaceSize(&totalSize, path));
        }

        const auto fileSize = totalSize / (GetThreadCount() * GetEntryCount());
        std::unique_ptr<char[]> buffer(new char[static_cast<size_t>(fileSize)]);
        std::iota(buffer.get(), buffer.get() + fileSize, static_cast<char>(threadIndex));
        for( auto entryIndex = 0; entryIndex < GetEntryCount(); ++entryIndex )
        {
            char path[PathLength];
            MakePath(path, GetEntryIndex(threadIndex, entryIndex));
            const auto file = GetFile(GetEntryIndex(threadIndex, entryIndex));
            int64_t freeSize = 0;

            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFreeSpaceSize(&freeSize, path));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(
                file,
                0,
                buffer.get(),
                static_cast<size_t>(std::min(fileSize, freeSize)),
                nn::fs::WriteOption()));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(file));
        }

        Succeed(threadIndex);
    }

protected:
    virtual void SetUpFile(bool* outSucceeded, int) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = true;
    }
};

class InBulkWriteSingleFile : public InBulkWriteFilesTestCase
{
public:
    InBulkWriteSingleFile() NN_NOEXCEPT
        : InBulkWriteFilesTestCase(FsStressTest::ThreadCountMax, 1)
    {
    }
};

class SequentialWriteSingleFile : public SequentialWriteFilesTestCase
{
public:
    SequentialWriteSingleFile() NN_NOEXCEPT
        : SequentialWriteFilesTestCase(FsStressTest::ThreadCountMax, 1, false, false)
    {
    }
};

class RandomWriteSingleFile : public RandomWriteFilesTestCase
{
public:
    RandomWriteSingleFile() NN_NOEXCEPT
        : RandomWriteFilesTestCase(FsStressTest::ThreadCountMax, 1, false, false)
    {
    }
};

class InBulkWriteNumerousFiles : public InBulkWriteFilesTestCase
{
public:
    InBulkWriteNumerousFiles() NN_NOEXCEPT
        : InBulkWriteFilesTestCase(FsStressTest::ThreadCountMax, 10)
    {
    }
};

class SequentialWriteNumerousFiles : public SequentialWriteFilesTestCase
{
public:
    SequentialWriteNumerousFiles() NN_NOEXCEPT
        : SequentialWriteFilesTestCase(FsStressTest::ThreadCountMax, 10, false, false)
    {
    }
};

class RandomWriteNumerousFiles : public RandomWriteFilesTestCase
{
public:
    RandomWriteNumerousFiles() NN_NOEXCEPT
        : RandomWriteFilesTestCase(FsStressTest::ThreadCountMax, 10, false, false)
    {
    }
};

class SequentialAppendFiles : public SequentialWriteFilesTestCase
{
public:
    SequentialAppendFiles() NN_NOEXCEPT
        : SequentialWriteFilesTestCase(FsStressTest::ThreadCountMax, 10, true, false)
    {
    }
};

class RandomAppendFiles : public RandomWriteFilesTestCase
{
public:
    RandomAppendFiles() NN_NOEXCEPT
        : RandomWriteFilesTestCase(FsStressTest::ThreadCountMax, 10, true, false)
    {
    }
};

class SequentialWriteFlushFiles : public SequentialWriteFilesTestCase
{
public:
    SequentialWriteFlushFiles() NN_NOEXCEPT
        : SequentialWriteFilesTestCase(FsStressTest::ThreadCountMax, 10, false, true)
    {
    }
};

class RandomWriteFlushFiles : public RandomWriteFilesTestCase
{
public:
    RandomWriteFlushFiles() NN_NOEXCEPT
        : RandomWriteFilesTestCase(FsStressTest::ThreadCountMax, 10, false, true)
    {
    }
};

class SequentialAppendFlushFiles : public SequentialWriteFilesTestCase
{
public:
    SequentialAppendFlushFiles() NN_NOEXCEPT
        : SequentialWriteFilesTestCase(FsStressTest::ThreadCountMax, 10, true, true)
    {
    }
};

class RandomAppendFlushFiles : public RandomWriteFilesTestCase
{
public:
    RandomAppendFlushFiles() NN_NOEXCEPT
        : RandomWriteFilesTestCase(FsStressTest::ThreadCountMax, 10, true, true)
    {
    }
};

class InBulkWriteFilesDifferentPriorities : public InBulkWriteNumerousFiles
{
public:
    virtual int GetPriority(int threadIndex) const NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::os::DefaultThreadPriority + GetThreadCount() / 2 - threadIndex;
    }
};

class SequentialWriteFilesDifferentPriorities : public SequentialWriteNumerousFiles
{
public:
    virtual int GetPriority(int threadIndex) const NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::os::DefaultThreadPriority + GetThreadCount() / 2 - threadIndex;
    }
};

class RandomWriteFilesDifferentPriorities : public RandomWriteNumerousFiles
{
public:
    virtual int GetPriority(int threadIndex) const NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::os::DefaultThreadPriority + GetThreadCount() / 2 - threadIndex;
    }
};

class InBulkWriteFewerFiles : public InBulkWriteFilesTestCase
{
public:
    InBulkWriteFewerFiles() NN_NOEXCEPT
        : InBulkWriteFilesTestCase(FsStressTest::ThreadCountMax / 2, 10)
    {
    }
};

class SequentialWriteFewerFiles : public SequentialWriteFilesTestCase
{
public:
    SequentialWriteFewerFiles() NN_NOEXCEPT
        : SequentialWriteFilesTestCase(FsStressTest::ThreadCountMax / 2, 10, false, false)
    {
    }
};

class RandomWriteFewerFiles : public RandomWriteFilesTestCase
{
public:
    RandomWriteFewerFiles() NN_NOEXCEPT
        : RandomWriteFilesTestCase(FsStressTest::ThreadCountMax / 2, 10, false, false)
    {
    }
};

TEST_F(SaveDataFsStressTest, InBulkWriteSingleFile)
{
    Test<InBulkWriteSingleFile>(GetMountName());
}

TEST_F(SaveDataFsStressTest, SequentialWriteSingleFile)
{
    Test<SequentialWriteSingleFile>(GetMountName());
}

TEST_F(SaveDataFsStressTest, RandomWriteSingleFile)
{
    Test<RandomWriteSingleFile>(GetMountName());
}

TEST_F(SaveDataFsStressTest, InBulkWriteNumerousFiles)
{
    Test<InBulkWriteNumerousFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, SequentialWriteNumerousFiles)
{
    Test<SequentialWriteNumerousFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, RandomWriteNumerousFiles)
{
    Test<RandomWriteNumerousFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, SequentialAppendFiles)
{
    Test<SequentialAppendFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, RandomAppendFiles)
{
    Test<RandomAppendFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, SequentialWriteFlushFiles)
{
    Test<SequentialWriteFlushFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, RandomWriteFlushFiles)
{
    Test<RandomWriteFlushFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, SequentialAppendFlushFiles)
{
    Test<SequentialAppendFlushFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, RandomAppendFlushFiles)
{
    Test<RandomAppendFlushFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, InBulkWriteFilesDifferentPriorities)
{
    Test<InBulkWriteFilesDifferentPriorities>(GetMountName());
}

TEST_F(SaveDataFsStressTest, SequentialWriteFilesDifferentPriorities)
{
    Test<SequentialWriteFilesDifferentPriorities>(GetMountName());
}

TEST_F(SaveDataFsStressTest, RandomWriteFilesDifferentPriorities)
{
    Test<RandomWriteFilesDifferentPriorities>(GetMountName());
}

#ifndef NN_BUILD_CONFIG_OS_WIN
TEST_F(SaveDataFsStressTest, LimitWriteFiles)
{
    Test<LimitWriteFiles>(GetMountName());
}
#endif

TEST_F(MultipleSaveDataFsStressTest, InBulkWriteNumerousFiles)
{
    Test<InBulkWriteFewerFiles, InBulkWriteFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, SequentialWriteNumerousFiles)
{
    Test<SequentialWriteFewerFiles, SequentialWriteFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, RandomWriteNumerousFiles)
{
    Test<RandomWriteFewerFiles, RandomWriteFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(OtherApplicationSaveDataFsStressTest, InBulkWriteSingleFile)
{
    Test<InBulkWriteSingleFile>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, SequentialWriteSingleFile)
{
    Test<SequentialWriteSingleFile>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, RandomWriteSingleFile)
{
    Test<RandomWriteSingleFile>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, InBulkWriteNumerousFiles)
{
    Test<InBulkWriteNumerousFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, SequentialWriteNumerousFiles)
{
    Test<SequentialWriteNumerousFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, RandomWriteNumerousFiles)
{
    Test<RandomWriteNumerousFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, SequentialAppendFiles)
{
    Test<SequentialAppendFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, RandomAppendFiles)
{
    Test<RandomAppendFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, SequentialWriteFlushFiles)
{
    Test<SequentialWriteFlushFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, RandomWriteFlushFiles)
{
    Test<RandomWriteFlushFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, SequentialAppendFlushFiles)
{
    Test<SequentialAppendFlushFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, RandomAppendFlushFiles)
{
    Test<RandomAppendFlushFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, InBulkWriteFilesDifferentPriorities)
{
    Test<InBulkWriteFilesDifferentPriorities>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, SequentialWriteFilesDifferentPriorities)
{
    Test<SequentialWriteFilesDifferentPriorities>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, RandomWriteFilesDifferentPriorities)
{
    Test<RandomWriteFilesDifferentPriorities>(GetMountName());
}

#ifndef NN_BUILD_CONFIG_OS_WIN
TEST_F(OtherApplicationSaveDataFsStressTest, LimitWriteFiles)
{
    Test<LimitWriteFiles>(GetMountName());
}
#endif

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)
TEST_F(HostFsStressTest, InBulkWriteSingleFile)
{
    Test<InBulkWriteSingleFile>(GetMountName());
}

TEST_F(HostFsStressTest, SequentialWriteSingleFile)
{
    Test<SequentialWriteSingleFile>(GetMountName());
}

TEST_F(HostFsStressTest, RandomWriteSingleFile)
{
    Test<RandomWriteSingleFile>(GetMountName());
}

TEST_F(HostFsStressTest, InBulkWriteNumerousFiles)
{
    Test<InBulkWriteNumerousFiles>(GetMountName());
}

TEST_F(HostFsStressTest, SequentialWriteNumerousFiles)
{
    Test<SequentialWriteNumerousFiles>(GetMountName());
}

TEST_F(HostFsStressTest, RandomWriteNumerousFiles)
{
    Test<RandomWriteNumerousFiles>(GetMountName());
}

TEST_F(HostFsStressTest, SequentialAppendFiles)
{
    Test<SequentialAppendFiles>(GetMountName());
}

TEST_F(HostFsStressTest, RandomAppendFiles)
{
    Test<RandomAppendFiles>(GetMountName());
}

TEST_F(HostFsStressTest, SequentialWriteFlushFiles)
{
    Test<SequentialWriteFlushFiles>(GetMountName());
}

TEST_F(HostFsStressTest, RandomWriteFlushFiles)
{
    Test<RandomWriteFlushFiles>(GetMountName());
}

TEST_F(HostFsStressTest, SequentialAppendFlushFiles)
{
    Test<SequentialAppendFlushFiles>(GetMountName());
}

TEST_F(HostFsStressTest, RandomAppendFlushFiles)
{
    Test<RandomAppendFlushFiles>(GetMountName());
}

TEST_F(HostFsStressTest, InBulkWriteFilesDifferentPriorities)
{
    Test<InBulkWriteFilesDifferentPriorities>(GetMountName());
}

TEST_F(HostFsStressTest, SequentialWriteFilesDifferentPriorities)
{
    Test<SequentialWriteFilesDifferentPriorities>(GetMountName());
}

TEST_F(HostFsStressTest, RandomWriteFilesDifferentPriorities)
{
    Test<RandomWriteFilesDifferentPriorities>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)

TEST_F(SdCardFsStressTest, InBulkWriteSingleFile)
{
    Test<InBulkWriteSingleFile>(GetMountName());
}

TEST_F(SdCardFsStressTest, SequentialWriteSingleFile)
{
    Test<SequentialWriteSingleFile>(GetMountName());
}

TEST_F(SdCardFsStressTest, RandomWriteSingleFile)
{
    Test<RandomWriteSingleFile>(GetMountName());
}

TEST_F(SdCardFsStressTest, InBulkWriteNumerousFiles)
{
    Test<InBulkWriteNumerousFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, SequentialWriteNumerousFiles)
{
    Test<SequentialWriteNumerousFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, RandomWriteNumerousFiles)
{
    Test<RandomWriteNumerousFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, SequentialAppendFiles)
{
    Test<SequentialAppendFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, RandomAppendFiles)
{
    Test<RandomAppendFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, SequentialWriteFlushFiles)
{
    Test<SequentialWriteFlushFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, RandomWriteFlushFiles)
{
    Test<RandomWriteFlushFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, SequentialAppendFlushFiles)
{
    Test<SequentialAppendFlushFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, RandomAppendFlushFiles)
{
    Test<RandomAppendFlushFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, InBulkWriteFilesDifferentPriorities)
{
    Test<InBulkWriteFilesDifferentPriorities>(GetMountName());
}

TEST_F(SdCardFsStressTest, SequentialWriteFilesDifferentPriorities)
{
    Test<SequentialWriteFilesDifferentPriorities>(GetMountName());
}

TEST_F(SdCardFsStressTest, RandomWriteFilesDifferentPriorities)
{
    Test<RandomWriteFilesDifferentPriorities>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)
TEST_F(TemporaryStorageFsStressTest, InBulkWriteSingleFile)
{
    Test<InBulkWriteSingleFile>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, SequentialWriteSingleFile)
{
    Test<SequentialWriteSingleFile>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, RandomWriteSingleFile)
{
    Test<RandomWriteSingleFile>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, InBulkWriteNumerousFiles)
{
    Test<InBulkWriteNumerousFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, SequentialWriteNumerousFiles)
{
    Test<SequentialWriteNumerousFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, RandomWriteNumerousFiles)
{
    Test<RandomWriteNumerousFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, SequentialAppendFiles)
{
    Test<SequentialAppendFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, RandomAppendFiles)
{
    Test<RandomAppendFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, SequentialWriteFlushFiles)
{
    Test<SequentialWriteFlushFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, RandomWriteFlushFiles)
{
    Test<RandomWriteFlushFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, SequentialAppendFlushFiles)
{
    Test<SequentialAppendFlushFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, RandomAppendFlushFiles)
{
    Test<RandomAppendFlushFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, InBulkWriteFilesDifferentPriorities)
{
    Test<InBulkWriteFilesDifferentPriorities>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, SequentialWriteFilesDifferentPriorities)
{
    Test<SequentialWriteFilesDifferentPriorities>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, RandomWriteFilesDifferentPriorities)
{
    Test<RandomWriteFilesDifferentPriorities>(GetMountName());
}

#ifndef NN_BUILD_CONFIG_OS_WIN
TEST_F(TemporaryStorageFsStressTest, LimitWriteFiles)
{
    Test<LimitWriteFiles>(GetMountName());
}
#endif
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)

}}
