﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_Stress_AccessFilesTestCase.h"

namespace nnt { namespace fs {

class ResizeFilesTestCase : public AccessFilesTestCase
{
public:
    ResizeFilesTestCase(int threadCount, int fileCount) NN_NOEXCEPT
        : AccessFilesTestCase(
            threadCount,
            fileCount,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write))
    {
    }

    virtual ~ResizeFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 5;
    }

    virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);

        ASSERT_FALSE(pTest->IsReadOnly(GetTestCaseIndex()));
        ASSERT_LE(GetEntryCount() * GetThreadCount(), FsStressTest::EntryCountMax);
        for( auto entryIndex = 0; entryIndex < GetEntryCount(); ++entryIndex )
        {
            auto succeeded = false;
            ResizeFile(
                &succeeded,
                GetEntryIndex(threadIndex, entryIndex),
                pTest->IsSaveData(GetTestCaseIndex()));
            ASSERT_TRUE(succeeded);
        }

        Succeed(threadIndex);
    }

protected:
    virtual void ResizeFile(
        bool* outSuccceeded,
        int entryIndex,
        bool ensuresFileClearedInitially) NN_NOEXCEPT = 0;
};

class ExpandFiles : public ResizeFilesTestCase
{
public:
    static const auto FileSizeMax = 1024;

public:
    ExpandFiles() NN_NOEXCEPT
        : ResizeFilesTestCase(FsStressTest::ThreadCountMax, 5)
    {
    }

    virtual ~ExpandFiles() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void SetUpFile(bool* outSucceeded, int) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = true;
    }

    virtual void ResizeFile(
        bool* outSucceeded,
        int entryIndex,
        bool ensuresFileClearedInitially) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        for( auto size = 1; size <= FileSizeMax; ++size )
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetFileSize(file, size));

            if( ensuresFileClearedInitially && entryIndex % 10 == 0 )
            {
                char buffer = -1;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(file, size - 1, &buffer, 1)) << size;
                EXPECT_EQ(0, buffer);
            }
        }

        *outSucceeded = true;
    }
};

NN_DEFINE_STATIC_CONSTANT(const int ExpandFiles::FileSizeMax);

class ShrinkFiles : public ResizeFilesTestCase
{
public:
    static const auto FileSizeMax = 1024;

public:
    ShrinkFiles() NN_NOEXCEPT
        : ResizeFilesTestCase(FsStressTest::ThreadCountMax, 5)
    {
    }

    virtual ~ShrinkFiles() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void SetUpFile(bool* outSucceeded, int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetFileSize(file, FileSizeMax));

        *outSucceeded = true;
    }

    virtual void ResizeFile(
        bool* outSucceeded,
        int entryIndex,
        bool ensuresFileClearedInitially) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);

        if( ensuresFileClearedInitially && entryIndex % 10 == 0 )
        {
            std::unique_ptr<char[]> buffer(new char[FileSizeMax]);
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, buffer.get(), FileSizeMax));
            const auto IsZero = [](char value) NN_NOEXCEPT
            {
                return value == 0;
            };
            EXPECT_TRUE(std::all_of(buffer.get(), buffer.get() + FileSizeMax, IsZero));
        }

        for( auto size = FileSizeMax; 0 <= size; --size )
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetFileSize(file, size));
        }

        *outSucceeded = true;
    }
};

NN_DEFINE_STATIC_CONSTANT(const int ShrinkFiles::FileSizeMax);

class RandomResizeFiles : public ResizeFilesTestCase
{
public:
    static const auto FileSizeMax = 4 * 1024;
    static const auto LoopCount = 256;

public:
    RandomResizeFiles() NN_NOEXCEPT
        : ResizeFilesTestCase(FsStressTest::ThreadCountMax, 5)
    {
    }

    virtual ~RandomResizeFiles() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void SetUpFile(bool* outSucceeded, int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetFileSize(file, entryIndex * 16));

        *outSucceeded = true;
    }

    virtual void ResizeFile(bool* outSucceeded, int entryIndex, bool) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        std::unique_ptr<std::mt19937> rng(new std::mt19937(nnt::fs::util::GetRandomSeed()));
        for( auto count = 0; count < LoopCount; --count )
        {
            const auto size = std::uniform_int_distribution<size_t>(0, FileSizeMax)(*rng);
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetFileSize(file, size));
        }

        *outSucceeded = true;
    }
};

NN_DEFINE_STATIC_CONSTANT(const int RandomResizeFiles::FileSizeMax);

TEST_F(SaveDataFsStressTest, ExpandFiles)
{
    Test<ExpandFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, ShrinkFiles)
{
    Test<ExpandFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, RandomResizeFiles)
{
    Test<ExpandFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, ExpandFiles)
{
    Test<ExpandFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, ShrinkFiles)
{
    Test<ExpandFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, RandomResizeFiles)
{
    Test<ExpandFiles>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)
TEST_F(HostFsStressTest, ExpandFiles)
{
    Test<ExpandFiles>(GetMountName());
}

TEST_F(HostFsStressTest, ShrinkFiles)
{
    Test<ExpandFiles>(GetMountName());
}

TEST_F(HostFsStressTest, RandomResizeFiles)
{
    Test<ExpandFiles>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)

TEST_F(SdCardFsStressTest, ExpandFiles)
{
    Test<ExpandFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, ShrinkFiles)
{
    Test<ExpandFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, RandomResizeFiles)
{
    Test<ExpandFiles>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)
TEST_F(TemporaryStorageFsStressTest, ExpandFiles)
{
    Test<ExpandFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, ShrinkFiles)
{
    Test<ExpandFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, RandomResizeFiles)
{
    Test<ExpandFiles>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)

}}
