﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>
#include <random>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_Stress_AccessFilesTestCase.h"

namespace nnt { namespace fs {

class ReadWriteFilesTestCase : public AccessFilesTestCase
{
public:
    static const auto StandardFileSize = 1 * 1024;
    static const auto LargeFileSize = 3 * 512 * 1024;

public:
    ReadWriteFilesTestCase(int threadCount, int fileCount, size_t fileSize) NN_NOEXCEPT
        : AccessFilesTestCase(
            threadCount,
            fileCount,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)),
            m_FileSize(fileSize)
    {
    }

    virtual ~ReadWriteFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);

        ASSERT_FALSE(pTest->IsReadOnly(GetTestCaseIndex()));
        ASSERT_LE(GetEntryCount() * GetThreadCount(), FsStressTest::EntryCountMax);
        std::unique_ptr<char[]> readBuffer(new char[GetFileSize()]);
        std::unique_ptr<char[]> writeBuffer(new char[GetFileSize()]);
        std::iota(writeBuffer.get(), writeBuffer.get() + GetFileSize(), static_cast<char>(threadIndex));
        for( auto entryIndex = 0; entryIndex < GetEntryCount(); ++entryIndex )
        {
            auto succeeded = false;
            ReadWriteFile(
                &succeeded,
                readBuffer.get(),
                writeBuffer.get(),
                GetFileSize(),
                GetEntryIndex(threadIndex, entryIndex));
        }

        Succeed(threadIndex);
    }

protected:
    virtual void SetUpFile(bool* outSucceeded, int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetFileSize(file, GetFileSize()));

        *outSucceeded = true;
    }

    virtual void ReadWriteFile(
        bool* outSucceeded,
        char* readBuffer,
        const char* writeBuffer,
        size_t size,
        int entryIndex) NN_NOEXCEPT = 0;

    size_t GetFileSize() const NN_NOEXCEPT
    {
        return m_FileSize;
    }

private:
    size_t m_FileSize;
};

NN_DEFINE_STATIC_CONSTANT(const int ReadWriteFilesTestCase::StandardFileSize);
NN_DEFINE_STATIC_CONSTANT(const int ReadWriteFilesTestCase::LargeFileSize);

class InBulkReadWriteFilesTestCase : public ReadWriteFilesTestCase
{
public:
    InBulkReadWriteFilesTestCase(int threadCount, int fileCount, size_t fileSize) NN_NOEXCEPT
        : ReadWriteFilesTestCase(threadCount, fileCount, fileSize)
    {
    }

    virtual ~InBulkReadWriteFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 30;
    }

protected:
    virtual void ReadWriteFile(
        bool* outSucceeded,
        char* readBuffer,
        const char* writeBuffer,
        size_t size,
        int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteFile(
            file,
            0,
            writeBuffer,
            size,
            nn::fs::WriteOption()));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, readBuffer, size));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(file));
        EXPECT_TRUE(std::equal(readBuffer, readBuffer + size, writeBuffer));

        *outSucceeded = true;
    }
};

class SequentialReadWriteFilesTestCase : public ReadWriteFilesTestCase
{
public:
    SequentialReadWriteFilesTestCase(int threadCount, int fileCount) NN_NOEXCEPT
        : ReadWriteFilesTestCase(threadCount, fileCount, StandardFileSize)
    {
    }

    virtual ~SequentialReadWriteFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 3;
    }

protected:
    virtual void ReadWriteFile(
        bool* outSucceeded,
        char* readBuffer,
        const char* writeBuffer,
        size_t bufferSize,
        int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        for( int64_t offset = 0; offset < static_cast<int64_t>(bufferSize); ++offset )
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteFile(
                file,
                offset,
                writeBuffer + offset,
                1,
                nn::fs::WriteOption()));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(file, offset, readBuffer + offset, 1));
            EXPECT_EQ(writeBuffer[offset], readBuffer[offset]);
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(file));

        *outSucceeded = true;
    }
};

class RandomReadWriteFilesTestCase : public ReadWriteFilesTestCase
{
public:
    RandomReadWriteFilesTestCase(int threadCount, int fileCount) NN_NOEXCEPT
        : ReadWriteFilesTestCase(threadCount, fileCount, StandardFileSize)
    {
    }

    virtual ~RandomReadWriteFilesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 3;
    }

protected:
    virtual void ReadWriteFile(
        bool* outSucceeded,
        char* readBuffer,
        const char* writeBuffer,
        size_t size,
        int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        std::unique_ptr<std::mt19937> rng(new std::mt19937(nnt::fs::util::GetRandomSeed()));
        for( size_t count = 0; count < size; ++count )
        {
            const auto offset = std::uniform_int_distribution<int64_t>(0, size - 1)(*rng);
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteFile(
                file,
                offset,
                writeBuffer + offset,
                1,
                nn::fs::WriteOption()));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(file, offset, readBuffer + offset, 1));
            EXPECT_EQ(writeBuffer[offset], readBuffer[offset]);
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(file));

        *outSucceeded = true;
    }
};

class InBulkReadWriteSingleFile : public InBulkReadWriteFilesTestCase
{
public:
    InBulkReadWriteSingleFile() NN_NOEXCEPT
        : InBulkReadWriteFilesTestCase(FsStressTest::ThreadCountMax, 1, StandardFileSize)
    {
    }
};

class SequentialReadWriteSingleFile : public SequentialReadWriteFilesTestCase
{
public:
    SequentialReadWriteSingleFile() NN_NOEXCEPT
        : SequentialReadWriteFilesTestCase(FsStressTest::ThreadCountMax, 1)
    {
    }
};

class RandomReadWriteSingleFile : public RandomReadWriteFilesTestCase
{
public:
    RandomReadWriteSingleFile() NN_NOEXCEPT
        : RandomReadWriteFilesTestCase(FsStressTest::ThreadCountMax, 1)
    {
    }
};

class InBulkReadWriteNumerousFiles : public InBulkReadWriteFilesTestCase
{
public:
    InBulkReadWriteNumerousFiles() NN_NOEXCEPT
        : InBulkReadWriteFilesTestCase(FsStressTest::ThreadCountMax, 10, StandardFileSize)
    {
    }
};

class SequentialReadWriteNumerousFiles : public SequentialReadWriteFilesTestCase
{
public:
    SequentialReadWriteNumerousFiles() NN_NOEXCEPT
        : SequentialReadWriteFilesTestCase(FsStressTest::ThreadCountMax, 10)
    {
    }
};

class RandomReadWriteNumerousFiles : public RandomReadWriteFilesTestCase
{
public:
    RandomReadWriteNumerousFiles() NN_NOEXCEPT
        : RandomReadWriteFilesTestCase(FsStressTest::ThreadCountMax, 10)
    {
    }
};

class InBulkReadWriteFewerFiles : public InBulkReadWriteFilesTestCase
{
public:
    InBulkReadWriteFewerFiles() NN_NOEXCEPT
        : InBulkReadWriteFilesTestCase(FsStressTest::ThreadCountMax / 2, 10, StandardFileSize)
    {
    }
};

class SequentialReadWriteFewerFiles : public SequentialReadWriteFilesTestCase
{
public:
    SequentialReadWriteFewerFiles() NN_NOEXCEPT
        : SequentialReadWriteFilesTestCase(FsStressTest::ThreadCountMax / 2, 10)
    {
    }
};

class RandomReadWriteFewerFiles : public RandomReadWriteFilesTestCase
{
public:
    RandomReadWriteFewerFiles() NN_NOEXCEPT
        : RandomReadWriteFilesTestCase(FsStressTest::ThreadCountMax / 2, 10)
    {
    }
};

class InBulkReadWriteLargeFiles : public InBulkReadWriteFilesTestCase
{
public:
    InBulkReadWriteLargeFiles() NN_NOEXCEPT
        : InBulkReadWriteFilesTestCase(1, 1, LargeFileSize)
    {
    }
};

class RandomReadWriteLargeFiles : public ReadWriteFilesTestCase
{
public:
    static const size_t MaxOffset = 128 * 1024;
    static const size_t ReadWriteCount = 1 * 1024;

public:
    RandomReadWriteLargeFiles() NN_NOEXCEPT
        : ReadWriteFilesTestCase(1, 1, LargeFileSize + MaxOffset)
    {
    }

    virtual ~RandomReadWriteLargeFiles() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 3;
    }

protected:
    virtual void ReadWriteFile(
        bool* outSucceeded,
        char* readBuffer,
        const char* writeBuffer,
        size_t size,
        int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        NN_SDK_REQUIRES_LESS_EQUAL(MaxOffset, size);

        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        std::unique_ptr<std::mt19937> rng(new std::mt19937(nnt::fs::util::GetRandomSeed()));
        for( size_t count = 0; count < ReadWriteCount; ++count )
        {
            const auto offset = std::uniform_int_distribution<int64_t>(0, MaxOffset)(*rng);
            const auto readWriteSize
                = std::uniform_int_distribution<size_t>(1, static_cast<size_t>(size - offset))(*rng);
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteFile(
                file,
                offset,
                writeBuffer + offset,
                readWriteSize,
                nn::fs::WriteOption()));
            NNT_EXPECT_RESULT_SUCCESS(
                nn::fs::ReadFile(file, offset, readBuffer + offset, readWriteSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer + offset, readBuffer + offset, readWriteSize);
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(file));

        *outSucceeded = true;
    }
};

NN_DEFINE_STATIC_CONSTANT(const size_t RandomReadWriteLargeFiles::MaxOffset);

TEST_F(SaveDataFsStressTest, InBulkReadWriteSingleFile)
{
    Test<InBulkReadWriteSingleFile>(GetMountName());
}

TEST_F(SaveDataFsStressTest, SequentialReadWriteSingleFile)
{
    Test<SequentialReadWriteSingleFile>(GetMountName());
}

TEST_F(SaveDataFsStressTest, RandomReadWriteSingleFile)
{
    Test<RandomReadWriteSingleFile>(GetMountName());
}

TEST_F(SaveDataFsStressTest, InBulkReadWriteNumerousFiles)
{
    Test<InBulkReadWriteNumerousFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, SequentialReadWriteNumerousFiles)
{
    Test<SequentialReadWriteNumerousFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, RandomReadWriteNumerousFiles)
{
    Test<RandomReadWriteNumerousFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, InBulkReadWriteLargeFiles)
{
    Test<InBulkReadWriteLargeFiles>(GetMountName());
}

TEST_F(SaveDataFsStressTest, RandomReadWriteLargeFiles)
{
    Test<RandomReadWriteLargeFiles>(GetMountName());
}

TEST_F(MultipleSaveDataFsStressTest, InBulkReadWriteNumerousFiles)
{
    Test<InBulkReadWriteFewerFiles, InBulkReadWriteFewerFiles>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, SequentialReadWriteNumerousFiles)
{
    Test<SequentialReadWriteFewerFiles, SequentialReadWriteFewerFiles>(
        GetMountName(0),
        GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, RandomReadWriteNumerousFiles)
{
    Test<RandomReadWriteFewerFiles, RandomReadWriteFewerFiles>(
        GetMountName(0),
        GetMountName(1));
}

TEST_F(OtherApplicationSaveDataFsStressTest, InBulkReadWriteSingleFile)
{
    Test<InBulkReadWriteSingleFile>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, SequentialReadWriteSingleFile)
{
    Test<SequentialReadWriteSingleFile>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, RandomReadWriteSingleFile)
{
    Test<RandomReadWriteSingleFile>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, InBulkReadWriteNumerousFiles)
{
    Test<InBulkReadWriteNumerousFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, SequentialReadWriteNumerousFiles)
{
    Test<SequentialReadWriteNumerousFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, RandomReadWriteNumerousFiles)
{
    Test<RandomReadWriteNumerousFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, InBulkReadWriteLargeFiles)
{
    Test<InBulkReadWriteLargeFiles>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, RandomReadWriteLargeFiles)
{
    Test<RandomReadWriteLargeFiles>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)
TEST_F(HostFsStressTest, InBulkReadWriteSingleFile)
{
    Test<InBulkReadWriteSingleFile>(GetMountName());
}

TEST_F(HostFsStressTest, SequentialReadWriteSingleFile)
{
    Test<SequentialReadWriteSingleFile>(GetMountName());
}

TEST_F(HostFsStressTest, RandomReadWriteSingleFile)
{
    Test<RandomReadWriteSingleFile>(GetMountName());
}

TEST_F(HostFsStressTest, InBulkReadWriteNumerousFiles)
{
    Test<InBulkReadWriteNumerousFiles>(GetMountName());
}

TEST_F(HostFsStressTest, SequentialReadWriteNumerousFiles)
{
    Test<SequentialReadWriteNumerousFiles>(GetMountName());
}

TEST_F(HostFsStressTest, RandomReadWriteNumerousFiles)
{
    Test<RandomReadWriteNumerousFiles>(GetMountName());
}

TEST_F(HostFsStressTest, InBulkReadWriteLargeFiles)
{
    Test<InBulkReadWriteLargeFiles>(GetMountName());
}

TEST_F(HostFsStressTest, RandomReadWriteLargeFiles)
{
    Test<RandomReadWriteLargeFiles>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)

TEST_F(SdCardFsStressTest, InBulkReadWriteSingleFile)
{
    Test<InBulkReadWriteSingleFile>(GetMountName());
}

TEST_F(SdCardFsStressTest, SequentialReadWriteSingleFile)
{
    Test<SequentialReadWriteSingleFile>(GetMountName());
}

TEST_F(SdCardFsStressTest, RandomReadWriteSingleFile)
{
    Test<RandomReadWriteSingleFile>(GetMountName());
}

TEST_F(SdCardFsStressTest, InBulkReadWriteNumerousFiles)
{
    Test<InBulkReadWriteNumerousFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, SequentialReadWriteNumerousFiles)
{
    Test<SequentialReadWriteNumerousFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, RandomReadWriteNumerousFiles)
{
    Test<RandomReadWriteNumerousFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, InBulkReadWriteLargeFiles)
{
    Test<InBulkReadWriteLargeFiles>(GetMountName());
}

TEST_F(SdCardFsStressTest, RandomReadWriteLargeFiles)
{
    Test<RandomReadWriteLargeFiles>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)
TEST_F(TemporaryStorageFsStressTest, InBulkReadWriteSingleFile)
{
    Test<InBulkReadWriteSingleFile>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, SequentialReadWriteSingleFile)
{
    Test<SequentialReadWriteSingleFile>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, RandomReadWriteSingleFile)
{
    Test<RandomReadWriteSingleFile>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, InBulkReadWriteNumerousFiles)
{
    Test<InBulkReadWriteNumerousFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, SequentialReadWriteNumerousFiles)
{
    Test<SequentialReadWriteNumerousFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, RandomReadWriteNumerousFiles)
{
    Test<RandomReadWriteNumerousFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, InBulkReadWriteLargeFiles)
{
    Test<InBulkReadWriteLargeFiles>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, RandomReadWriteLargeFiles)
{
    Test<RandomReadWriteLargeFiles>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)

}}
