﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testFs_Stress_OpenEntriesTestCase.h"
#include "testFs_Stress_AccessDeepDirectoriesTestCase.h"

namespace nnt { namespace fs {

class CreateDirectoriesTestCase : public OpenEntriesTestCase
{
public:
    CreateDirectoriesTestCase(int threadCount, int directoryCount) NN_NOEXCEPT
        : OpenEntriesTestCase(threadCount, directoryCount, nn::fs::DirectoryEntryType_Directory)
    {
    }

    virtual ~CreateDirectoriesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual void SetUp(FsStressTest* pTest) NN_NOEXCEPT NN_OVERRIDE
    {
        OpenEntriesTestCase::SetUp(pTest);
        ASSERT_FALSE(FailedAny());

        ASSERT_FALSE(pTest->IsReadOnly(GetTestCaseIndex()));
        FailAll();
        for( auto threadIndex = 0; threadIndex < GetThreadCount(); ++threadIndex )
        {
            for( auto entryIndex = 0; entryIndex < GetEntryCount(); ++entryIndex )
            {
                char path[PathLength];
                MakePath(path, GetEntryIndex(threadIndex, entryIndex));
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(path));
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(path));
            }
        }
        SucceedAll();
    }

    virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        ASSERT_FALSE(pTest->IsReadOnly(GetTestCaseIndex()));
        Fail(threadIndex);
        for( auto entryIndex = 0; entryIndex < GetEntryCount(); ++entryIndex )
        {
            char path[PathLength];
            MakePath(path, GetEntryIndex(threadIndex, entryIndex));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteDirectory(path));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(path));
        }
        Succeed(threadIndex);
    }
};

class CreateDirectories : public CreateDirectoriesTestCase
{
public:
    CreateDirectories() NN_NOEXCEPT
        : CreateDirectoriesTestCase(FsStressTest::ThreadCountMax, 10)
    {
    }
};

class CreateFewerDirectories : public CreateDirectoriesTestCase
{
public:
    CreateFewerDirectories() NN_NOEXCEPT
        : CreateDirectoriesTestCase(FsStressTest::ThreadCountMax / 2, 10)
    {
    }
};

class DeleteDeepDirectoriesRecursivelyTestCase : public AccessDeepDirectoriesTestCase
{
public:
    explicit DeleteDeepDirectoriesRecursivelyTestCase(int threadCount) NN_NOEXCEPT
        : AccessDeepDirectoriesTestCase(threadCount, 30)
    {
    }

    virtual ~DeleteDeepDirectoriesRecursivelyTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 3;
    }

    virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);

        ASSERT_FALSE(pTest->IsReadOnly(GetTestCaseIndex()));
        std::unique_ptr<char[]> path(new char[PathLength]);
        MakePath(path.get(), PathLength, threadIndex, 0);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(path.get()));
        CreateDirectories(path.get(), PathLength, threadIndex);

        Succeed(threadIndex);
    }
};

class DeleteDeepDirectoriesRecursively : public DeleteDeepDirectoriesRecursivelyTestCase
{
public:
    DeleteDeepDirectoriesRecursively() NN_NOEXCEPT
        : DeleteDeepDirectoriesRecursivelyTestCase(FsStressTest::ThreadCountMax)
    {
    }
};

class DeleteFewerDeepDirectoriesRecursively : public DeleteDeepDirectoriesRecursivelyTestCase
{
public:
    DeleteFewerDeepDirectoriesRecursively() NN_NOEXCEPT
        : DeleteDeepDirectoriesRecursivelyTestCase(FsStressTest::ThreadCountMax / 2)
    {
    }
};

TEST_F(SaveDataFsStressTest, CreateDirectories)
{
    Test<CreateDirectories>(GetMountName());
}

TEST_F(SaveDataFsStressTest, DeleteDeepDirectoriesRecursively)
{
    Test<DeleteDeepDirectoriesRecursively>(GetMountName());
}

TEST_F(MultipleSaveDataFsStressTest, CreateDirectories)
{
    Test<CreateFewerDirectories, CreateFewerDirectories>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, DeleteDeepDirectoriesRecursively)
{
    Test<DeleteFewerDeepDirectoriesRecursively, DeleteFewerDeepDirectoriesRecursively>(
        GetMountName(0),
        GetMountName(1));
}

TEST_F(OtherApplicationSaveDataFsStressTest, CreateDirectories)
{
    Test<CreateDirectories>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, DeleteDeepDirectoriesRecursively)
{
    Test<DeleteDeepDirectoriesRecursively>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)
TEST_F(HostFsStressTest, CreateDirectories)
{
    Test<CreateDirectories>(GetMountName());
}

TEST_F(HostFsStressTest, DeleteDeepDirectoriesRecursively)
{
    Test<DeleteDeepDirectoriesRecursively>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)

TEST_F(SdCardFsStressTest, CreateDirectories)
{
    Test<CreateDirectories>(GetMountName());
}

TEST_F(SdCardFsStressTest, DeleteDeepDirectoriesRecursively)
{
    Test<DeleteDeepDirectoriesRecursively>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)
TEST_F(TemporaryStorageFsStressTest, CreateDirectories)
{
    Test<CreateDirectories>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, DeleteDeepDirectoriesRecursively)
{
    Test<DeleteDeepDirectoriesRecursively>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)

}}
