﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <limits>

#include <nn/util/util_Base64.h>
#include <nn/fs/fs_Rom.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_SaveDataForDebug.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_AddOnContent.h>
#include <nn/sf/sf_Result.h>

#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

namespace
{

static nnt::fs::util::String g_TestDirectoryPath;
const char DefaultMountNamePrefix = 'p';
static ::nn::Result g_MountLimitResult;

}

namespace nn
{
    bool operator==(const nn::Result& lhs, const nn::Result& rhs) NN_NOEXCEPT
    {
        return lhs.GetInnerValueForDebug() == rhs.GetInnerValueForDebug();
    }
}

class DoToFailTest : public ::testing::Test
{
public:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        m_HostDirectory.Create();
        g_TestDirectoryPath = m_HostDirectory.GetPath().c_str();
    }
    static void TearDownTestCase() NN_NOEXCEPT
    {
        m_HostDirectory.Delete();
    }
private:
    static nnt::fs::util::TemporaryHostDirectory m_HostDirectory;
};

nnt::fs::util::TemporaryHostDirectory DoToFailTest::m_HostDirectory;

nnt::fs::util::String MakeMountName(char prefix, int64_t bit) NN_NOEXCEPT
{
    char name[nn::fs::MountNameLengthMax + 1];
    name[0] = prefix;
    nn::util::Base64::ToBase64String(name + 1, sizeof(name) - 1, &bit, sizeof(bit), nn::util::Base64::Mode_UrlSafe);
    return name;
}

template<typename MountFunction>
int64_t DoToFailTestGetMountLimit(MountFunction mount, char prefix) NN_NOEXCEPT
{
    int64_t limit = ::std::numeric_limits<int64_t>::max();
    for( int64_t i = 0; i < limit; ++i )
    {
        const nnt::fs::util::String name = MakeMountName(prefix, i);
        nn::Result result = mount(name.c_str());
        if( result.IsFailure() )
        {
            EXPECT_TRUE(nn::fs::ResultAllocationMemoryFailed::Includes(result)
                     || nn::fs::ResultOpenCountLimit::Includes(result)
                     || nn::sf::ResultMemoryAllocationFailed::Includes(result));
            NN_LOG("mount limit : %s\n", testing::PrintToString(result).c_str());
            limit = i;
            g_MountLimitResult = result;
            break;
        }
    }
    return limit;
}

template<typename MountFunction>
int64_t DoToFailTestGetMountLimit(MountFunction mount) NN_NOEXCEPT
{
    return DoToFailTestGetMountLimit(mount, DefaultMountNamePrefix);
}

template<typename MountFunction>
void DoToFailTestMount(MountFunction mount, int64_t limit, char prefix) NN_NOEXCEPT
{
    NN_LOG("mount count : %d\n", limit - 1);

    // もう一回マウントできるはず
    {
        for( int64_t i = 0; i < limit; ++i )
        {
            const nnt::fs::util::String name = MakeMountName(prefix, i);
            NNT_ASSERT_RESULT_SUCCESS(mount(name.c_str())) << name.c_str() << ", " << i;
        }

        // limit のときは同じリザルトで失敗するはず
        {
            const nnt::fs::util::String name = MakeMountName(prefix, limit);
            ASSERT_EQ(g_MountLimitResult, mount(name.c_str())) << name.c_str();
        }
    }
}

template<typename MountFunction>
void DoToFailTestMount(MountFunction mount, int64_t limit) NN_NOEXCEPT
{
    DoToFailTestMount(mount, limit, DefaultMountNamePrefix);
}

void DoToFailTestUnmount(int64_t limit, char prefix) NN_NOEXCEPT
{
    for( int64_t i = 0; i < limit; ++i )
    {
        nn::fs::Unmount(MakeMountName(prefix, i).c_str());
    }
}

void DoToFailTestUnmount(int64_t limit) NN_NOEXCEPT
{
    DoToFailTestUnmount(limit, DefaultMountNamePrefix);
}

TEST_F(DoToFailTest, MountRom)
{
    size_t cacheBufferSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheBufferSize));
    auto cacheBuffer = nnt::fs::util::AllocateBuffer(cacheBufferSize);
    ASSERT_NE(nullptr, cacheBuffer);

    auto mountFunction = [&](const char* name) NN_NOEXCEPT
    {
        return nn::fs::MountRom(name, cacheBuffer.get(), cacheBufferSize);
    };

    static int64_t s_Limit = -1;
    if( s_Limit < 0 )
    {
        // 限度確定
        s_Limit = DoToFailTestGetMountLimit(mountFunction);
        // Unmount
        DoToFailTestUnmount(s_Limit);
    }

    DoToFailTestMount(mountFunction, s_Limit);
    DoToFailTestUnmount(s_Limit);
}

TEST_F(DoToFailTest, MountRomDoubleCache)
{
    size_t cacheBufferSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheBufferSize));
    auto cacheBuffer1 = nnt::fs::util::AllocateBuffer(cacheBufferSize);
    ASSERT_NE(nullptr, cacheBuffer1);
    auto cacheBuffer2 = nnt::fs::util::AllocateBuffer(cacheBufferSize);
    ASSERT_NE(nullptr, cacheBuffer2);

    auto mountFunction1 = [&](const char* name) NN_NOEXCEPT
    {
        return nn::fs::MountRom(name, cacheBuffer1.get(), cacheBufferSize);
    };
    auto mountFunction2 = [&](const char* name) NN_NOEXCEPT
    {
        return nn::fs::MountRom(name, cacheBuffer2.get(), cacheBufferSize);
    };

    static int64_t s_Limit1 = -1;
    static int64_t s_Limit2 = -1;
    if( s_Limit1 < 0 )
    {
        // 限度確定
        s_Limit1 = DoToFailTestGetMountLimit(mountFunction1, 'a');
        s_Limit2 = DoToFailTestGetMountLimit(mountFunction2, 'b');
        // Unmount
        DoToFailTestUnmount(s_Limit1, 'a');
        DoToFailTestUnmount(s_Limit2, 'b');
    }

    DoToFailTestMount(mountFunction1, s_Limit1, 'a');
    DoToFailTestMount(mountFunction2, s_Limit2, 'b');

    DoToFailTestUnmount(s_Limit1, 'a');
    DoToFailTestUnmount(s_Limit2, 'b');
}

TEST_F(DoToFailTest, MountSdCard)
{
    static int64_t s_Limit = -1;
    if( s_Limit < 0 )
    {
        // 限度確定
        s_Limit = DoToFailTestGetMountLimit(nn::fs::MountSdCard);
        // Unmount
        DoToFailTestUnmount(s_Limit);
    }
    DoToFailTestMount(nn::fs::MountSdCard, s_Limit);
    DoToFailTestUnmount(s_Limit);
}

TEST_F(DoToFailTest, DISABLED_MountSaveData)
{
    static int64_t s_Limit = -1;
    auto mountFunction = [&](const char* name) NN_NOEXCEPT
    {
        return nn::fs::MountSaveDataForDebug(name);
    };
    if( s_Limit < 0 )
    {
        // 限度確定
        s_Limit = DoToFailTestGetMountLimit(mountFunction);
        // Unmount
        DoToFailTestUnmount(s_Limit);
    }
    DoToFailTestMount(mountFunction, s_Limit);
    DoToFailTestUnmount(s_Limit);
}
