﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SaveDataTransferVersion2.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>

#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

namespace
{
    const nn::Bit64 CurrentApplicationIdValue = 0x0100000000010000;
}

namespace nn { namespace fs { namespace detail {
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> GetFileSystemProxyServiceObject() NN_NOEXCEPT;
}}}

TEST(SaveDataTransferForCloudBackUpTest, ListSaveDataOwnedId)
{
    int appIdCount = 0;
    nn::ncm::ApplicationId appIds[8];

    int offsets[] = { 0, 3, 7, 8 };
    for (auto& offset : offsets)
    {
        appIdCount = 0;
        std::memset(appIds, 0x0, sizeof(nn::ncm::ApplicationId) * 8);
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ListApplicationAccessibleSaveDataOwnerId(&appIdCount, appIds, { CurrentApplicationIdValue }, 0, offset, 8));
        int expectIdCount = std::max(7 - offset, 0);
        EXPECT_EQ(appIdCount, expectIdCount);
        for (int i = 0; i < expectIdCount; i++)
        {
            EXPECT_EQ(appIds[i].value, CurrentApplicationIdValue + expectIdCount - i);
        }
    }

    int k = 0;
    while (NN_STATIC_CONDITION(true))
    {
        appIdCount = 0;
        nn::ncm::ApplicationId appId = { 0 };
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ListApplicationAccessibleSaveDataOwnerId(&appIdCount, &appId, { CurrentApplicationIdValue }, 0, k, 1));
        if (appIdCount == 0)
        {
            break;
        }
        EXPECT_EQ(appId.value, CurrentApplicationIdValue + 7 - k);
        k += appIdCount;
    }
    EXPECT_EQ(k, 7);

    {
        appIdCount = -1;
        nn::ncm::ApplicationId appId = { 0 };
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ListApplicationAccessibleSaveDataOwnerId(&appIdCount, &appId, { CurrentApplicationIdValue }, 0, 0, 0));
        EXPECT_EQ(appIdCount, 0);
    }

    {
        appIdCount = 0;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetCountOfApplicationAccessibleSaveDataOwnerId(&appIdCount, { CurrentApplicationIdValue }, 0));
        EXPECT_EQ(appIdCount, 7);
    }
}

TEST(SaveDataTransferForCloudBackUpTest, ListSaveDataOwnedIdFailure)
{
    int appIdCount = 0;
    nn::ncm::ApplicationId appId = { 0 };
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, nn::fs::ListApplicationAccessibleSaveDataOwnerId(&appIdCount, &appId, { CurrentApplicationIdValue }, 0, -1, 1));
    // NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultNullptrArgument, nn::fs::ListApplicationAccessibleSaveDataOwnerId(&appIdCount, nullptr, { CurrentApplicationIdValue }, 0, 0, 1));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultNullptrArgument, nn::fs::ListApplicationAccessibleSaveDataOwnerId(&appIdCount, nullptr, { CurrentApplicationIdValue }, 0, 0, 1));
    {
        auto fileSystemProxy = nn::fs::detail::GetFileSystemProxyServiceObject();
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidSize, fileSystemProxy->ListAccessibleSaveDataOwnerId(nn::sf::Out<int32_t>(&appIdCount), nn::sf::OutBuffer(reinterpret_cast<char*>(&appId), 4), appId, 0, 1));
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
