﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkLog.h>
#include <memory>
#include <functional>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_SignedSystemPartition.h>
#include <nn/lmem/lmem_ExpHeap.h>


using namespace nn;
using namespace nn::fs;


namespace {

const int ExpHeapSize = 128 * 1024;

NN_ALIGNAS(4096) char g_ExpHeapStack[ExpHeapSize];
nn::lmem::HeapHandle g_ExpHeapHandle = nullptr;

void InitializeExpHeap()
{
    if (g_ExpHeapHandle == nullptr)
    {
        g_ExpHeapHandle = nn::lmem::CreateExpHeap(&g_ExpHeapStack, ExpHeapSize, nn::lmem::CreationOption_DebugFill | nn::lmem::CreationOption_ThreadSafe);
        NN_ABORT_UNLESS(g_ExpHeapHandle);
    }
}

void* Allocate(size_t size)
{
    NN_ABORT_UNLESS(g_ExpHeapHandle);
    void* buffer = nn::lmem::AllocateFromExpHeap(g_ExpHeapHandle, size);
    return buffer;
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    NN_ABORT_UNLESS(g_ExpHeapHandle);
    nn::lmem::FreeToExpHeap(g_ExpHeapHandle, p);
}

template<typename EnterDirectory, typename ExitDirectory, typename FindFile>
Result IterateDirectoryRecursiveInternal(char* path, const size_t size, DirectoryEntry* pDirectoryEntry,
    EnterDirectory enterDirectory,
    ExitDirectory exitDirectory,
    FindFile findFile
) NN_NOEXCEPT
{
    DirectoryHandle handle;
    NN_RESULT_DO(OpenDirectory(&handle, path, OpenDirectoryMode_All));
    NN_UTIL_SCOPE_EXIT
    {
        CloseDirectory(handle);
    };

    while (NN_STATIC_CONDITION(true))
    {
        int64_t readNum = 0;
        NN_RESULT_DO(ReadDirectory(&readNum, pDirectoryEntry, handle, 1));

        if (readNum == 0)
        {
            break;
        }

        auto parentPathSize = strnlen(path, size);
        strncat(path, pDirectoryEntry->name, size - parentPathSize);
        {
            if (pDirectoryEntry->directoryEntryType == DirectoryEntryType_Directory)
            {
                NN_RESULT_DO(enterDirectory(path, *pDirectoryEntry));
                strncat(path, "/", size - parentPathSize - strnlen(pDirectoryEntry->name, size));
                NN_RESULT_DO(IterateDirectoryRecursiveInternal(path, size, pDirectoryEntry, enterDirectory, exitDirectory, findFile));
                NN_RESULT_DO(exitDirectory(path, *pDirectoryEntry));
            }
            else
            {
                NN_RESULT_DO(findFile(path, *pDirectoryEntry));
            }
        }
        path[parentPathSize] = '\0';

    }

    NN_RESULT_SUCCESS;
}


template<typename EnterDirectory, typename ExitDirectory, typename FindFile>
Result IterateDirectoryRecursive(const char* path,
    EnterDirectory enterDirectory,
    ExitDirectory exitDirectory,
    FindFile findFile
) NN_NOEXCEPT
{
    DirectoryEntry directoryEntryBuffer = { { 0 } }; // スタック節約用
    char pathBuffer[768];
    strncpy(pathBuffer, path, nn::fs::EntryNameLengthMax);

    return IterateDirectoryRecursiveInternal(pathBuffer, nn::fs::EntryNameLengthMax, &directoryEntryBuffer, enterDirectory, exitDirectory, findFile);
}



Result ListDirectoryRecursive(const char* path) NN_NOEXCEPT
{
    return IterateDirectoryRecursive(path,
        [](const char* path, const DirectoryEntry& entry) {
        NN_UNUSED(entry);
        NN_SDK_LOG("DIR  <%s>\n", path);
        NN_RESULT_SUCCESS;
        },
        [](const char* path, const DirectoryEntry& entry) {
            NN_UNUSED(path);
            NN_UNUSED(entry);
            NN_RESULT_SUCCESS;
        },
        [](const char* path, const DirectoryEntry& entry) {
        NN_SDK_LOG("FILE <%-40s> %10lld\n", path, entry.fileSize);
        NN_RESULT_SUCCESS;
    }
    );
}


void DumpBuffer(const void* buffer, size_t size)
{
    for (size_t i = 0; i < size; ++i)
    {
        if (i % 4 == 0)
        {
            NN_SDK_LOG(" ");
        }
        if (i % 32 == 0)
        {
            NN_SDK_LOG("\n");
        }

        uint8_t value8 = static_cast<const uint8_t*>(buffer)[i];
        NN_SDK_LOG("%02x", value8);
        NN_UNUSED(value8);
    }

    NN_SDK_LOG("\n\n");
}



// FsSaveData 流用
void CreateAndUpdateFile(const char* path)
{
    bool isSignedSystemPartitionOnSdCardValid = nn::fs::IsSignedSystemPartitionOnSdCardValid();
    NN_SDK_LOG("isSignedSystemPartitionOnSdCardValid: %d\n", isSignedSystemPartitionOnSdCardValid);
    NN_UNUSED(isSignedSystemPartitionOnSdCardValid);

    struct SaveDataStructure
    {
        int value;
    };

    const size_t FileSize = sizeof(SaveDataStructure);

    // ファイルの存在を確認します。
    {
        nn::fs::DirectoryEntryType directoryEntryType;
        auto result = nn::fs::GetEntryType(&directoryEntryType, path);
        if (nn::fs::ResultPathNotFound().Includes(result))
        {
            // 対象ファイルが存在しません。
            // ファイルを作成します。
            {
                result = nn::fs::CreateFile(path, FileSize);
                if (nn::fs::ResultPathNotFound::Includes(result))
                {
                    // 対象ファイルの親ディレクトリが存在しません。
                    // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
                }
                else if (nn::fs::ResultPathAlreadyExists::Includes(result)
                    || nn::fs::ResultTargetLocked::Includes(result))
                {
                    // 対象ファイルが既に存在しています。
                    // ファイルが既に存在していても構わない場合は、このエラーハンドリングは不要です。
                    // エラーハンドリングしない場合、ファイルのサイズが FileSize である保証が無いことに注意してください。
                    // 必要であれば対象ファイルを削除してから再度作成してください。
                }
                else if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
                {
                    // セーブデータのデータ保存領域が不足しています。
                    NN_ABORT("Usable space not enough.\n");
                    return;
                }
                // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
                // これ以上のエラーハンドリングは不要です。
            }

            // 作成したファイルに初期データを書き込みます。
            {
                nn::fs::FileHandle fileHandle;

                // ファイルをオープンします。
                result = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Write);
                if (nn::fs::ResultPathNotFound::Includes(result))
                {
                    // 対象ファイルが存在しません。
                    // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
                }
                else if (nn::fs::ResultTargetLocked::Includes(result))
                {
                    // 対象ファイルが既にオープンされています。
                    // ファイルが既にオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
                }
                // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
                // これ以上のエラーハンドリングは不要です。

                // ファイルに書き込みます。
                SaveDataStructure data = { 0 };
                result = nn::fs::WriteFile(fileHandle, 0, &data, sizeof(data), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
                if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
                {
                    // セーブデータのデータ保存領域が不足しています。
                    // ファイルをオープンする際に OpenMode_AllowAppend を指定していない場合は、このエラーハンドリングは不要です。
                    NN_ABORT("Usable space not enough.\n");
                }
                // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
                // これ以上のエラーハンドリングは不要です。

                nn::fs::CloseFile(fileHandle);
            }

            NN_SDK_LOG("Create initial data as %s\n", path);
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。
    }

    // ファイルのデータを読み込み、更新します。
    {
        SaveDataStructure data;

        nn::fs::FileHandle fileHandle;

        // ファイルをオープンします
        {
            auto result = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write);
            if (nn::fs::ResultPathNotFound::Includes(result))
            {
                // 対象ファイルが存在しません。
                // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
            }
            else if (nn::fs::ResultTargetLocked::Includes(result))
            {
                // 対象ファイルが既にオープンされています。
                // ファイルが既にオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
            }
            // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
            // これ以上のエラーハンドリングは不要です。
        }

        // ファイルを読み込みます。
        {
            size_t readSize;

            // ファイル読み込み失敗時はライブラリ内でアボートするため、エラーハンドリングは不要です。
            NN_SDK_LOG("Read %s\n", path);
            (void)nn::fs::ReadFile(&readSize, fileHandle, 0, &data, sizeof(data));
        }

        NN_SDK_LOG("data.value: 0x%x\n", data.value);

        // 読み込んだデータを更新します。
        NN_SDK_LOG("Increment data.value: 0x%x -> 0x%x\n", data.value, data.value + 1);
        data.value++;

        // 更新したデータを書き込みます。
        {
            NN_SDK_LOG("Write %s\n", path);
            auto result = nn::fs::WriteFile(fileHandle, 0, &data, sizeof(data), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
            if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
            {
                // セーブデータのデータ保存領域が不足しています。
                // ファイルをオープンする際に OpenMode_AllowAppend を指定していない場合は、このエラーハンドリングは不要です。
                NN_ABORT("Usable space not enough.\n");
            }
            // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
            // これ以上のエラーハンドリングは不要です。
        }

        NN_SDK_LOG("Close %s\n", path);
        nn::fs::CloseFile(fileHandle);

    }

}


}





extern "C" void nninitStartup()
{
}
extern "C" void nndiagStartup()
{
}


extern "C" void nnMain()
{
    NN_SDK_LOG("**** safemode test *****\n");

    InitializeExpHeap();
    nn::fs::SetAllocator(Allocate, Deallocate);


    Result testResultArray[5];
    memset(testResultArray, 0xFF, sizeof(testResultArray));

    // bisPartition (normal p2)
    {
        testResultArray[0] = [&]() -> Result
        {
            NN_SDK_LOG("Open and read BisPartitionId::BootConfigAndPackage2Part1:\n");
            std::unique_ptr<IStorage> storage;
            NN_RESULT_DO(OpenBisPartition(&storage, nn::fs::BisPartitionId::BootConfigAndPackage2Part1));

            char buffer[512];
            NN_RESULT_DO(storage->Read(0, buffer, 512));

            DumpBuffer(buffer, 512);

            NN_RESULT_SUCCESS;
        }();
    }

    // bis の system
    {
        testResultArray[1] = [&]() -> Result
        {
            NN_SDK_LOG("Mount and access BisPartitionId::SystemProperPartition:\n");
            NN_RESULT_DO(MountBis("bisSystem", nn::fs::BisPartitionId::SystemProperPartition));
            NN_RESULT_DO(ListDirectoryRecursive("bisSystem:/"));

            CreateAndUpdateFile("bisSystem:/bisTestFile");

            Unmount("bisSystem");
            NN_RESULT_SUCCESS;
        }();
    }

    // ncm のシステムセーブ
    {
        testResultArray[2] = [&]() -> Result
        {
            NN_SDK_LOG("Mount and access system save data of ncm on SaveDataSpaceId::ProperSystem:\n");

            NN_RESULT_DO(MountSystemSaveData("ssave", nn::fs::SaveDataSpaceId::ProperSystem, 0x8000000000000120));
            NN_RESULT_DO(ListDirectoryRecursive("ssave:/"));

            // 内容更新
            CreateAndUpdateFile("ssave:/fsSaveTestFile");
            NN_SDK_LOG("Commit \"ssave\"\n");
            (void)nn::fs::CommitSaveData("ssave");

            Unmount("ssave");
            NN_RESULT_SUCCESS;
        }();
    }

    // アカウント毎システムセーブ
    {
        const nn::fs::UserId TestUserId1 = { { 2 } };
        testResultArray[3] = [&]() -> Result
        {
            NN_SDK_LOG("Mount and access account system save data on SaveDataSpaceId::ProperSystem:\n");

            NN_RESULT_DO(MountSystemSaveData("ssave", nn::fs::SaveDataSpaceId::ProperSystem, 0x8000000000004000, TestUserId1));
            NN_RESULT_DO(ListDirectoryRecursive("ssave:/"));

            // 内容更新
            CreateAndUpdateFile("ssave:/fsSaveTestFile");
            NN_SDK_LOG("Commit \"ssave\"\n");
            (void)nn::fs::CommitSaveData("ssave");

            Unmount("ssave");
            NN_RESULT_SUCCESS;
        }();
    }

    // fs::IsSignedSystemPartitionOnSdCardValid()
    {
        testResultArray[4] = [&]() -> Result
        {
            bool isSignedSystemPartitionOnSdCardValid = nn::fs::IsSignedSystemPartitionOnSdCardValid();
            NN_SDK_LOG("isSignedSystemPartitionOnSdCardValid: %d\n", isSignedSystemPartitionOnSdCardValid);

            if (!isSignedSystemPartitionOnSdCardValid)
            {
                return nn::fs::ResultUnknown();
            }

            NN_RESULT_SUCCESS;
        }();
    }


    // テスト結果を sd に書き出し
    NN_ABORT_UNLESS_RESULT_SUCCESS(MountSdCardForDebug("sd"));
    (void)(CreateFile("sd:/safeModeTestResult.bin", sizeof(testResultArray)));
    FileHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(OpenFile(&handle, "sd:/safeModeTestResult.bin", nn::fs::OpenMode_Write));
    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteFile(handle, 0, &testResultArray, sizeof(testResultArray), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    nn::fs::CloseFile(handle);
    Unmount("sd");

    for (auto result : testResultArray)
    {
        NN_SDK_LOG("safemode test : %08x (%s)\n", result.GetInnerValueForDebug(), result.IsSuccess() ? "Success" : "Failure");
        NN_UNUSED(result);
    }
    NN_SDK_LOG("**** safemode test done *****\n");
}

