﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>

#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_ImageDirectory.h>
#include <nn/fs/fs_FatPrivate.h>
#include <nn/nn_Log.h>

#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/nnt_Argument.h>

using namespace nn::fs;
using namespace nnt::fs::util;

namespace{

const char MountName[]          = "perftest";
const char DirectoryPath[]      = "perftest:/testdir/";
const char RenamePath[]         = "perftest:/testdir/Rename/";
const int  FileCount       = 4096;
const int  DirCount        = 256;
const int  SetSizeBig      = 1 * 1024;
const int  SetSizeSmall    = 512;
const int  RandomFileSize  = 3 * 1024;

char g_Path[FileCount][NameMaxLength];
char g_RenamePath[FileCount][NameMaxLength];

class PerformanceTestCreateOpen : public ::testing::Test, public nnt::fs::util::PrepareWorkDirFixture
{
protected:

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        CreateWorkRootPath();
        NNT_EXPECT_RESULT_SUCCESS(MountHost(MountName, GetWorkRootPath().c_str()));
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
    }
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        Unmount(MountName);
        DeleteWorkRootPath();
    }
};
}

/**
* @brief テスト対象のパスを配列に設定します。
* @param[in/out] pPath     テスト対象となるパスの配列
* @param[in]     TestCount テスト対象となるパスの配列の数
* @param[in]     isRandom  ランダムなパス番号を設定するか
*/
void SetTestPath(char (*pPath)[NameMaxLength], const int TestCount, const bool isRandom) NN_NOEXCEPT
{
    if(isRandom == true)
    {
        std::unique_ptr<int[]> pathCount(new int[TestCount]);
        CreateRandomArray(pathCount.get(), TestCount, 1);
        for(int i = 0; i < TestCount; i++)
        {
            nn::util::SNPrintf(pPath[i], NameMaxLength, "%s%04d", DirectoryPath, pathCount[i]);
        }
    }
    else
    {
        for(int i = 0; i < TestCount; i++)
        {
            nn::util::SNPrintf(pPath[i], NameMaxLength, "%s%04d", DirectoryPath, i);
        }
    }

    if(TestCount == FileCount)
    {
        String path;
        for(int i = 0; i < TestCount; i++)
        {
            path = pPath[i];
            path.append(".file");
            nn::util::SNPrintf(pPath[i], NameMaxLength, "%s", path.c_str());
        }
    }
}

nn::Result CreateSequentialFile(const char* path, const int start, const int count, const int size) NN_NOEXCEPT
{
    char file[256];
    nn::Result ret = nn::fs::ResultUnknown();
    for (int32_t i = start; i < (start + count); i++)
    {
        nn::util::SNPrintf(file, sizeof(file), "%s/file_xxxxxxxxxxxxxxxxxxxxxxxxxxxxxx_%04d.bin", path, i);

        ret = nn::fs::CreateFile(file, size);

        if(ret.IsSuccess() != true)
        {
            NN_LOG("loop count: %d, file path: %s\n", i, file);
            return nn::fs::ResultUnknown();
        }
    }
    return ret;
}

void PerformanceCreateFile(const char *path, int num)
{
    char dir[256];
    nn::util::SNPrintf(dir, sizeof(dir), "%sManyFiles", path);
    nn::fs::DeleteDirectoryRecursively(dir);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(dir));

    for(int i = 0; i < num; i += 100)
    {
        nnt::fs::util::TimeCount timeCount;

        timeCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(CreateSequentialFile(dir, i, 100, 0));
        timeCount.StopTime();

        NN_LOG("%04d: ", i);
        timeCount.ViewTime(timeCount.GetTotalTime());

        /* 100ファイル作るのに60秒以上かかっていたら検出 */
        EXPECT_GT(60, timeCount.GetTotalTime() / 1000 / 1000 / 1000);

        timeCount.ResetTime();
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(dir));
}

void TestTooManyFilesOpen(const char* path, int count)
{
    FileHandle handle[FileCount];
    nn::Result result;
    int opened = 0;
    int i;

    ASSERT_GT(FileCount, count + 1);
    NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(nullptr, g_Path, path, count + 1, 0));

    NN_UTIL_SCOPE_EXIT
    {
        for (i = 0; i < opened; i++)
        {
            CloseFile(handle[i]);
        }
    };

    for (i = 0; i < count - 1; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(result = OpenFile(&handle[i], g_Path[i], OpenMode_Read));
        if (result.IsFailure())
        {
            NN_LOG("OpenFile(%s) failed\n", g_Path[i]);
            return;
        }
        opened++;
    }

    // ncm がSDのコンテンツメタデータベース保存用に一つオープンしている場合がある
    i = count - 1;
    result = OpenFile(&handle[i], g_Path[i], OpenMode_Read);
    if (result.IsSuccess())
    {
        opened++;
    }

    i = count;
    NNT_EXPECT_RESULT_FAILURE(ResultFatFsTooManyFilesOpenedU, result = OpenFile(&handle[i], g_Path[0], OpenMode_Read));
    if (result.IsSuccess())
    {
        CloseFile(handle[i]);
    }

    NNT_EXPECT_RESULT_FAILURE(ResultFatFsTooManyFilesOpenedS, result = OpenFile(&handle[i], g_Path[i], OpenMode_Read));
    if (result.IsSuccess())
    {
        CloseFile(handle[i]);
    }
}

//!< @brief exFATのサポート状態の取得
TEST_F(PerformanceTestCreateOpen, IsExFatSupported)
{
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    EXPECT_FALSE(nn::fs::IsExFatSupported());
#else
    EXPECT_TRUE(nn::fs::IsExFatSupported());
#endif
}

//!< @brief ImageDirectory(Nand)に多数のファイルを作成する時間の計測
TEST_F(PerformanceTestCreateOpen, CreateFileBis)
{
    char path[256];
    char mountname[256];

    nn::util::SNPrintf(mountname, sizeof(mountname), "ImgDirNand");
    nn::util::SNPrintf(path, sizeof(path), "%s:/", mountname);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory(mountname, nn::fs::ImageDirectoryId::Nand));
    // 本体メニューの上限枚数
    PerformanceCreateFile(path, 1000);
    nn::fs::Unmount(mountname);
}

//!< @brief ImageDirectory(SdCard)に多数のファイルを作成する時間の計測
TEST_F(PerformanceTestCreateOpen, CreateFileSd)
{
    char path[256];
    char mountname[256];

    nn::util::SNPrintf(mountname, sizeof(mountname), "ImgDirSd");
    nn::util::SNPrintf(path, sizeof(path), "%s:/", mountname);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory(mountname, nn::fs::ImageDirectoryId::SdCard));
    // スクリーンショトは日付ごとにディレクトリが切られるので、
    // 1日のスクリーンショット撮影数がフラットに置かれるファイル数になる。
    // 1分間に1回スクリーンショットを撮ったとして、10時間続けたしても600枚+α
    PerformanceCreateFile(path, 1000);
    nn::fs::Unmount(mountname);
}

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
//!< @brief SdCardで256個のファイルをオープン出来ること、257個目で失敗すること
TEST_F(PerformanceTestCreateOpen, TooManyFilesOpen)
{
    const int fileCount = 256;
    char path[fileCount];
    char mountname[fileCount];
    nn::util::SNPrintf(mountname, sizeof(mountname), "sd");
    nn::util::SNPrintf(path, sizeof(path), "%s:/test/", mountname);

    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug(mountname));
    DeleteDirectoryRecursively(path);
    NNT_EXPECT_RESULT_SUCCESS(CreateDirectory(path));

    TestTooManyFilesOpen(path, fileCount);

    DeleteDirectoryRecursively(path);
    nn::fs::Unmount(mountname);
}
#endif

//!< @brief 4096 個のファイルを作成・開閉・削除する時間の計測
TEST_F(PerformanceTestCreateOpen, CreateFile_OpenFile_CloseFile_DeleteFile)
{
    // ファイルを作成する時間を計測
    {
        NN_LOG("\nCreate File Performance Test\n");

        SetTestPath(g_Path, FileCount, false);
        TimeCount timeCountCreate;
        for(int i = 0; i < FileCount; i++)
        {
            NNT_FS_SCOPED_TRACE("loop count: %d, file path: %s\n", i, g_Path[i]);
            timeCountCreate.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(CreateFile(g_Path[i], 0));
            timeCountCreate.StopTime();
        }
        timeCountCreate.ViewTime(timeCountCreate.GetTotalTime());
    }

    // ファイルを作成した順に Open/Close する時間を計測
    {
        NN_LOG("\nSequencial Open/Close File Performance Test\n");

        NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(DirectoryPath));
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
        NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(nullptr, g_Path, DirectoryPath, FileCount, 0));

        TimeCount timeCountSequencialOpen;
        TimeCount timeCountSequencialClose;
        FileHandle handle;
        for(int i = 0; i < FileCount; i++)
        {
            NNT_FS_SCOPED_TRACE("loop count: %d, file path: %s\n", i, g_Path[i]);
            // ファイルを Open する時間を取得
            timeCountSequencialOpen.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, g_Path[i], OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend));
            timeCountSequencialOpen.StopTime();

            // ファイルを Close する時間を取得
            timeCountSequencialClose.StartTime();
            CloseFile(handle);
            timeCountSequencialClose.StopTime();
        }
        NN_LOG(" Open  Files\n");
        timeCountSequencialOpen.ViewTime(timeCountSequencialOpen.GetTotalTime());
        NN_LOG(" Close Files\n");
        timeCountSequencialClose.ViewTime(timeCountSequencialClose.GetTotalTime());
    }

    // ファイルをランダムな順に Open/Close する時間を計測
    {
        NN_LOG("\nRandom Open/Close File Performance Test\n");

        NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(DirectoryPath));
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
        NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(nullptr, g_Path, DirectoryPath, FileCount, 0));

        SetTestPath(g_Path, FileCount, true);
        TimeCount timeCountRandomOpen;
        TimeCount timeCountRandomClose;
        FileHandle handle;
        for(int i = 0; i < FileCount; i++)
        {
            NNT_FS_SCOPED_TRACE("loop count: %d, file path: %s\n", i, g_Path[i]);
            timeCountRandomOpen.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, g_Path[i], OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend));
            timeCountRandomOpen.StopTime();

            timeCountRandomClose.StartTime();
            CloseFile(handle);
            timeCountRandomClose.StopTime();
        }
        NN_LOG(" Open  Files\n");
        timeCountRandomOpen.ViewTime(timeCountRandomOpen.GetTotalTime());
        NN_LOG(" Close Files\n");
        timeCountRandomClose.ViewTime(timeCountRandomClose.GetTotalTime());
    }

    // ファイルを削除する時間を計測
    {
        NN_LOG("\nDelete File Performance Test\n");

        SetTestPath(g_Path, FileCount, false);
        TimeCount timeCountDelete;
        for(int i = 0; i < FileCount; i++)
        {
            NNT_FS_SCOPED_TRACE("loop count: %d, file path: %s\n", i, g_Path[i]);
            timeCountDelete.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(DeleteFile(g_Path[i]));
            timeCountDelete.StopTime();
        }
        timeCountDelete.ViewTime(timeCountDelete.GetTotalTime());
    }
}

//!< @brief 256 個のディレクトリを作成・開閉・削除する時間の計測
TEST_F(PerformanceTestCreateOpen, CreateDirectory_OpenDirectory_CloseDirectory_DeleteDirectory)
{
    // ディレクトリを作成する時間を計測
    {
        NN_LOG("\nCreate Directory Performance Test\n");

        SetTestPath(g_Path, DirCount, false);
        TimeCount timeCountCreate;
        for(int i = 0; i < DirCount; i++)
        {
            NNT_FS_SCOPED_TRACE("loop count: %d, directory path: %s\n", i, g_Path[i]);
            timeCountCreate.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(g_Path[i]));
            timeCountCreate.StopTime();
        }
        timeCountCreate.ViewTime(timeCountCreate.GetTotalTime());
    }

    // ディレクトリを作成した順に Open/Close する時間を計測
    {
        NN_LOG("\nSequencial Open/Close Directory Performance Test");

        NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(DirectoryPath));
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
        NNT_ASSERT_RESULT_SUCCESS(CreateTestDirectory(nullptr, g_Path, DirectoryPath, DirCount, DIRTYPE_FLAT, false, 0));

        DirectoryHandle dirHandle;
        TimeCount timeCountSequencialOpen;
        TimeCount timeCountSequencialClose;
        for(int i = 0; i < DirCount; i++)
        {
            NNT_FS_SCOPED_TRACE("loop count: %d, directory path: %s\n", i, g_Path[i]);
            timeCountSequencialOpen.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(OpenDirectory(&dirHandle, g_Path[i], OpenDirectoryMode_All));
            timeCountSequencialOpen.StopTime();

            timeCountSequencialClose.StartTime();
            CloseDirectory(dirHandle);
            timeCountSequencialClose.StopTime();
        }
        NN_LOG("\n Open Directory\n");
        timeCountSequencialOpen.ViewTime(timeCountSequencialOpen.GetTotalTime());
        NN_LOG("\n Close Directory\n");
        timeCountSequencialClose.ViewTime(timeCountSequencialClose.GetTotalTime());
    }

    // ディレクトリをランダムな順に Open/Close する時間を計測
    {
        NN_LOG("\nRandom Open/Close Directory Performance Test");

        NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(DirectoryPath));
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
        NNT_ASSERT_RESULT_SUCCESS(CreateTestDirectory(nullptr, g_Path, DirectoryPath, DirCount, DIRTYPE_FLAT, false, 0));

        SetTestPath(g_Path, DirCount, true);
        DirectoryHandle dirHandle;
        TimeCount timeCountRandomOpen;
        TimeCount timeCountRandomClose;
        for(int i = 0; i < DirCount; i++)
        {
            NNT_FS_SCOPED_TRACE("loop count: %d, directory path: %s\n", i, g_Path[i]);
            timeCountRandomOpen.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(OpenDirectory(&dirHandle, g_Path[i], OpenDirectoryMode_All));
            timeCountRandomOpen.StopTime();

            timeCountRandomClose.StartTime();
            CloseDirectory(dirHandle);
            timeCountRandomClose.StopTime();
        }
        NN_LOG("\n Open Directory\n");
        timeCountRandomOpen.ViewTime(timeCountRandomOpen.GetTotalTime());
        NN_LOG("\n Close Directory\n");
        timeCountRandomClose.ViewTime(timeCountRandomClose.GetTotalTime());
    }

    // ディレクトリを削除する時間を計測
    {
        NN_LOG("\nDelete Directory Performance Test\n");

        SetTestPath(g_Path, DirCount, false);
        TimeCount timeCountDelete;
        for(int i = 0; i < DirCount; i++)
        {
            NNT_FS_SCOPED_TRACE("loop count: %d, directory path: %s\n", i, g_Path[i]);
            timeCountDelete.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(DeleteDirectory(g_Path[i]));
            timeCountDelete.StopTime();
        }
        timeCountDelete.ViewTime(timeCountDelete.GetTotalTime());
    }
}

//!< @brief ディレクトリを再帰的に削除する時間の計測（1階層に4096ファイル）
TEST_F(PerformanceTestCreateOpen, DeleteDirectoryRecursivelyFlat)
{
    NN_LOG("\nDelete Flat Directory Recursively Performance Test\n");

    // 4096個のファイルを作成
    NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(nullptr, g_Path, DirectoryPath, FileCount, 0));

    TimeCount timeCountDeleteRecursivelyFlat;

    // 4096個のファイルを持ったディレクトリを再帰的に削除する時間を計測
    timeCountDeleteRecursivelyFlat.StartTime();
    NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(DirectoryPath));
    timeCountDeleteRecursivelyFlat.StopTime();

    NN_LOG("\n Recursively (1 Directory %d Files)\n", FileCount);
    timeCountDeleteRecursivelyFlat.ViewTime(timeCountDeleteRecursivelyFlat.GetEndTime());
}

//!< @brief 4096 個のファイルのファイルサイズを変更する時間の計測
TEST_F(PerformanceTestCreateOpen, SetFileSize)
{
    NN_LOG("\nSet File Size Performance Test\n");

    FileHandle fileHandle;

    // ファイルサイズを 0B から 1KB に変更する時間を計測
    {
        NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(nullptr, g_Path, DirectoryPath, FileCount, 0));
        TimeCount timeCountSetBig;
        for(int i = 0; i < FileCount; i++)
        {
            NNT_FS_SCOPED_TRACE("loop count: %d, file path: %s\n", i, g_Path[i]);
            NNT_ASSERT_RESULT_SUCCESS(OpenFile(&fileHandle, g_Path[i], OpenMode_Write));

            // ファイルサイズ変更(1KB)
            timeCountSetBig.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(SetFileSize(fileHandle, SetSizeBig));
            timeCountSetBig.StopTime();

            CloseFile(fileHandle);
        }
        NN_LOG("\n Set File Size (0B >> 1KB)\n");
        timeCountSetBig.ViewTime(timeCountSetBig.GetTotalTime());
    }

    // ファイルサイズを 1KB から 0.5KB に変更する時間を計測
    {
        NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(DirectoryPath));
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
        NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(nullptr, g_Path, DirectoryPath, FileCount, SetSizeBig));

        TimeCount timeCountSetSmall;
        for(int i = 0; i < FileCount; i++)
        {
            NNT_FS_SCOPED_TRACE("loop count: %d, file path: %s\n", i, g_Path[i]);
            NNT_ASSERT_RESULT_SUCCESS(OpenFile(&fileHandle, g_Path[i], OpenMode_Write));

            // ファイルサイズ変更(0.5KB)
            timeCountSetSmall.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(SetFileSize(fileHandle, SetSizeSmall));
            timeCountSetSmall.StopTime();

            CloseFile(fileHandle);
        }
        NN_LOG("\n Set File Size (1KB >> 0.5KB)\n");
        timeCountSetSmall.ViewTime(timeCountSetSmall.GetTotalTime());
    }
}

//!< @brief 4096 個のファイルのファイルサイズを取得する時間の計測
TEST_F(PerformanceTestCreateOpen, GetFileSize)
{
    NN_LOG("\nGet File Size Performance Test\n");

    FileHandle fileHandle;

    // 1KB のファイルサイズを取得する時間を計測
    {
        NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(nullptr, g_Path, DirectoryPath, FileCount, SetSizeBig));
        TimeCount timeCountGetSize;
        for(int i = 0; i < FileCount; i++)
        {
            NNT_FS_SCOPED_TRACE("loop count: %d, file path: %s\n", i, g_Path[i]);
            NNT_ASSERT_RESULT_SUCCESS(OpenFile(&fileHandle, g_Path[i], OpenMode_Write));

            // 1KBのファイルサイズを取得
            int64_t readSize = 0;

            timeCountGetSize.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(GetFileSize(&readSize, fileHandle));
            timeCountGetSize.StopTime();

            CloseFile(fileHandle);
        }
        NN_LOG("\n Get File Size\n");
        timeCountGetSize.ViewTime(timeCountGetSize.GetTotalTime());
    }

    // 0～3KB のランダムなファイルサイズ取得する時間を計測
    {
        NNT_ASSERT_RESULT_SUCCESS(DeleteDirectoryRecursively(DirectoryPath));
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(DirectoryPath));
        NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(nullptr, g_Path, DirectoryPath, FileCount, 0));

        TimeCount timeCountGetRandomSize;
        for(int i = 0; i < FileCount; i++)
        {
            NNT_FS_SCOPED_TRACE("loop count: %d, file path: %s\n", i, g_Path[i]);
            NNT_ASSERT_RESULT_SUCCESS(OpenFile(&fileHandle, g_Path[i], OpenMode_Write));

            // ランダムなファイルサイズに変更
            int randomFileSize = rand() % (RandomFileSize + 1);
            NNT_ASSERT_RESULT_SUCCESS(SetFileSize(fileHandle, randomFileSize));

            // ランダムなファイルサイズを取得
            int64_t readSize = 0;

            timeCountGetRandomSize.StartTime();
            NNT_ASSERT_RESULT_SUCCESS(GetFileSize(&readSize, fileHandle));
            timeCountGetRandomSize.StopTime();

            CloseFile(fileHandle);
        }
        NN_LOG("\n Get File Random Size\n");
        timeCountGetRandomSize.ViewTime(timeCountGetRandomSize.GetTotalTime());
    }
}

//!< @brief 4096 個のファイルの名称を変更する時間の計測
TEST_F(PerformanceTestCreateOpen, RenameFile)
{
    NN_LOG("\nRename File Performance Test\n");

    // ファイルをリネームする時間を計測
    NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(nullptr, g_Path, DirectoryPath, FileCount, 0));

    TimeCount timeCountRenameFile;
    for(int i = 0; i < FileCount; i++)
    {
        nn::util::SNPrintf(g_RenamePath[i], NameMaxLength, "%snew%04d", DirectoryPath, i);
        NNT_FS_SCOPED_TRACE("loop count: %d, file path: %s, rename file path: %s", i, g_Path[i], g_RenamePath[i]);

        timeCountRenameFile.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(RenameFile(g_Path[i], g_RenamePath[i]));
        timeCountRenameFile.StopTime();
    }
    NN_LOG(" Rename Files\n");
    timeCountRenameFile.ViewTime(timeCountRenameFile.GetTotalTime());
}

//!< @brief 4096 個のファイルを移動する時間の計測
TEST_F(PerformanceTestCreateOpen, MoveFile)
{
    NN_LOG("\nRename File Performance Test\n");

    // ファイルを移動する時間を計測
    NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(nullptr, g_Path, DirectoryPath, FileCount, 0));

    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(RenamePath));
    TimeCount timeCountMoveFile;
    for(int i = 0; i < FileCount; i++)
    {
        nn::util::SNPrintf(g_RenamePath[i], NameMaxLength, "%s%04d", RenamePath, i);
        NNT_FS_SCOPED_TRACE("loop count: %d, file path: %s, moved file path: %s", i, g_Path[i], g_RenamePath[i]);

        timeCountMoveFile.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(RenameFile(g_Path[i], g_RenamePath[i]));
        timeCountMoveFile.StopTime();
    }
    NN_LOG(" Move Files\n");
    timeCountMoveFile.ViewTime(timeCountMoveFile.GetTotalTime());
}

//!< @brief 256 個のディレクトリの名称を変更する時間の計測
TEST_F(PerformanceTestCreateOpen, RenameDirectory)
{
    NN_LOG("\nRename Directory Performance Test\n");

    // ディレクトリをリネームする時間を計測
    NNT_ASSERT_RESULT_SUCCESS(CreateTestDirectory(nullptr, g_Path, DirectoryPath, DirCount, DIRTYPE_FLAT, false, 0));

    TimeCount timeCountRenameDirectory;
    for(int i = 0; i < DirCount; i++)
    {
        nn::util::SNPrintf(g_RenamePath[i], NameMaxLength, "%snew%04d", DirectoryPath, i);
        NNT_FS_SCOPED_TRACE("loop count: %d, directory path: %s, rename directory path: %s", i, g_Path[i], g_RenamePath[i]);

        timeCountRenameDirectory.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(RenameDirectory(g_Path[i], g_RenamePath[i]));
        timeCountRenameDirectory.StopTime();
    }
    NN_LOG(" Rename Directories\n");
    timeCountRenameDirectory.ViewTime(timeCountRenameDirectory.GetTotalTime());
}

//!< @brief 256 個のディレクトリを移動する時間の計測
TEST_F(PerformanceTestCreateOpen, MoveDirectory)
{
    NN_LOG("\nRename Directory Performance Test\n");

    // ディレクトリを移動する時間を計測
    NNT_ASSERT_RESULT_SUCCESS(CreateTestDirectory(nullptr, g_Path, DirectoryPath, DirCount, DIRTYPE_FLAT, false, 0));

    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(RenamePath));
    TimeCount timeCountMoveDirectory;
    for(int i = 0; i < DirCount; i++)
    {
        nn::util::SNPrintf(g_RenamePath[i], NameMaxLength, "%s%04d", RenamePath, i);
        NNT_FS_SCOPED_TRACE("loop count: %d, directory path: %s, moved directory path: %s", i, g_Path[i], g_RenamePath[i]);

        timeCountMoveDirectory.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(RenameDirectory(g_Path[i], g_RenamePath[i]));
        timeCountMoveDirectory.StopTime();
    }
    NN_LOG(" Move Directories\n");
    timeCountMoveDirectory.ViewTime(timeCountMoveDirectory.GetTotalTime());
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nnt::fs::util::SetFsTestPerformanceConfiguration();

    ::testing::InitGoogleTest(&argc, argv);

    SetAllocator(Allocate, Deallocate);
    ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    // MmcPatrol 休止
    nn::fs::SuspendMmcPatrol();

    auto testResult = RUN_ALL_TESTS();

    // MmcPatrol 再開
    nn::fs::ResumeMmcPatrol();

    if (CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
