﻿# coding: UTF-8

import os
import subprocess
import xml.dom.minidom
import sys
import patchtest
import time

g_NspExtension = '.nsp'

# 指定バージョンのファイル名を取得
def getFileName(version):
    return patchtest.g_OutputDirectoryPath + '/v' + str(version)

# 指定したバージョンの nsp ファイルのパスを取得
def getNspPath(version):
    return os.path.normpath("{0}/v{1}{2}".format(patchtest.g_OutputDirectoryPath, version, g_NspExtension))

# 指定したバージョンのパッチ nsp ファイルのパスを取得
def getPatchedNspPath(version):
    return os.path.normpath("{0}/v{1}_patch{2}".format(patchtest.g_OutputDirectoryPath, version, g_NspExtension))

# 指定したバージョンの最適化パッチ nsp ファイルのパスを取得
def getOptimizedNspPath(version):
    return os.path.normpath("{0}/v{1}_opt_patch{2}".format(patchtest.g_OutputDirectoryPath, version, g_NspExtension))

# 指定したバージョンのデフラグパッチ nsp ファイルのパスを取得
def getDefragmentedNspPath(version):
    return os.path.normpath("{0}/v{1}_defrag_patch{2}".format(patchtest.g_OutputDirectoryPath, version, g_NspExtension))

# スパース化した nsp ファイルのパスを取得
def getSparseNspPath(filename):
    return os.path.normpath("{0}/{1}{2}".format(patchtest.g_OutputDirectoryPath, filename, g_NspExtension))

# バージョン指定のメタを作成
def makeVersionedMeta(version):
    dom = xml.dom.minidom.parse(patchtest.META_BASE_FILE_PATH)
    metaElement = dom.getElementsByTagName('NintendoSdkMeta')[0]

    applicationElements = metaElement.getElementsByTagName('Application')
    if applicationElements:
        applicationElement = applicationElements[0]

        # IconPath を更新
        iconPathNode = applicationElement.getElementsByTagName('Icon')[0].getElementsByTagName('IconPath')[0].childNodes[0]
        iconPathNode.nodeValue = os.path.normpath(os.path.join(patchtest.SIGLO_ROOT, iconPathNode.nodeValue.replace('../', '')))
    else:
        applicationElement = metaElement.appendChild(dom.createElement('Application'))

    versionElements = applicationElement.getElementsByTagName('Version')
    if versionElements:
        applicationElement.removeChild(versionElements[0])

    versionElement = applicationElement.appendChild(dom.createElement('Version'))
    versionElement.appendChild(dom.createTextNode(str(version)))

    outputPath = os.path.normpath(getFileName(version) + '.meta')
    with open(outputPath, 'w') as outputFile:
        dom.writexml(outputFile)

    return outputPath

# nsp の作成
def createNsp(version, metaPath):
    programPath = os.path.normpath(patchtest.g_OutputDirectoryPath + '/code')
    dataPath = patchtest.g_ResourceDirectoryPath
    outputPath = getNspPath(version)

    patchtest.printf('Create nsp %s', os.path.basename(outputPath))
    # 作成
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'creatensp', '-o', outputPath, '--type', 'Application', '--meta', metaPath, '--desc', patchtest.DESC_FILE_PATH, '--program', programPath, dataPath])
    # 検証
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'verify', outputPath])

    return outputPath

# オリジナルの nsp を作成
def createOriginalNsp():
    return createNsp(0, patchtest.META_BASE_FILE_PATH)

# パッチを作成
def makePatch(version, shiftSize=0):
    return makePatchWithOriginal(getNspPath(0), version, shiftSize)

# 指定のオリジナルファイルでパッチを作成
def makePatchWithOriginal(originalNspPath, version, shiftSize=0):
    metaPath = makeVersionedMeta(version)
    currentNspPath = createNsp(version, metaPath)
    outputPath = getPatchedNspPath(version)

    args = []
    if 0 < shiftSize:
        args = ['--optimize-size-strongly', str(shiftSize)]

    patchtest.printf('Make patch %s from [%s, %s]', os.path.basename(outputPath), os.path.basename(originalNspPath), os.path.basename(currentNspPath))
    # 作成
    start = time.time()
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'makepatch', '-o', outputPath, '--desc', patchtest.DESC_FILE_PATH, '--original', originalNspPath, '--current', currentNspPath, '--save-patch-build-log'] + args)
    elapsed = time.time() - start
    print ("Elapsed time : {0} sec".format(elapsed))
    print ("File size : {0} bytes".format(os.path.getsize(outputPath)))
    # 検証
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'verify', outputPath])

    os.remove(metaPath)
    os.remove(currentNspPath)

    return outputPath

# パッチを最適化
def optimizePatch(version, defragSize=0, shiftSize=0):
    currentPatchPath = getPatchedNspPath(version)
    if version < 2:
        return currentPatchPath

    if version == 2:
        previousPatchPath = getPatchedNspPath(version - 1)
    else:
        previousPatchPath = getOptimizedNspPath(version - 1)

    outputPath = getOptimizedNspPath(version)

    args = []
    if 0 < shiftSize:
        args = ['--optimize-size-strongly', str(shiftSize)]

    patchtest.printf('Optimize patch %s from [%s, %s]', os.path.basename(outputPath), os.path.basename(previousPatchPath), os.path.basename(currentPatchPath))
    # 作成
    start = time.time()
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'optimizepatch', '-o', outputPath, '--desc', patchtest.DESC_FILE_PATH, '--previous', previousPatchPath, '--current', currentPatchPath, '--save-patch-build-log'] + args)
    elapsed = time.time() - start
    print ("Elapsed time : {0} sec".format(elapsed))
    print ("File size : {0} bytes".format(os.path.getsize(outputPath)))
    # 検証
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'verify', outputPath])
    # パッチ間の検証
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'verify', outputPath, '--previous', previousPatchPath])

    # デフラグしないで終了
    if defragSize <= 0:
        os.remove(currentPatchPath)

        return outputPath

    bdiffPath = outputPath.replace('.nsp', '_bdiff.nsp')

    # パッチ間差分を作成
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'bdiff', '-o', bdiffPath, '--source', previousPatchPath, '--destination', outputPath])

    defragPatchPath = getDefragmentedNspPath(version - 1)
    if os.path.exists(defragPatchPath):
        previousPatchPath = defragPatchPath

    defragOutputPath = getDefragmentedNspPath(version)

    patchtest.printf('Defragment patch %s from [%s, %s]', os.path.basename(defragOutputPath), os.path.basename(previousPatchPath), os.path.basename(currentPatchPath))
    # 作成
    start = time.time()
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'optimizepatch', '-o', defragOutputPath, '--desc', patchtest.DESC_FILE_PATH, '--previous', previousPatchPath, '--current', currentPatchPath, '--defragment', '--defragment-size', str(defragSize), '--save-patch-build-log'] + args)
    elapsed = time.time() - start
    print ("Elapsed time : {0} sec".format(elapsed))
    print ("File size : {0} bytes".format(os.path.getsize(defragOutputPath)))
    # 検証
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'verify', defragOutputPath])
    # パッチ間の検証
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'verify', defragOutputPath, '--previous', previousPatchPath])

    bdiffPath = defragOutputPath.replace('.nsp', '_bdiff.nsp')

    # パッチ間差分を作成
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'bdiff', '-o', bdiffPath, '--source', previousPatchPath, '--destination', defragOutputPath])

    os.remove(currentPatchPath)

    return outputPath, defragOutputPath

# スパース化 nsp の作成
def sparsifyNsp(version):
    currentPath = createNsp(version, makeVersionedMeta(version))
    originalPath = getNspPath(0)
    outputPatchPath = getSparseNspPath("v{0}_sparse_patch".format(version))
    outputOriginalPath = getSparseNspPath("v{0}_sparse_patch.original".format(version))

    args = []
    if 1 < version:
        args = ['--previous', getSparseNspPath("v{0}_sparse_patch".format(version - 1))]

    patchtest.printf("Sparsify nsp {0} + {1}".format(os.path.basename(outputOriginalPath), os.path.basename(outputPatchPath)))
    # 作成
    start = time.time()
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'makepatch', '-o', outputPatchPath, '--desc', patchtest.DESC_FILE_PATH, '--original', originalPath, '--current', currentPath, '--do-application-compaction', '--compaction-block-size', '1', '--compaction-erase-size', '2', '--save-patch-build-log'] + args)
    elapsed = time.time() - start
    print ("Elapsed time : {0} sec".format(elapsed))
    print ("File size : {0} + {1} bytes".format(os.path.getsize(outputOriginalPath), os.path.getsize(outputPatchPath)))

    # 検証
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'verify', outputOriginalPath])
    subprocess.check_output([patchtest.AUTHORING_TOOL_PATH, 'verify', outputPatchPath])

    return outputOriginalPath, outputPatchPath
