﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/web.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_Content.h>
#include <nn/lr/lr_Result.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ncm/ncm_Result.h>
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

namespace {

const int ReadFileStep = 16 * 1024;

// ファイルの読み込みを行う関数オブジェクト
class FileReader
{
public:
    nn::Result operator()(const char* filePath, int64_t fileSize) NN_NOEXCEPT
    {
        nn::fs::FileHandle file;
        NN_RESULT_DO(nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        // ハッシュのブロック単位で読み込みが発生するようにする
        for( int64_t offset = 0; offset < fileSize; offset += ReadFileStep )
        {
            const auto size =
                static_cast<size_t>(std::min<int64_t>(sizeof(m_Buffer), fileSize - offset));

            NN_RESULT_DO(nn::fs::ReadFile(file, offset, m_Buffer, size));
        }

        NN_RESULT_SUCCESS;
    }

private:
    char m_Buffer[256];
};

}

// ディレクトリを巡回してすべてのファイルをログ出力します。
TEST(PatchAddHtmlDocumentTest, Traverse)
{
    const nn::ncm::ApplicationId applicationId = {0x0005000C10000000};

    nn::ns::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::ns::Finalize();
    };

    NNT_ASSERT_RESULT_SUCCESS(
        nn::ns::MountApplicationHtmlDocument("content", applicationId));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("content");
    };

    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::TraverseDirectory<FileReader>("content:/"));
}

// html document がないことをテストします。
TEST(PatchAddHtmlDocumentTest, NoHtmlDocument)
{
    const nn::ncm::ApplicationId applicationId = {0x0005000C10000000};

    nn::ns::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::ns::Finalize();
    };

    NNT_ASSERT_RESULT_FAILURE(
        nn::ns::ResultApplicationContentNotFound,
        nn::ns::MountApplicationHtmlDocument("content", applicationId));
}

// ↓手動操作が必要なことに注意
TEST(PatchAddHtmlDocumentTest, WebAppletViewer)
{
    static const char* const DocumentPath = "dir.htdocs/test.html";

    nn::web::ShowOfflineHtmlPageArg showOfflinePageArg(DocumentPath);

    nn::web::OfflineHtmlPageReturnValue offlinePageReturnValue;
    nn::Result result = nn::web::ShowOfflineHtmlPage(&offlinePageReturnValue, showOfflinePageArg);
    NN_ABORT_UNLESS(result.IsSuccess());
    NN_LOG("Exit Reason %d\n", offlinePageReturnValue.GetOfflineExitReason());
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nn::fs::SetEnabledAutoAbort(false);

    int result = RUN_ALL_TESTS();

    nnt::Exit(result);
}
