﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <random>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>

#include <nn/fs/fs_SystemData.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/util/util_ScopeExit.h>

#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

#include "testFs_Integration_FileDataCache_Util.h"


namespace {

const char* RomMountName = "rom";

struct AocInfo
{
    const char* name;
    nn::aoc::AddOnContentIndex index;
};
const AocInfo Aoc1Info = { "aoc1", 1 };
const AocInfo Aoc2Info = { "aoc2", 2 };
const AocInfo AocRegressionSIGLO80958Info = { "aoc3", 3 };
const AocInfo Aoc6gInfo = { "aoc4", 4 };
const AocInfo AocMinimumMetadataRomInfo = { "aoc5", 5 };

const char* SystemDataMountName = "sys";
const nn::ncm::SystemDataId SystemDataId = { 0x0005000C10000001 };

const nn::TimeSpan ZeroTimeSpan(0);

const int MaxThreadInfoCount = 4;
ThreadInfo g_ThreadInfo[MaxThreadInfoCount];

}

// ----------------------------------------------------------------------------

TEST(CacheEffectiveness, CheckRomFsCacheEffectiveness)
{
    const char* targetRomFsFileName = "8m";
    char path[16];
    ::snprintf(path, 16, "%s:/%s", RomMountName, targetRomFsFileName);

    const size_t fileBufferSize = 8 * 1024 * 1024;
    void* fileBuffer = std::malloc(fileBufferSize);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(fileBuffer);
    };

    int failureCount = 0;
    for (int trial = 0; trial < 8; trial++)
    {
        nn::TimeSpan noCacheTime;
        {
            ScopedRomFsMounter romFs(RomMountName);

            nn::fs::FileHandle fileHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read));

            nn::os::Tick startTick = nn::os::GetSystemTick();
            for (int i = 0; i < 4; i++)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, fileBuffer, fileBufferSize));
                EXPECT_TRUE(nnt::fs::util::IsFilledWith8BitCount(fileBuffer, fileBufferSize, 0));
            }
            nn::os::Tick endTick = nn::os::GetSystemTick();

            nn::fs::CloseFile(fileHandle);

            noCacheTime = (endTick - startTick).ToTimeSpan();
        }
        nn::TimeSpan cacheTime;
        {
            ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
            ScopedRomFsMounter romFs(RomMountName);

            nn::fs::FileHandle fileHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read));

            nn::os::Tick startTick = nn::os::GetSystemTick();
            for (int i = 0; i < 4; i++)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, fileBuffer, fileBufferSize));
                EXPECT_TRUE(nnt::fs::util::IsFilledWith8BitCount(fileBuffer, fileBufferSize, 0));
            }
            nn::os::Tick endTick = nn::os::GetSystemTick();

            nn::fs::CloseFile(fileHandle);

            cacheTime = (endTick - startTick).ToTimeSpan();
        }
        if (cacheTime >= noCacheTime)
        {
            failureCount++;

            // Develop のファームウェアを使用してテストすると、キャッシュなしの方が処理時間で勝ることが稀にある (SIGLO-81094)
            // UART 出力の影響が考えられるので、失敗した場合は再実行までの間隔を空けてテストと UART 出力がなるべく被らないようにする
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }
    EXPECT_LE(failureCount, 1);  // 1 回までは失敗を許容する (SIGLO-81094)
    ExpectNoMemoryLeak();
}

TEST(CacheEffectiveness, CheckAocCacheEffectiveness)
{
    const char* targetRomFsFileName = "6m";
    char path[16];
    ::snprintf(path, 16, "%s:/%s", Aoc1Info.name, targetRomFsFileName);

    const size_t fileBufferSize = 6 * 1024 * 1024;
    void* fileBuffer = std::malloc(fileBufferSize);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(fileBuffer);
    };

    int failureCount = 0;
    for (int trial = 0; trial < 8; trial++)
    {
        nn::TimeSpan noCacheTime;
        {
            ScopedAddOnContentMounter aoc1(Aoc1Info.name, Aoc1Info.index);

            nn::fs::FileHandle fileHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read));

            nn::os::Tick startTick = nn::os::GetSystemTick();
            for (int i = 0; i < 4; i++)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, fileBuffer, fileBufferSize));
                EXPECT_TRUE(nnt::fs::util::IsFilledWith8BitCount(fileBuffer, fileBufferSize, 0));
            }
            nn::os::Tick endTick = nn::os::GetSystemTick();

            nn::fs::CloseFile(fileHandle);

            noCacheTime = (endTick - startTick).ToTimeSpan();
        }
        nn::TimeSpan cacheTime;
        {
            ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
            ScopedAddOnContentMounter aoc1(Aoc1Info.name, Aoc1Info.index);

            nn::fs::FileHandle fileHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read));

            nn::os::Tick startTick = nn::os::GetSystemTick();
            for (int i = 0; i < 4; i++)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, fileBuffer, fileBufferSize));
                EXPECT_TRUE(nnt::fs::util::IsFilledWith8BitCount(fileBuffer, fileBufferSize, 0));
            }
            nn::os::Tick endTick = nn::os::GetSystemTick();

            nn::fs::CloseFile(fileHandle);

            cacheTime = (endTick - startTick).ToTimeSpan();
        }
        if (cacheTime >= noCacheTime)
        {
            failureCount++;

            // Develop のファームウェアを使用してテストすると、キャッシュなしの方が処理時間で勝ることが稀にある (SIGLO-81094)
            // UART 出力の影響が考えられるので、失敗した場合は再実行までの間隔を空けてテストと UART 出力がなるべく被らないようにする
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }
    EXPECT_LE(failureCount, 1);  // 1 回までは失敗を許容する (SIGLO-81094)
    ExpectNoMemoryLeak();
}

TEST(CacheEffectiveness, CheckSystemDataWillNotBeCached)
{
    const size_t fileDataCacheSize = 8 * 1024 * 1024;

    const char* targetRomFsFileName = "8m";
    char path[16];
    ::snprintf(path, 16, "%s:/%s", RomMountName, targetRomFsFileName);

    const size_t fileBufferSize = fileDataCacheSize;
    void* fileBuffer = std::malloc(fileDataCacheSize);
    NN_UTIL_SCOPE_EXIT
    {
        std::free(fileBuffer);
    };

    for (int i = 0; i < 4; i++)
    {
        int64_t seed = nn::os::GetSystemTick().GetInt64Value();

        ScopedGlobalFileDataCacheEnabler fileDataCache(fileDataCacheSize);
        ScopedRomFsMounter romFs(RomMountName);
        ScopedSystemDataMounter systemData(SystemDataMountName, SystemDataId);
        {
            nn::fs::FileHandle fileHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(fileHandle);
            };

            nn::TimeSpan beforeCacheTime;
            {
                nn::os::Tick startTick = nn::os::GetSystemTick();
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0LL, fileBuffer, fileBufferSize));
                nn::os::Tick endTick = nn::os::GetSystemTick();

                beforeCacheTime = (endTick - startTick).ToTimeSpan();

                EXPECT_TRUE(nnt::fs::util::IsFilledWith8BitCount(fileBuffer, fileBufferSize, 0));
            }
            {
                ASSERT_TRUE(RandomReadTest(SystemDataMountName, seed, 128, ZeroTimeSpan));
            }
            nn::TimeSpan afterCacheTime;
            {
                nn::os::Tick startTick = nn::os::GetSystemTick();
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0LL, fileBuffer, fileBufferSize));
                nn::os::Tick endTick = nn::os::GetSystemTick();

                afterCacheTime = (endTick - startTick).ToTimeSpan();

                EXPECT_TRUE(nnt::fs::util::IsFilledWith8BitCount(fileBuffer, fileBufferSize, 0));
            }
            EXPECT_LT(afterCacheTime, beforeCacheTime);
        }
    }
    ExpectNoMemoryLeak();
}

// ----------------------------------------------------------------------------

TEST(Reading, SingleThreadMultipleFileSystemReading)
{
    ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

    ScopedRomFsMounter romFs(RomMountName);
    ScopedAddOnContentMounter aoc1(Aoc1Info.name, Aoc1Info.index);
    ScopedAddOnContentMounter aoc2(Aoc2Info.name, Aoc2Info.index);

    for (int i = 0; i < 5; i++)
    {
        EXPECT_TRUE(RandomReadTest(RomMountName, i * 3 + 0, 64, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(Aoc1Info.name, i * 3 + 1, 32, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(Aoc2Info.name, i * 3 + 2, 32, ZeroTimeSpan));
    }
}

TEST(Reading, MultiThreadMultipleFileSystemReading)
{
    ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

    ScopedRomFsMounter romFs(RomMountName);
    ScopedAddOnContentMounter aoc1(Aoc1Info.name, Aoc1Info.index);
    ScopedAddOnContentMounter aoc2(Aoc2Info.name, Aoc2Info.index);

    const int ThreadCount = 3;

    RandomReadTestArg testArgs[ThreadCount];
    for (int i = 0; i < ThreadCount; i++)
    {
        switch (i)
        {
        case 0:
            testArgs[i].mountName = RomMountName;
            break;
        case 1:
            testArgs[i].mountName = Aoc1Info.name;
            break;
        case 2:
            testArgs[i].mountName = Aoc2Info.name;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        testArgs[i].seed = i;
        testArgs[i].trialCount = 1024;
        testArgs[i].trialTime = ZeroTimeSpan;
        testArgs[i].trialSize = 0;
    }

    for (int i = 0; i < ThreadCount; i++)
    {
        nn::os::CreateThread(
            &g_ThreadInfo[i].threadType,
            RandomReadTest,
            &testArgs[i],
            g_ThreadInfo[i].threadStack,
            g_ThreadInfo[i].ThreadStackSize,
            nn::os::DefaultThreadPriority,
            i);
        nn::os::StartThread(&g_ThreadInfo[i].threadType);
    }
    for (int i = 0; i < ThreadCount; i++)
    {
        nn::os::WaitThread(&g_ThreadInfo[i].threadType);
        nn::os::DestroyThread(&g_ThreadInfo[i].threadType);
    }
    for (int i = 0; i < ThreadCount; i++)
    {
        EXPECT_TRUE(testArgs[i].success);
    }
}

// ----------------------------------------------------------------------------

TEST(OperationOrder, AocEnableBeforeMountDisableAfterUnmount)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        {
            ScopedAddOnContentMounter aoc(Aoc1Info.name, Aoc1Info.index);
            EXPECT_TRUE(RandomReadTest(Aoc1Info.name, 0, 128, ZeroTimeSpan));
        }
    }
    ExpectNoMemoryLeak();
}

TEST(OperationOrder, AocEnableAfterMountDisableAfterUnmount)
{
    {
        ScopedAddOnContentMounter aoc(Aoc2Info.name, Aoc2Info.index);

        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

        EXPECT_TRUE(RandomReadTest(Aoc2Info.name, 0, 128, ZeroTimeSpan));

        aoc.Unmount();
    }
    ExpectNoMemoryLeak();
}

TEST(OperationOrder, AocEnableBeforeMountDisableBeforeUnmount)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

        ScopedAddOnContentMounter aoc(Aoc1Info.name, Aoc1Info.index);

        EXPECT_TRUE(RandomReadTest(Aoc1Info.name, 0, 128, ZeroTimeSpan));

        fileDataCache.Disable();
    }
    ExpectNoMemoryLeak();
}

TEST(OperationOrder, AocEnableAfterMountDisableBeforeUnmount)
{
    {
        ScopedAddOnContentMounter aoc(Aoc2Info.name, Aoc2Info.index);
        {
            ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

            EXPECT_TRUE(RandomReadTest(Aoc2Info.name, 0, 128, ZeroTimeSpan));
        }
    }
    ExpectNoMemoryLeak();
}

TEST(OperationOrder, AocRepeatCacheSwitchingWhileKeepingMounted)
{
    {
        ScopedAddOnContentMounter aoc(Aoc1Info.name, Aoc1Info.index);

        for (int i = 0; i < 5; i++)
        {
            EXPECT_TRUE(RandomReadTest(Aoc1Info.name, i * 2, 16, ZeroTimeSpan));
            {
                ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

                EXPECT_TRUE(RandomReadTest(Aoc1Info.name, i * 2 + 1, 16, ZeroTimeSpan));
            }
        }
    }
    ExpectNoMemoryLeak();
}

TEST(OperationOrder, AocRepeatMountSwitchingWhileKeepingCacheEnabled)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

        for (int i = 0; i < 5; i++)
        {
            ScopedAddOnContentMounter aoc(Aoc2Info.name, Aoc2Info.index);

            EXPECT_TRUE(RandomReadTest(Aoc2Info.name, i, 32, ZeroTimeSpan));
        }
    }
    ExpectNoMemoryLeak();
}

TEST(OperationOrder, RepeatCacheSwitchingWhileKeepingMultipleFileSystemMounted)
{
    {
        ScopedRomFsMounter romFs(RomMountName);
        ScopedAddOnContentMounter aoc1(Aoc1Info.name, Aoc1Info.index);
        ScopedAddOnContentMounter aoc2(Aoc2Info.name, Aoc2Info.index);

        for (int i = 0; i < 5; i++)
        {
            EXPECT_TRUE(RandomReadTest(RomMountName, i * 12 + 0, 16, ZeroTimeSpan));
            EXPECT_TRUE(RandomReadTest(Aoc1Info.name, i * 12 + 1, 16, ZeroTimeSpan));
            EXPECT_TRUE(RandomReadTest(Aoc2Info.name, i * 12 + 2, 16, ZeroTimeSpan));
            EXPECT_TRUE(RandomReadTest(RomMountName, i * 12 + 3, 16, ZeroTimeSpan));
            EXPECT_TRUE(RandomReadTest(Aoc1Info.name, i * 12 + 4, 16, ZeroTimeSpan));
            EXPECT_TRUE(RandomReadTest(Aoc2Info.name, i * 12 + 5, 16, ZeroTimeSpan));
            {
                ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

                EXPECT_TRUE(RandomReadTest(RomMountName, i * 12 + 6, 16, ZeroTimeSpan));
                EXPECT_TRUE(RandomReadTest(Aoc1Info.name, i * 12 + 7, 16, ZeroTimeSpan));
                EXPECT_TRUE(RandomReadTest(Aoc2Info.name, i * 12 + 8, 16, ZeroTimeSpan));
                EXPECT_TRUE(RandomReadTest(RomMountName, i * 12 + 9, 16, ZeroTimeSpan));
                EXPECT_TRUE(RandomReadTest(Aoc1Info.name, i * 12 + 10, 16, ZeroTimeSpan));
                EXPECT_TRUE(RandomReadTest(Aoc2Info.name, i * 12 + 11, 16, ZeroTimeSpan));
            }
        }
    }
    ExpectNoMemoryLeak();
}

TEST(OperationOrder, RepeatMultipleFileSystemMountSwitchingWhileKeepingCacheEnabled)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

        for (int i = 0; i < 3; i++)
        {
            {
                ScopedAddOnContentMounter aoc2(Aoc2Info.name, Aoc2Info.index);
                EXPECT_TRUE(RandomReadTest(Aoc2Info.name, i * 18, 32, ZeroTimeSpan));

                ScopedAddOnContentMounter aoc1(Aoc1Info.name, Aoc1Info.index);
                EXPECT_TRUE(RandomReadTest(Aoc1Info.name, i * 18 + 1, 32, ZeroTimeSpan));

                ScopedRomFsMounter romFs(RomMountName);
                EXPECT_TRUE(RandomReadTest(RomMountName, i * 18 + 2, 32, ZeroTimeSpan));

                aoc2.Unmount();

                EXPECT_TRUE(RandomReadTest(Aoc1Info.name, i * 18 + 3, 32, ZeroTimeSpan));
                EXPECT_TRUE(RandomReadTest(RomMountName, i * 18 + 4, 32, ZeroTimeSpan));

                romFs.Unmount();

                EXPECT_TRUE(RandomReadTest(Aoc1Info.name, i * 18 + 5, 32, ZeroTimeSpan));
            }
            {
                ScopedRomFsMounter romFs(RomMountName);
                EXPECT_TRUE(RandomReadTest(RomMountName, i * 18 + 6, 32, ZeroTimeSpan));

                ScopedAddOnContentMounter aoc2(Aoc2Info.name, Aoc2Info.index);
                EXPECT_TRUE(RandomReadTest(Aoc2Info.name, i * 18 + 7, 32, ZeroTimeSpan));

                ScopedAddOnContentMounter aoc1(Aoc1Info.name, Aoc1Info.index);
                EXPECT_TRUE(RandomReadTest(Aoc1Info.name, i * 18 + 8, 32, ZeroTimeSpan));

                romFs.Unmount();

                EXPECT_TRUE(RandomReadTest(Aoc2Info.name, i * 18 + 9, 32, ZeroTimeSpan));
                EXPECT_TRUE(RandomReadTest(Aoc1Info.name, i * 18 + 10, 32, ZeroTimeSpan));

                aoc1.Unmount();

                EXPECT_TRUE(RandomReadTest(Aoc2Info.name, i * 18 + 11, 32, ZeroTimeSpan));
            }
            {
                ScopedAddOnContentMounter aoc1(Aoc1Info.name, Aoc1Info.index);
                EXPECT_TRUE(RandomReadTest(Aoc1Info.name, i * 18 + 12, 32, ZeroTimeSpan));

                ScopedRomFsMounter romFs(RomMountName);
                EXPECT_TRUE(RandomReadTest(RomMountName, i * 18 + 13, 32, ZeroTimeSpan));

                ScopedAddOnContentMounter aoc2(Aoc2Info.name, Aoc2Info.index);
                EXPECT_TRUE(RandomReadTest(Aoc2Info.name, i * 18 + 14, 32, ZeroTimeSpan));

                aoc1.Unmount();

                EXPECT_TRUE(RandomReadTest(RomMountName, i * 18 + 15, 32, ZeroTimeSpan));
                EXPECT_TRUE(RandomReadTest(Aoc2Info.name, i * 18 + 16, 32, ZeroTimeSpan));

                aoc2.Unmount();

                EXPECT_TRUE(RandomReadTest(RomMountName, i * 18 + 17, 32, ZeroTimeSpan));
            }
        }
    }
    ExpectNoMemoryLeak();
}

// ----------------------------------------------------------------------------

TEST(CacheSize, SmallCacheSizeMultipleFileSystem)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(MinAcceptableCacheMemorySize);
        ScopedRomFsMounter romFs(RomMountName);
        ScopedAddOnContentMounter aoc1(Aoc1Info.name, Aoc1Info.index);
        ScopedAddOnContentMounter aoc2(Aoc2Info.name, Aoc2Info.index);

        EXPECT_TRUE(RandomReadTest(RomMountName,  0, 32, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(Aoc1Info.name, 1, 32, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(Aoc2Info.name, 2, 32, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(RomMountName,  3, 32, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(Aoc1Info.name, 4, 32, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(Aoc2Info.name, 5, 32, ZeroTimeSpan));
    }
    ExpectNoMemoryLeak();
}

TEST(CacheSize, LargeCacheSizeMultipleFileSystem)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(GetMaxAllocatableCacheMemorySize());
        ScopedRomFsMounter romFs(RomMountName);
        ScopedAddOnContentMounter aoc1(Aoc1Info.name, Aoc1Info.index);
        ScopedAddOnContentMounter aoc2(Aoc2Info.name, Aoc2Info.index);

        EXPECT_TRUE(RandomReadTest(RomMountName,  0, 32, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(Aoc1Info.name, 1, 32, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(Aoc2Info.name, 2, 32, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(RomMountName,  3, 32, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(Aoc1Info.name, 4, 32, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(Aoc2Info.name, 5, 32, ZeroTimeSpan));
    }
    ExpectNoMemoryLeak();
}

TEST(CacheSize, RandomCacheSizeMultipleFileSystem)
{
    int64_t seed = nn::os::GetSystemTick().GetInt64Value();
    std::mt19937 mt(static_cast<unsigned int>(seed));
    std::uniform_int_distribution<size_t> dist(MinAcceptableCacheMemorySize, GetMaxAllocatableCacheMemorySize());

    for (int i = 0; i < 16; i++)
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(dist(mt));
        ScopedRomFsMounter romFs(RomMountName);
        ScopedAddOnContentMounter aoc1(Aoc1Info.name, Aoc1Info.index);
        ScopedAddOnContentMounter aoc2(Aoc2Info.name, Aoc2Info.index);

        EXPECT_TRUE(RandomReadTest(RomMountName,  0, 4, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(Aoc1Info.name, 1, 4, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(Aoc2Info.name, 2, 4, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(RomMountName,  3, 4, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(Aoc1Info.name, 4, 4, ZeroTimeSpan));
        EXPECT_TRUE(RandomReadTest(Aoc2Info.name, 5, 4, ZeroTimeSpan));
    }
    ExpectNoMemoryLeak();
}

TEST(CacheSize, LargeCacheSizeHeavyTest)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(GetMaxAllocatableCacheMemorySize());
        ScopedAddOnContentMounter aoc6g(Aoc6gInfo.name, Aoc6gInfo.index);

        const int ThreadCount = 3;

        RandomReadTestArg testArgs[ThreadCount];
        for (int i = 0; i < ThreadCount; i++)
        {
            testArgs[i].mountName = Aoc6gInfo.name;
            testArgs[i].seed = i;
            testArgs[i].trialCount = 0;
            testArgs[i].trialTime = ZeroTimeSpan;
            testArgs[i].trialSize = 6LL * 1024 * 1024 * 1024;
        }

        for (int i = 0; i < ThreadCount; i++)
        {
            nn::os::CreateThread(
                &g_ThreadInfo[i].threadType,
                RandomReadTest,
                &testArgs[i],
                g_ThreadInfo[i].threadStack,
                g_ThreadInfo[i].ThreadStackSize,
                nn::os::DefaultThreadPriority,
                i);
            nn::os::StartThread(&g_ThreadInfo[i].threadType);
        }
        for (int i = 0; i < ThreadCount; i++)
        {
            nn::os::WaitThread(&g_ThreadInfo[i].threadType);
            nn::os::DestroyThread(&g_ThreadInfo[i].threadType);
        }
        for (int i = 0; i < ThreadCount; i++)
        {
            EXPECT_TRUE(testArgs[i].success);
        }
    }
    ExpectNoMemoryLeak();
}

// ----------------------------------------------------------------------------

namespace {

struct RomFsReadThreadArg
{
    int64_t seed;
    nn::TimeSpan trialTime;
};

void RomFsReadThreadFunc(void* theArg)
{
    RomFsReadThreadArg* arg = static_cast<RomFsReadThreadArg*>(theArg);

    nn::os::Tick startTick = nn::os::GetSystemTick();

    while (nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - startTick) < arg->trialTime)
    {
        ScopedRomFsMounter romFs(RomMountName);
        int64_t seed = nn::os::GetSystemTick().GetInt64Value();
        EXPECT_TRUE(RandomReadTest(RomMountName, seed, 16, ZeroTimeSpan));
    }
}

struct AocReadThreadArg
{
    const char* mountName;
    nn::aoc::AddOnContentIndex index;

    int64_t seed;
    nn::TimeSpan trialTime;
};

void AocReadThreadFunc(void* theArg)
{
    AocReadThreadArg* arg = static_cast<AocReadThreadArg*>(theArg);

    nn::os::Tick startTick = nn::os::GetSystemTick();

    while (nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - startTick) < arg->trialTime)
    {
        ScopedAddOnContentMounter aoc(arg->mountName, arg->index);
        int64_t seed = nn::os::GetSystemTick().GetInt64Value();
        EXPECT_TRUE(RandomReadTest(arg->mountName, seed, 16, ZeroTimeSpan));
    }
}

struct CacheSwitchingThreadArg
{
    int64_t seed;
    nn::TimeSpan trialTime;
};
void CacheSwitchingThreadFunc(void* theArg)
{
    CacheSwitchingThreadArg* arg = static_cast<CacheSwitchingThreadArg*>(theArg);

    std::mt19937 mt(arg->seed);
    std::uniform_int_distribution<int64_t> msSelector(0, 20);

    nn::os::Tick startTick = nn::os::GetSystemTick();

    while (nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - startTick) < arg->trialTime)
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(msSelector(mt)));
    }
}

}  // namespace unnamed

TEST(ThreadSafety, MultipleFileSystemReadAndCacheSwitching)
{
    CacheSwitchingThreadArg cacheSwitchingThreadArg;
    cacheSwitchingThreadArg.seed = nn::os::GetSystemTick().GetInt64Value();
    cacheSwitchingThreadArg.trialTime = nn::TimeSpan::FromSeconds(10);

    RomFsReadThreadArg romFsReadThreadArg;
    romFsReadThreadArg.seed = nn::os::GetSystemTick().GetInt64Value();
    romFsReadThreadArg.trialTime = nn::TimeSpan::FromSeconds(10);

    AocReadThreadArg aocReadThreadArg[2];
    aocReadThreadArg[0].index = Aoc1Info.index;
    aocReadThreadArg[0].mountName = Aoc1Info.name;
    aocReadThreadArg[0].seed = nn::os::GetSystemTick().GetInt64Value();
    aocReadThreadArg[0].trialTime = nn::TimeSpan::FromSeconds(10);
    aocReadThreadArg[1].index = Aoc2Info.index;
    aocReadThreadArg[1].mountName = Aoc2Info.name;
    aocReadThreadArg[1].seed = nn::os::GetSystemTick().GetInt64Value();
    aocReadThreadArg[1].trialTime = nn::TimeSpan::FromSeconds(10);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(
            &g_ThreadInfo[0].threadType,
            CacheSwitchingThreadFunc,
            &cacheSwitchingThreadArg,
            g_ThreadInfo[0].threadStack,
            g_ThreadInfo[0].ThreadStackSize,
            nn::os::DefaultThreadPriority - 1,
            0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(
            &g_ThreadInfo[1].threadType,
            RomFsReadThreadFunc,
            &romFsReadThreadArg,
            g_ThreadInfo[1].threadStack,
            g_ThreadInfo[1].ThreadStackSize,
            nn::os::DefaultThreadPriority,
            0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(
            &g_ThreadInfo[2].threadType,
            AocReadThreadFunc,
            &aocReadThreadArg[0],
            g_ThreadInfo[2].threadStack,
            g_ThreadInfo[2].ThreadStackSize,
            nn::os::DefaultThreadPriority,
            1));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(
            &g_ThreadInfo[3].threadType,
            AocReadThreadFunc,
            &aocReadThreadArg[1],
            g_ThreadInfo[3].threadStack,
            g_ThreadInfo[3].ThreadStackSize,
            nn::os::DefaultThreadPriority,
            2));

    for (int i = 0; i < 4; i++)
    {
        nn::os::StartThread(&g_ThreadInfo[i].threadType);
    }
    for (int i = 0; i < 4; i++)
    {
        nn::os::WaitThread(&g_ThreadInfo[i].threadType);
        nn::os::DestroyThread(&g_ThreadInfo[i].threadType);
    }

    ExpectNoMemoryLeak();
}

// ----------------------------------------------------------------------------

TEST(RomSize, MinimumMetadataRom)
{
    ScopedAddOnContentMounter aoc(AocMinimumMetadataRomInfo.name, AocMinimumMetadataRomInfo.index);
    {
        char path[16];
        ::snprintf(path, sizeof(path), "%s:/a", AocMinimumMetadataRomInfo.name);

        nn::fs::FileHandle fileHandle;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(fileHandle);
        };

        int64_t fileSize;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, fileHandle));
        ASSERT_EQ(fileSize, 64);
        {
            ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
            char buffer;
            const int64_t offset = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, offset, &buffer, sizeof(buffer)));
            EXPECT_EQ(buffer, static_cast<char>(offset & 0xff));
        }
        {
            ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
            char buffer;
            const int64_t offset = fileSize - 1;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, offset, &buffer, sizeof(buffer)));
            EXPECT_EQ(buffer, static_cast<char>(offset & 0xff));
        }
    }
}

// ----------------------------------------------------------------------------

TEST(Regression, SIGLO80958)
{
    ScopedAddOnContentMounter aoc(AocRegressionSIGLO80958Info.name, AocRegressionSIGLO80958Info.index);
    ScopedGlobalFileDataCacheEnabler fileDataCache(1 * 1024 * 1024);

    char path[16];
    ::snprintf(path, 16, "%s:/2097153", AocRegressionSIGLO80958Info.name);

    nn::fs::FileHandle file;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };

    int64_t fileSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));
    ASSERT_EQ(2 * 1024 * 1024 + 1, fileSize);

    auto buffer = nnt::fs::util::AllocateBuffer(2 * 1024 * 1024);
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, buffer.get(), 2 * 1024 * 1024));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(file, 2 * 1024 * 1024, buffer.get(), 1));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, buffer.get(), 2 * 1024 * 1024));
}
