﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <memory>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>

using namespace nn::fs;
using namespace nnt::fs::util;

namespace {
    void WaitAttachDetach(bool isWaitAttach)
    {
        if( IsSdCardInserted() == isWaitAttach )
        {
            return;
        }

        std::unique_ptr<nn::fs::IEventNotifier> deviceDetectionEventNotifier;
        NNT_ASSERT_RESULT_SUCCESS(OpenSdCardDetectionEventNotifier(&deviceDetectionEventNotifier));

        nn::os::SystemEventType deviceDetectionEvent;
        NNT_ASSERT_RESULT_SUCCESS(deviceDetectionEventNotifier->BindEvent(&deviceDetectionEvent, nn::os::EventClearMode_ManualClear));

        while(true)
        {

            NN_LOG("Waiting %s...\n", isWaitAttach ? "attach" : "detach");
            nn::os::WaitSystemEvent(&deviceDetectionEvent);
            nn::os::ClearSystemEvent(&deviceDetectionEvent);

            bool isSdCardInserted = IsSdCardInserted();
            NN_LOG("Detect %s.\n", isSdCardInserted ? "attach" : "detach");
            if( isSdCardInserted == isWaitAttach )
            {
                break;
            }
        }
    }

    void WaitAttach()
    {
        WaitAttachDetach(true);
    }

    void WaitDetach()
    {
        WaitAttachDetach(false);
    }
}

// 抜去後、正しくエラーが返るマウントになること
TEST(SdAttachDetach, FileAccessAfterDetach)
{
    WaitAttach();

    // sd をマウントしてファイルオープン
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd"));
    CreateFile("sd:/file", 1024);
    FileHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, "sd:/file", OpenMode_Write | OpenMode_Read));

    // デタッチ
    WaitDetach();

    // デタッチ後の新規マウントは失敗
    NNT_EXPECT_RESULT_FAILURE(ResultSdCardAccessFailed, MountSdCardForDebug("sd_fail"));

    // 既オープンファイルのアクセスは失敗
    char buffer[32];
    NNT_EXPECT_RESULT_FAILURE(ResultSdCardAccessFailed, WriteFile(handle, 0, buffer, sizeof(buffer), WriteOption()));

    // クローズは可能
    CloseFile(handle);

    // アンマウント可能
    Unmount("sd");
}


// 同じ SD で再アタッチ後、別マウントで再アクセス可能になること
// 古いマウントは復活しないこと
TEST(SdReattach, FileAccessAfterReAttach)
{
    WaitAttach();

    // sd をマウントしてファイルオープン
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd"));
    CreateFile("sd:/file", 1024);
    FileHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, "sd:/file", OpenMode_Write | OpenMode_Read));

    // デタッチ
    WaitDetach();

    // 同じ SD をアタッチ
    WaitAttach();

    // 別マウント
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd2"));

    // 古いマウントから開いたハンドルでのファイルアクセスは失敗
    char buffer[32];
    NNT_EXPECT_RESULT_FAILURE(ResultSdCardAccessFailed, WriteFile(handle, 0, buffer, sizeof(buffer), WriteOption()));

    // クローズは可能
    CloseFile(handle);

    // 新しいマウントから開いたハンドルでのファイルアクセスは可能
    FileHandle handle2;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle2, "sd2:/file", OpenMode_Write | OpenMode_Read));
    char buffer2[32];
    NNT_EXPECT_RESULT_SUCCESS(WriteFile(handle2, 0, buffer2, sizeof(buffer2), WriteOption()));
    NNT_EXPECT_RESULT_SUCCESS(FlushFile(handle2));
    CloseFile(handle2);


    // アンマウント可能
    Unmount("sd");
    Unmount("sd2");
}

// 同じ SD で再アタッチ後、別マウントで再アクセス可能になり、先のアンマウントにより阻害されないこと
// 古いマウントは復活しないこと
TEST(SdReattach, FileAccessAfterReAttachUnmount)
{
    WaitAttach();

    // sd をマウントしてファイルオープン
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd"));
    CreateFile("sd:/file", 1024);
    FileHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, "sd:/file", OpenMode_Write | OpenMode_Read));

    // デタッチ
    WaitDetach();

    // 同じ SD をアタッチ
    WaitAttach();

    // 別マウント
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd2"));

    // 新しいマウントからファイルオープン
    FileHandle handle2;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle2, "sd2:/file", OpenMode_Write | OpenMode_Read));

    // 古いマウントから開いたハンドルをクローズ
    CloseFile(handle);

    // 古いマウントをアンマウント
    Unmount("sd");

    // 新しいマウントから開いたハンドルでのファイルアクセスは可能
    char buffer2[32];
    NNT_EXPECT_RESULT_SUCCESS(WriteFile(handle2, 0, buffer2, sizeof(buffer2), WriteOption()));
    NNT_EXPECT_RESULT_SUCCESS(FlushFile(handle2));
    CloseFile(handle2);

    Unmount("sd2");
}

// 別 SD を再アタッチ後、新しい内容が読めること
// 複数マウントでも新旧での有効無効が正しく振る舞うこと
TEST(SdReattach, FileAccessAfterReAttachAnotherSd)
{
    WaitAttach();

    // sd をマウント
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd1"));
    CreateFile("sd1:/file1", 1024);

    // アクセス可能
    NNT_ASSERT_RESULT_SUCCESS(DumpDirectoryRecursive("sd1:/"));

    // ファイルオープン
    FileHandle handle1;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle1, "sd1:/file", OpenMode_Write | OpenMode_Read));

    // 別マウント
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd2"));


    // デタッチ
    WaitDetach();

    // 別の SD をアタッチ
    WaitAttach();

    // さらに別マウント
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd3"));

    // 古いマウントから開いたハンドルでのファイルアクセスは失敗
    char buffer[32];
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSdCardAccessFailed, WriteFile(handle1, 0, buffer, sizeof(buffer), WriteOption()));

    // 古いマウントへのアクセスは失敗
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSdCardAccessFailed, CreateFile("sd1:/fail", 1024));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSdCardAccessFailed, CreateFile("sd2:/fail", 1024));

    // 古いハンドルのクローズは可能
    CloseFile(handle1);

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSdCardAccessFailed, DumpDirectoryRecursive("sd1:/"));

    // 新しいマウントへのアクセスは成功
    NNT_EXPECT_RESULT_SUCCESS(CreateFile("sd3:/success", 1024));
    NNT_EXPECT_RESULT_SUCCESS(DeleteFile("sd3:/success"));

    // 内容が別になっていることを確認
    NNT_ASSERT_RESULT_SUCCESS(DumpDirectoryRecursive("sd3:/"));

    // さらに別マウント
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd4"));

    // 新しいマウントへのアクセスは成功
    NNT_EXPECT_RESULT_SUCCESS(CreateFile("sd4:/success", 1024));
    NNT_EXPECT_RESULT_SUCCESS(DeleteFile("sd4:/success"));


    // 新しいマウントから開いたハンドルでのファイルアクセスは可能
    FileHandle handle2;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle2, "sd3:/file", OpenMode_Write | OpenMode_Read));
    char buffer2[32];
    NNT_EXPECT_RESULT_SUCCESS(WriteFile(handle2, 0, buffer2, sizeof(buffer), WriteOption()));
    NNT_EXPECT_RESULT_SUCCESS(FlushFile(handle2));
    CloseFile(handle2);

    // アンマウント可能
    Unmount("sd1");
    Unmount("sd2");
    Unmount("sd3");
    Unmount("sd4");

}


namespace {
    void CreateSystemSaveDataWithFile(SaveDataId saveDataId, int index)
    {
        const int FileSize = 32;

        NNT_ASSERT_RESULT_SUCCESS(CreateSystemSaveData(SaveDataSpaceId::SdSystem, saveDataId, InvalidUserId, 0, 48 * 1024, 48 * 1024, 0));
        NNT_ASSERT_RESULT_SUCCESS(MountSystemSaveData("save", SaveDataSpaceId::SdSystem, saveDataId, InvalidUserId));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith32BitCount("save:/file", FileSize, index * 4));
        NNT_ASSERT_RESULT_SUCCESS(CommitSaveData("save"));
        Unmount("save");
    }

    void VerifyFile(SaveDataId saveDataId, int index)
    {
        NNT_ASSERT_RESULT_SUCCESS(MountSystemSaveData("save", SaveDataSpaceId::SdSystem, saveDataId, InvalidUserId));
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("save:/file", index * 4));
        Unmount("save");
    }
}


// 別の SD カードで別のシステムセーブを扱えること
TEST(SdAttachDetach, SystemSaveData)
{
    WaitDetach();

    const SaveDataId Id[4] =
    {
        0x8000000000004000,
        0x8000000000004001,
        0x8000000000004002,
        0x8000000000004003,
    };

    // SD A
    NN_LOG("*** Waiting SDCard A\n");
    WaitAttach();
    DeleteAllTestSaveData();

    // A にセーブ 0, 1 作成
    CreateSystemSaveDataWithFile(Id[0], 0xA0);
    CreateSystemSaveDataWithFile(Id[1], 0xA1);


    // 列挙結果と中身を確認
    Vector<nn::fs::SaveDataInfo> infos;
    FindSaveData(&infos, SaveDataSpaceId::SdSystem,
        [](const nn::fs::SaveDataInfo& info)
        {
            return true;
        }
    );
    EXPECT_EQ(2, infos.size());
    EXPECT_EQ(Id[0], infos[0].saveDataId);
    EXPECT_EQ(Id[1], infos[1].saveDataId);

    VerifyFile(Id[0], 0xA0);
    VerifyFile(Id[1], 0xA1);


    // SD B に差し替え
    WaitDetach();
    NN_LOG("*** Waiting SDCard B\n");
    WaitAttach();
    DeleteAllTestSaveData();

    // B にセーブ 0, 2 作成
    CreateSystemSaveDataWithFile(Id[0], 0xB0);
    CreateSystemSaveDataWithFile(Id[2], 0xB2);


    // 列挙結果と中身を確認
    FindSaveData(&infos, SaveDataSpaceId::SdSystem,
        [](const nn::fs::SaveDataInfo& info)
    {
        return true;
    }
    );
    EXPECT_EQ(2, infos.size());
    EXPECT_EQ(Id[0], infos[0].saveDataId);
    EXPECT_EQ(Id[2], infos[1].saveDataId);

    VerifyFile(Id[0], 0xB0);
    VerifyFile(Id[2], 0xB2);


    // SD A 再
    WaitDetach();
    NN_LOG("*** Waiting SDCard A\n");
    WaitAttach();


    // 列挙結果と中身を確認
    FindSaveData(&infos, SaveDataSpaceId::SdSystem,
        [](const nn::fs::SaveDataInfo& info)
    {
        return true;
    }
    );
    EXPECT_EQ(2, infos.size());
    EXPECT_EQ(Id[0], infos[0].saveDataId);
    EXPECT_EQ(Id[1], infos[1].saveDataId);

    VerifyFile(Id[0], 0xA0);
    VerifyFile(Id[1], 0xA1);

    DeleteAllTestSaveData(); // teardown



    // SD B 再
    WaitDetach();
    NN_LOG("*** Waiting SDCard B\n");
    WaitAttach();

    // 列挙結果と中身を確認
    FindSaveData(&infos, SaveDataSpaceId::SdSystem,
        [](const nn::fs::SaveDataInfo& info)
    {
        return true;
    }
    );
    EXPECT_EQ(2, infos.size());
    EXPECT_EQ(Id[0], infos[0].saveDataId);
    EXPECT_EQ(Id[2], infos[1].saveDataId);

    VerifyFile(Id[0], 0xB0);
    VerifyFile(Id[2], 0xB2);

    DeleteAllTestSaveData(); // teardown
}

// SD カード差し替えでシステムセーブのマウントが無効化すること
TEST(SdAttachDetach, SystemSaveDataInvalidated)
{
    WaitDetach();

    const SaveDataId Id = 0x8000000000004000;

    // SD A
    NN_LOG("*** Waiting SDCard A\n");
    WaitAttach();
    DeleteAllTestSaveData();
    CreateSystemSaveDataWithFile(Id, 0xA0);

    NNT_ASSERT_RESULT_SUCCESS(MountSystemSaveData("saveA", SaveDataSpaceId::SdSystem, Id, InvalidUserId));

    // SD B に差し替え
    WaitDetach();
    NN_LOG("*** Waiting SDCard B\n");
    WaitAttach();

    // 古いマウントへのアクセスは失敗
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSdCardAccessFailed, CreateFile("saveA:/fail", 1024));

    // Save B 作成＆アクセス可能
    CreateSystemSaveDataWithFile(Id, 0xB0);
    NNT_ASSERT_RESULT_SUCCESS(MountSystemSaveData("saveB", SaveDataSpaceId::SdSystem, Id, InvalidUserId));

    // 古いマウントへのアクセスは引き続き失敗
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSdCardAccessFailed, CreateFile("saveA:/fail", 1024));
    Unmount("saveA");

    Unmount("saveB");

    DeleteAllTestSaveData();
}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nnt::fs::util::ResetAllocateCount();

    SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
