﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fssystem/dbm/fs_DbmParameters.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h.>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util_function.h>
#include <nnt/fsApi/testFs_Integration_Api.h>
#include <nnt/fsApiRo/testFs_ApiRo.h>

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {
    const int HeapSize = 10 * 1024 * 1024;
    char g_HeapStack[HeapSize];

    // SaveData Format Parameters
    const int64_t SaveDataSize = 352 * 1024 * 1024;
    const int64_t SaveDataJournalSize = 352 * 1024 * 1024;

    const auto StorageSize = SaveDataSize;

    const nn::ncm::ApplicationId TestApplicationId = {0x0001000C10000000};
    const char* MountName = "testsave";
    const char* RootPath = "testsave:";
    nn::fs::UserId g_UserId;

    void TestCreateDirectory(const char* mountName, const char* dirPath)
    {
        char path[256];
        nn::util::SNPrintf(path, sizeof(path), "%s:/%s", mountName, dirPath);
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory(path));
    }

    void TestCreateZeroFile(const char* mountName, const char* filePath, unsigned int size)
    {
        char path[256];
        nn::util::SNPrintf(path, sizeof(path), "%s:/%s", mountName, filePath);
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWithZero(path, size, 0));
    }

    void TestCreateFilledFile(const char* mountName, const char* filePath, unsigned int size)
    {
        char path[256];
        nn::util::SNPrintf(path, sizeof(path), "%s:/%s", mountName, filePath);
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith8BitCount(path, size, 0));
    }

    void TestCreateFilesInOrder(const char* mountName, const char* basePath, unsigned int size, unsigned int num)
    {
        char path[256];
        for (unsigned int i = 0; i < num; i++)
        {
            nn::util::SNPrintf(path, sizeof(path), "%s/%04u.file", basePath, i);
            TestCreateFilledFile(mountName, path, size);
        }
    }

    void CreateTestResource(const char* mountName)
    {
        char path[256];

        TestCreateDirectory(mountName, "dir0");
        TestCreateZeroFile(mountName, "file0.dat", 0x100);
        TestCreateZeroFile(mountName, "dir0/file1.dat", 0x100);

        TestCreateDirectory(mountName, "BufferAddressAlignmentRo");
        TestCreateFilledFile(mountName, "BufferAddressAlignmentRo/test.file", 0x800);

        TestCreateDirectory(mountName, "EntryCompositionRo");
        nn::util::SNPrintf(path, sizeof(path), "%s:/EntryCompositionRo/", mountName);
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateTestDirectory(nullptr, nullptr, path, 13, nnt::fs::util::DIRTYPE_NEST, true, 0x400));
        TestCreateFilesInOrder(mountName, "EntryCompositionRo/0", 0x400, 4095);

        TestCreateDirectory(mountName, "OperateRangeRo");
        TestCreateZeroFile(mountName, "OperateRangeRo/test.file", 0x400);

        const char Utf8String[] = {static_cast<char>(-29), static_cast<char>(-127), static_cast<char>(-126), '\0'}; // "あ\0" in utf-8
        TestCreateDirectory(mountName, "PathArgumentRo");
        nn::util::SNPrintf(path, sizeof(path), "PathArgumentRo/%s", Utf8String);
        TestCreateDirectory(mountName, path);
        nn::util::SNPrintf(path, sizeof(path), "PathArgumentRo/%s.file", Utf8String);
        TestCreateZeroFile(mountName, path, 0x20);
        nn::util::SNPrintf(path, sizeof(path), "PathArgumentRo/%s/dummy", Utf8String);
        TestCreateZeroFile(mountName, path, 0x0);
        TestCreateDirectory(mountName, "PathArgumentRo/RootExcess");
        TestCreateZeroFile(mountName, "PathArgumentRo/RootExcess/dummy", 0x0);

        TestCreateDirectory(mountName, "PointerArgumentRo");
        TestCreateDirectory(mountName, "PointerArgumentRo/test");
        TestCreateZeroFile(mountName, "PointerArgumentRo/test.file", 0x20);
        TestCreateZeroFile(mountName, "PointerArgumentRo/test/dummy", 0x0);

        TestCreateDirectory(mountName, "PostConditionRo");
        TestCreateDirectory(mountName, "PostConditionRo/test");
        TestCreateDirectory(mountName, "PostConditionRo/test/test");
        TestCreateFilledFile(mountName, "PostConditionRo/test.file", 0x20);
        TestCreateZeroFile(mountName, "PostConditionRo/test/test.file", 0x20);
        TestCreateZeroFile(mountName, "PostConditionRo/test/test/dummy", 0x0);

        TestCreateDirectory(mountName, "PreConditionRo");
        TestCreateDirectory(mountName, "PreConditionRo/dir");
        TestCreateZeroFile(mountName, "PreConditionRo/test.file", 0x20);
        TestCreateZeroFile(mountName, "PreConditionRo/dir/test.file", 0x0);

        TestCreateDirectory(mountName, "RaceConditionRo");
        TestCreateDirectory(mountName, "RaceConditionRo/Test");
        TestCreateFilesInOrder(mountName, "RaceConditionRo", 0x800, 2);
        TestCreateFilesInOrder(mountName, "RaceConditionRo/Test", 0x800, 10);
        TestCreateFilledFile(mountName, "RaceConditionRo/file.dat", 0x10000000);

        TestCreateDirectory(mountName, "ResourceOpenCountRo");
        TestCreateDirectory(mountName, "ResourceOpenCountRo/Test");
        TestCreateFilesInOrder(mountName, "ResourceOpenCountRo/Test", 0x400, 1024);
        nn::util::SNPrintf(path, sizeof(path), "%s:/ResourceOpenCountRo/Test/", mountName);
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateTestDirectory(nullptr, nullptr, path, 64, nnt::fs::util::DIRTYPE_FLAT, true, 0));

        TestCreateDirectory(mountName, "SizeArgumentRo");
        TestCreateFilledFile(mountName, "SizeArgumentRo/0.file", 0x0);
        TestCreateFilledFile(mountName, "SizeArgumentRo/32.file", 0x20);
        TestCreateFilledFile(mountName, "SizeArgumentRo/4096.file", 0x1000);
        TestCreateFilledFile(mountName, "SizeArgumentRo/16384.file", 0x4000);

        TestCreateDirectory(mountName, "TargetExistenceRo");
        TestCreateDirectory(mountName, "TargetExistenceRo/testDir");
        TestCreateZeroFile(mountName, "TargetExistenceRo/test.file", 0);
        TestCreateZeroFile(mountName, "TargetExistenceRo/testDir/test.file", 0);

        TestCreateDirectory(mountName, "TargetInUseRo");
        TestCreateDirectory(mountName, "TargetInUseRo/lockedDir");
        TestCreateZeroFile(mountName, "TargetInUseRo/locked.file", 0);
        TestCreateZeroFile(mountName, "TargetInUseRo/lockedDir/locked.file", 0);
    }

    void CreateTestSaveData(nn::fs::UserId userId)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(TestApplicationId, userId, TestApplicationId.value, SaveDataSize, SaveDataJournalSize, 0));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName, userId));

        CreateTestResource(MountName);

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
        nn::fs::Unmount(MountName);
    }

    nn::fs::UserId GetUserId()
    {
        nn::account::Uid uid;
        nn::fs::UserId userId;

        // アカウントライブラリを初期化
        nn::account::Initialize();

        nn::Result result;
        uid = nn::account::InvalidUid;
        int userCount = 0;
        result = nn::account::ListAllUsers(&userCount, &uid, 1);
        NN_ASSERT(result.IsSuccess() && userCount > 0);

        userId._data[0] = uid._data[0];
        userId._data[1] = uid._data[1];
        return userId;
    }
}

namespace nnt { namespace fs { namespace api {

    class FsApiSaveDataReadOnlyFileSystem : public FsApiIntegrationTestFileSystem
    {
    public:
        FsApiSaveDataReadOnlyFileSystem() NN_NOEXCEPT
             : FsApiIntegrationTestFileSystem(MountName)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSaveDataReadOnly(MountName, TestApplicationId, g_UserId));
        }

        virtual ~FsApiSaveDataReadOnlyFileSystem() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::fs::Unmount(MountName);
        }
    };

    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index)
    {
        NN_UNUSED(index);
        std::unique_ptr<FsApiSaveDataReadOnlyFileSystem> saveDataRoFs(new FsApiSaveDataReadOnlyFileSystem());

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Integration));

        info->fileSystem = std::move(saveDataRoFs);
        info->type = FileSystemType_SaveDataFileSystem;
        info->rootDirPath = RootPath;

        auto& attribute = info->attribute;

        attribute.directoryNameLengthMax = nn::fssystem::dbm::MaxDirectoryNameLength;
        attribute.fileNameLengthMax      = nn::fssystem::dbm::MaxFileNameLength;
        // EntryNameLengthMax 以上のパスが fs_SavePath で弾かれるので設定できるのは EntryNameLengthMax - 1 まで
        attribute.directoryPathLengthMax = nn::fs::EntryNameLengthMax - 1;
        attribute.filePathLengthMax      = nn::fs::EntryNameLengthMax - 1;
        attribute.fileSizeMax = 0xFFFFFFFF;
        attribute.storageSize = StorageSize;

        attribute.fileOpenMax      = 1024;
        attribute.directoryOpenMax = 64;

        attribute.speedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::MmcStorageContextEnabled);

        attribute.isSupportedMultiBytePath = true;
        attribute.isSupportedGetFreeSpaceSize = true;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedQueryRange = true;

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllRoTests();

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    nn::fs::SetEnabledAutoAbort(false);

    nnt::fs::util::DeleteAllTestSaveData();
    g_UserId = GetUserId();
    CreateTestSaveData(g_UserId);

    auto ret = RUN_ALL_TESTS();

    nnt::fs::util::DeleteAllTestSaveData();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
