﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_DbmRomPathTool.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Integration_Api.h>
#include <nnt/fsApiRo/testFs_ApiRo.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h.>

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {
    const char* const MountName = "rom";
    const char* const RootPath = "rom:";
}

namespace nnt { namespace fs { namespace api {
    class FsApiRomFileSystem : public FsApiIntegrationTestFileSystem
    {
    public:
        FsApiRomFileSystem() NN_NOEXCEPT
            : FsApiIntegrationTestFileSystem(MountName),
              m_CacheBuffer(nullptr, util::DeleterBuffer)
        {
            size_t cacheBufferSize = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::QueryMountRomCacheSize(&cacheBufferSize)
            );
            m_CacheBuffer = util::AllocateBuffer(cacheBufferSize);
            NN_ABORT_UNLESS_NOT_NULL(m_CacheBuffer);

            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::MountRom(
                    MountName,
                    m_CacheBuffer.get(),
                    cacheBufferSize
                )
            );
        }

        virtual ~FsApiRomFileSystem() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::fs::Unmount(MountName);
        }

    private:
        decltype(util::AllocateBuffer(0)) m_CacheBuffer;
    };

    // HostFs 版 Rom（raw 起動）かどうかを判定
    bool IsEmulatedRom() NN_NOEXCEPT
    {
        size_t size;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&size));
        return size <= 32;
    }

    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index) NN_NOEXCEPT
    {
        NN_UNUSED(index);
        std::unique_ptr<FsApiRomFileSystem> romFs(new FsApiRomFileSystem());

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Integration));
        info->fileSystem = std::move(romFs);
        info->type = FileSystemType_RomFileSystem;
        info->rootDirPath = RootPath;

        auto& attribute = info->attribute;

        static const int NameLengthMax  = nn::fs::RomPathTool::MaxPathLength;
        static const int PathLengthMax  = nn::fs::RomPathTool::MaxPathLength;
        attribute.directoryNameLengthMax = NameLengthMax;
        attribute.fileNameLengthMax      = NameLengthMax;
        attribute.directoryPathLengthMax = PathLengthMax;
        attribute.filePathLengthMax      = PathLengthMax;

        attribute.fileSizeMax = 0x0000100000000000;
        attribute.storageSize = 0x0000100000000000;

        attribute.fileOpenMax      = 1024;
        attribute.directoryOpenMax = 64;

        attribute.isSupportedMultiBytePath = true;
        attribute.isSupportedGetFreeSpaceSize = false;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedQueryRange = true;

        attribute.fileSizeAlignment = 1;

        attribute.speedEmulationTypeFlag = 0;
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_WIN)
        attribute.aesCtrKeyTypeFlag = 0;
#else
        if( IsEmulatedRom() )
        {
            attribute.aesCtrKeyTypeFlag = 0;
        }
        else
        {
#if defined(FS_TEST_SYSTEM_PROGRAM)
            attribute.aesCtrKeyTypeFlag = static_cast<int32_t>(nn::fs::AesCtrKeyTypeFlag::InternalKeyForSoftwareAes);
#else
            attribute.aesCtrKeyTypeFlag = static_cast<int32_t>(nn::fs::AesCtrKeyTypeFlag::InternalKeyForHardwareAes);
#endif // FS_TEST_SYSTEM_PROGRAM
        }
#endif // NN_BUILD_CONFIG_OS_SUPPORTS_WIN

        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
#if defined(FS_TEST_SYSTEM_PROGRAM)
    nn::fs::InitializeWithMultiSessionForSystem();
#endif // defined(FS_TEST_SYSTEM_PROGRAM)

    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllRoTests();

    const int HeapSize = 1024 * 1024;
    std::unique_ptr<char[]> heapStack(new char[HeapSize]);

    nnt::fs::util::InitializeTestLibraryHeap(heapStack.get(), HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto ret = 0;
    {
        nnt::fs::util::GlobalFileDataCache dataCache(argc, argv);

        ret = RUN_ALL_TESTS();
    }

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
