﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_DbmRomPathTool.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Integration_Api.h>
#include <nnt/fsApiRo/testFs_ApiRo.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h.>

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {

const char* const MountName = "rom";
const char* const RootPath = "rom:";

class FsApiAocFileSystem : public nnt::fs::api::FsApiIntegrationTestFileSystem
{
public:
    FsApiAocFileSystem() NN_NOEXCEPT
        : FsApiIntegrationTestFileSystem(MountName),
          m_CacheBuffer(nullptr, nnt::fs::util::DeleterBuffer)
    {
        const nn::aoc::AddOnContentIndex index = 1;
        size_t cacheSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountAddOnContentCacheSize(&cacheSize, index));
        m_CacheBuffer = nnt::fs::util::AllocateBuffer(cacheSize);
        NN_ABORT_UNLESS_NOT_NULL(m_CacheBuffer);
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::MountAddOnContent(MountName, index, m_CacheBuffer.get(), cacheSize));
    }

    virtual ~FsApiAocFileSystem() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::Unmount(MountName);
    }

private:
    decltype(nnt::fs::util::AllocateBuffer(0)) m_CacheBuffer;
};

}

namespace nnt { namespace fs { namespace api {

void GetTestFileSystemInfo(std::unique_ptr<nnt::fs::api::TestFileSystemInfo>* outValue, int index)
{
    NN_UNUSED(index);
    std::unique_ptr<FsApiAocFileSystem> aoc(new FsApiAocFileSystem());

    std::unique_ptr<nnt::fs::api::TestFileSystemInfo> info(new nnt::fs::api::TestFileSystemInfo(FsApiTestType::Integration));
    info->fileSystem = std::move(aoc);
    info->type = nnt::fs::api::FileSystemType_RomFileSystem;
    info->rootDirPath = RootPath;

    auto& attribute = info->attribute;

    static const int NameLengthMax  = nn::fs::RomPathTool::MaxPathLength;
    static const int PathLengthMax  = nn::fs::RomPathTool::MaxPathLength;
    attribute.directoryNameLengthMax = NameLengthMax;
    attribute.fileNameLengthMax      = NameLengthMax;
    attribute.directoryPathLengthMax = PathLengthMax;
    attribute.filePathLengthMax      = PathLengthMax;

    attribute.fileSizeMax = 0x0000100000000000;
    attribute.storageSize = 0x0000100000000000;

    attribute.fileOpenMax      = 1024;
    attribute.directoryOpenMax = 64;

    attribute.isSupportedMultiBytePath = true;
    attribute.isSupportedGetFreeSpaceSize = false;
    attribute.isConcatenationFileSystem = false;
    attribute.isSupportedQueryRange = true;

    attribute.fileSizeAlignment = 1;

    attribute.aesCtrKeyTypeFlag = static_cast<int32_t>(nn::fs::AesCtrKeyTypeFlag::InternalKeyForSoftwareAes);
    attribute.speedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::MmcStorageContextEnabled)
                                     | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::SdCardStorageContextEnabled)
                                     | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::GameCardStorageContextEnabled)
                                     | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::UsbStorageContextEnabled);

    *outValue = std::move(info);
}

}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllRoTests();

    const int HeapSize = 1024 * 1024;
    std::unique_ptr<char[]> heapStack(new char[HeapSize]);

    nnt::fs::util::InitializeTestLibraryHeap(heapStack.get(), HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto ret = 0;
    {
        nnt::fs::util::GlobalFileDataCache dataCache(argc, argv);

        ret = RUN_ALL_TESTS();
    }

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
