﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/aoc.h>
#include <nn/util/util_Optional.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/fs/fs_ResultHandler.h>


using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nn::util;
using namespace nnt::fs::util;


namespace {

const int MountAocCountMax = 128;


//! @pre Aoc がインストール済み
TEST(AddOnContent, ListAndMount)
{
    nn::Result result;
    NN_LOG("AocSimple\n");

    // 追加コンテンツ数の取得
    int aocCount = nn::aoc::CountAddOnContent();
    NN_LOG("CountAddOnCountent -> %d\n", aocCount);

    // 追加コンテンツのリストアップ
    const int MaxListupCount = 256;
    nn::aoc::AddOnContentIndex aocList[MaxListupCount];
    int listupCount = nn::aoc::ListAddOnContent(aocList, 0, MaxListupCount);
    NN_LOG("ListAddOnCountent  -> %d\n", listupCount);
    for (int i = 0; i < listupCount; ++i)
    {
        NN_LOG("  Index[%d]: %d\n", i, aocList[i]);
    }

    // リストアップされた追加コンテンツを読み取り、表示する
    for (int i = 0; i < listupCount; ++i)
    {
        NNT_FS_SCOPED_TRACE("%s %d: %d\n", __FUNCTION__, __LINE__, i);

        // アクセス対象のインデックス
        auto index = aocList[i];

        // 追加コンテンツのマウント
        size_t cacheSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(QueryMountAddOnContentCacheSize(&cacheSize, index));

        auto mountCacheBuffer = AllocateBuffer(cacheSize);
        NN_ASSERT_NOT_NULL(mountCacheBuffer);

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountAddOnContent("aoc", index, mountCacheBuffer.get(), cacheSize));
        NNT_EXPECT_RESULT_SUCCESS(DumpDirectoryRecursive("aoc:/"));
        nn::fs::Unmount("aoc");
    }
}

//! @pre index 1-2048 の Aoc がインストール済み、public なシステムデータインストール済み
TEST(AddOnContent, Mount128)
{
    // 128個同時にマウントできること
    optional<decltype(AllocateBuffer(0))> mountCacheBufferArray[MountAocCountMax + 1];

    for (int i = 0; i < MountAocCountMax; i++)
    {
        NNT_FS_SCOPED_TRACE("%s %d: %d\n", __FUNCTION__, __LINE__, i);

        size_t cacheSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(QueryMountAddOnContentCacheSize(&cacheSize, static_cast<nn::aoc::AddOnContentIndex>(i + 1)));

        auto buffer = AllocateBuffer(cacheSize);
        mountCacheBufferArray[i].emplace(std::move(buffer));

        char mountName[32];
        util::SNPrintf(mountName, sizeof(mountName), "aoc%d", i);

        NNT_ASSERT_RESULT_SUCCESS(MountAddOnContent(mountName, static_cast<nn::aoc::AddOnContentIndex>(i + 1), mountCacheBufferArray[i]->get(), cacheSize));
    }

    // アクセスできること
    for (int i = 0; i < MountAocCountMax; i++)
    {
        char path[32];
        util::SNPrintf(path, sizeof(path), "aoc%d:/", i);
        NNT_EXPECT_RESULT_SUCCESS(DumpDirectoryRecursive(path));
    }

    // 128個マウント状態でもシステムデータはマウントできること
    {
        const ncm::SystemDataId NgWordId = { 0x0100000000000806 };
        NNT_ASSERT_RESULT_SUCCESS(MountSystemData("systemData", NgWordId));
        Unmount("systemData");
    }


    // 128個までのマウント数制限がかかっていること
    {
        size_t cacheSize = 0;

        // 実装上、 QueryMountAddOnContentCacheSize() も出来ない
        NNT_EXPECT_RESULT_FAILURE(ResultOpenCountLimit, QueryMountAddOnContentCacheSize(&cacheSize, static_cast<nn::aoc::AddOnContentIndex>(MountAocCountMax + 1)));

        cacheSize = 1024 * 1024;
        auto buffer = AllocateBuffer(cacheSize);
        NNT_EXPECT_RESULT_FAILURE(ResultOpenCountLimit, MountAddOnContent("aoc128", static_cast<nn::aoc::AddOnContentIndex>(MountAocCountMax + 1), buffer.get(), cacheSize));
    }

    // アンマウントで枠が空くこと
    {
        Unmount("aoc0");

        size_t cacheSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(QueryMountAddOnContentCacheSize(&cacheSize, MountAocCountMax));
        auto buffer = AllocateBuffer(cacheSize);
        NNT_EXPECT_RESULT_SUCCESS(MountAddOnContent("aoc128", static_cast<nn::aoc::AddOnContentIndex>(MountAocCountMax + 1), buffer.get(), cacheSize));

        Unmount("aoc128");
    }


    // teardown
    for (int i = 1; i < MountAocCountMax; i++)
    {
        char mountName[32];
        util::SNPrintf(mountName, sizeof(mountName), "aoc%d", i);
        Unmount(mountName);
    }

}

const int LoopCount = 1024;

void ReadThreadFunction(void* arg) NN_NOEXCEPT
{
    int index = static_cast<int>(reinterpret_cast<intptr_t>(arg));

    char path[32];
    util::SNPrintf(path, sizeof(path), "aoc%d:/", index);

    const int BufferSize = 16 * 1024;
    auto buffer = AllocateBuffer(BufferSize);

    for (int i = 0; i < LoopCount; i++)
    {
        if (index == 0)
        {
            NN_LOG("%s %d: %d/%d\n", __FUNCTION__, __LINE__, i, LoopCount);
        }

        auto result = DumpDirectoryRecursive(path, true, buffer.get(), BufferSize, true);
        if (result.IsFailure())
        {
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        nn::os::YieldThread();
    }
}


void ReadRom() NN_NOEXCEPT
{
    const int RomBufferSize = 16 * 1024;
    auto romBuffer = AllocateBuffer(RomBufferSize);

    NNT_ASSERT_RESULT_SUCCESS(MountRom("rom", romBuffer.get(), RomBufferSize));

    const int BufferSize = 16 * 1024;
    auto buffer = AllocateBuffer(BufferSize);

    for (int i = 0; i < LoopCount; i++)
    {
        auto result = DumpDirectoryRecursive("rom:/", true, buffer.get(), BufferSize, true);
        if (result.IsFailure())
        {
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        nn::os::YieldThread();
    }

    Unmount("rom");
}

const int ThreadCount = 32;
const size_t StackSize = 8192;
static NN_ALIGNAS(4096) uint8_t g_Stack[ThreadCount][StackSize];
nn::os::ThreadType g_Threads[ThreadCount];

//! @pre
//!   - 0 - 127 の Aoc がインストール済み
//!   - アプリと Aoc が外部鍵版である
// TODO: 外部鍵パッチも足す
TEST(AddOnContent, Mount128ConcurrentAccessHeavy)
{
    // 最大 32 個同時にマウントし、同時アクセスできること

    optional<decltype(AllocateBuffer(0))> mountCacheBufferArray[MountAocCountMax + 1];

    for (int i = 0; i < MountAocCountMax; i++)
    {
        NNT_FS_SCOPED_TRACE("%s %d: %d\n", __FUNCTION__, __LINE__, i);

        size_t cacheSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(QueryMountAddOnContentCacheSize(&cacheSize, static_cast<nn::aoc::AddOnContentIndex>(i + 1)));

        auto buffer = AllocateBuffer(cacheSize);
        mountCacheBufferArray[i].emplace(std::move(buffer));

        char mountName[32];
        util::SNPrintf(mountName, sizeof(mountName), "aoc%d", i);

        NNT_ASSERT_RESULT_SUCCESS(MountAddOnContent(mountName, static_cast<nn::aoc::AddOnContentIndex>(i + 1), mountCacheBufferArray[i]->get(), cacheSize));
    }

    // 各スレッドで read アクセス
    for (int i = 0; i < ThreadCount; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&g_Threads[i], ReadThreadFunction, reinterpret_cast<void*>(i), g_Stack[i], StackSize, nn::os::DefaultThreadPriority));
    }
    for (int i = 0; i < ThreadCount; i++)
    {
        nn::os::StartThread(&g_Threads[i]);
    }

    // Rom の読み出しも並列で行う
    ReadRom();

    for (int i = 0; i < ThreadCount; i++)
    {
        NN_LOG("%s %d: %d/%d\n", __FUNCTION__, __LINE__, i, ThreadCount);
        nn::os::WaitThread(const_cast<nn::os::ThreadType*>(&g_Threads[i]));
        nn::os::DestroyThread(const_cast<nn::os::ThreadType*>(&g_Threads[i]));
    }

}


}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    SetEnabledAutoAbort(false);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
