﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/fs.h>
#include <fsa/fs_FileSystemAccessor.h>
#include <fsa/fs_FileAccessor.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>

#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util_function.h>

namespace nnt { namespace fs {

namespace
{

    class DummyFileSystem : public nn::fs::fsa::IFileSystem, public nn::fs::detail::Newable
    {
    public:
        explicit DummyFileSystem() NN_NOEXCEPT
        {
        }

    private:
        virtual nn::Result DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoDeleteFile(const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoCreateDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoDeleteDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoCommit() NN_NOEXCEPT NN_OVERRIDE;
    };

    class DummyFile : public nn::fs::fsa::IFile, public nn::fs::detail::Newable
    {
    public:
        explicit DummyFile() NN_NOEXCEPT
        {
        }

    private:
        virtual nn::Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoOperateRange(void* outBuffer,
            size_t outBufferSize,
            nn::fs::OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;
    };

    class DummyDirectory : public nn::fs::fsa::IDirectory, public nn::fs::detail::Newable
    {
    public:
        explicit DummyDirectory() NN_NOEXCEPT
        {
        }

    private:
        virtual nn::Result DoRead(int64_t* outValue, nn::fs::DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::Result DoGetEntryCount(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;
    };

    nn::Result DummyFileSystem::DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_UNUSED(size);
        NN_UNUSED(option);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFileSystem::DoDeleteFile(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFileSystem::DoCreateDirectory(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFileSystem::DoDeleteDirectory(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFileSystem::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFileSystem::DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFileSystem::DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT
    {
        NN_UNUSED(currentPath);
        NN_UNUSED(newPath);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFileSystem::DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT
    {
        NN_UNUSED(currentPath);
        NN_UNUSED(newPath);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFileSystem::DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_UNUSED(path);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFileSystem::DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_UNUSED(mode);

        outValue->reset(new DummyFile());

        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFileSystem::DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_UNUSED(mode);

        outValue->reset(new DummyDirectory());

        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFileSystem::DoCommit() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFile::DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_UNUSED(offset);
        NN_UNUSED(buffer);
        NN_UNUSED(size);
        NN_UNUSED(option);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFile::DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT
    {
        NN_UNUSED(offset);
        NN_UNUSED(buffer);
        NN_UNUSED(size);
        NN_UNUSED(option);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFile::DoFlush() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFile::DoSetSize(int64_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFile::DoGetSize(int64_t* outValue) NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyFile::DoOperateRange(void* outBuffer,
        size_t outBufferSize,
        nn::fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT
    {
        NN_UNUSED(outBufferSize);
        NN_UNUSED(operationId);
        NN_UNUSED(offset);
        NN_UNUSED(size);
        NN_UNUSED(inBuffer);
        NN_UNUSED(inBufferSize);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyDirectory::DoRead(int64_t* outValue, nn::fs::DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_UNUSED(outEntries);
        NN_UNUSED(count);
        NN_RESULT_SUCCESS;
    }

    nn::Result DummyDirectory::DoGetEntryCount(int64_t* outValue) NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_RESULT_SUCCESS;
    }

}

}}

class FileSystemAccessorConstructor : public ::testing::Test
{
};

TEST_F(FileSystemAccessorConstructor, MountNameEmpty)
{
    std::unique_ptr<nn::fs::fsa::IFileSystem> pFileSystem(new nnt::fs::DummyFileSystem());
    nn::fs::detail::FileSystemAccessor accessor("", std::move(pFileSystem));
}

TEST_F(FileSystemAccessorConstructor, MountNameOverLength)
{
    const auto BufferLength = nn::fs::MountName::LENGTH + 2;

    auto buffer = nnt::fs::util::AllocateBuffer(BufferLength);
    std::memset(buffer.get(), 'a', BufferLength - 1);
    buffer.get()[BufferLength - 1] = 0;

    std::unique_ptr<nn::fs::fsa::IFileSystem> pFileSystem(new nnt::fs::DummyFileSystem());
    nn::fs::detail::FileSystemAccessor accessor(buffer.get(), std::move(pFileSystem));
}

class FileSystemAccessorDestractor : public ::testing::Test
{
};

TEST_F(FileSystemAccessorDestractor, OpenFileListNotEmpty)
{
    std::unique_ptr<nn::fs::fsa::IFileSystem> pFileSystem(new nnt::fs::DummyFileSystem());
    std::unique_ptr<nn::fs::detail::FileAccessor> file;
    {
        nn::fs::detail::FileSystemAccessor fileSystem("a", std::move(pFileSystem));
        fileSystem.OpenFile(&file, "a", nn::fs::OpenMode_Read);
    }
}

TEST_F(FileSystemAccessorDestractor, OpenDirectoryListNotEmpty)
{
    std::unique_ptr<nn::fs::fsa::IFileSystem> pFileSystem(new nnt::fs::DummyFileSystem());
    std::unique_ptr<nn::fs::detail::DirectoryAccessor> directory;
    {
        nn::fs::detail::FileSystemAccessor fileSystem("a", std::move(pFileSystem));
        fileSystem.OpenDirectory(&directory, "a", nn::fs::OpenDirectoryMode_All);
    }
}

class FileSystemAccessorCommit : public ::testing::Test
{
};

TEST_F(FileSystemAccessorCommit, OpenFileListHasWritableFile)
{
    std::unique_ptr<nn::fs::fsa::IFileSystem> pFileSystem(new nnt::fs::DummyFileSystem());
    std::unique_ptr<nn::fs::detail::FileAccessor> file;
    nn::fs::detail::FileSystemAccessor fileSystem("a", std::move(pFileSystem));
    fileSystem.OpenFile(&file, "a", nn::fs::OpenMode_Write);
    fileSystem.Commit();
}
