﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>

namespace nn { namespace fs {
    Result MountSaveDataInternalStorage(const char* name, SaveDataSpaceId spaceId, SaveDataId saveDataId) NN_NOEXCEPT;
}}

using namespace nn;

/* テストケースのアクセスを行ったセーブデータをダンプ */
/*
 * HostSaveDataPath 以下にディレクトリ名が Size<saveDataSize>MB_Journal<saveDataJournalSize>MB_0 のディレクトリを作成する。
 * その下に <saveDataId> のディレクトリを作成する。
 * <saveDataId> ディレクトリ下に、ディレクトリ名 "0" のディレクトリ名を作成。（既にあれば数値をインクリメント）
 * <saveDataId>/<Count> ディレクトリ下にセーブデータイメージファイルを
 * "AllocationTableControlArea", "AllocationTableMeta", "AllocationTableData", "Raw" の名前で作成する。
 */

namespace {

const int MaxPathLength = 256;
const char HostSaveDataPath[] = {"D:/TempSave"};

const int BufferSize = 8 * 1024 * 1024;

const int64_t SaveDataSizeArray[] =
{
    64 * 1024 * 1024,
    512 * 1024 * 1024,
    2LL * 1024 * 1024 * 1024,
};

const nn::fs::UserId TestUserId = {{0, 1}};

const int64_t SaveDataJournalSize = 256 * 1024 * 1024;
const int64_t SaveDataReserveSize = 128 * 1024;
const int TestFileCount = 4;
const int TestOperationCount = 19;

int g_RandomValue[256];

}

void CopyFile(const char* srcFile, const char* destFile)
{
    nn::fs::FileHandle srcHandle;
    nn::fs::FileHandle destHandle;
    int64_t fileSize = 0;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&srcHandle, srcFile, nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, srcHandle));

    nn::fs::CreateFile(destFile, fileSize);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&destHandle, destFile, nn::fs::OpenMode_Write));

    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);

    int64_t offset = 0;
    while (offset < fileSize)
    {
        int readSize = ((fileSize - offset) > BufferSize) ? BufferSize : (fileSize - offset);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(srcHandle, offset, buffer.get(), readSize));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(destHandle, offset, buffer.get(), readSize, nn::fs::WriteOption::MakeValue(0)));
        offset += readSize;
    }
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(destHandle));
    nn::fs::CloseFile(destHandle);
    nn::fs::CloseFile(srcHandle);
}

bool ExistsDirectory(const char* path)
{
    nn::fs::DirectoryHandle outValue;
    auto result = nn::fs::OpenDirectory(&outValue, path, nn::fs::OpenDirectoryMode_All);
    if (result.IsFailure())
    {
        return false;
    }
    nn::fs::CloseDirectory(outValue);
    return true;
}

void CleanSaveDataCache()
{
    /* SaveDataFs のキャッシュ追い出し */
    nn::fs::MountSystemSaveData("save", 0x8000000000000000);
    nn::fs::Unmount("save");
}

void DumpSaveDataRaw(char* dstBasePath, nn::fs::SaveDataId saveDataId)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::User, nullptr));

    char srcPath[MaxPathLength];
    char dstPath[MaxPathLength];
    nn::util::SNPrintf(srcPath, sizeof(srcPath), "%s:/save/%016llx", nn::fs::GetBisMountName(nn::fs::BisPartitionId::User), saveDataId);
    nn::util::SNPrintf(dstPath, sizeof(dstPath), "%s/Raw", dstBasePath);
    CopyFile(srcPath, dstPath);

    nn::fs::Unmount(nn::fs::GetBisMountName(nn::fs::BisPartitionId::User));
}

void DumpSaveDataInternalStorage(char* dstBasePath, nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId saveDataId)
{
    const char* Paths[] =
    {
        "AllocationTableControlArea",
        "AllocationTableMeta",
        "AllocationTableData",
    };
    const char* MountName = "internal";

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName, spaceId, saveDataId));

    for (auto entryName : Paths)
    {
        char srcPath[MaxPathLength];
        char dstPath[MaxPathLength];
        nn::util::SNPrintf(srcPath, sizeof(srcPath), "%s:/%s", MountName, entryName);
        nn::util::SNPrintf(dstPath, sizeof(dstPath), "%s/%s", dstBasePath, entryName);

        CopyFile(srcPath, dstPath);
    }
    nn::fs::Unmount(MountName);
    CleanSaveDataCache();
}

void DumpSaveDataImage(const char* dstBaseDir)
{
    nnt::fs::util::Vector<nn::fs::SaveDataInfo> infoArray;
    nnt::fs::util::FindSaveData(&infoArray, nn::fs::SaveDataSpaceId::User,
        [](const nn::fs::SaveDataInfo& info) { return info.applicationId == nnt::fs::util::ApplicationId; }
    );

    for (auto info : infoArray)
    {
        char dstDir[MaxPathLength];
        nn::util::SNPrintf(dstDir, sizeof(dstDir), "%s/%016llx", dstBaseDir, info.saveDataId);
        nn::fs::CreateDirectory(dstDir);

        int count = 0;
        do
        {
            nn::util::SNPrintf(dstDir, sizeof(dstDir), "%s/%016llx/%d", dstBaseDir, info.saveDataId, count);
            count++;
        } while (ExistsDirectory(dstDir));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(dstDir));

        DumpSaveDataRaw(dstDir, info.saveDataId);
        DumpSaveDataInternalStorage(dstDir, info.saveDataSpaceId, info.saveDataId);
    }
}

/* Operation */
void TestCreateFile(const char* filePath, int64_t fileSize)
{
    nn::fs::DeleteFile(filePath);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(filePath, fileSize));
}

void TestModifyFiles(const char* filePath, int64_t startOffset, int64_t accessSize)
{
    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);
    nn::fs::FileHandle fileHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, filePath, nn::fs::OpenMode_Write | nn::fs::OpenMode_Read));

    int64_t offset = 0;
    while (offset < accessSize)
    {
        int bytes = ((accessSize - offset) > BufferSize) ? BufferSize : (accessSize - offset);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, offset + startOffset, buffer.get(), bytes));
        for (int i = 0; i < bytes; i++)
        {
            (buffer.get())[i] = (buffer.get())[i] + 1;
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(fileHandle, offset + startOffset, buffer.get(), bytes, nn::fs::WriteOption::MakeValue(0)));
        offset += bytes;
    }
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(fileHandle));
    nn::fs::CloseFile(fileHandle);
}

void TestInitialize(const char* mountName, int64_t saveDataSize, int64_t journalSize, int count)
{
    int64_t fileSize = (saveDataSize - SaveDataReserveSize) / TestFileCount;
    for (int i = 0; i < TestFileCount; i++)
    {
        char filePath[MaxPathLength];
        nn::util::SNPrintf(filePath, sizeof(filePath), "%s:/TestFile%d", mountName, i);

        TestCreateFile(filePath, fileSize);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(mountName));
    }

    for (int i = 0; i < 256; i++)
    {
        g_RandomValue[i] = random();
    }
}

void TestModifyFilesRandom(const char* mountName, int64_t saveDataSize, int64_t journalSize, int count)
{
    int64_t fileSize = (saveDataSize - SaveDataReserveSize) / TestFileCount;

    NN_LOG("count=%d\n", count);
    for (int loop = 0; loop < count; loop++)
    {
        int64_t journalRemain = journalSize - SaveDataReserveSize;
        int n = random() % 8 + 8;
        for (int i = 0; i < n; i++)
        {
            int64_t writeSize = random() % (std::min(fileSize, static_cast<int64_t>(64 * 1024)) - 1) + 1;
            int64_t writeOffset = random() % (fileSize - writeSize);

            if (journalRemain < writeSize)
            {
                break;
            }
            journalRemain -= writeSize;

            char filePath[MaxPathLength];
            nn::util::SNPrintf(filePath, sizeof(filePath), "%s:/TestFile%d", mountName, random() % TestFileCount);

            TestModifyFiles(filePath, writeOffset, writeSize);
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(mountName));
    }
}

void TestModifyFilesRandomFixed(const char* mountName, int64_t saveDataSize, int64_t journalSize, int count)
{
    int64_t fileSize = (saveDataSize - SaveDataReserveSize) / TestFileCount;
    int randomCount = 0;

    int64_t journalRemain = journalSize - SaveDataReserveSize;
    int n = g_RandomValue[randomCount++] % 8 + 8;
    for (int i = 0; i < n; i++)
    {
        int64_t writeSize = g_RandomValue[randomCount++] % (std::min(fileSize, static_cast<int64_t>(64 * 1024)) - 1) + 1;
        int64_t writeOffset = g_RandomValue[randomCount++] % (fileSize - writeSize);

        if (journalRemain < writeSize)
        {
            break;
        }
        journalRemain -= writeSize;

        char filePath[MaxPathLength];
        nn::util::SNPrintf(filePath, sizeof(filePath), "%s:/TestFile%d", mountName, g_RandomValue[randomCount++] % TestFileCount);

        TestModifyFiles(filePath, writeOffset, writeSize);
    }
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(mountName));
}

struct TestParam {
    std::function<void(const char*, int64_t, int64_t, int)> operation;
    int arg;
};

void TestSaveData(const char* dstMountName)
{
    const TestParam testParams[][TestOperationCount] =
    {
        {
            {TestInitialize, 0},
            {TestModifyFilesRandom, 3},
            {TestModifyFilesRandomFixed, 1},
            {TestModifyFilesRandom, 7},
            {TestModifyFilesRandomFixed, 1},
            {TestModifyFilesRandom, 15},
            {TestModifyFilesRandomFixed, 1},
            {TestModifyFilesRandom, 31},
            {TestModifyFilesRandomFixed, 1},
            {TestModifyFilesRandom, 63},
            {TestModifyFilesRandomFixed, 1},
            {TestModifyFilesRandom, 127},
            {TestModifyFilesRandomFixed, 1},
            {TestModifyFilesRandom, 255},
            {TestModifyFilesRandomFixed, 1},
            {TestModifyFilesRandom, 511},
            {TestModifyFilesRandomFixed, 1},
            {TestModifyFilesRandom, 1023},
            {TestModifyFilesRandomFixed, 1},
        },
    };

    for (auto saveDataSize : SaveDataSizeArray)
    {
        int64_t journalSize = (saveDataSize < SaveDataJournalSize) ? saveDataSize : SaveDataJournalSize;

        for (int testGroup = 0; testGroup < (sizeof(testParams) / sizeof(testParams[0])); testGroup++)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::ApplicationId, TestUserId, 0, saveDataSize, journalSize, 0));

            nnt::fs::util::Vector<nn::fs::SaveDataInfo> infoArray;
            nnt::fs::util::FindSaveData(&infoArray, nn::fs::SaveDataSpaceId::User,
                [](const nn::fs::SaveDataInfo& info) { return info.applicationId == nnt::fs::util::ApplicationId; }
            );

            for (int i = 0; i < TestOperationCount; i++)
            {
                const char* MountName = "save";
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName, nnt::fs::util::ApplicationId, TestUserId));
                testParams[testGroup][i].operation(MountName, saveDataSize, journalSize, testParams[testGroup][i].arg);
                nn::fs::Unmount(MountName);
                CleanSaveDataCache();

                char dstDir[MaxPathLength];
                nn::util::SNPrintf(dstDir, sizeof(dstDir), "%s:/Size%dMB_Journal%dMB_%d", dstMountName, saveDataSize / 1024 / 1024, journalSize / 1024 / 1024, testGroup);
                nn::fs::CreateDirectory(dstDir);
                NN_LOG("%s\n", dstDir);

                DumpSaveDataImage(dstDir);
            }

            for (auto& info : infoArray)
            {
                nn::fs::DeleteSaveData(info.saveDataId);
            }
        }
    }
}

TEST(SaveData, DumpSaveData)
{
    std::mt19937 random(nnt::fs::util::GetRandomSeed());

    const char* MountName = "host";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost(MountName, HostSaveDataPath));

    TestSaveData(MountName);

    nn::fs::Unmount(MountName);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
