﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_Debug.h>
#include <nn/fs/fs_Context.h>
#include <nn/fs/fs_SystemSaveData.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

namespace {

const size_t StackSize = 4 * 1024;
static NN_ALIGNAS(4096) uint8_t stack[StackSize];

template <typename Func>
void RunOnAnotherThread(Func func, void* pArg = nullptr)
{
    os::ThreadType thread;

    NNT_ASSERT_RESULT_SUCCESS(os::CreateThread(&thread, func, pArg, stack, StackSize, os::DefaultThreadPriority));
    os::StartThread(&thread);
    os::WaitThread(&thread);
    os::DestroyThread(&thread);
}


TEST(SetEnabledAutoAbort, Disabled)
{
    SetEnabledAutoAbort(false);

    NNT_ASSERT_RESULT_SUCCESS(MountSaveDataForDebug("save"));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile("save:/file", 0));

    FileHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, "save:/file", OpenMode_Read));

    char buffer[32];
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidOperationForOpenMode, WriteFile(handle, 0, buffer, sizeof(buffer), WriteOption()));

    {
        // 他スレッドにも作用する
        RunOnAnotherThread(
            [](void* pArg)
            {
                char buffer[32];
                auto pHandle = reinterpret_cast<FileHandle*>(pArg);
                NNT_EXPECT_RESULT_FAILURE(ResultInvalidOperationForOpenMode, WriteFile(*pHandle, 0, buffer, sizeof(buffer), WriteOption()));
            },
            &handle
        );
    }


    CloseFile(handle);
    Unmount("save");
    SetEnabledAutoAbort(true);
}


TEST(SetEnabledAutoAbort, EnabledDeath)
{
    SetResultHandledByApplication(true); // appli

    NNT_ASSERT_RESULT_SUCCESS(MountSaveDataForDebug("save"));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile("save:/file", 0));

    FileHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, "save:/file", OpenMode_Read));

    char buffer[32];
    EXPECT_DEATH_IF_SUPPORTED(WriteFile(handle, 0, buffer, sizeof(buffer), WriteOption()), "");

    CloseFile(handle);
    Unmount("save");

    // アプリはハンドリングしない result
    EXPECT_DEATH_IF_SUPPORTED(MountSystemSaveData("save2", 0x8000000000000005), "");
}


TEST(DefaultHandling, System)
{
    SetResultHandledByApplication(false); // system

    NNT_ASSERT_RESULT_SUCCESS(MountSaveDataForDebug("save"));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile("save:/file", 0));

    FileHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, "save:/file", OpenMode_Read));

    char buffer[32];
    EXPECT_DEATH_IF_SUPPORTED(WriteFile(handle, 0, buffer, sizeof(buffer), WriteOption()), "");

    CloseFile(handle);
    Unmount("save");

    // システムはハンドリングする result
    NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, MountSystemSaveData("save2", 0x8000000000000005));

    SetResultHandledByApplication(true);
}



TEST(ScopedContextSetter, Disabled)
{
    NNT_ASSERT_RESULT_SUCCESS(MountSaveDataForDebug("save"));

    {
        // スコープ内のみのコンテキスト変更

        // abort 全抑制
        ScopedAutoAbortDisabler scopedDisabler;
        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, CreateFile(nullptr, 0));
        NNT_EXPECT_RESULT_FAILURE(ResultOutOfRange,  CreateFile("save:/file", -1));


        {
            // 生存期間でも他スレッドには作用しない
            RunOnAnotherThread(
                [](void* pArg)
                {
                    NN_UNUSED(pArg);
                    EXPECT_DEATH_IF_SUPPORTED(CreateFile(nullptr, 0), "");
                    EXPECT_DEATH_IF_SUPPORTED(CreateFile("save:/file", -1), "");
                }
            );
        }

        {
            // コンテキスト変更のネスト

            // 選択して abort 抑制
            FsContext context(
                [](Result result) { // ResultInvalidPath のみハンドリング、他はデフォルト挙動
                    if (ResultInvalidPath::Includes(result))
                    {
                        return AbortSpecifier::ReturnResult;
                    }
                    return AbortSpecifier::Default;
                }
            );
            ScopedFsContextSetter scopedContext(context);

            NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, CreateFile(nullptr, 0));
            EXPECT_DEATH_IF_SUPPORTED(CreateFile("save:/file", -1), "");
        }

        // スコープに従い pop すること
        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, CreateFile(nullptr, 0));
        NNT_EXPECT_RESULT_FAILURE(ResultOutOfRange,  CreateFile("save:/file", -1));
    }

    // デフォルト
    EXPECT_DEATH_IF_SUPPORTED(CreateFile(nullptr, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(CreateFile("save:/file", -1), "");

    Unmount("save");
}


// プロセス全体の挙動の変更
TEST(SetDefaultFsContextResultHandler, Set)
{
    NNT_ASSERT_RESULT_SUCCESS(MountSaveDataForDebug("save"));

    {
        SetDefaultFsContextResultHandler(
            [](Result result) { // ResultInvalidPath のみハンドリング、他はデフォルト挙動
            if (ResultInvalidPath::Includes(result))
            {
                return AbortSpecifier::ReturnResult;
            }
            return AbortSpecifier::Default;
        }
        );

        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, CreateFile(nullptr, 0));
        EXPECT_DEATH_IF_SUPPORTED(CreateFile("save:/file", -1), "");

        // 他スレッドにも作用する
        RunOnAnotherThread(
            [](void* pArg)
        {
            NN_UNUSED(pArg);
            NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, CreateFile(nullptr, 0));
            EXPECT_DEATH_IF_SUPPORTED(CreateFile("save:/file", -1), "");
        }
        );
    }

    {
        // デフォルトに戻す
        SetDefaultFsContextResultHandler(nullptr);

        EXPECT_DEATH_IF_SUPPORTED(CreateFile(nullptr, 0), "");
        EXPECT_DEATH_IF_SUPPORTED(CreateFile("save:/file", -1), "");

        // 他スレッドにも作用する
        RunOnAnotherThread(
            [](void* pArg)
        {
            NN_UNUSED(pArg);
            EXPECT_DEATH_IF_SUPPORTED(CreateFile(nullptr, 0), "");
            EXPECT_DEATH_IF_SUPPORTED(CreateFile("save:/file", -1), "");
        }
        );

    }


    Unmount("save");
}


}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
