﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/fs/fs_ResultHandler.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util_adapter.h>
#include <nnt/nnt_Argument.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;
using ::testing::Return;
using ::testing::SetArgPointee;


namespace {

class MockFile : public fsa::IFile
{
public:
    MOCK_METHOD5(DoRead, Result(size_t* outValue, int64_t offset, void* buffer, size_t size, const ReadOption& option));
    MOCK_METHOD4(DoWrite, Result(int64_t offset, const void* buffer, size_t size, const WriteOption& option));
    MOCK_METHOD0(DoFlush, Result());
    MOCK_METHOD1(DoSetSize, Result(int64_t size));
    MOCK_METHOD1(DoGetSize, Result(int64_t* outValue));
    MOCK_METHOD7(DoOperateRange, Result(void* outBuffer, size_t outBufferSize, OperationId operationId, int64_t offset, int64_t size, const void* inBuffer, size_t inBufferSize));
};

TEST(PlacementOpenFile, simple)
{
    DeleteAllTestSaveData();

    const size_t FileSize = 128 * 1024;

    const UserId userId = { { 0, 1 } };
    NNT_ASSERT_RESULT_SUCCESS(CreateAndMountSaveData("save", userId));
    NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/file", FileSize, 0));

    FileHandle handleBase;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handleBase, "save:/file", OpenMode_Write | OpenMode_Read));

    // handleBase は委譲
    std::unique_ptr<FileHandleFile> fileHandleFile(new FileHandleFile(handleBase));

    FileHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, std::move(fileHandleFile), OpenMode_Write | OpenMode_Read));

    auto buffer = AllocateBuffer(FileSize);
    NNT_EXPECT_RESULT_SUCCESS(ReadFile(handle, 0, buffer.get(), FileSize));
    EXPECT_TRUE(IsFilledWith32BitCount(buffer.get(), FileSize, 0));

    CloseFile(handle);
    Unmount("save");

    DeleteAllTestSaveData();
}

// IFile を直接 handle 化する OpenFile で開いた際に、 handle と FileAccessor が正しく IFile を叩けていること
TEST(PlacementOpenFile, mock)
{
    std::unique_ptr<MockFile> mockFile(new MockFile);

    size_t outReadSize;
    int64_t readOffset = 1;
    auto readBuffer = nnt::fs::util::AllocateBuffer(32);
    size_t readSize = 2;
    ReadOption readOption;
    size_t expectedOutReadSize = 3;

    int64_t writeOffset = 11;
    auto writeBuffer = nnt::fs::util::AllocateBuffer(32);
    size_t writeSize = 12;
    WriteOption writeOption = WriteOption::MakeValue(WriteOptionFlag_Flush);


    int64_t setSize = 21;

    int64_t outGetSize = 22;
    int64_t expectedOutGetSize = 23;


    QueryRangeInfo queryRangeInfo;
    int64_t queryRangeOffset = 34;
    size_t queryRangeSize = 35;


    {
        EXPECT_CALL(*mockFile.get(), DoRead(&outReadSize, readOffset, readBuffer.get(), readSize, readOption))
            .WillOnce(
                DoAll(
                    SetArgPointee<0>(expectedOutReadSize),
                    Return(ResultSuccess())
                )
            );

        EXPECT_CALL(*mockFile.get(), DoWrite(writeOffset, writeBuffer.get(), writeSize, writeOption))
            .WillOnce(Return(ResultSuccess()));

        EXPECT_CALL(*mockFile.get(), DoFlush())
            .WillOnce(Return(ResultSuccess()));

        EXPECT_CALL(*mockFile.get(), DoSetSize(setSize))
            .WillOnce(Return(ResultSuccess()));

        EXPECT_CALL(*mockFile.get(), DoGetSize(&outGetSize))
            .WillOnce(
                DoAll(
                    SetArgPointee<0>(expectedOutGetSize),
                    Return(ResultSuccess())
                )
            );

        EXPECT_CALL(*mockFile.get(), DoOperateRange(
            &queryRangeInfo,
            sizeof(queryRangeInfo),
            OperationId::QueryRange,
            queryRangeOffset,
            queryRangeSize,
            nullptr,
            0
        ))
            .WillOnce(Return(ResultSuccess()));
    }

    {
        FileHandle handle;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, std::move(mockFile), OpenMode_Read));

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(&outReadSize, handle, readOffset, readBuffer.get(), readSize, readOption));
        EXPECT_EQ(expectedOutReadSize, outReadSize);

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteFile(handle, writeOffset, writeBuffer.get(), writeSize, writeOption));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::FlushFile(handle));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::SetFileSize(handle, setSize));

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetFileSize(&outGetSize, handle));
        EXPECT_EQ(expectedOutGetSize, outGetSize);

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::QueryRange(&queryRangeInfo, handle, queryRangeOffset, queryRangeSize));

        nn::fs::CloseFile(handle);
    }
}


}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}

