﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_RomOnFile.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_Argument.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {

nnt::fs::util::String g_RomFilePath;
nnt::fs::util::String g_SourceDirectoryPath;

class MountRomOnFileTest
    : public nnt::fs::util::CheckMemoryLeakFixture, public ::testing::Test
{
protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        nnt::fs::util::CheckMemoryLeakFixture::StartMemoryCheck();

        // 引数で与えられた RomFs アーカイブファイルを開く
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::OpenFile(&m_RomFile, g_RomFilePath.c_str(), nn::fs::OpenMode_Read));
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 開いていた RomFs アーカイブファイルを閉じる
        nn::fs::CloseFile(m_RomFile);
        nn::fs::UnmountHostRoot();

        nnt::fs::util::CheckMemoryLeakFixture::StopMemoryLeakCheck();
    }

    const nn::fs::FileHandle GetRomFile() const NN_NOEXCEPT
    {
        return m_RomFile;
    }

private:
    nn::fs::FileHandle m_RomFile;
};

// 一連のマウント動作をテストし、グローバルな new, delete が呼ばれないことをテスト
TEST_F(MountRomOnFileTest, CheckGlobalNewDeleteFlag)
{
    nnt::fs::util::ResetGlobalNewDeleteFlag();
    nnt::fs::util::SetStackTraceDumpOnGlobalNewDeleteCallEnabled(true);
    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::SetStackTraceDumpOnGlobalNewDeleteCallEnabled(false);
    };

    {
        // キャッシュバッファを確保
        size_t cacheBufferSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::QueryMountRomOnFileCacheSize(&cacheBufferSize, GetRomFile()));
        ASSERT_LT(static_cast<size_t>(0), cacheBufferSize);

        auto cacheBuffer = nnt::fs::util::AllocateBuffer(cacheBufferSize);

        // マウント
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::MountRomOnFile("rom", GetRomFile(), cacheBuffer.get(), cacheBufferSize));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("rom");
        };

        // 別のマウント名で二重にマウント可能
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::MountRomOnFile("rom2", GetRomFile(), cacheBuffer.get(), cacheBufferSize));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("rom2");
        };

        // ファイルオープン
        nn::fs::FileHandle file;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::OpenFile(&file, "rom:/file0.dat", nn::fs::OpenMode_Read));
        nn::fs::CloseFile(file);

        // ディレクトリオープン
        nn::fs::DirectoryHandle directory;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::OpenDirectory(&directory, "rom:/dir0", nn::fs::OpenDirectoryMode_All));
        nn::fs::CloseDirectory(directory);
    }

    EXPECT_FALSE(nnt::fs::util::IsGlobalNewDeleteCalled());
}

// マウントした RomFs アーカイブの内容が元データと一致するかテスト
TEST_F(MountRomOnFileTest, VerifyMount)
{
    size_t cacheBufferSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomOnFileCacheSize(&cacheBufferSize, GetRomFile()));
    ASSERT_LT(static_cast<size_t>(0), cacheBufferSize);

    auto cacheBuffer = nnt::fs::util::AllocateBuffer(cacheBufferSize);

    bool verifyResult = false;
    NNT_EXPECT_RESULT_SUCCESS(
        nnt::fs::util::VerifyMount(
            &verifyResult,
            [&](const char* mountName) NN_NOEXCEPT
            {
                return nn::fs::MountRomOnFile(
                    mountName, GetRomFile(), cacheBuffer.get(), cacheBufferSize);
            },
            g_SourceDirectoryPath.c_str(),
            "mount",
            "mount:/"
        )
    );
    EXPECT_TRUE(verifyResult);
}

typedef MountRomOnFileTest MountRomOnFileDeathTest;

// 失敗時の動作をテスト
TEST_F(MountRomOnFileDeathTest, Failure)
{
    DisableCheck();

    nn::fs::FileHandle nullFile = {};

    size_t cacheBufferSize = 0;

    // QueryMountRomOnFileCacheSize で不正な引数なら失敗
    {
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultNullptrArgument,
            nn::fs::QueryMountRomOnFileCacheSize(nullptr, GetRomFile()));
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::QueryMountRomOnFileCacheSize(&cacheBufferSize, nullFile), "");
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomOnFileCacheSize(&cacheBufferSize, GetRomFile()));
    ASSERT_LT(static_cast<size_t>(0), cacheBufferSize);

    auto cacheBuffer = nnt::fs::util::AllocateBuffer(cacheBufferSize);

    // MountRomOnFile で不正な引数なら失敗
    {
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidMountName,
            nn::fs::MountRomOnFile(nullptr, GetRomFile(), cacheBuffer.get(), cacheBufferSize));
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::MountRomOnFile("rom", nullFile, cacheBuffer.get(), cacheBufferSize), "");
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultNullptrArgument,
            nn::fs::MountRomOnFile("rom", GetRomFile(), nullptr, cacheBufferSize));
#if defined(NN_SDK_BUILD_RELEASE)
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultPreconditionViolation,
            nn::fs::MountRomOnFile("rom", GetRomFile(), cacheBuffer.get(), 0));
#else
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::MountRomOnFile("rom", GetRomFile(), cacheBuffer.get(), 0), "");
#endif
    }

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fs::MountRomOnFile("rom", GetRomFile(), cacheBuffer.get(), cacheBufferSize));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("rom");
    };

    // MountRomOnFile で既に使用されているマウント名なら失敗
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultMountNameAlreadyExists,
        nn::fs::MountRomOnFile("rom", GetRomFile(), cacheBuffer.get(), cacheBufferSize));
}

}

namespace {
    class MountRomOnFileTestForFsUtilMountTest : public MountRomOnFileTest
    {
    public:
        MountRomOnFileTestForFsUtilMountTest() NN_NOEXCEPT
            : m_Buffer(nullptr, nnt::fs::util::DeleterBuffer)
        {
        }

        virtual void TestBody() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            MountRomOnFileTest::SetUp();
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            MountRomOnFileTest::TearDown();
            m_Buffer.reset();
        }

        nn::fs::FileHandle GetRomFile() NN_NOEXCEPT
        {
            return MountRomOnFileTest::GetRomFile();
        }

        char* AllocateBuffer(size_t size) NN_NOEXCEPT
        {
            m_Buffer = nnt::fs::util::AllocateBuffer(size);
            return m_Buffer.get();
        }

    private:
        decltype(nnt::fs::util::AllocateBuffer(0)) m_Buffer;
    };

    MountRomOnFileTestForFsUtilMountTest g_MountRomOnFileTest;

    void SetUpForMountRomOnFile() NN_NOEXCEPT
    {
        g_MountRomOnFileTest.SetUp();
    }

    void TearDownForMountRomOnFile() NN_NOEXCEPT
    {
        g_MountRomOnFileTest.TearDown();
    }

    const nnt::fs::util::MountTestAttribute GetAttributeForSystemData() NN_NOEXCEPT
    {
        nnt::fs::util::MountTestAttribute attribute = {};
        attribute.setUp = SetUpForMountRomOnFile;
        attribute.tearDowm = TearDownForMountRomOnFile;
        return attribute;
    }

    nn::Result MountRomOnFileForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        size_t cacheBufferSize;
        NN_RESULT_DO(nn::fs::QueryMountRomOnFileCacheSize(&cacheBufferSize, g_MountRomOnFileTest.GetRomFile()));

        auto pCacheBuffer = g_MountRomOnFileTest.AllocateBuffer(cacheBufferSize);

        NN_RESULT_DO(nn::fs::MountRomOnFile(mountName, g_MountRomOnFileTest.GetRomFile(), pCacheBuffer, cacheBufferSize));
        NN_RESULT_SUCCESS;
    }

    const nnt::fs::util::MountTestParameter MountTestParameters[] = {
        { "MountRomOnFile", MountRomOnFileForMountNameTest, nullptr, GetAttributeForSystemData },
    };
}

namespace nnt { namespace fs { namespace util {
    NNT_FS_INSTANTIATE_TEST_CASE_MOUNT(WithMountRomOnFile, ::testing::ValuesIn(MountTestParameters));
}}}

extern "C" void nnMain()
{
    nnt::fs::util::LoadMountTest();

    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nnt::fs::util::mock::InitializeForFsTest(&argc, argv);

    if( argc <= 2 )
    {
        NN_LOG("Usage: testFs_FsLib_MountRomOnFile <source directory path> <rom file path>\n");
        nnt::Exit(0);
    }
    g_SourceDirectoryPath.assign(argv[1]);
    g_RomFilePath.assign(argv[2]);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
