﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_MountPrivate.h>
#include <nn/fs/fs_SdCardForDebug.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
    #include <detail/fssrv_RuntimeConfiguration.h>
#endif

#include "testFs_FsLib_Mount.h"
#include "testFs_FsLib_Mount_Bis.h"

using namespace nn::fs;

TEST(ConvertToFsCommonPath, MountContentStorage)
{
    const nnt::fs::util::String UserMountName = "mount";
    const ContentStorageId Ids[] =
    {
        ContentStorageId::User,
        ContentStorageId::System,
        ContentStorageId::SdCard,
    };

    for( auto id : Ids )
    {
        NNT_ASSERT_RESULT_SUCCESS(MountContentStorage(UserMountName.c_str(), id));
        char commonPath[EntryNameLengthMax];
        NNT_EXPECT_RESULT_SUCCESS(ConvertToFsCommonPath(commonPath, sizeof(commonPath), (UserMountName + ":/path").c_str()));
        EXPECT_STREQ((nnt::fs::util::String() + GetContentStorageMountName(id) + ":/path").c_str(), commonPath);
        Unmount(UserMountName.c_str());
    }
}

typedef nnt::fs::util::CheckGlobalNewDeleteFlagTestFixture MountContentStorageTest;

TEST_F(MountContentStorageTest, MountContentStorage)
{
    // 古いデータベースのファイルが残っているとテストが落ちるので削除
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd"));
    DeleteDirectoryRecursively("sd:/Nintendo/Contents");
    Unmount("sd");

    const nn::fs::ContentStorageId Ids[] =
    {
        nn::fs::ContentStorageId::System,
        nn::fs::ContentStorageId::User,
        nn::fs::ContentStorageId::SdCard,
    };

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    SetBisRootForHostHelper setBisRootSystem(BisPartitionId::System, (g_TestDirPath.GetPath() + "/system").c_str());
    SetBisRootForHostHelper setBisRootUser(BisPartitionId::User, (g_TestDirPath.GetPath() + "/user").c_str());

    {
        NNT_ASSERT_RESULT_SUCCESS(MountHostRoot());
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory((g_TestDirPath.GetPath() + "/system").c_str()));
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory((g_TestDirPath.GetPath() + "/user").c_str()));
        UnmountHostRoot();
    }
#endif

    for( auto id : Ids )
    {
        NNT_ASSERT_RESULT_SUCCESS(MountContentStorage(id));

        nnt::fs::util::String mountName;
        mountName += GetContentStorageMountName(id);

        DeleteFile((mountName + ":/test.file").c_str());

        NNT_EXPECT_RESULT_SUCCESS(DumpDirectoryRecursiveWithWhiteList((mountName + ":/").c_str()));
        NNT_EXPECT_RESULT_SUCCESS(CreateFile((mountName + ":/test.file").c_str(), 32));
        NNT_EXPECT_RESULT_SUCCESS(DumpDirectoryRecursiveWithWhiteList((mountName + ":/").c_str()));

        NNT_EXPECT_RESULT_SUCCESS(DeleteFile((mountName + ":/test.file").c_str()));

        Unmount(mountName.c_str());
    }
}

#if 0
TEST_F(MountContentStorageTest, AccessViaMountCode) // Todo: 自動化
{

    const nn::fs::ContentStorageId Ids[] =
    {
        nn::fs::ContentStorageId::System,
        nn::fs::ContentStorageId::User,
        nn::fs::ContentStorageId::SdCard,
    };


#if defined(NN_BUILD_CONFIG_OS_WIN32)
    SetBisRootForHostHelper setBisRootSystem(BisPartitionId::System, (g_TestDirPath.GetPath() + "/system").c_str());
    SetBisRootForHostHelper setBisRootUser(BisPartitionId::User, (g_TestDirPath.GetPath() + "/user").c_str());
#endif

    {
        NNT_ASSERT_RESULT_SUCCESS(MountHostRoot());
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory((g_TestDirPath.GetPath() + "/system").c_str()));
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory((g_TestDirPath.GetPath() + "/user").c_str()));
        UnmountHostRoot();
    }

    for( auto id : Ids )
    {
        NNT_ASSERT_RESULT_SUCCESS(MountContentStorage(id));
        NNT_ASSERT_RESULT_SUCCESS(MountHostRoot());

        nnt::fs::util::String mountName;
        mountName += GetContentStorageMountName(id);

        const size_t BufferSize = 128 * 1024;
        std::unique_ptr<char[]> buffer(new char[BufferSize]);

        NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CopyFile("d:/vol/test.nca", (mountName + ":/test.nca").c_str(), buffer.get(), BufferSize));

        {
#if defined(NN_BUILD_CONFIG_OS_WIN)
            nn::fssrv::detail::SetEnabledProgramVerification(false);
            NN_UTIL_SCOPE_EXIT
            {
                nn::fssrv::detail::SetEnabledProgramVerification(true);
            };
#endif

            NNT_ASSERT_RESULT_SUCCESS(MountCode("code", (mountName + ":/test.nca").c_str()));
            NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("code:/"));
            Unmount("code");
        }

        NNT_ASSERT_RESULT_SUCCESS(DeleteFile((mountName + ":/test.nca").c_str()));

        Unmount(mountName.c_str());
        UnmountHostRoot();
    }
}
#endif

TEST_F(MountFailure, MountContentStorageAlreadyExistForDefaultMountName)
{
    static const nn::fs::ContentStorageId StorageIds[] = {
        nn::fs::ContentStorageId::User,
        nn::fs::ContentStorageId::System,
        nn::fs::ContentStorageId::SdCard
    };

    for( auto id : StorageIds )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage(id));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultMountNameAlreadyExists, nn::fs::MountContentStorage(id));
        nn::fs::Unmount(nn::fs::GetContentStorageMountName(id));
    }
}

namespace
{
    const nnt::fs::util::MountTestAttribute GetAttribute() NN_NOEXCEPT
    {
        nnt::fs::util::MountTestAttribute attribute = {};
        attribute.isReservedMountNameSupported = true;
        return attribute;
    }

    nn::Result MountContentStorageForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountContentStorage(mountName, nn::fs::ContentStorageId::User));
        NN_RESULT_SUCCESS;
    }

    const nnt::fs::util::MountTestParameter MountTestParameters[] = {
        { "MountContentStorage", MountContentStorageForMountNameTest, nullptr, GetAttribute },
    };
}

NNT_FS_INSTANTIATE_TEST_CASE_MOUNT(WithMountContentStorage, ::testing::ValuesIn(MountTestParameters));
