﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_MountPrivate.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
    #include <nn/fssystem/fs_PathOnExecutionDirectory.h>
#endif

#include "testFs_FsLib_Mount.h"
#include "testFs_FsLib_Mount_Bis.h"

using namespace nn::fs;

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || \
    defined(NN_BUILD_CONFIG_HARDWARE_NX) || \
    defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2)

TEST(MountBisTest, MountBis)
{
    NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::System, nullptr));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("@System:/test.file", 1024));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::ListDirectoryRecursive("@System:/"));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile("@System:/test.file"));

    NNT_EXPECT_RESULT_FAILURE(ResultMountNameAlreadyExists, MountBis(BisPartitionId::System, nullptr));
    Unmount("@System");

    NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::System, nullptr));
    ValidateFileSystem("@System:/");
    Unmount("@System");

    NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::System, nullptr));
    ValidateFileSystem("@System:/");
    Unmount("@System");
}

TEST(MountBisTest, MountUserBis)
{
    nnt::fs::util::SaveDataFileSystemMounter::ClearSaveDataFileSystemCache();

    NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::User, nullptr));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("@User:/test.file", 1024));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("@User:/"));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile("@User:/test.file"));

    Unmount("@User");
}

TEST(MountBisTest, BigFileOnUserBis)
{
    NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::User, nullptr));

    const int64_t FileSize = 8ULL * 1024 * 1024 * 1024;
    const char FileName[] = "@User:/test.file";
    NNT_EXPECT_RESULT_SUCCESS(CreateFile(FileName, FileSize, CreateFileOptionFlag_BigFile));
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::ListDirectoryRecursive("@User:/"));
        FileHandle handle;
        NNT_EXPECT_RESULT_SUCCESS(OpenFile(&handle, FileName, OpenMode_Read));

        int64_t size;
        NNT_EXPECT_RESULT_SUCCESS(GetFileSize(&size, handle));
        EXPECT_EQ(FileSize, size);

        CloseFile(handle);
    }
    NNT_EXPECT_RESULT_SUCCESS(DeleteFile(FileName));

    NNT_EXPECT_RESULT_FAILURE(ResultMountNameAlreadyExists, MountBis(BisPartitionId::User, nullptr));
    Unmount("@User");
}

#elif defined(NN_BUILD_CONFIG_OS_WIN)

TEST_F(MountBasic, MountBis)
{
    auto testDirPath = g_TestDirPath.GetPath();

    // TODO: SetBisRootForHostHelper の削除

    {
        SetBisRootForHostHelper setBisRoot(BisPartitionId::System, testDirPath.c_str());
        NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::System, nullptr));
        ValidateFileSystem("@System:/");
        Unmount("@System");
    }

    NNT_ASSERT_RESULT_SUCCESS(MountHost("host", testDirPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory("host:/bis"));

    {
        auto testSubDirPath = g_TestDirPath.GetPath();
        testSubDirPath.append("/bis");
        SetBisRootForHostHelper setBisRoot(BisPartitionId::System, testSubDirPath.c_str());
        NNT_EXPECT_RESULT_SUCCESS(MountBis(BisPartitionId::System, nullptr));
        ValidateFileSystem("@System:/");
        Unmount("@System");
    }

    NNT_EXPECT_RESULT_SUCCESS(DeleteDirectory("host:/bis"));
    Unmount("host");
}

#else

TEST(MountBisTest, MountBis)
{
    NNT_EXPECT_RESULT_FAILURE(ResultPartitionNotFound, MountBis(BisPartitionId::System, nullptr));
}

#endif

#if defined(NN_BUILD_CONFIG_OS_WIN)

TEST_F(MountFailure, MountBisPathNotFound)
{
    // TODO: SetBisRootForHostHelper の削除

    {
        auto testDirPathStartSlash = nnt::fs::util::String("/");
        testDirPathStartSlash.append(g_TestDirPath.GetPath().c_str());
        SetBisRootForHostHelper setBisRoot(BisPartitionId::System, testDirPathStartSlash.c_str());
        NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, MountBis(BisPartitionId::System, nullptr));
    }

    {
        SetBisRootForHostHelper setBisRoot(BisPartitionId::System, "c:/notfound");
        NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, MountBis(BisPartitionId::System, nullptr));
    }
}

TEST_F(MountFailure, MountBisAlreadyExistForDefaultMountName)
{
    // TODO: SetBisRootForHostHelper の削除

    SetBisRootForHostHelper setBisRoot(nn::fs::BisPartitionId::System, g_TestDirPath.GetPath().c_str());
    NNT_ASSERT_RESULT_SUCCESS(MountBis(nn::fs::BisPartitionId::System, nullptr));
    NNT_EXPECT_RESULT_FAILURE(ResultMountNameAlreadyExists, nn::fs::MountBis(BisPartitionId::System, nullptr));
    nn::fs::Unmount(nn::fs::GetBisMountName(nn::fs::BisPartitionId::System));
}

#endif

TEST(ConvertToFsCommonPath, MountBis)
{
    const nnt::fs::util::String UserMountName = "mount";
    const BisPartitionId Ids[] =
    {
        BisPartitionId::User,
        BisPartitionId::System,
        BisPartitionId::SafeMode,
        BisPartitionId::CalibrationFile,
    };

    for( auto id : Ids )
    {
        NNT_ASSERT_RESULT_SUCCESS(MountBis(UserMountName.c_str(), id));
        char commonPath[EntryNameLengthMax];
        NNT_EXPECT_RESULT_SUCCESS(ConvertToFsCommonPath(commonPath, sizeof(commonPath), (UserMountName + ":/path").c_str()));
        EXPECT_STREQ((nnt::fs::util::String() + GetBisMountName(id) + ":/path").c_str(), commonPath);
        Unmount(UserMountName.c_str());
    }
}

#if defined(NN_BUILD_CONFIG_OS_WIN32)
const char* GetExecutionDirectoryPath()
{
    static const nn::fssystem::PathOnExecutionDirectory s_ExecutionDirectoryPath("");
    return s_ExecutionDirectoryPath.Get();
}

TEST(MountBisBasic, MountEachBis)
{
    const BisPartitionId Ids[] =
    {
        BisPartitionId::User,
        BisPartitionId::System,
        BisPartitionId::SafeMode,
        BisPartitionId::CalibrationFile,
    };
    for( auto id : Ids )
    {
        // まず MountBis でマウントし、Id ごとに異なるファイルを作成
        NNT_ASSERT_RESULT_SUCCESS(MountBis("mount", id));
        char filePath[EntryNameLengthMax];
        memset(filePath, 0x00, sizeof(filePath));
        switch( id )
        {
        case nn::fs::BisPartitionId::User: strncpy(filePath, "mount:/fileUsr", 6 + 1 + 7 + 1); break;
        case nn::fs::BisPartitionId::System: strncpy(filePath, "mount:/fileSys", 6 + 1 + 7 + 1); break;
        case nn::fs::BisPartitionId::SafeMode: strncpy(filePath, "mount:/fileSaf", 6 + 1 + 7 + 1); break;
        case nn::fs::BisPartitionId::CalibrationFile: strncpy(filePath, "mount:/fileCal", 6 + 1 + 7 + 1); break;
        default: strncat(filePath, "", 0);
        }
        nn::fs::DirectoryEntryType directoryEntryType;
        if( nn::fs::GetEntryType(&directoryEntryType, filePath).IsFailure() )
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(filePath, 4));
        }
        Unmount("mount");

        // 次に作ったファイルの Host 上のパスを指定しファイル削除(存在確認)
        memset(filePath, 0x00, sizeof(filePath));
        size_t copyLength = strnlen(GetExecutionDirectoryPath(), EntryNameLengthMax);
        strncpy(filePath, GetExecutionDirectoryPath(), copyLength);
        filePath[copyLength] = '\0';
        switch( id )
        {
        case nn::fs::BisPartitionId::User: strncat(filePath, "bis/user/fileUsr", 3 + 1 + 4 + 1 + 7 + 1); break;
        case nn::fs::BisPartitionId::System: strncat(filePath, "bis/system/fileSys", 3 + 1 + 6 + 1 + 7 + 1); break;
        case nn::fs::BisPartitionId::SafeMode: strncat(filePath, "bis/safe/fileSaf", 3 + 1 + 4 + 1 + 7 + 1); break;
        case nn::fs::BisPartitionId::CalibrationFile: strncat(filePath, "bis/cal/fileCal", 3 + 1 + 3 + 1 + 7 + 1); break;
        default: strncat(filePath, "", 0);
        }
        MountHostRoot();
        NNT_EXPECT_RESULT_SUCCESS(DeleteFile(filePath));
        UnmountHostRoot();
    }
}
#endif

namespace
{
    const nnt::fs::util::MountTestAttribute GetAttribute() NN_NOEXCEPT
    {
        nnt::fs::util::MountTestAttribute attribute = {};
        attribute.isReservedMountNameSupported = true;
        return attribute;
    }

    nn::Result MountBisForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountBis(mountName, nn::fs::BisPartitionId::System));
        NN_RESULT_SUCCESS;
    }

    const nnt::fs::util::MountTestParameter MountTestParameters[] = {
        { "MountBis", MountBisForMountNameTest, nullptr, GetAttribute },
    };
}

NNT_FS_INSTANTIATE_TEST_CASE_MOUNT(WithMountBis, ::testing::ValuesIn(MountTestParameters));
