﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <memory>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

using namespace nn::fs;

namespace
{
    static const int ThreadsNum = 2;
    nn::os::ThreadType g_Threads[ThreadsNum];
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_ThreadStacks[ThreadsNum][4096];

    template <nn::Result OpenNotifierFunction(std::unique_ptr<IEventNotifier>*), bool IsInsertedFunction()>
    void ClientThread(void* argument) NN_NOEXCEPT
    {
        NN_UNUSED(argument);

        std::unique_ptr<nn::fs::IEventNotifier> deviceDetectionEventNotifier;
        NNT_ASSERT_RESULT_SUCCESS(OpenNotifierFunction(&deviceDetectionEventNotifier));

        nn::os::SystemEventType deviceDetectionEvent;
        NNT_ASSERT_RESULT_SUCCESS(deviceDetectionEventNotifier->BindEvent(&deviceDetectionEvent, nn::os::EventClearMode_ManualClear));

        for (int i = 0; i < 10; i++)    // 10 回 挿抜イベントをチェックしたら終わる
        {
            NN_LOG("Wait (handle: 0x%08X)\n", nn::os::GetReadableHandleOfSystemEvent(&deviceDetectionEvent));
            nn::os::WaitSystemEvent(&deviceDetectionEvent);
            nn::os::ClearSystemEvent(&deviceDetectionEvent);

            NN_LOG("%s Detection! (#%d)\n", IsInsertedFunction() ? "Inserted" : "Removed", i);
        }

        nn::os::DestroySystemEvent(&deviceDetectionEvent);
    }
}

// 複数スレッドで SD カード挿抜検知を行うテスト
TEST(SdCardDetectionTest, MultiThread)
{
    for (int i = 0; i < ThreadsNum; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&g_Threads[i], ClientThread<nn::fs::OpenSdCardDetectionEventNotifier, nn::fs::IsSdCardInserted>, nullptr, g_ThreadStacks[i], sizeof(g_ThreadStacks[i]), 10));
        nn::os::StartThread(&g_Threads[i]);
    }

    for (int i = 0; i < ThreadsNum; i++)
    {
        nn::os::WaitThread(&g_Threads[i]);
        nn::os::DestroyThread(&g_Threads[i]);
    }
}

// 複数スレッドでゲームカード挿抜検知を行うテスト
TEST(GameCardDetectionTest, MultiThread)
{
    for (int i = 0; i < ThreadsNum; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&g_Threads[i], ClientThread<nn::fs::OpenGameCardDetectionEventNotifier, nn::fs::IsGameCardInserted>, nullptr, g_ThreadStacks[i], sizeof(g_ThreadStacks[i]), 10));
        nn::os::StartThread(&g_Threads[i]);
    }

    for (int i = 0; i < ThreadsNum; i++)
    {
        nn::os::WaitThread(&g_Threads[i]);
        nn::os::DestroyThread(&g_Threads[i]);
    }
}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nnt::fs::util::ResetAllocateCount();

    SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
