﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>

#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_SaveDataForDebug.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>

#include "testFs_FsLib_AccessLog.h"

namespace
{
    nn::fs::SaveDataId GetSaveDataId(const nn::fs::SystemSaveDataId systemSaveDataId) NN_NOEXCEPT
    {
        nn::util::optional<nn::fs::SaveDataInfo> saveDataInfo;
        nnt::fs::util::FindSaveData(
            &saveDataInfo,
            nn::fs::SaveDataSpaceId::System,
            [&](const nn::fs::SaveDataInfo info)
            {
                return info.systemSaveDataId == systemSaveDataId;
            });
        EXPECT_TRUE(saveDataInfo != nn::util::nullopt);
        return saveDataInfo->saveDataId;
    }
}

TEST(CommitSaveData, Success)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Commit", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CommitSaveData", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Commit", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CommitSaveData", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Commit", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CommitSaveData", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Commit", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CommitSaveData", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
*/
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug(nnt::fs::MountName));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::Commit(nnt::fs::MountName));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(nnt::fs::MountName));
    nn::fs::Unmount(nnt::fs::MountName);
}

TEST(CommitSaveData, Failure)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "Commit", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "CommitSaveData", name: "test" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "Commit", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "CommitSaveData", name: "test" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Commit", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CommitSaveData", name: "test" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Commit", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CommitSaveData", name: "test" }
*/
{
    ASSERT_TRUE(nn::fs::Commit(nnt::fs::MountName).IsFailure());
    ASSERT_TRUE(nn::fs::CommitSaveData(nnt::fs::MountName).IsFailure());
}

// NX のとき Failure、Generic のとき Success
#if defined(NN_BUILD_CONFIG_OS_WIN)
TEST(GetSaveDataSize, Success)
#else
TEST(GetSaveDataSize, Failure)
#endif
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "GetSaveDataSize", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "GetSaveDataSize", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetSaveDataSize", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetSaveDataSize", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
*/
{
    int64_t savedataSize;
    int64_t journalSize;
    (void)nn::fs::GetSaveDataSize(&savedataSize, &journalSize, nnt::fs::GetDefaultUid());
}

TEST(ExtendSaveData, Success)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ExtendSaveData", userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_size: 65536, save_data_journal_size: 65536 }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ExtendSaveData", userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_size: 65536, save_data_journal_size: 65536 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSaveData", savedataid: 0x1234 }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::ExtendSaveData", userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_size: 65536, save_data_journal_size: 65536 }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::ExtendSaveData", userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_size: 65536, save_data_journal_size: 65536 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSaveData", savedataid: 0x1234 }
*/
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::ApplicationId, nnt::fs::GetDefaultFsUid(), 1, 0x10000, 0x10000, 0));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(nnt::fs::GetSaveDataId(nnt::fs::GetDefaultFsUid())));
    };
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ExtendSaveData(nnt::fs::GetDefaultUid(), 0x10000, 0x10000));
}

TEST(ExtendSaveData, DISABLED_Failure)
{}

TEST(ExtendSystemSaveData, Success)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateSystemSaveData", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x0, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ExtendSaveData", savedataspaceid: System, savedataid: 0x1234, save_data_size: 65536, save_data_journal_size: 65536 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSystemSaveData", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateSystemSaveData", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x0, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::ExtendSaveData", savedataspaceid: System, savedataid: 0x1234, save_data_size: 65536, save_data_journal_size: 65536 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSystemSaveData", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
*/
{
    const nn::fs::SystemSaveDataId systemSaveDataId = 0x8000000000004000;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(nn::fs::SaveDataSpaceId::System, systemSaveDataId, nnt::fs::GetDefaultFsUid(), 0, 0x10000, 0x10000, 0));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSystemSaveData(nn::fs::SaveDataSpaceId::System, systemSaveDataId, nnt::fs::GetDefaultFsUid()));
    };
    nn::fs::SaveDataId saveDataId = GetSaveDataId(systemSaveDataId);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ExtendSaveData(nn::fs::SaveDataSpaceId::System, saveDataId, 0x10000, 0x10000));
}

TEST(ExtendSystemSaveData, Failure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "ExtendSaveData", savedataspaceid: User, savedataid: 0x1234, save_data_size: 1024, save_data_journal_size: 1024 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::ExtendSaveData", savedataspaceid: User, savedataid: 0x1234, save_data_size: 1024, save_data_journal_size: 1024 }
*/
{
    const nn::fs::SaveDataId saveDataId = 9999;
    ASSERT_TRUE(nn::fs::ExtendSaveData(nn::fs::SaveDataSpaceId::User, saveDataId, 1024, 1024).IsFailure());
}

// 有効化すると、他のテストが失敗するようになる
TEST(SetSaveDataRootPath, DISABLED_Success)
{
    nn::fs::SetSaveDataRootPath(nnt::fs::util::GetHostTemporaryPath());
}

TEST(SetSaveDataRootPath, DISABLED_Failure)
{}

TEST(SetSaveDataFlags, Success)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SetSaveDataFlags", savedataid: 0x1234, savedataspaceid: User, save_data_flags: 0x00000004 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSaveData", savedataid: 0x1234 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SetSaveDataFlags", savedataid: 0x1234, savedataspaceid: User, save_data_flags: 0x00000004 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSaveData", savedataid: 0x1234 }
*/
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::ApplicationId, nnt::fs::GetDefaultFsUid(), 1, 0x10000, 0x10000, 0));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(nnt::fs::GetSaveDataId(nnt::fs::GetDefaultFsUid())));
    };
    nn::fs::SaveDataId saveDataId = nnt::fs::GetSaveDataId(nnt::fs::GetDefaultFsUid());
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSaveDataFlags(saveDataId, nn::fs::SaveDataSpaceId::User, 0x4));
}

TEST(SetSaveDataFlags, Failure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "SetSaveDataFlags", savedataid: 0x1234, savedataspaceid: User, save_data_flags: 0x00000004 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SetSaveDataFlags", savedataid: 0x1234, savedataspaceid: User, save_data_flags: 0x00000004 }
*/
{
    nn::fs::SaveDataId saveDataId = 9999;
    ASSERT_TRUE(nn::fs::SetSaveDataFlags(saveDataId, nn::fs::SaveDataSpaceId::User, 0x4).IsFailure());
}

TEST(QuerySaveDataTotalSize, Success)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "QuerySaveDataTotalSize", save_data_size: 65536, save_data_journal_size: 32768 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::QuerySaveDataTotalSize", save_data_size: 65536, save_data_journal_size: 32768 }
*/
{
    int64_t totalSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QuerySaveDataTotalSize(&totalSize, 0x10000, 0x8000));
}

TEST(GetSaveDataStatus, Success)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "GetSaveDataOwnerId", save_data_size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "GetSaveDataFlags", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "GetSaveDataFlags", savedataspaceid: User, savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "GetSaveDataTimeStamp", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "GetSaveDataAvailableSize", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "GetSaveDataJournalSize", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "QuerySaveDataInternalStorageTotalSize", savedataspaceid: User, savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSaveData", savedataid: 0x1234 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetSaveDataOwnerId", save_data_size: 22 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetSaveDataFlags", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetSaveDataFlags", savedataspaceid: User, savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetSaveDataTimeStamp", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetSaveDataAvailableSize", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetSaveDataJournalSize", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00313802, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::QuerySaveDataInternalStorageTotalSize", savedataspaceid: User, savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSaveData", savedataid: 0x1234 }
*/
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::ApplicationId, nnt::fs::GetDefaultFsUid(), 1, 0x10000, 0x10000, 0));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(nnt::fs::GetSaveDataId(nnt::fs::GetDefaultFsUid())));
    };

    nn::fs::SaveDataId saveDataId = nnt::fs::GetSaveDataId(nnt::fs::GetDefaultFsUid());

    nn::Bit64 ownerId;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataOwnerId(&ownerId, saveDataId));

    uint32_t flags;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataFlags(&flags, saveDataId));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataFlags(&flags, nn::fs::SaveDataSpaceId::User, saveDataId));

    nn::time::PosixTime tm;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataTimeStamp(&tm, saveDataId));

    int64_t size;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataAvailableSize(&size, saveDataId));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataJournalSize(&size, saveDataId));
    (void)nn::fs::QuerySaveDataInternalStorageTotalSize(&size, nn::fs::SaveDataSpaceId::User, saveDataId);
}

TEST(GetSaveDataStatus, Failure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "GetSaveDataOwnerId", save_data_size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "GetSaveDataFlags", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "GetSaveDataFlags", savedataspaceid: User, savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "GetSaveDataTimeStamp", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "GetSaveDataAvailableSize", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "GetSaveDataJournalSize", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "QuerySaveDataInternalStorageTotalSize", savedataspaceid: User, savedataid: 0x1234 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetSaveDataOwnerId", save_data_size: 9999 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetSaveDataFlags", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetSaveDataFlags", savedataspaceid: User, savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetSaveDataTimeStamp", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetSaveDataAvailableSize", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetSaveDataJournalSize", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::QuerySaveDataInternalStorageTotalSize", savedataspaceid: User, savedataid: 0x1234 }
*/
{
    nn::fs::SaveDataId saveDataId = 9999;
    nn::Bit64 ownerId;
    ASSERT_TRUE(nn::fs::GetSaveDataOwnerId(&ownerId, saveDataId).IsFailure());

    uint32_t flags;
    ASSERT_TRUE(nn::fs::GetSaveDataFlags(&flags, saveDataId).IsFailure());
    ASSERT_TRUE(nn::fs::GetSaveDataFlags(&flags, nn::fs::SaveDataSpaceId::User, saveDataId).IsFailure());

    nn::time::PosixTime tm;
    ASSERT_TRUE(nn::fs::GetSaveDataTimeStamp(&tm, saveDataId).IsFailure());

    int64_t size;
    ASSERT_TRUE(nn::fs::GetSaveDataAvailableSize(&size, saveDataId).IsFailure());
    ASSERT_TRUE(nn::fs::GetSaveDataJournalSize(&size, saveDataId).IsFailure());
    ASSERT_TRUE(nn::fs::QuerySaveDataInternalStorageTotalSize(&size, nn::fs::SaveDataSpaceId::User, saveDataId).IsFailure());
}

TEST(SaveDataIteration, Success)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x2, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x3, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 4 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSaveData", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSaveData", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSaveData", savedataid: 0x1234 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x2, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x3, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 4 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSaveData", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSaveData", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSaveData", savedataid: 0x1234 }
*/
{
    std::unique_ptr<nn::fs::SaveDataIterator> iterator;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iterator, nn::fs::SaveDataSpaceId::User));

    nn::fs::UserId userId1 = {{0, 101}};
    nn::fs::UserId userId2 = {{0, 102}};
    nn::fs::UserId userId3 = {{0, 103}};
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::ApplicationId, userId1, 1, 0x10000, 0x10000, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::ApplicationId, userId2, 2, 0x10000, 0x10000, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::ApplicationId, userId3, 3, 0x10000, 0x10000, 0));

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iterator, nn::fs::SaveDataSpaceId::User));
    nn::fs::SaveDataInfo info[4] = {};
    int64_t count = 0;
    NNT_ASSERT_RESULT_SUCCESS(iterator->ReadSaveDataInfo(&count, info, 4));

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(nnt::fs::GetSaveDataId(userId1)));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(nnt::fs::GetSaveDataId(userId2)));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(nnt::fs::GetSaveDataId(userId3)));
}

TEST(SaveDataIteration, DISABLED_Failure)
{}
