﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat.h>

#include <nn/fs/fs_BcatSaveData.h>
#include <nn/fs/fs_SystemBcatSaveData.h>

#include "testFs_FsLib_AccessLog.h"

namespace {
    void DeleteBcatSaveData(const nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        nn::util::optional<nn::fs::SaveDataInfo> saveDataInfo;
        nnt::fs::util::FindSaveData(
            &saveDataInfo,
            nn::fs::SaveDataSpaceId::System,
            [&](const nn::fs::SaveDataInfo info)
            {
                return info.applicationId == applicationId
                    && info.saveDataType == nn::fs::SaveDataType::Bcat;
            });
        if (saveDataInfo != nn::util::nullopt)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(saveDataInfo->saveDataId));
        }
    }

    void DeleteSystemBcatSaveData(const nn::fs::SystemBcatSaveDataId systemSaveDataId) NN_NOEXCEPT
    {
        nn::util::optional<nn::fs::SaveDataInfo> saveDataInfo;
        nnt::fs::util::FindSaveData(
            &saveDataInfo,
            nn::fs::SaveDataSpaceId::System,
            [&](const nn::fs::SaveDataInfo info)
            {
                return info.systemSaveDataId == systemSaveDataId
                    && info.saveDataType == nn::fs::SaveDataType::SystemBcat;
            });
        if (saveDataInfo != nn::util::nullopt)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(saveDataInfo->saveDataId));
        }
    }
}

class BcatSaveData : public testing::Test
{
public:
    static void SetUpTestCase();
    static void TearDownTestCase();
};

void BcatSaveData::SetUpTestCase()
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountDeliveryCacheStorage" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "CreateBcatSaveData", applicationid: 0x100000000003006, save_data_size: 6291456 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountDeliveryCacheStorage" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountHost", name: "bcat-dc" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::bcat::MountDeliveryCacheStorage" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateBcatSaveData", applicationid: 0x100000000003006, save_data_size: 6291456 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountHost", name: "bcat-dc" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::bcat::MountDeliveryCacheStorage" }
*/
{
    nn::bcat::Initialize();
    int64_t saveDataSize = 0x0000000000600000;
    nn::fs::CreateBcatSaveData(nnt::fs::ApplicationId, saveDataSize);
    NNT_ASSERT_RESULT_SUCCESS(nn::bcat::MountDeliveryCacheStorage());
}

TEST_F(BcatSaveData, EnumerateDeliveryCacheDirectory_Success)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnumerateDeliveryCacheDirectory" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnumerateDeliveryCacheDirectory" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenDirectory", path: "bcat-dc:/" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::ReadDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::ReadDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::bcat::EnumerateDeliveryCacheDirectory" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenDirectory", path: "bcat-dc:/" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::ReadDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::ReadDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::bcat::EnumerateDeliveryCacheDirectory" }
*/
{
    static nn::bcat::DirectoryName s_DirNames[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
    int dirCount = 0;

    NNT_ASSERT_RESULT_SUCCESS(nn::bcat::EnumerateDeliveryCacheDirectory(&dirCount, s_DirNames, NN_ARRAY_SIZE(s_DirNames)));
}

TEST_F(BcatSaveData, DISABLED_EnumerateDeliveryCacheDirectory_Failure)
{}

#if defined(NN_BUILD_CONFIG_OS_WIN)
TEST_F(BcatSaveData, DeliveryCacheDirectory_Success)
#else
TEST_F(BcatSaveData, DeliveryCacheDirectory_Failure)
#endif
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x0000047A, handle: 0x0000000000000000, priority: Normal, function: "Open", class_name: DeliveryCacheDirectory, path: "dir1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000E7A, handle: 0x0000000000000000, priority: Normal, function: "Read", class_name: DeliveryCacheDirectory }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Close", class_name: DeliveryCacheDirectory }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0000047A, handle: 0x0000000000000000, priority: Normal, function: "Open", class_name: DeliveryCacheDirectory, path: "dir1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000E7A, handle: 0x0000000000000000, priority: Normal, function: "Read", class_name: DeliveryCacheDirectory }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Close", class_name: DeliveryCacheDirectory }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenDirectory", path: "bcat-dc:/dir1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheDirectory::Open", class_name: DeliveryCacheDirectory, path: "dir1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenDirectory", path: "bcat-dc:/dir1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::ReadDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::ReadDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheDirectory::Read", class_name: DeliveryCacheDirectory }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheDirectory::Close", class_name: DeliveryCacheDirectory }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenDirectory", path: "bcat-dc:/dir1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheDirectory::Open", class_name: DeliveryCacheDirectory, path: "dir1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenDirectory", path: "bcat-dc:/dir1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::ReadDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::ReadDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheDirectory::Read", class_name: DeliveryCacheDirectory }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheDirectory::Close", class_name: DeliveryCacheDirectory }
*/
{
    nn::bcat::DeliveryCacheDirectory directory;
    (void)directory.Open("dir1");

    static nn::bcat::DeliveryCacheDirectoryEntry s_Entries[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};
    int entryCount = 0;
    (void)directory.Read(&entryCount, s_Entries, NN_ARRAY_SIZE(s_Entries));

    directory.Close();
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
TEST_F(BcatSaveData, DeliveryCacheFile_Success)
#else
TEST_F(BcatSaveData, DeliveryCacheFile_Failure)
#endif
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x0000047A, handle: 0x0000000000000000, priority: Normal, function: "Open", path: "dir1/test.txt", class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x00000E7A, handle: 0x0000000000000000, priority: Normal, function: "Read", offset: 0, size: 8, class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Close", class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x0000047A, handle: 0x0000000000000000, priority: Normal, function: "Open", path: "dir1/test.txt", class_name: DeliveryCacheFile }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0000047A, handle: 0x0000000000000000, priority: Normal, function: "Open", path: "dir1/test.txt", class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x00000E7A, handle: 0x0000000000000000, priority: Normal, function: "Read", offset: 0, size: 8, class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Close", class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x0000047A, handle: 0x0000000000000000, priority: Normal, function: "Open", path: "dir1/test.txt", class_name: DeliveryCacheFile }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "bcat-dc:/dir1/test.txt", open_mode: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheFile::Open", path: "dir1/test.txt", class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 8 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheFile::Read", offset: 0, size: 8, class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheFile::Close", class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "bcat-dc:/dir1/test.txt", open_mode: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheFile::Open", path: "dir1/test.txt", class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheFile::Close", class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheFile::~DeliveryCacheFile", class_name: DeliveryCacheFile }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "bcat-dc:/dir1/test.txt", open_mode: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheFile::Open", path: "dir1/test.txt", class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 8 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheFile::Read", offset: 0, size: 8, class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheFile::Close", class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "bcat-dc:/dir1/test.txt", open_mode: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheFile::Open", path: "dir1/test.txt", class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheFile::Close", class_name: DeliveryCacheFile }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::bcat::DeliveryCacheFile::~DeliveryCacheFile", class_name: DeliveryCacheFile }
*/
{
    nn::bcat::DeliveryCacheFile file;

    (void)file.Open("dir1/test.txt");
    nn::Bit8 data[8] = {};
    size_t read;
    (void)file.Read(&read, 0, data, NN_ARRAY_SIZE(data));

    file.Close();
    (void)file.Open("dir1/test.txt");
}

void BcatSaveData::TearDownTestCase()
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "UnmountDeliveryCacheStorage" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "UnmountDeliveryCacheStorage" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "bcat-dc" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::bcat::UnmountDeliveryCacheStorage" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "bcat-dc" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::bcat::UnmountDeliveryCacheStorage" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
*/
{
    nn::bcat::UnmountDeliveryCacheStorage();
    DeleteBcatSaveData(nnt::fs::ApplicationId);
}

TEST(MountBcatSaveData, SuccessFailure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "CreateBcatSaveData", applicationid: 0x100000000003006, save_data_size: 6291456 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountBcatSaveData", name: "test", applicationid: 0x100000000003006 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000E02, handle: 0x0000000000000000, priority: Normal, function: "MountBcatSaveData", name: "test", applicationid: 0x100000000003006 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateBcatSaveData", applicationid: 0x100000000003006, save_data_size: 6291456 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountBcatSaveData", name: "test", applicationid: 0x100000000003006 }
FS_ACCESS: { start: 0, end: 0, result: 0x00007802, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountBcatSaveData", name: "test", applicationid: 0x100000000003006 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
*/
{
    int64_t saveDataSize = 0x600000;
    nn::fs::CreateBcatSaveData(nnt::fs::ApplicationId, saveDataSize);
    NN_UTIL_SCOPE_EXIT
    {
        DeleteBcatSaveData(nnt::fs::ApplicationId);
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBcatSaveData(nnt::fs::MountName, nnt::fs::ApplicationId));
    ASSERT_TRUE(nn::fs::MountBcatSaveData(nnt::fs::MountName, nnt::fs::ApplicationId).IsFailure());
    nn::fs::Unmount(nnt::fs::MountName);
}

TEST(MountSystemBcatSaveData, SuccessFailure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateSystemBcatSaveData", savedataid: 0x1234, save_data_size: 6291456, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSystemBcatSaveData", name: "test", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000E02, handle: 0x0000000000000000, priority: Normal, function: "MountSystemBcatSaveData", name: "test", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSaveData", savedataid: 0x1234 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateSystemBcatSaveData", savedataid: 0x1234, save_data_size: 6291456, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSystemBcatSaveData", name: "test", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00007802, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSystemBcatSaveData", name: "test", savedataid: 0x1234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSaveData", savedataid: 0x1234 }
*/
{
    nn::fs::SystemBcatSaveDataId systemBcatSaveDataId = 0x8000000000001000;
    int64_t saveDataSize = 0x600000;
    nn::fs::CreateSystemBcatSaveData(systemBcatSaveDataId, saveDataSize, 0);
    NN_UTIL_SCOPE_EXIT
    {
        DeleteSystemBcatSaveData(systemBcatSaveDataId);
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSystemBcatSaveData(nnt::fs::MountName, systemBcatSaveDataId));
    ASSERT_TRUE(nn::fs::MountSystemBcatSaveData(nnt::fs::MountName, systemBcatSaveDataId).IsFailure());
    nn::fs::Unmount(nnt::fs::MountName);
}

TEST(CreateBcatSaveData, Failure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "CreateBcatSaveData", applicationid: 0x100000000003006, save_data_size: 16384 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateBcatSaveData", applicationid: 0x100000000003006, save_data_size: 16384 }
*/
{
    ASSERT_TRUE(nn::fs::CreateBcatSaveData(nnt::fs::ApplicationId, 0x4000).IsFailure());
}

TEST(CreateSystemBcatSaveData, Failure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "CreateSystemBcatSaveData", savedataid: 0x1234, save_data_size: 8192, save_data_flags: 0x00000000 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateSystemBcatSaveData", savedataid: 0x1234, save_data_size: 8192, save_data_flags: 0x00000000 }
*/
{
    ASSERT_TRUE(nn::fs::CreateSystemBcatSaveData(1, 0x2000, 0x0).IsFailure());
}
