﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account.h>
#include <nn/fs.h>

#include <nn/account/account_ApiForAdministrators.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SaveDataForDebug.h>
#include <nn/pdm/pdm_QueryApi.h>

#include <nnt.h>

#include "testFs_FsLib_AccessLog.h"

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    #include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>
#endif

namespace nnt{ namespace fs{
    nnt::fs::util::String g_RomFilePath;
}}

nn::account::Uid nnt::fs::GetDefaultUid() NN_NOEXCEPT
{
    nn::account::Uid user;
#if defined(NN_BUILD_CONFIG_OS_WIN)
    int userCount = 0;
    auto result = nn::account::ListAllUsers(&userCount, &user, 1);
    if( result.IsFailure() || userCount <= 0 )
    {
        nn::account::BeginUserRegistration(&user);
        nn::account::CompleteUserRegistrationForcibly(user);
    }
#else
    int userCount = 0;
    auto result = nn::account::ListAllUsers(&userCount, &user, 1);
    NN_ABORT_UNLESS(result.IsSuccess() && userCount > 0);
#endif
    return user;
}

nn::fs::UserId nnt::fs::GetDefaultFsUid() NN_NOEXCEPT
{
    return nn::fs::ConvertAccountUidToFsUserId(GetDefaultUid());
}

nn::fs::SaveDataId nnt::fs::GetSaveDataId(nn::account::Uid uid) NN_NOEXCEPT
{
    nn::fs::UserId userId = {{uid._data[0], uid._data[1]}};
    return GetSaveDataId(userId);
}

nn::fs::SaveDataId nnt::fs::GetSaveDataId(nn::fs::UserId userId) NN_NOEXCEPT
{
    nn::util::optional<nn::fs::SaveDataInfo> saveDataInfo;
    nnt::fs::util::FindSaveData(
        &saveDataInfo,
        nn::fs::SaveDataSpaceId::User,
        [&](const nn::fs::SaveDataInfo info)
        {
            return info.saveDataUserId == userId;
        });
    EXPECT_TRUE(saveDataInfo != nn::util::nullopt);
    return saveDataInfo->saveDataId;
}

/*
AccessLogTest: NX
FS_ACCESS: { sdk_version: 0.0.0, spec: NX }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SetAllocator" }
AccessLogTest: NX-system
FS_ACCESS: { sdk_version: 0.0.0, spec: NX, for_system: true }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SetAllocator" }
AccessLogTest: Generic
FS_ACCESS: { sdk_version: 0.0.0, spec: Generic }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SetAllocator" }
AccessLogTest: Generic-system
FS_ACCESS: { sdk_version: 0.0.0, spec: Generic, for_system: true }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SetAllocator" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSystemSaveData", name: "pdm", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "pdm:/PlayEvent.dat", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "pdm:/BaseTimePoint.bin", open_mode: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Background, function: "nn::fs::SetPriorityRawOnCurrentThread" }
*/

TEST(Access, Priority)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "GetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "SetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "SetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SetPriorityOnCurrentThread" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "GetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "SetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "SetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SetPriorityOnCurrentThread" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "nn::fs::SetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "nn::fs::SetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SetPriorityOnCurrentThread" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "nn::fs::SetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "nn::fs::SetPriorityOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SetPriorityOnCurrentThread" }
*/
{
    nn::fs::Priority defaultPriority = nn::fs::GetPriorityOnCurrentThread();
    nn::fs::Priority priorityList[] = {
        nn::fs::Priority_Realtime,
        nn::fs::Priority_Normal,
        nn::fs::Priority_Low
    };
    for(nn::fs::Priority priority : priorityList)
    {
        nn::fs::SetPriorityOnCurrentThread(priority);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
        nn::fs::Unmount("test");
    }
    nn::fs::SetPriorityOnCurrentThread(defaultPriority);
}

TEST(Access, PriorityRaw)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Background, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Background, function: "Unmount", name: "test" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "GetPriorityRawOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "SetPriorityRawOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SetPriorityRawOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "SetPriorityRawOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Background, function: "SetPriorityRawOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Background, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Background, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SetPriorityRawOnCurrentThread" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Background, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Background, function: "nn::fs::Unmount", name: "test" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetPriorityRawOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "nn::fs::SetPriorityRawOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Realtime, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SetPriorityRawOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "nn::fs::SetPriorityRawOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Low, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Background, function: "nn::fs::SetPriorityRawOnCurrentThread" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Background, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Background, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SetPriorityRawOnCurrentThread" }
*/
{
    nn::fs::PriorityRaw defaultPriority = nn::fs::GetPriorityRawOnCurrentThread();
    nn::fs::PriorityRaw priorityList[] = {
        nn::fs::PriorityRaw_Realtime,
        nn::fs::PriorityRaw_Normal,
        nn::fs::PriorityRaw_Low,
        nn::fs::PriorityRaw_Background
    };
    for(nn::fs::PriorityRaw priority : priorityList)
    {
        nn::fs::SetPriorityRawOnCurrentThread(priority);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
        nn::fs::Unmount("test");
    }
    nn::fs::SetPriorityRawOnCurrentThread(defaultPriority);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    if( argc > 2 )
    {
        if( strcmp(argv[2], "fs_system_access_log_enable") == 0 )
        {
#if !defined(NN_SDK_BUILD_RELEASE)
            nn::fs::SetLocalSystemAccessLogForDebug(true);
#else
            NN_LOG("release build is not supported fs_system_access_log_enable.");
#endif
        }
    }

    nn::fs::SetLocalAccessLog(true);

#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::fs::SetGlobalAccessLogMode(nn::fs::AccessLogMode_Log);
#endif

    nn::fs::SetEnabledAutoAbort(false);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    if( argc > 1 )
    {
        nnt::fs::g_RomFilePath.assign(argv[1]);
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::account::InitializeForAdministrator();
    nn::pdm::InitializeForQuery();
#else
    nn::account::InitializeForSystemService();
#endif

    nnt::fs::util::ResetAllocateCount();
    nnt::fs::util::SetStackTraceDumpOnGlobalNewDeleteCallEnabled(true);

    auto result = RUN_ALL_TESTS();

    if( nnt::fs::util::CheckMemoryLeak())
    {
        NN_LOG("[ERROR] memory leak detected\n");
        nnt::Exit(1);
    }

    if( nnt::fs::util::IsGlobalNewDeleteCalled() )
    {
        NN_LOG("[ERROR] unexpected global new call is detected\n");
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
