﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/fsUtil/testFs_util_RecordingAccessRangeMock.h>

namespace nnt { namespace fs { namespace util {

void AsynchronousAccessAlignmentAndSplitSizeChecker::CheckForReading() NN_NOEXCEPT
{
    const bool isLowerPriority = IsLowerPriority();
    static const int64_t AccessOffset = 512 * 1024 - 1;

    {
        m_pRecorder->Clear();

        const size_t accessSize = (isLowerPriority ? 2 * 128 * 1024 : 2 * 512 * 1024);

        std::unique_ptr<char[]> buffer(new char[accessSize]);
        ASSERT_NE(nullptr, buffer.get());

        NNT_ASSERT_RESULT_SUCCESS(Read(AccessOffset, buffer.get(), accessSize));

        struct AccessRange
        {
            int64_t position;
            size_t size;
        };
        static const AccessRange HigherAccessRanges[] =
        {
            // パイプラインによる分割
            //   分割サイズ 512 KiB、アライメント 4 KiB のため
            //   初期アクセスサイズは 512 KiB - 4 KiB + 1
            { 524287, 520193 },
            { 1044480, 524288 },
            { 1568768, 4095 }
        };
        static const AccessRange LowerAccessRanges[] =
        {
            // パイプラインによらない分割
            //   分割サイズ 128 KiB、アライメント 16 KiB のため
            //   初期アクセスサイズは 128 KiB - 16 KiB + 1
            { 524287, 114689 },
            { 638976, 131072 },
            { 770048, 16383 },
        };
        auto pAccessRanges = (isLowerPriority ? LowerAccessRanges : HigherAccessRanges);
        auto sizeAccessRanges = (
            isLowerPriority
                ? (sizeof(LowerAccessRanges) / sizeof(LowerAccessRanges[0]))
                : (sizeof(HigherAccessRanges) / sizeof(HigherAccessRanges[0]))
            );

        decltype(sizeAccessRanges) index = 0;
        NNT_FS_ASSERT_NO_FATAL_FAILURE(
            m_pRecorder->Check([&](nnt::fs::util::AccessRangeRecord& record) NN_NOEXCEPT
            {
                ASSERT_LT(index, sizeAccessRanges);
                ASSERT_FALSE(record.IsWriteAccess());
                ASSERT_EQ(pAccessRanges[index].position, record.GetPosition());
                ASSERT_EQ(pAccessRanges[index].size, record.GetSize());
                ++index;
            })
        );
    }

    if( !isLowerPriority )
    {
        m_pRecorder->Clear();

        // パイプラインを使用しないアクセスはアライメント制約がない
        const size_t AccessSize = 512 * 1024;

        std::unique_ptr<char[]> buffer(new char[AccessSize]);
        ASSERT_NE(nullptr, buffer.get());

        NNT_ASSERT_RESULT_SUCCESS(Read(AccessOffset, buffer.get(), AccessSize));

        NNT_FS_ASSERT_NO_FATAL_FAILURE(
            m_pRecorder->Check([&](nnt::fs::util::AccessRangeRecord& record) NN_NOEXCEPT
            {
                ASSERT_FALSE(record.IsWriteAccess());
                ASSERT_EQ(AccessOffset, record.GetPosition());
                ASSERT_EQ(AccessSize, record.GetSize());
            })
        );
    }
}

void AsynchronousAccessAlignmentAndSplitSizeChecker::CheckForWriting() NN_NOEXCEPT
{
    const bool isLowerPriority = IsLowerPriority();
    static const int64_t AccessOffset = 512 * 1024 - 1;

    m_pRecorder->Clear();

    const size_t accessSize = (isLowerPriority ? 2 * 64 * 1024 : 2 * 512 * 1024);

    std::unique_ptr<char[]> buffer(new char[accessSize]);
    ASSERT_NE(nullptr, buffer.get());

    NNT_ASSERT_RESULT_SUCCESS(Write(AccessOffset, buffer.get(), accessSize));

    struct AccessRange
    {
        int64_t position;
        size_t size;
    };
    static const AccessRange HigherAccessRanges[] =
    {
        // パイプラインによらない分割
        //   分割サイズ 512 KiB、アライメント 16 KiB のため
        //   初期アクセスサイズは 512 KiB - 16 KiB + 1
        { 524287, 507905 },
        { 1032192, 524288 },
        { 1556480, 16383 }
    };
    static const AccessRange LowerAccessRanges[] =
    {
        // パイプラインによらない分割
        //   分割サイズ 64 KiB、アライメント 16 KiB のため
        //   初期アクセスサイズは 64 KiB - 16 KiB + 1
        { 524287, 49153 },
        { 573440, 65536 },
        { 638976, 16383 }
    };
    auto pAccessRanges = (isLowerPriority ? LowerAccessRanges : HigherAccessRanges);
    auto sizeAccessRanges = (
        isLowerPriority
            ? (sizeof(LowerAccessRanges) / sizeof(LowerAccessRanges[0]))
            : (sizeof(HigherAccessRanges) / sizeof(HigherAccessRanges[0]))
        );

    decltype(sizeAccessRanges) index = 0;
    NNT_FS_ASSERT_NO_FATAL_FAILURE(
        m_pRecorder->Check([&](nnt::fs::util::AccessRangeRecord& record) NN_NOEXCEPT
        {
            ASSERT_TRUE(record.IsWriteAccess());
            ASSERT_LT(index, sizeAccessRanges);
            ASSERT_EQ(pAccessRanges[index].position, record.GetPosition());
            ASSERT_EQ(pAccessRanges[index].size, record.GetSize());
            ++index;
        })
    );
}

}}} // namespace nnt::fs::util
