﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nnt/nntest.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace {
    const int FileSize = 32;

    typedef nn::Result(*ReadFunction)(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, size_t expectedSize);

    nn::Result ReadFunction0(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, size_t expectedSize) NN_NOEXCEPT
    {
        NN_UNUSED(size);

        NN_RESULT_DO(pTestFile->Read(offset, buffer, expectedSize, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction1(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, size_t expectedSize) NN_NOEXCEPT
    {
        NN_UNUSED(size);

        NN_RESULT_DO(pTestFile->Read(offset, buffer, expectedSize));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction2(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, size_t expectedSize) NN_NOEXCEPT
    {
        size_t readSize;
        nnt::fs::util::InvalidateVariable(&readSize);
        NN_RESULT_DO(pTestFile->Read(&readSize, offset, buffer, size, nn::fs::ReadOption()));
        EXPECT_EQ(expectedSize, readSize);
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction3(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, size_t expectedSize) NN_NOEXCEPT
    {
        size_t readSize;
        nnt::fs::util::InvalidateVariable(&readSize);
        NN_RESULT_DO(pTestFile->Read(&readSize, offset, buffer, size));
        EXPECT_EQ(expectedSize, readSize);
        NN_RESULT_SUCCESS;
    }

    bool IsReadOverload(ReadFunction read) NN_NOEXCEPT
    {
        return read != ReadFunction2;
    }

    const ReadFunction ReadFunctions[] = {
        ReadFunction0,
        ReadFunction1,
        ReadFunction2,
        ReadFunction3
    };
}

namespace nnt { namespace fs { namespace api {
    void LoadPostConditionRoTests()
    {
        return;
    }

    class PostConditionRo : public GetFileSystemTestFixture
    {
    };

    class PostConditionRoFile : public GetFileSystemTestFixture, public ::testing::WithParamInterface<ReadFunction>
    {
    public:
        String m_FileName;
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::SetUp();
            m_FileName = GetTestRootPath().append("/PostConditionRo/test.file");
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::TearDown();
        }

        void MakeExpectImage(char* pBuffer, const int bufferSize) NN_NOEXCEPT
        {
            for (int cnt = 0; cnt < bufferSize; cnt++)
            {
                pBuffer[cnt] = cnt & 0xffU;
            }
        }
    };

    class PostConditionRoDirectory : public GetFileSystemTestFixture
    {
    public:
        String m_DirName;
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::SetUp();
            m_DirName = GetTestRootPath().append("/PostConditionRo/test");
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::TearDown();
        }
    };

    TEST_F(PostConditionRoFile, GetEntryType)
    {
        DirectoryEntryType deType;
        util::InvalidateVariable(&deType);
        NNT_EXPECT_RESULT_SUCCESS(GetFs().GetEntryType(&deType, m_FileName.c_str()));
        EXPECT_EQ(DirectoryEntryType_File, deType);
    }

    TEST_F(PostConditionRoDirectory, GetEntryType)
    {
        DirectoryEntryType deType;
        util::InvalidateVariable(&deType);
        NNT_EXPECT_RESULT_SUCCESS(GetFs().GetEntryType(&deType, m_DirName.c_str()));
        EXPECT_EQ(DirectoryEntryType_Directory, deType);
    }

    TEST_P(PostConditionRoFile, OpenFile_WriteFile_Flush_ReadFile)
    {
        auto read = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        std::unique_ptr<ITestFile> file;
        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, m_FileName.c_str(), OpenMode_Read));
        char writeBuffer[FileSize];
        MakeExpectImage(writeBuffer, FileSize);
        char readBuffer[FileSize];
        util::InvalidateVariable(readBuffer, FileSize);
        NNT_EXPECT_RESULT_SUCCESS(read(file.get(), 0, readBuffer, FileSize, static_cast<size_t>(FileSize)));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer, FileSize);
        file.reset(nullptr);
    }

    TEST_F(PostConditionRoFile, OpenFile_SetSize_GetSize)
    {
        std::unique_ptr<ITestFile> file;
        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, m_FileName.c_str(), OpenMode_Read));
        int64_t actualSize;
        util::InvalidateVariable(&actualSize);
        NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&actualSize));
        EXPECT_EQ(actualSize, static_cast<int64_t>(FileSize));
        file.reset(nullptr);
    }

    TEST_F(PostConditionRoDirectory, OpenDirectory_ReadDirectory_GetDirectoryEntryCount)
    {
        String fileName = m_DirName;
        fileName.append("/test.file");
        String dirName = m_DirName;
        dirName.append("/test");

        DirectoryEntry de[2];
        util::InvalidateVariable(de, 2);

        {
            std::unique_ptr<ITestDirectory> dir;
            NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&dir, m_DirName.c_str(), OpenDirectoryMode_File));
            int64_t entryCount;
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->GetEntryCount(&entryCount));
            EXPECT_EQ(1, entryCount);
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, de, 1));
            EXPECT_EQ(1, entryCount);
            EXPECT_TRUE(strncmp(de[0].name, "test.file", sizeof(de[0].name)) == 0);
            EXPECT_EQ(DirectoryEntryType_File, de[0].directoryEntryType);
            EXPECT_EQ(FileSize, de[0].fileSize);
            // すべて読み出した後は 0 が返る
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, de, 1));
            EXPECT_EQ(0, entryCount);
        }
        {
            std::unique_ptr<ITestDirectory> dir;
            NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&dir, m_DirName.c_str(), OpenDirectoryMode_Directory));
            int64_t entryCount;
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->GetEntryCount(&entryCount));
            EXPECT_EQ(1, entryCount);
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, de, 1));
            EXPECT_EQ(1, entryCount);
            EXPECT_TRUE(strncmp(de[0].name, "test", sizeof(de[0].name)) == 0);
            EXPECT_EQ(DirectoryEntryType_Directory, de[0].directoryEntryType);
            EXPECT_EQ(0, de[0].fileSize);
            // すべて読み出した後は 0 が返る
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, de, 1));
            EXPECT_EQ(0, entryCount);
        }
        {
            std::unique_ptr<ITestDirectory> dir;
            NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&dir, m_DirName.c_str(), static_cast<OpenDirectoryMode>(OpenDirectoryMode_File | OpenDirectoryMode_Directory)));
            int64_t entryCount;
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->GetEntryCount(&entryCount));
            EXPECT_EQ(2, entryCount);
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, de, 2));
            EXPECT_EQ(2, entryCount);
            for (int i = 0; i < 2; i++)
            {
                if (de[i].directoryEntryType == DirectoryEntryType_File)
                {
                    EXPECT_TRUE(strncmp(de[i].name, "test.file", sizeof(de[i].name)) == 0);
                    EXPECT_EQ(FileSize, de[i].fileSize);
                }
                else
                {
                    EXPECT_TRUE(strncmp(de[i].name, "test", sizeof(de[i].name)) == 0);
                    EXPECT_EQ(0, de[i].fileSize);
                }
            }
            // すべて読み出した後は 0 が返る
            util::InvalidateVariable(&entryCount);
            NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, de, 1));
            EXPECT_EQ(0, entryCount);
        }
    }

    INSTANTIATE_TEST_CASE_P(WithReadOverloads,
        PostConditionRoFile,
        ::testing::ValuesIn(ReadFunctions));
}}}
