﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nnt/nntest.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

/**
* @file
* @brief    ファイルシステム関数、アライメント観点を定義します。
*/
namespace {
    // 0～64までのオフセット用MAX値
    const size_t g_offSetSize = 64;

    // 各サイズ定義
    const size_t g_readSize    = 1024;
    const size_t g_compareSize = 1024;

    // バッファ初期値の隣接比較サイズ
    const size_t g_initRelatedCompSize = 32;

    // バッファ初期値の比較定義(InvalidateVariable()によって初期値に設定される値)
    const uint8_t g_initValue  = 0x5A;

    // リライト用バッファ
    static uint8_t g_readBuffer[2 * 1024];
    static uint8_t g_writeBuffer[2 * 1024];

    typedef nn::Result(*ReadFunction)(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size);

    nn::Result ReadFunction0(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction1(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction2(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction3(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    bool IsReadOverload(ReadFunction read) NN_NOEXCEPT
    {
        return read != ReadFunction2;
    }

    const ReadFunction ReadFunctions[] =
    {
        ReadFunction0,
        ReadFunction1,
        ReadFunction2,
        ReadFunction3
    };
}

namespace nnt { namespace fs { namespace api {
    void LoadBufferAddressAlignmentRoTests()
    {
        return;
    }

    /**
     * @brief   BufferAddressAlignmentRo で利用するテストフィクスチャです。
     */
    class BufferAddressAlignmentRo : public GetFileSystemTestFixture, public ::testing::WithParamInterface<ReadFunction>
    {
    protected:

        /**
         * @brief テスト開始時に毎回呼び出される関数です。
         */
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::SetUp();
        }

        /**
         * @brief テスト終了時に毎回呼び出される関数です。
         */
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::TearDown();
        }

        /**
         * @brief バッファの初期値領域を確認する関数です。
         *   @param[in]  pBuffer                初期値を確認するバッファの先頭ポインタを指定します
         *   @param[in]  bufferSize             指定しているバッファのサイズ（バイト単位）を指定します
         *   @param[in]  offsetSize             バッファをオフセットするサイズ（バイト単位）を指定します
         *
         *   @return     指定バッファの比較結果が返ります。
         *   @retval     true                   指定バッファには初期値のみ含まれています。
         *   @retval     false                  指定バッファには初期値以外の内容が含まれています。
         */
        bool BufferInitBoundComp(const uint8_t* pBuffer, const size_t bufferSize, const size_t offsetSize) NN_NOEXCEPT
        {
            uint8_t* pBaseAddress    = const_cast<uint8_t*>(pBuffer);
            uint8_t* pAddressPos     = nullptr;
            size_t compSize          = 0;

            // 比較終端までのサイズ設定
            compSize        = std::min(offsetSize, g_initRelatedCompSize);

            // 比較する先頭バッファアドレスを設定（隣接サイズ分のみ比較するため、先頭アドレス移動）
            pAddressPos     = (pBaseAddress + offsetSize) - compSize;

            // 設定した先頭バッファアドレスから初期値かを比較
            for (size_t checkPos = 0; checkPos < compSize; checkPos++)
            {
                if (pAddressPos[checkPos] != g_initValue)
                {
                    return false;
                }
            }

            // 比較終端までのサイズ設定
            compSize        = std::min(bufferSize - (offsetSize + g_readSize), g_initRelatedCompSize);

            // 比較する先頭バッファアドレスを設定（Read領域後の先頭アドレス）
            pAddressPos     = pBaseAddress + offsetSize + g_readSize;

            // Read領域後の先頭バッファアドレスから初期値かを比較
            for (size_t checkPos = 0; checkPos < compSize; checkPos++)
            {
                if (pAddressPos[checkPos] != g_initValue)
                {
                    return false;
                }
            }

            return true;
        }

        void MakeExpectImage(uint8_t* pBuffer, const size_t bufferSize) NN_NOEXCEPT
        {
            for (unsigned int cnt = 0; cnt < bufferSize; cnt++)
            {
                pBuffer[cnt] = cnt & 0xffU;
            }
        }
    };

    //!< @brief 2048バイトのファイルを0～64バイト範囲でオフセットしつつ、1024バイト単位の読み込みが正しくできること
    TEST_P(BufferAddressAlignmentRo, ReadFile_WriteFile)
    {
        auto read = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        std::unique_ptr<ITestFile> file;
        String fileName = GetTestRootPath().append("/BufferAddressAlignmentRo/test.file");

        // 期待値データのパターンデータを作成
        this->MakeExpectImage(&g_writeBuffer[0], sizeof(g_writeBuffer));

        for (size_t changePos = 0; changePos <= g_offSetSize; changePos++)
        {
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Read));

            util::InvalidateVariable(g_readBuffer, sizeof(g_readBuffer));

            NNT_EXPECT_RESULT_SUCCESS(read(file.get(), 0, &g_readBuffer[changePos], g_readSize));

            NNT_FS_UTIL_EXPECT_MEMCMPEQ(&g_writeBuffer[0], &g_readBuffer[changePos], g_compareSize);

            EXPECT_TRUE(this->BufferInitBoundComp(&g_readBuffer[0], sizeof(g_readBuffer), changePos) == true);

            file.reset(nullptr);
        }
    }

    INSTANTIATE_TEST_CASE_P(WithReadOverloads,
        BufferAddressAlignmentRo,
        ::testing::ValuesIn(ReadFunctions));
}}}
