﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

// 2 つの NA をリンクする必要がある。
//
// 0: NA とフレンド候補取得用 SNS アカウントの紐付け解除 API を実行する（紐付け解除されては困る NA を使用しないよう注意）
//    → 自身の適当な NA を使用する。
// 1: NA とフレンド候補取得用 SNS アカウントの紐付け状態・フレンドリスト取得 API を実行する
//    → 動作確認用 SNS アカウント（https://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=270185076）

namespace
{
    nn::account::Uid s_Users[nn::account::UserCountMax] = {};
    nn::account::NetworkServiceAccountId s_AccountIds[nn::account::UserCountMax] = {};
    int s_UserCount = 0;
}

class FriendsSnsAccount : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());

        nnt::friends::LoadAccounts(&s_UserCount, s_Users, s_AccountIds, NN_ARRAY_SIZE(s_Users));
        NN_ABORT_UNLESS_GREATER_EQUAL(s_UserCount, 2);

        nn::friends::SetOption(nn::friends::OptionAdmin_CheckUserStatus, 0);
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }
};

TEST_F(FriendsSnsAccount, GetLinkWebPageUrl)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    static nn::friends::WebPageUrl s_Url;

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountLinkPageUrl(&context, &s_Url, s_Users[0],
            nn::friends::SnsAccountType_Facebook));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        NN_LOG("Facebook: %s\n", s_Url.value);
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountLinkPageUrl(&context, &s_Url, s_Users[0],
            nn::friends::SnsAccountType_Twitter));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        NN_LOG("Twitter: %s\n", s_Url.value);
    }
}

TEST_F(FriendsSnsAccount, Unlink)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::UnlinkSnsAccount(&context, s_Users[0],
            nn::friends::SnsAccountType_Facebook));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::UnlinkSnsAccount(&context, s_Users[0],
            nn::friends::SnsAccountType_Twitter));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
}

TEST_F(FriendsSnsAccount, GetLinkage)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    {
        nn::friends::SnsAccountLinkage linkage = {};

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountLinkage(&context, &linkage, s_Users[0]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        // 紐付け解除しているので、false になっていることは保証される。
        EXPECT_FALSE(linkage.isFacebookLinked);
        EXPECT_FALSE(linkage.isTwitterLinked);
    }

    {
        nn::friends::SnsAccountLinkage linkage = {};

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountLinkage(&context, &linkage, s_Users[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        NN_LOG("Linkage:\n");
        NN_LOG("    Facabook = %s\n", linkage.isFacebookLinked ? "Linked" : "Not linked");
        NN_LOG("    Twitter  = %s\n", linkage.isTwitterLinked ? "Linked" : "Not linked");
    }
}

TEST_F(FriendsSnsAccount, GetProfile)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::SnsAccountProfile profile = {};

    // 紐付け解除しているので、何も取れない。
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountProfile(&context, &profile, s_Users[0],
            s_AccountIds[0], nn::friends::SnsAccountType_Facebook));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNintendoAccountIsNotLinkedToSnsAccount, context.GetResult());
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountProfile(&context, &profile, s_Users[0],
            s_AccountIds[0], nn::friends::SnsAccountType_Twitter));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNintendoAccountIsNotLinkedToSnsAccount, context.GetResult());
    }

    nn::friends::SnsAccountLinkage linkage = {};

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountLinkage(&context, &linkage, s_Users[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }

    // 0 と 1 は SNS 上でフレンドではないので情報は取れない。
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountProfile(&context, &profile, s_Users[1],
            s_AccountIds[0], nn::friends::SnsAccountType_Facebook));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        if (linkage.isFacebookLinked)
        {
            NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultSnsProfileNotGet, context.GetResult());
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNintendoAccountIsNotLinkedToSnsAccount, context.GetResult());
        }
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountProfile(&context, &profile, s_Users[1],
            s_AccountIds[0], nn::friends::SnsAccountType_Twitter));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        if (linkage.isTwitterLinked)
        {
            NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultSnsProfileNotGet, context.GetResult());
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNintendoAccountIsNotLinkedToSnsAccount, context.GetResult());
        }
    }

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountProfile(&context, &profile, s_Users[1],
            s_AccountIds[1], nn::friends::SnsAccountType_Facebook));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        if (linkage.isFacebookLinked)
        {
            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            NN_LOG("Facebook profile:\n");
            NN_LOG("    name1    = %s\n", profile.name1.value);
            NN_LOG("    name2    = %s\n", profile.name2.value);
            NN_LOG("    imageUrl = %s\n", profile.imageUrl.value);
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNintendoAccountIsNotLinkedToSnsAccount, context.GetResult());
        }
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountProfile(&context, &profile, s_Users[1],
            s_AccountIds[1], nn::friends::SnsAccountType_Twitter));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        if (linkage.isTwitterLinked)
        {
            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            NN_LOG("Twitter profile:\n");
            NN_LOG("    name1    = %s\n", profile.name1.value);
            NN_LOG("    name2    = %s\n", profile.name2.value);
            NN_LOG("    imageUrl = %s\n", profile.imageUrl.value);
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNintendoAccountIsNotLinkedToSnsAccount, context.GetResult());
        }
    }
} // NOLINT(impl/function_size)

TEST_F(FriendsSnsAccount, GetFriendList)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    static nn::friends::SnsAccountFriend s_Friends[300];
    int count = 0;

    // 紐付け解除しているので、何も取れない。
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountFriendList(&context, &count, s_Friends,
            s_Users[0], NN_ARRAY_SIZE(s_Friends), nn::friends::SnsAccountType_Facebook));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNintendoAccountIsNotLinkedToSnsAccount, context.GetResult());
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountFriendList(&context, &count, s_Friends,
            s_Users[0], NN_ARRAY_SIZE(s_Friends), nn::friends::SnsAccountType_Twitter));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNintendoAccountIsNotLinkedToSnsAccount, context.GetResult());
    }

    nn::friends::SnsAccountLinkage linkage = {};

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountLinkage(&context, &linkage, s_Users[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountFriendList(&context, &count, s_Friends,
            s_Users[1], NN_ARRAY_SIZE(s_Friends), nn::friends::SnsAccountType_Facebook));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        if (linkage.isFacebookLinked)
        {
            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            for (int i = 0; i < count; i++)
            {
                EXPECT_EQ(s_Friends[i].GetType(), nn::friends::SnsAccountType_Facebook);
            }

            nnt::friends::Dump(s_Friends, count);
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNintendoAccountIsNotLinkedToSnsAccount, context.GetResult());
        }
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetSnsAccountFriendList(&context, &count, s_Friends,
            s_Users[1], NN_ARRAY_SIZE(s_Friends), nn::friends::SnsAccountType_Twitter));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        if (linkage.isTwitterLinked)
        {
            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            for (int i = 0; i < count; i++)
            {
                EXPECT_EQ(s_Friends[i].GetType(), nn::friends::SnsAccountType_Twitter);
            }

            nnt::friends::Dump(s_Friends, count);
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNintendoAccountIsNotLinkedToSnsAccount, context.GetResult());
        }
    }
}
