﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/os.h>
#include <nn/fgm.h>
#include <nn/nn_Log.h>
#include <nnt/nntest.h>


class FgmRequestBasicTest: public ::testing::TestWithParam<nn::fgm::Module>{};
const nn::fgm::Module modules[] =
{
    nn::fgm::Module_Cpu,
    nn::fgm::Module_Gpu,
    //nn::fgm::Module_Emc,
    nn::fgm::Module_SysBus,
    nn::fgm::Module_Mselect,
    nn::fgm::Module_Nvdec,
    nn::fgm::Module_Nvenc,
    nn::fgm::Module_Nvjpg,
    nn::fgm::Module_Test,
};

INSTANTIATE_TEST_CASE_P(FgmBasicTestModules, FgmRequestBasicTest, testing::ValuesIn(modules));


// *** Initialize basic testing ***//
TEST_P(FgmRequestBasicTest, Initialize_DefaultClear)
{
    nn::Result result;
    nn::fgm::Request request;

    result = request.Initialize(GetParam(), nn::fgm::Priority_Default);
    EXPECT_TRUE(result.IsSuccess());

    result = request.Finalize();
    ASSERT_TRUE(result.IsSuccess());

}

TEST_P(FgmRequestBasicTest, Initialize_ManualClear)
{
    nn::Result result;
    nn::fgm::Request request;

    result = request.Initialize(GetParam(), nn::fgm::Priority_Default, nn::os::EventClearMode_ManualClear);
    EXPECT_TRUE(result.IsSuccess());

    result = request.Finalize();
    ASSERT_TRUE(result.IsSuccess());

}

TEST_P(FgmRequestBasicTest, Initialize_AutoClear)
{
    nn::Result result;
    nn::fgm::Request request;

    result = request.Initialize(GetParam(), nn::fgm::Priority_Default, nn::os::EventClearMode_AutoClear);
    EXPECT_TRUE(result.IsSuccess());

    result = request.Finalize();
    ASSERT_TRUE(result.IsSuccess());

}

// Breaks test, os system even crashes and stops test from running
// Error: システムイベントが初期化されていません。
TEST_P(FgmRequestBasicTest, DISABLED_Initialize_InvalidModule)
{
    nn::Result result;
    nn::fgm::Request request;

    result = request.Initialize(nn::fgm::Module_NumModule, nn::fgm::Priority_Default);
    EXPECT_FALSE(result.IsSuccess());

    result = request.Finalize();
    ASSERT_FALSE(result.IsSuccess());
}

// Breaks Test, fgm Aborts unless priority <= Priority_Lowest
TEST_P(FgmRequestBasicTest, DISABLED_Initialize_InvalidPriorities)
{
    nn::Result result;
    nn::fgm::Request request;

    result = request.Initialize(nn::fgm::Module_Test, nn::fgm::Priority_Lowest + 1);
    EXPECT_FALSE(result.IsSuccess());

    result = request.Finalize();
    ASSERT_FALSE(result.IsSuccess());
}

// *** Get settings basic testing *** //
TEST_P(FgmRequestBasicTest, GetSettings)
{
    nn::Result result;
    nn::fgm::Request request;
    nn::fgm::Setting currentRate;

    result = request.Initialize(GetParam(), nn::fgm::Priority_Default);
    EXPECT_TRUE(result.IsSuccess());

    result = request.Get(&currentRate);
    EXPECT_TRUE(result.IsSuccess());
    NN_LOG("Module %d: %.2f MHz\n", GetParam(), currentRate / (1000.0 * 1000.0));

    result = request.Finalize();
    ASSERT_TRUE(result.IsSuccess());
}

// *** Set range basic testing *** /
TEST_P(FgmRequestBasicTest, SetRange_MinLessMax)
{
    nn::Result result;
    nn::fgm::Request request;
    nn::fgm::Setting min, max, currentRate, newRate;

    result = request.Initialize(GetParam(), nn::fgm::Priority_Default);
    EXPECT_TRUE(result.IsSuccess());

    result = request.Get(&currentRate);
    EXPECT_TRUE(result.IsSuccess());

    srand((int)nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds());

    max = rand();
    min = rand() % max;
    EXPECT_GE(max, min);
    result = request.SetAndWait(min, max);
    ASSERT_TRUE(result.IsSuccess());

    newRate = 0;
    result = request.Get(&newRate);
    EXPECT_TRUE(result.IsSuccess());

    NN_LOG("Set range to %.2f to %.2f MHz\n",
        min / (1000.0 * 1000.0),
        max / (1000.0 * 1000.0));

    if (newRate != currentRate)
    {
        NN_LOG("Setting changed from %.2f to %.2f MHz\n",
            currentRate / (1000.0 * 1000.0),
            newRate / (1000.0 * 1000.0));
    }
    else
    {
        NN_LOG("Request did not change current setting.");
    }
    result = request.Finalize();
    ASSERT_TRUE(result.IsSuccess());
}

TEST_P(FgmRequestBasicTest, SetRange_MinEqualMax)
{
    nn::Result result;
    nn::fgm::Request request;
    nn::fgm::Setting rate, currentRate, newRate;

    result = request.Initialize(GetParam(), nn::fgm::Priority_Default);
    EXPECT_TRUE(result.IsSuccess());

    result = request.Get(&currentRate);
    EXPECT_TRUE(result.IsSuccess());

    srand((int)nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds());
    rate = rand();
    result = request.SetAndWait(rate, rate);
    ASSERT_TRUE(result.IsSuccess());

    newRate = 0;
    result = request.Get(&newRate);
    EXPECT_TRUE(result.IsSuccess());

    NN_LOG("Set range min and max to %.2f MHz\n",
        rate / (1000.0 * 1000.0));

    if (newRate != currentRate)
    {
        NN_LOG("Setting changed from %.2f to %.2f MHz\n",
            currentRate / (1000.0 * 1000.0),
            newRate / (1000.0 * 1000.0));
    }
    else
    {
        NN_LOG("Request did not change current setting.");
    }

    result = request.Finalize();
    ASSERT_TRUE(result.IsSuccess());
}

TEST_P(FgmRequestBasicTest, SetRange_MinMoreMax)
{
    nn::Result result;
    nn::fgm::Request request;
    nn::fgm::Setting min, max, currentRate, newRate;

    result = request.Initialize(GetParam(), nn::fgm::Priority_Default);
    EXPECT_TRUE(result.IsSuccess());

    result = request.Get(&currentRate);
    EXPECT_TRUE(result.IsSuccess());

    min = rand();
    max = rand() % min + 1;
    EXPECT_GT(min, max);
    result = request.SetAndWait(min, max);
    EXPECT_FALSE(result.IsSuccess());
    if (result.IsFailure())
        NN_LOG("Min greater than max test, error %d:%d\n", result.GetModule(), result.GetDescription());

    result = request.Get(&newRate);
    EXPECT_TRUE(result.IsSuccess());

    NN_LOG("Set range to %.2f to %.2f MHz\n",
        min / (1000.0 * 1000.0),
        max / (1000.0 * 1000.0));

    NN_LOG("Setting changed from %.2f to %.2f MHz\n",
        currentRate / (1000.0 * 1000.0),
        newRate / (1000.0 * 1000.0));

    result = request.Finalize();
    ASSERT_TRUE(result.IsSuccess());

}

// ** Priority Tests **//
TEST_P(FgmRequestBasicTest, DifferentPriority_NoIntersection)
{
    nn::Result result;
    nn::fgm::Request defaultRequest, highRequest;
    nn::fgm::Setting lowRate, highRate;

    EXPECT_TRUE((result = defaultRequest.Initialize(GetParam(), nn::fgm::Priority_Default)).IsSuccess());
    EXPECT_TRUE((result = highRequest.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());

    result = highRequest.SetAndWait(nn::fgm::Setting_Max, nn::fgm::Setting_Max);
    EXPECT_TRUE(result.IsSuccess());

    result = highRequest.Get(&highRate);
    EXPECT_TRUE(result.IsSuccess());

    result = defaultRequest.SetAndWait(nn::fgm::Setting_Min, nn::fgm::Setting_Min);
    EXPECT_TRUE(result.IsSuccess());

    result = defaultRequest.Get(&lowRate);
    EXPECT_TRUE(result.IsSuccess());

    // Higher priorty request wins
    EXPECT_EQ(highRate, lowRate);
    EXPECT_TRUE((result = defaultRequest.Finalize()).IsSuccess());
    ASSERT_TRUE((result = highRequest.Finalize()).IsSuccess());
}

TEST_P(FgmRequestBasicTest, SamePriority_NoIntersection)
{
    nn::Result result;
    nn::fgm::Request firstRequest, finalRequest;
    nn::fgm::Setting firstRate, finalRate;

    EXPECT_TRUE((result = firstRequest.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());
    EXPECT_TRUE((result = finalRequest.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());

    EXPECT_TRUE((result = firstRequest.SetAndWait(nn::fgm::Setting_Max, nn::fgm::Setting_Max)).IsSuccess());
    EXPECT_TRUE((result = firstRequest.Get(&firstRate)).IsSuccess());

    EXPECT_TRUE((result = finalRequest.SetAndWait(nn::fgm::Setting_Min, nn::fgm::Setting_Min)).IsSuccess());
    EXPECT_TRUE((result = finalRequest.Get(&finalRate)).IsSuccess());

    // Final Request takes priority
    EXPECT_LT(finalRate, firstRate);

    EXPECT_TRUE((result = firstRequest.Finalize()).IsSuccess());
    ASSERT_TRUE((result = finalRequest.Finalize()).IsSuccess());
}

TEST_P(FgmRequestBasicTest, DifferentPriority_FullIntersection)
{
    nn::Result result;
    nn::fgm::Request firstRequest, finalRequest;
    nn::fgm::Setting firstRate, finalRate;

    EXPECT_TRUE((result = firstRequest.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());
    EXPECT_TRUE((result = finalRequest.Initialize(GetParam(), nn::fgm::Priority_Default)).IsSuccess());

    EXPECT_TRUE((result = firstRequest.SetAndWait(nn::fgm::Setting_Min, nn::fgm::Setting_Max)).IsSuccess());
    EXPECT_TRUE((result = firstRequest.Get(&firstRate)).IsSuccess());

    EXPECT_TRUE((result = finalRequest.SetAndWait(nn::fgm::Setting_Min, nn::fgm::Setting_Max)).IsSuccess());
    EXPECT_TRUE((result = finalRequest.Get(&finalRate)).IsSuccess());

    // Range stays the same, setting remains the same
    EXPECT_EQ(finalRate, firstRate);

    EXPECT_TRUE((result = firstRequest.Finalize()).IsSuccess());
    ASSERT_TRUE((result = finalRequest.Finalize()).IsSuccess());
}

TEST_P(FgmRequestBasicTest, SamePriority_FullIntersection)
{
    nn::Result result;
    nn::fgm::Request firstRequest, finalRequest;
    nn::fgm::Setting firstRate, finalRate;

    EXPECT_TRUE((result = firstRequest.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());
    EXPECT_TRUE((result = finalRequest.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());

    EXPECT_TRUE((result = firstRequest.SetAndWait(nn::fgm::Setting_Min, nn::fgm::Setting_Max)).IsSuccess());
    EXPECT_TRUE((result = firstRequest.Get(&firstRate)).IsSuccess());

    EXPECT_TRUE((result = finalRequest.SetAndWait(nn::fgm::Setting_Min, nn::fgm::Setting_Max)).IsSuccess());
    EXPECT_TRUE((result = finalRequest.Get(&finalRate)).IsSuccess());

    // Range stays the same, setting remains the same
    EXPECT_EQ(finalRate, firstRate);

    EXPECT_TRUE((result = firstRequest.Finalize()).IsSuccess());
    ASSERT_TRUE((result = finalRequest.Finalize()).IsSuccess());
}

TEST_P(FgmRequestBasicTest, DifferentPriority_LeftIntersection)
{
    nn::Result result;
    nn::fgm::Request firstRequest, finalRequest;
    nn::fgm::Setting initialRate, firstRate, finalRate, newMaxRate;

    EXPECT_TRUE((result = firstRequest.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());
    EXPECT_TRUE((result = finalRequest.Initialize(GetParam(), nn::fgm::Priority_Default)).IsSuccess());

    EXPECT_TRUE((result = firstRequest.Get(&initialRate)).IsSuccess());

    EXPECT_TRUE((result = firstRequest.SetAndWait(initialRate, nn::fgm::Setting_Max)).IsSuccess());
    EXPECT_TRUE((result = firstRequest.Get(&firstRate)).IsSuccess());

    newMaxRate = initialRate + 100 * 1000 * 1000;
    EXPECT_TRUE((result = finalRequest.SetAndWait(nn::fgm::Setting_Min, newMaxRate)).IsSuccess());
    EXPECT_TRUE((result = finalRequest.Get(&finalRate)).IsSuccess());

    // Range stays the same, setting remains the same
    EXPECT_EQ(finalRate, firstRate);

    EXPECT_TRUE((result = firstRequest.Finalize()).IsSuccess());
    ASSERT_TRUE((result = finalRequest.Finalize()).IsSuccess());

}

TEST_P(FgmRequestBasicTest, SamePriority_LeftIntersection)
{
    nn::Result result;
    nn::fgm::Request firstRequest, finalRequest;
    nn::fgm::Setting initialRate, firstRate, finalRate, newMaxRate;

    EXPECT_TRUE((result = firstRequest.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());
    EXPECT_TRUE((result = finalRequest.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());

    EXPECT_TRUE((result = firstRequest.Get(&initialRate)).IsSuccess());

    EXPECT_TRUE((result = firstRequest.SetAndWait(initialRate, nn::fgm::Setting_Max)).IsSuccess());
    EXPECT_TRUE((result = firstRequest.Get(&firstRate)).IsSuccess());

    newMaxRate = initialRate + 100 * 1000 * 1000;
    EXPECT_TRUE((result = finalRequest.SetAndWait(nn::fgm::Setting_Min, newMaxRate)).IsSuccess());
    EXPECT_TRUE((result = finalRequest.Get(&finalRate)).IsSuccess());

    // Setting remains the same
    EXPECT_EQ(finalRate, firstRate);

    EXPECT_TRUE((result = firstRequest.Finalize()).IsSuccess());
    ASSERT_TRUE((result = finalRequest.Finalize()).IsSuccess());

}

TEST_P(FgmRequestBasicTest, DifferentPriority_RightIntersection)
{
    nn::Result result;
    nn::fgm::Request firstRequest, finalRequest;
    nn::fgm::Setting initialRate, firstRate, finalRate, newMaxRate;

    EXPECT_TRUE((result = firstRequest.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());
    EXPECT_TRUE((result = finalRequest.Initialize(GetParam(), nn::fgm::Priority_Default)).IsSuccess());

    EXPECT_TRUE((result = firstRequest.Get(&initialRate)).IsSuccess());

    EXPECT_TRUE((result = firstRequest.SetAndWait(nn::fgm::Setting_Min, initialRate)).IsSuccess());
    EXPECT_TRUE((result = firstRequest.Get(&firstRate)).IsSuccess());

    newMaxRate = initialRate - 100 * 1000 * 1000;
    EXPECT_TRUE((result = finalRequest.SetAndWait(newMaxRate, nn::fgm::Setting_Max)).IsSuccess());
    EXPECT_TRUE((result = finalRequest.Get(&finalRate)).IsSuccess());

    // Range stays the same, setting remains the same
    EXPECT_EQ(finalRate, firstRate);

    EXPECT_TRUE((result = firstRequest.Finalize()).IsSuccess());
    ASSERT_TRUE((result = finalRequest.Finalize()).IsSuccess());

}

TEST_P(FgmRequestBasicTest, SamePriority_RightIntersection)
{
    nn::Result result;
    nn::fgm::Request firstRequest, finalRequest;
    nn::fgm::Setting initialRate, firstRate, finalRate, newMaxRate;

    EXPECT_TRUE((result = firstRequest.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());
    EXPECT_TRUE((result = finalRequest.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());

    EXPECT_TRUE((result = firstRequest.Get(&initialRate)).IsSuccess());

    EXPECT_TRUE((result = firstRequest.SetAndWait(nn::fgm::Setting_Min, initialRate)).IsSuccess());
    EXPECT_TRUE((result = firstRequest.Get(&firstRate)).IsSuccess());

    newMaxRate = initialRate - 100 * 1000 * 1000;
    EXPECT_TRUE((result = finalRequest.SetAndWait(newMaxRate, nn::fgm::Setting_Max)).IsSuccess());
    EXPECT_TRUE((result = finalRequest.Get(&finalRate)).IsSuccess());

    // Second requests wins
    EXPECT_GE(finalRate, firstRate);

    EXPECT_TRUE((result = firstRequest.Finalize()).IsSuccess());
    ASSERT_TRUE((result = finalRequest.Finalize()).IsSuccess());

}

// ** Wait tests ** //
TEST_P(FgmRequestBasicTest, WaitWithTimeout)
{
    nn::Result result;
    nn::Result waitResult;
    nn::fgm::Request request;
    nn::fgm::Setting currentSetting;
    nn::fgm::Setting newSetting;
    nn::fgm::Setting min = 1200 * 1000 * 1000;
    nn::fgm::Setting max = nn::fgm::Setting_Max; // Highest possible for this resource

    if ((result = request.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsFailure() ||
        (result = request.Get(&currentSetting)).IsFailure() ||
        (result = request.Set(min, max)).IsFailure())
    {
        EXPECT_TRUE(false);
        NN_LOG("failed to request new range, error %d:%d\n", result.GetModule(), result.GetDescription());
        goto out;
    }

    NN_LOG("Request range %.2f - %.2f MHz\n", min / (1000.0 * 1000.0), max / (1000.0 * 1000.0));

    while ((waitResult = request.WaitWithTimeout(nn::TimeSpan::FromSeconds(5))).IsSuccess() &&
        (result = request.Get(&newSetting)).IsSuccess())
    {
        EXPECT_TRUE(waitResult.IsSuccess());
        if (newSetting != currentSetting)
        {
            NN_LOG("state change detected, from %.2f to %.2f\n", currentSetting / (1000.0 * 1000.0),
                newSetting / (1000.0 * 1000.0));
            currentSetting = newSetting;
        }
    }
    EXPECT_FALSE(waitResult.IsSuccess());
out:
    ASSERT_TRUE((result = request.Finalize()).IsSuccess());
}

TEST_P(FgmRequestBasicTest, Wait)
{
    nn::Result result;
    nn::fgm::Request request;
    nn::fgm::Setting currentRate;

    result = request.Initialize(GetParam(), nn::fgm::Priority_Highest);
    EXPECT_TRUE(result.IsSuccess());

    result = request.Set(nn::fgm::Setting_Min, nn::fgm::Setting_Min);
    EXPECT_TRUE(result.IsSuccess());

    result = request.Wait();
    EXPECT_TRUE(result.IsSuccess());

    result = request.Get(&currentRate);
    EXPECT_TRUE(result.IsSuccess());

    NN_LOG("Set to min rate of IP: %.2f MHz\n", currentRate / (1000.0 * 1000.0));

    ASSERT_TRUE((result = request.Finalize()).IsSuccess());
}

// *** Removing request tests *** //
TEST_P(FgmRequestBasicTest, RemoveRequest)
{
    nn::Result result;
    nn::fgm::Request request, validateRequest;
    nn::fgm::Setting initialRate, finalRate;

    EXPECT_TRUE((result = request.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());
    EXPECT_TRUE((result = request.Get(&initialRate)).IsSuccess());

    EXPECT_TRUE((result = request.SetAndWait(nn::fgm::Setting_Min, nn::fgm::Setting_Min)).IsSuccess());
    EXPECT_TRUE((result = request.Finalize()).IsSuccess());

    EXPECT_TRUE((result = validateRequest.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());
    EXPECT_TRUE((result = validateRequest.Get(&finalRate)).IsSuccess());
    EXPECT_EQ(initialRate, finalRate);

}

// *** Cancel test *** //
TEST_P(FgmRequestBasicTest, Cancel)
{
    nn::Result result;
    nn::fgm::Request request;
    nn::fgm::Setting initialRate, finalRate;

    EXPECT_TRUE((result = request.Initialize(GetParam(), nn::fgm::Priority_Highest)).IsSuccess());
    EXPECT_TRUE((result = request.Get(&initialRate)).IsSuccess());
    EXPECT_TRUE((result = request.Set(nn::fgm::Setting_Min, nn::fgm::Setting_Max)).IsSuccess());
    EXPECT_TRUE((result = request.Cancel()).IsSuccess());
    EXPECT_TRUE((result = request.Get(&finalRate)).IsSuccess());
    EXPECT_EQ(initialRate, finalRate);

}

// *** Port basic test *** //
TEST_P(FgmRequestBasicTest, Permissions)
{
    nn::Result result;
    nn::fgm::Request request;

    const nn::fgm::PriorityDefinitions priorities[] = {
        nn::fgm::Priority_Highest,
        nn::fgm::Priority_Lowest
    };

    for (int i = 0; i < sizeof(priorities) / sizeof(priorities[0]); ++i)
    {
        EXPECT_TRUE((result = request.Initialize(nn::fgm::Module_Test, priorities[i])).IsSuccess());
        EXPECT_TRUE((result = request.SetAndWait(nn::fgm::Setting_Min, nn::fgm::Setting_Min)).IsSuccess());
        EXPECT_TRUE((result = request.Finalize()).IsSuccess());
    }
}
