﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/os.h>

#include <nn/fgm/fgm.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/init/init_Malloc.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>

#include <nv/nv_MemoryManagement.h>

// Contains necessary initialization code, such as setting heap size and initializing FGM

extern "C" int NvOsDrvOpen(const char *pathname, int flags);

namespace {

const size_t TotalHeapSize = 32 * 1024 * 1024;

const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;

class FgmEnvironment : public ::testing::Environment
{
private:
    nn::fgm::Request request;

public:
    virtual void SetUp() override
    {
        nn::Result r = request.Initialize(nn::fgm::Module_Test, nn::fgm::Priority_Default);
        r = request.SetAndWait(nn::fgm::Setting_Max / 2, nn::fgm::Setting_Max);
        NN_LOG("Requests initialized.\n");
    }
    virtual void TearDown() override
    {
        request.Finalize();
        NN_LOG("Requests finalized. \n");
    }
};

} // namespace

extern "C" void nninitStartup()
{
    nn::Result result;

    /* set heap size */
    result = nn::os::SetMemoryHeapSize(TotalHeapSize);
    if (!result.IsSuccess()) {
        NN_LOG("Failed SetMemoryHeapSize\n");
        return;
    }
    NN_LOG("SetMemoryHeapSize 0x%x OK\n", TotalHeapSize);

    uintptr_t address;
    result = nn::os::AllocateMemoryBlock(&address, TotalHeapSize / 2);
    NN_ASSERT(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), TotalHeapSize / 2);

    ::testing::AddGlobalTestEnvironment(new FgmEnvironment);
}

extern "C" void nnMain()
{
    nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);

    // Power on GPU
    NvOsDrvOpen("/dev/nvhost-ctrl-gpu", 0);

    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    static auto alloc = [](size_t size) -> void* {
        return std::malloc(size);
    };

    static auto dealloc = [](void* p, size_t size) {
        NN_UNUSED(size);
        std::free(p);
    };

    nn::fs::SetAllocator(alloc, dealloc);

    int result = RUN_ALL_TESTS();
    ::nnt::Exit(result);
}
