﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init/init_Malloc.h>
#include <nn/fatalsrv/fatalsrv_Server.h>

#include <nn/fatal/fatal_ApiPrivate.h>

#include "../../../../Programs/Eris/Sources/Libraries/fatalsrv/fatalsrv_PowerControl.h"

#include <nn/psm/psm.h>
#include <nn/psm/psm_SystemProcess.h>

namespace {
    class PowerControl : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
        }

        static void TearDownTestCase()
        {
        }


    private:
    };

    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_DrawThreadStack[4 * 1024];

    void TestPowerControl(nn::fatalsrv::PowerControlTask& task)
    {
        // 実際のタスクスレッドと同じスタックサイズで動かす
        nn::os::ThreadType drawTaskThread;
        nn::os::CreateThread(&drawTaskThread, [](void* arg) {
            auto task = reinterpret_cast<nn::fatalsrv::PowerControlTask*>(arg);
            task->Run();
        }, &task, g_DrawThreadStack, sizeof(g_DrawThreadStack), nn::os::DefaultThreadPriority);

        nn::os::StartThread(&drawTaskThread);
        nn::os::WaitThread(&drawTaskThread);
        nn::os::DestroyThread(&drawTaskThread);
    }
}

namespace
{
}

// INFO: このテストは目視で自動的に電源が切れることを確認するためのものです。
//       テストの実行中に Externals\HostBridge\tools\HostBridgeController\HostBridgeController.exe を動かして
//       I2C Slave の AverageVCell の値を小さくして、自動シャットダウンが働くことを確認してください。
//
// TODO: 自動テストで自動電源断の成功を確認する。
#if 0
TEST_F(PowerControl, Default)
{
    nn::os::Event errorReportWrittenEvent(nn::os::EventClearMode_AutoClear);
    errorReportWrittenEvent.Signal();

    nn::fatalsrv::PowerControlTask task(errorReportWrittenEvent);

    TestPowerControl(task);
}
#endif

class PowerControllerBase : public nn::fatalsrv::IPowerController
{
public:
    NN_IMPLICIT PowerControllerBase(nn::os::Event& shutdownEvent) NN_NOEXCEPT : m_ShutdownEvent(shutdownEvent) {}
    virtual void Shutdown() NN_NOEXCEPT { m_ShutdownEvent.Signal(); }
    virtual void Reboot() NN_NOEXCEPT {  }
    virtual nn::psm::BatteryVoltageState GetBatteryVoltageState() NN_NOEXCEPT { return nn::psm::BatteryVoltageState_ShutdownRequired; }
    virtual nn::bpc::SleepButtonState GetSleepButtonState() NN_NOEXCEPT { return nn::bpc::SleepButtonState_Released; }
    virtual int GetMaxShutdownWaitSec() const NN_NOEXCEPT { return 1; }
private:
    nn::os::Event& m_ShutdownEvent;
};


// 既に電圧が低い状態の場合は、バッテリー残量確認イベントがシグナルされずにシャットダウンすることを確認する。
TEST_F(PowerControl, AlreadyShutdownState)
{
    nn::os::Event shutdownEvent(nn::os::EventClearMode_AutoClear);
    PowerControllerBase controller(shutdownEvent);


    nn::os::Event errorReportWrittenEvent(nn::os::EventClearMode_AutoClear);
    errorReportWrittenEvent.Signal();

    nn::os::Event batterCheckedEvent(nn::os::EventClearMode_AutoClear);

    nn::fatalsrv::PowerControlTask task;
    task.Initialize(&errorReportWrittenEvent, &batterCheckedEvent, &controller);

    TestPowerControl(task);
    EXPECT_EQ(false, batterCheckedEvent.TryWait());
    EXPECT_EQ(true, shutdownEvent.TryWait());
}

// 電圧が good -> req_sleep -> req_shutdown の遷移の確認
TEST_F(PowerControl, Normal)
{
    class PowerController : public PowerControllerBase
    {
    public:
        NN_IMPLICIT PowerController(nn::os::Event& shutdownEvent) NN_NOEXCEPT : PowerControllerBase(shutdownEvent) {}
        virtual nn::psm::BatteryVoltageState GetBatteryVoltageState() NN_NOEXCEPT
        {
            NN_FUNCTION_LOCAL_STATIC(int, count, = 0);
            nn::psm::BatteryVoltageState v;
            if (count == 0)
            {
                v = nn::psm::BatteryVoltageState_Good;
            }
            else if (count == 1)
            {
                v = nn::psm::BatteryVoltageState_SleepRequired;
            }
            else
            {
                v = nn::psm::BatteryVoltageState_ShutdownRequired;
            }
            count++;
            return v;
        }
    };


    nn::os::Event shutdownEvent(nn::os::EventClearMode_AutoClear);
    PowerController controller(shutdownEvent);

    nn::os::Event errorReportWrittenEvent(nn::os::EventClearMode_AutoClear);
    errorReportWrittenEvent.Signal();

    nn::os::Event batterCheckedEvent(nn::os::EventClearMode_AutoClear);

    nn::fatalsrv::PowerControlTask task;
    task.Initialize(&errorReportWrittenEvent, &batterCheckedEvent, &controller);

    TestPowerControl(task);
    EXPECT_EQ(true, batterCheckedEvent.TryWait());
    EXPECT_EQ(true, shutdownEvent.TryWait());
}

// 電圧が good -> req_sleep の遷移の確認
TEST_F(PowerControl, ReqSleep)
{
    class PowerController : public PowerControllerBase
    {
    public:
        NN_IMPLICIT PowerController(nn::os::Event& shutdownEvent) NN_NOEXCEPT : PowerControllerBase(shutdownEvent) {}
        virtual nn::psm::BatteryVoltageState GetBatteryVoltageState() NN_NOEXCEPT
        {
            NN_FUNCTION_LOCAL_STATIC(int, count, = 0);
            nn::psm::BatteryVoltageState v;
            if (count == 0)
            {
                v = nn::psm::BatteryVoltageState_Good;
            }
            else
            {
                v = nn::psm::BatteryVoltageState_SleepRequired;
            }
            count++;
            return v;
        }
    };

    nn::os::Event shutdownEvent(nn::os::EventClearMode_AutoClear);
    PowerController controller(shutdownEvent);

    nn::os::Event errorReportWrittenEvent(nn::os::EventClearMode_AutoClear);
    errorReportWrittenEvent.Signal();

    nn::os::Event batterCheckedEvent(nn::os::EventClearMode_AutoClear);

    nn::fatalsrv::PowerControlTask task;
    task.Initialize(&errorReportWrittenEvent, &batterCheckedEvent, &controller);

    TestPowerControl(task);
    EXPECT_EQ(true, batterCheckedEvent.TryWait());
    EXPECT_EQ(true, shutdownEvent.TryWait());
}

// 電圧が good -> req_sleep -> good -> req_sleep の遷移の確認
TEST_F(PowerControl, Cancel)
{
    class PowerController : public PowerControllerBase
    {
    public:
        NN_IMPLICIT PowerController(nn::os::Event& shutdownEvent) NN_NOEXCEPT : PowerControllerBase(shutdownEvent) {}
        virtual nn::psm::BatteryVoltageState GetBatteryVoltageState() NN_NOEXCEPT
        {
            NN_FUNCTION_LOCAL_STATIC(int, count, = 0);
            nn::psm::BatteryVoltageState v;
            if (count == 0)
            {
                v = nn::psm::BatteryVoltageState_Good;
            }
            else if (count == 1)
            {
                v = nn::psm::BatteryVoltageState_SleepRequired;
            }
            else if (count == 2)
            {
                v = nn::psm::BatteryVoltageState_Good;
            }
            else
            {
                v = nn::psm::BatteryVoltageState_SleepRequired;
            }
            count++;
            return v;
        }
    };

    nn::os::Event shutdownEvent(nn::os::EventClearMode_AutoClear);
    PowerController controller(shutdownEvent);

    nn::os::Event errorReportWrittenEvent(nn::os::EventClearMode_AutoClear);
    errorReportWrittenEvent.Signal();

    nn::os::Event batterCheckedEvent(nn::os::EventClearMode_AutoClear);

    nn::fatalsrv::PowerControlTask task;
    task.Initialize(&errorReportWrittenEvent, &batterCheckedEvent, &controller);

    auto start = nn::os::GetSystemTick();
    TestPowerControl(task);
    auto end = nn::os::GetSystemTick();

    EXPECT_EQ(true, batterCheckedEvent.TryWait());
    EXPECT_EQ(true, shutdownEvent.TryWait());
    EXPECT_TRUE((end - start).ToTimeSpan().GetSeconds() >= 5);
}
