﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *               Copyright (C) 2010, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished  proprietary information of BroadOn Communications Corp.,
 *  and  are protected by Federal copyright law. They may not be disclosed
 *  to  third  parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>

#include <nn/escore/escore.h>
#include <nnt/escoreUtil/testEscore_util_istorage.h>

USING_ESCORE_UTIL_NAMESPACE
USING_ES_NAMESPACE
USING_ISTORAGE_NAMESPACE
//#include <esitypes.h>
#include <nn/ioscrypto/iosctypes.h>

#include "../Common/testEs_Est_utils.h"

/* TODO: dynamically generate the following files */
#include "../Common/testEs_Ca_cert_new.cpp"
#include "../Common/testEs_Cp_cert_new.cpp"
#include "../Common/testEs_Xs_cert_new.cpp"

#include "../Common/testEs_Tkt3.cpp"
#include "../Common/testEs_Tkt6.cpp"

#include "../Common/testEs_Data5_enc.cpp"
#include "../Common/testEs_Data5_dec.cpp"

#if !defined( ES_TEST_MEMORY_CONSTRAINT )
#include "../Common/testEs_Tkt5.cpp"
#endif

/* Certificates */
const int ES_TEST_CERT_CHAIN_LEN = 3; /* CA, XS, CP */

static const void *__certs[ES_TEST_CERT_CHAIN_LEN] = {caNewCert, xsNewCert, cpNewCert};
static u32 __nCerts = ES_TEST_CERT_CHAIN_LEN;

#ifdef ES_TEST_ENABLE_TIMING
#include <nn/os.h>

#undef ES_TEST_LOG
#define ES_TEST_LOG nn::dbg::detail::Printf

const int ES_TEST_MAX_RECORDED_TIMES = 64;

static nn::os::Tick __startTick;
static nn::os::Tick __endTick;

static u64 __recordedTimes[ES_TEST_MAX_RECORDED_TIMES];
static u32 __curRecordedTime = 0;
static u32 __recordedTimesOverflow = 0;

static void __resetRecordedTimes()
{
    __curRecordedTime = 0;
}

static void __startTiming()
{
    __startTick = nn::os::Tick::GetSystemCurrent();
}

static void __endTiming( const char *desc )
{
    u64 elapsed;

    __endTick = nn::os::Tick::GetSystemCurrent();
    elapsed = ( ( __endTick - __startTick ) * 1000000 ) / nn::os::Tick::TICKS_PER_SECOND;
    ES_TEST_LOG( "%s: %llu microseconds\n", desc, elapsed );

    if( __curRecordedTime >= ES_TEST_MAX_RECORDED_TIMES )
    {
        __curRecordedTime = 0;
        __recordedTimesOverflow = 1;
    }

    __recordedTimes[__curRecordedTime] = elapsed;
    __curRecordedTime++;
}

static ESError __checkRecordedTimes( u64 *limits, u32 nLimits )
{
    ESError rv = ES_ERR_OK;
    u32 i;

    if( nLimits > __curRecordedTime || __recordedTimesOverflow )
    {
        ES_TEST_LOG( "Too many time limits, %u:%u:%u\n", nLimits, __curRecordedTime, __recordedTimesOverflow );
        rv = ES_ERR_INVALID;
        goto end;
    }

    for( i = 0; i < nLimits; i++ )
    {
        if( __recordedTimes[i] > limits[i] )
        {
            ES_TEST_LOG( "Time limit exceeded %u:%llu:%llu\n", i, __recordedTimes[i], limits[i] );
            rv = ES_ERR_FAIL;
            goto end;
        }
    }

end:
    return rv;
}
#else
static void __resetRecordedTimes()
{
}

static void __startTiming()
{
    return;
}

static u64 __endTiming( const char *desc )
{
    (void)desc;

    return 0;
}

static ESError __checkRecordedTimes( u64 *limits, u32 nLimits )
{
    (void)limits;
    (void)nLimits;

    return ES_ERR_OK;
}
#endif

#if defined( DO_LARGE_TICKET_TEST )
static const char *makeAbsPath( const char *fileName )
{
    static std::string path;

    //#define NNT_ES_SIGLO_ROOT set by nact script or vcxporj to the root path of Siglo.
    const char* kResourcePath = "/Tests/Escore/Sources/Tests/Performance/testEs_Performance/";
    path = std::string(NNT_ES_SIGLO_ROOT) + std::string(kResourcePath) + fileName;

    return path.c_str();
}
#endif

static ESError __testSetup()
{
    return ES_ERR_OK;
}


static ESError __testTicketPerf()
{
    ESError rv = ES_ERR_OK;
    u32 size;

    ETicket ticket;
    MemoryInputStream ticketStream( tkt6, sizeof( tkt6 ) );

    u64 limits[] = {40, 50, 5000, 10, 5000, 700000, 650000};

#if defined( DO_LARGE_TICKET_TEST )
#if !defined( ES_TEST_MEMORY_CONSTRAINT )
    MemoryInputStream largeTicketStream( tkt5, sizeof( tkt5 ) );
#else
    FileStream largeTicketStream;

    if( !( largeTicketStream.TryInitialize( makeAbsPath("00000005.tik"), OPEN_MODE_READ ).IsSuccess() ) )
    {
        ES_TEST_LOG( "Failed to initialize ticket stream\n" );
        rv = ES_ERR_FAIL;
        goto end;
    }
#endif
#endif

    __resetRecordedTimes();

    __startTiming();
    rv = ticket.Set( ticketStream, __certs, __nCerts, false );
    __endTiming( "Ticket set without verification" );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to set the ticket, rv=%d\n", rv );
        goto end;
    }

    __startTiming();
    rv = ticket.Set( ticketStream, __certs, __nCerts, true );
    __endTiming( "Ticket set with verification" );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to set the ticket, rv=%d\n", rv );
        goto end;
    }

    __startTiming();
    rv = ticket.GetSize( &size );
    __endTiming( "Ticket get size" );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to get the ticket size, rv=%d\n", rv );
        goto end;
    }

#if defined( DO_LARGE_TICKET_TEST )
    __startTiming();
    rv = ticket.Set( largeTicketStream, __certs, __nCerts, true );
    __endTiming( "Ticket set with verification for a maximum size ticket" );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to set the ticket, rv=%d\n", rv );
        goto end;
    }

    __startTiming();
    rv = ticket.Set( largeTicketStream, __certs, __nCerts, false );
    __endTiming( "Ticket set without verification for a maximum size ticket" );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to set the ticket, rv=%d\n", rv );
        goto end;
    }
#endif

    rv = __checkRecordedTimes( limits, sizeof( limits ) / sizeof( limits[0] ) );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    ES_TEST_LOG( "Passed the ticket performance test\n" );

end:
    return rv;
}  // NOLINT (readability/fn_size)

#ifdef TEST_PERSONALIZATION_PERF
static ESError __testPersonalizationPerf()
{
    ESError rv = ES_ERR_OK;

    ETicketService es;
    MemoryInputStream ticketStream( tkt6, sizeof( tkt6 ) );
    MemoryOutputStream outTicketStream( __outTicketBuf, sizeof( __outTicketBuf ) );

    u64 limits[] = {350000};

    __resetRecordedTimes();

    __startTiming();
    rv = es.ImportTicket( ticketStream, __certs, __nCerts, outTicketStream );
    __endTiming( "Ticket de-personalization" );
    if( rv != ES_ERR_OK )
    {
        ES_TEST_LOG( "Failed to import personalized ticket, rv=%d\n", rv );
        goto end;
    }

    rv = __checkRecordedTimes( limits, sizeof( limits ) / sizeof( limits[0] ) );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    ES_TEST_LOG( "Passed the personalization performance test\n" );

end:
    return rv;
}
#endif


static ESError __testCleanup()
{
    return ES_ERR_OK;
}

/**
    @brief FS用アロケータ
 */
void* Allocate(size_t size)
{
    return std::malloc(size);
}

/**
    @brief FS用デアロケータ
 */
void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    std::free(p);
}


TEST( PerformanceTest, Performance )
{
    ESError rv = ES_ERR_OK;

    nn::fs::SetAllocator(Allocate, Deallocate);
    nn::fs::MountHostRoot();

    rv = __testSetup();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

    rv = __testTicketPerf();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }


#ifdef TEST_PERSONALIZATION_PERF
    rv = __testPersonalizationPerf();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }
#endif

    rv = __testCleanup();
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        goto end;
    }

end:
    if( rv == ES_ERR_OK )
    {
        ES_TEST_LOG( "***** Passed performance tests *****\n" );
    }

    return;
}
