﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *               Copyright (C) 2008, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished proprietary information of BroadOn Communications Corp.,
 *  and are protected by Federal copyright law. They may not be disclosed
 *  to third parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */

/* test of functions enc/dec */

#include <nnt/nntest.h>
#include <nn/nn_Macro.h>

#define TESTNAME "RandAesTest"
#define NEED_NUMTICKS

#include "testEs_Testcommon.h"

#include <nn/csl/aes.h>

#define ERROR_CHECK( err )                         \
    if( err != IOSC_ERROR_OK )                     \
    {                                              \
        LOG_FAIL( "error in IOSC call: %d", err ); \
        EXIT( -1 );                                \
    }

const int MAX_BLOCKS = 64;

static unsigned long testseed = 1;

unsigned long encDecRand()
{
    testseed = 0x343fd * testseed + 0x269ec3;
    return ( (testseed)&0xffffffff );
}

namespace
{
    NN_ALIGNAS( ALIGN_SIZE ) u8 g_AesData[( 64 * 1024 )];
    NN_ALIGNAS( ALIGN_SIZE ) u8 g_AesEncData[( 64 * 1024 )];
    NN_ALIGNAS( ALIGN_SIZE ) u8 g_AesSwEncData[( 64 * 1024 )];
    NN_ALIGNAS( ALIGN_SIZE ) u8 g_AesSwDecData[( 64 * 1024 )];
    NN_ALIGNAS( ALIGN_SIZE ) u8 g_AesDecData[( 64 * 1024 )];
    NN_ALIGNAS( ALIGN_SIZE ) u8 g_AesDupData[( 64 * 1024 )];
    NN_ALIGNAS( ALIGN_SIZE ) u8 g_AesKey[16];
    NN_ALIGNAS( ALIGN_SIZE ) u8 g_AesIv[16];
}

/*
 * Compare results of IOSC_Encrypt and software AES
 */
IOSCError doEncryptTest( u8 *aesKey, u8 *aesIv, u8 *aesInput, u32 dataSize )
{
    IOSCError err;
    IOSCSecretKeyHandle ench;
    u8 aesSwIv[16];
    u8 aesDupIv[16];
    u32 blockSize;
    u32 totalSize;

    /*
     * Copy the input data and IV, so that it can be reused
     */
    memcpy( aesDupIv, aesIv, sizeof( aesDupIv ) );
    memcpy( g_AesDupData, aesInput, dataSize );

    /*
     * Paranoia!
     */
    memset( g_AesEncData, 0x27, sizeof( g_AesEncData ) );
    memset( g_AesSwEncData, 0x42, sizeof( g_AesSwEncData ) );

    /*
     * Do the software encryption for reference
     */
    memcpy( aesSwIv, aesDupIv, sizeof( aesSwIv ) );
    aes_SwEncrypt( aesKey, aesSwIv, aesInput, dataSize, g_AesSwEncData );

    IOSC_CreateObject( &ench, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
    IOSC_ImportSecretKey( ench, 0, 0, IOSC_NOSIGN_NOENC, 0, 0, aesKey );

    /*
     * Do encryption to a different output buffer
     */
    err = IOSC_Encrypt( ench, aesIv, aesInput, dataSize, g_AesEncData );
    ERROR_CHECK( err );

    if( memcmp( g_AesEncData, g_AesSwEncData, dataSize ) == 0 )
    {
        err = 0;
    }
    else
    {
        err = -1;
        LOG_FAIL( "Encrypt in one shot" );
        goto out;
    }

    /*
     * Repeat the encrypt a block at a time
     */
    memset( g_AesEncData, 0x99, sizeof( g_AesEncData ) );
    memcpy( aesIv, aesDupIv, sizeof( aesDupIv ) );

    blockSize = 16;
    for( totalSize = 0; totalSize < dataSize; totalSize += blockSize )
    {
        err = IOSC_Encrypt( ench, aesIv, aesInput + totalSize, blockSize, g_AesEncData + totalSize );
        ERROR_CHECK( err );
    }

    if( memcmp( g_AesEncData, g_AesSwEncData, dataSize ) == 0 )
    {
        err = 0;
    }
    else
    {
        err = -1;
        LOG_FAIL( "Encrypt block-at-a-time" );
        goto out;
    }

    /*
     * Do encryption again, but in-place this time
     */
    memcpy( aesIv, aesDupIv, sizeof( aesDupIv ) );
    err = IOSC_Encrypt( ench, aesIv, aesInput, dataSize, aesInput );
    ERROR_CHECK( err );

    if( memcmp( aesInput, g_AesSwEncData, dataSize ) == 0 )
    {
        err = 0;
    }
    else
    {
        err = -1;
        LOG_FAIL( "Encrypt in one shot in-place" );
        goto out;
    }

    /*
     * Repeat the encrypt a block at a time in-place
     */
    memcpy( aesIv, aesDupIv, sizeof( aesDupIv ) );
    memcpy( aesInput, g_AesDupData, dataSize );

    blockSize = 16;
    for( totalSize = 0; totalSize < dataSize; totalSize += blockSize )
    {
        err = IOSC_Encrypt( ench, aesIv, aesInput + totalSize, blockSize, aesInput + totalSize );
        ERROR_CHECK( err );
    }

    if( memcmp( aesInput, g_AesSwEncData, dataSize ) == 0 )
    {
        err = 0;
    }
    else
    {
        err = -1;
        LOG_FAIL( "Encrypt block-at-a-time in-place" );
        goto out;
    }

out:
    IOSC_DeleteObject( ench );
    return err;
}


IOSCError doDecryptTest( u8 *aesKey, u8 *aesIv, u8 *aesInput, u32 dataSize )
{
    IOSCError err;
    IOSCSecretKeyHandle ench;
    u8 aesSwIv[16];
    u8 aesDupIv[16];
    u32 blockSize;
    u32 totalSize;

    /*
     * Copy the input data and IV, so that it can be reused
     */
    memcpy( aesDupIv, aesIv, sizeof( aesDupIv ) );
    memcpy( g_AesDupData, aesInput, dataSize );

    /*
     * Paranoia!
     */
    memset( g_AesDecData, 0xa5, sizeof( g_AesDecData ) );
    memset( g_AesSwDecData, 0x5a, sizeof( g_AesSwDecData ) );

    /*
     * Do the software decryption for reference
     */
    memcpy( aesSwIv, aesIv, sizeof( aesSwIv ) );
    aes_SwDecrypt( aesKey, aesSwIv, aesInput, dataSize, g_AesSwDecData );

    IOSC_CreateObject( &ench, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
    IOSC_ImportSecretKey( ench, 0, 0, IOSC_NOSIGN_NOENC, 0, 0, aesKey );

    err = IOSC_Decrypt( ench, aesIv, aesInput, dataSize, g_AesDecData );
    ERROR_CHECK( err );

    if( memcmp( g_AesDecData, g_AesSwDecData, dataSize ) == 0 )
    {
        err = 0;
    }
    else
    {
        err = -1;
        LOG_FAIL( "Decrypt in one shot" );
        goto out;
    }

    /*
     * Repeat the decrypt a block at a time
     */
    memset( g_AesDecData, 0x77, sizeof( g_AesDecData ) );
    memcpy( aesIv, aesDupIv, sizeof( aesDupIv ) );

    blockSize = 16;
    for( totalSize = 0; totalSize < dataSize; totalSize += blockSize )
    {
        err = IOSC_Decrypt( ench, aesIv, aesInput + totalSize, blockSize, g_AesDecData + totalSize );
        ERROR_CHECK( err );
    }

    if( memcmp( g_AesDecData, g_AesSwDecData, dataSize ) == 0 )
    {
        err = 0;
    }
    else
    {
        err = -1;
        LOG_FAIL( "Decrypt block-at-a-time" );
        goto out;
    }

    /*
     * Repeat the decryption in-place
     */
    memcpy( aesIv, aesDupIv, sizeof( aesDupIv ) );
    err = IOSC_Decrypt( ench, aesIv, aesInput, dataSize, aesInput );
    ERROR_CHECK( err );

    if( memcmp( aesInput, g_AesSwDecData, dataSize ) == 0 )
    {
        err = 0;
    }
    else
    {
        err = -1;
        LOG_FAIL( "Decrypt in one shot in-place" );
        goto out;
    }

    /*
     * Now do it a block at a time in-place
     */
    memcpy( aesIv, aesDupIv, sizeof( aesDupIv ) );
    memcpy( aesInput, g_AesDupData, dataSize );
    blockSize = 16;
    for( totalSize = 0; totalSize < dataSize; totalSize += blockSize )
    {
        err = IOSC_Decrypt( ench, aesIv, aesInput + totalSize, blockSize, aesInput + totalSize );
        ERROR_CHECK( err );
    }

    if( memcmp( aesInput, g_AesSwDecData, dataSize ) == 0 )
    {
        err = 0;
    }
    else
    {
        err = -1;
        LOG_FAIL( "Decrypt block-at-a-time in-place" );
        goto out;
    }

out:
    IOSC_DeleteObject( ench );
    return err;
}


int doEncDecTests()
{
    IOSCError err;
    int i, iter, numblocks;
    int size;

    /*
     * Test of IOSC_Encrypt/IOSC_Decrypt random
     */
    for( iter = 0; iter < 30; iter++ )
    {
        numblocks = ( ( encDecRand() ) & ( 0xf ) ) + 1;
        for( i = 0; i < sizeof( IOSCAesKey ); i++ )
        {
            g_AesKey[i] = ( iter + 3 ) & 0xff;
            g_AesIv[i] = ( iter + 4 ) & 0xff;
        }
        for( i = 0; i < ( 16 * MAX_BLOCKS ); i++ )
        {
            g_AesData[i] = ( i + iter ) & 0xff;
        }

        err = doEncryptTest( g_AesKey, g_AesIv, g_AesData, numblocks * 16 );

        if( err == 0 )
        {
            LOG_PASS( "IOSC ENC TEST Iteration %d: %d Blocks", iter, numblocks );
        }
        else
        {
            LOG_FAIL( "IOSC ENC TEST Iteration %d: %d Blocks", iter, numblocks );
        }

        /* reinit key and iv */
        for( i = 0; i < 16; i++ )
        {
            g_AesKey[i] = ( iter + 3 ) & 0xff;
            g_AesIv[i] = ( iter + 4 ) & 0xff;
        }

        err = doDecryptTest( g_AesKey, g_AesIv, g_AesData, numblocks * 16 );

        if( err == 0 )
        {
            LOG_PASS( "IOSC DEC TEST Iteration %d: %d Blocks", iter, numblocks );
        }
        else
        {
            LOG_FAIL( "IOSC DEC TEST Iteration %d: %d Blocks", iter, numblocks );
        }
    }

    /*
     * Size test for Encrypt/Decrypt
     */
    for( size = 16; size <= ( 64 * 1024 ); size = size * 2 )
    {
        for( i = 0; i < sizeof( IOSCAesKey ); i++ )
        {
            g_AesKey[i] = ( i + 3 ) & 0xff;
            g_AesIv[i] = ( i + 4 ) & 0xff;
        }
        for( i = 0; i < size; i++ )
        {
            g_AesData[i] = encDecRand() & 0xff;
        }

        err = doEncryptTest( g_AesKey, g_AesIv, g_AesData, size );

        if( err == 0 )
        {
            LOG_PASS( "IOSC ENC TEST PASS Size = %d", size );
        }
        else
        {
            LOG_FAIL( "IOSC ENC TEST FAIL Size = %d", size );
            EXIT( -1 );
        }

        /* reinit key and iv */
        for( i = 0; i < 16; i++ )
        {
            g_AesKey[i] = ( i + 3 ) & 0xff;
            g_AesIv[i] = ( i + 4 ) & 0xff;
        }

        err = doDecryptTest( g_AesKey, g_AesIv, g_AesData, size );

        if( err == 0 )
        {
            LOG_PASS( "IOSC DEC TEST PASS Size = %d", size );
        }
        else
        {
            LOG_FAIL( "IOSC DEC TEST FAIL Size = %d", size );
            EXIT( -1 );
        }
    }

    return 0;
}

#if defined( SEPARATE_MAIN_PROGRAM )

int main( int argc, char **argv )
{
    IOSC_Initialize();
    doEncDecTests();
    EXIT( 0 );
    return 0;
}
#endif
