﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/es.h>
#include <nn/fs.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/es/es_MaxTicketSize.h>
#include <nn/es/es_MountName.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>
#include <nnt/esUtil/testEs_CertificateData.h>
#include <nnt/esUtil/testEs_Publish.h>
#include <nnt/esUtil/testEs_Utility.h>

#define ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(r) \
    do \
    { \
        const auto& _nn_result_do_temporary = r; \
        if (!_nn_result_do_temporary.IsSuccess()) \
        { \
            EXPECT_TRUE(false); \
        } \
    } while (NN_STATIC_CONDITION(0))

namespace {
    // チケットの最大数
    const int MaxTicketCount = 128;

    // チケットを事前にインポートする数
    // テスト開始前にこの数だけチケットをインポートし、存在していることを前提とする
    // Common, Personalized それぞれこの数だけインポートする
    const int ImportTicketCount = 3;

    const int ImportPrepurchaseRecordCount = 5;

    // テスト用チケットを読み込むためのマウント名
    const char* MountName = "ticketdata";

    // テスト用コモンチケットの保存パス
    const char* CommonTicketTestDataPath = "ticketdata:/Tests/Es/Sources/Tests/ETicketService/testEs_ETicketService/TestTicketData/Common/";

    // テスト用パーソナライズドチケットの保存パス
    const char* PersonalizedTicketTestDataPath = "ticketdata:/Tests/Es/Sources/Tests/ETicketService/testEs_ETicketService/TestTicketData/Personalized/";

    // テスト用予約購入情報のレコード
    // テスト用パーソナライズドチケットを同時にインポートした場合、RightsId と TicketId と AccountId が一致するパーソナライズドチケットが存在するかを
    // コメントに示しており、それがテストの前提となっている
    const nn::es::PrepurchaseRecord TestPrepurchaseRecord[ImportPrepurchaseRecordCount] =
    {
        { {{ 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }}, 1, 1, {}, { 1483196400 } }, // [0]: 存在する
        { {{ 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }}, 2, 1, {}, { 1483196400 } }, // [1]: 存在する
        { {{ 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }}, 3, 1, {}, { 1483196400 } }, // [2]: 存在する
        { {{ 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }}, 4, 2, {}, { 1483196400 } }, // [3]: 存在しない
        { {{ 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }}, 5, 2, {}, { 1483196400 } }, // [4]: 存在しない
    };
}

class TicketTestBase : public ::testing::Test
{
public:
    void CheckCommonTicketCount(int expected)
    {
        int count;
        count = nn::es::CountCommonTicket();
        EXPECT_EQ(expected, count);

        nn::es::RightsIdIncludingKeyId list[MaxTicketCount];
        count = nn::es::ListCommonTicketRightsIds(list, MaxTicketCount);
        EXPECT_LE(count, expected);
    }

    void CheckPersonalizedTicketCount(int expected)
    {
        int count;
        count = nn::es::CountPersonalizedTicket();
        EXPECT_EQ(expected, count);

        nn::es::RightsIdIncludingKeyId list[MaxTicketCount];
        count = nn::es::ListPersonalizedTicketRightsIds(list, MaxTicketCount);
        EXPECT_LE(count, expected);
    }

    void CheckPrepurchaseRecordCount(int expected)
    {
        int count;
        count = nn::es::CountPrepurchaseRecord();
        EXPECT_EQ(expected, count);

        nn::es::RightsIdIncludingKeyId list[MaxTicketCount];
        count = nn::es::ListPrepurchaseRecordRightsIds(list, MaxTicketCount);
        EXPECT_LE(count, expected);
    }

    void DeleteAllTicketForTest()
    {
        nn::es::DeleteAllCommonTicket();
        nn::es::DeleteAllPersonalizedTicket();
        nn::es::DeleteAllPrepurchaseRecord();

        CheckCommonTicketCount(0);
        CheckPersonalizedTicketCount(0);
        CheckPrepurchaseRecordCount(0);
    }

    void ImportCommonTicketForTest(char outTicketBuffer[][nn::es::MaxTicketSize], nn::es::TicketInfo outTicketInfoList[], int importTicketCount)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost(MountName, NNT_ES_SIGLO_ROOT));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        // テスト用チケットデータをインポートする
        for (int i = 0; i < importTicketCount; i++)
        {
            char ticketPath[200];
            nn::util::SNPrintf(ticketPath, sizeof(ticketPath), "%s%04d", CommonTicketTestDataPath, i + 1);

            nn::fs::FileHandle fileHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, ticketPath, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(fileHandle);
            };

            // リストファイルサイズ取得
            int64_t ticketSize;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&ticketSize, fileHandle));
            NN_SDK_ASSERT(ticketSize <= nn::es::MaxTicketSize);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, outTicketBuffer[i], static_cast<size_t>(ticketSize)));

            char certificateBuffer[nnt::es::CertificateSize];
            std::memcpy(certificateBuffer, nnt::es::caNewCert, nnt::es::caNewCertSize);
            std::memcpy(certificateBuffer + nnt::es::caNewCertSize, nnt::es::xsNewCert, nnt::es::xsNewCertSize);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::ImportTicket(outTicketBuffer[i], static_cast<size_t>(ticketSize), certificateBuffer, nnt::es::CertificateSize));

            outTicketInfoList[i] = nnt::es::CreateTicketInfo(outTicketBuffer[i], static_cast<size_t>(ticketSize));
        }

        CheckCommonTicketCount(importTicketCount);
    }

    void ImportPersonalizedTicketForTest(char outTicketBuffer[][nn::es::MaxTicketSize], nn::es::TicketInfo outTicketInfoList[], int importTicketCount)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost(MountName, NNT_ES_SIGLO_ROOT));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        // テスト用チケットデータをインポートする
        for (int i = 0; i < importTicketCount; i++)
        {
            char ticketPath[200];
            nn::util::SNPrintf(ticketPath, sizeof(ticketPath), "%s%04d", PersonalizedTicketTestDataPath, i + 1);

            nn::fs::FileHandle fileHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, ticketPath, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(fileHandle);
            };

            // リストファイルサイズ取得
            int64_t ticketSize;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&ticketSize, fileHandle));
            NN_SDK_ASSERT(ticketSize <= nn::es::MaxTicketSize);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, outTicketBuffer[i], static_cast<size_t>(ticketSize)));

            char certificateBuffer[nnt::es::CertificateSize];
            std::memcpy(certificateBuffer, nnt::es::caNewCert, nnt::es::caNewCertSize);
            std::memcpy(certificateBuffer + nnt::es::caNewCertSize, nnt::es::xsNewCert, nnt::es::xsNewCertSize);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::ImportTicket(outTicketBuffer[i], static_cast<size_t>(ticketSize), certificateBuffer, nnt::es::CertificateSize));

            outTicketInfoList[i] = nnt::es::CreateTicketInfo(outTicketBuffer[i], static_cast<size_t>(ticketSize));
        }

        CheckPersonalizedTicketCount(importTicketCount);
    }

    void ImportPrepurchaseRecordForTest(int importPrepurchaseRecordCount)
    {
        for (int i = 0; i < importPrepurchaseRecordCount; i++)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::ImportPrepurchaseRecord(TestPrepurchaseRecord[i]));
        }

        CheckPrepurchaseRecordCount(importPrepurchaseRecordCount);
    }

protected:
    virtual void SetUp()
    {
    }

    virtual void TearDown()
    {
    }

    static void SetUpTestCase()
    {
        // サービスオブジェクトへの参照の取得
        nn::es::Initialize();
    }

    static void TearDownTestCase()
    {
        // 共有ポインタの解放
        nn::es::Finalize();
    }
};

class Ticket : public TicketTestBase
{
protected:
    char commonTicketData[ImportTicketCount][nn::es::MaxTicketSize];
    char personalizedTicketData[ImportTicketCount][nn::es::MaxTicketSize];

    nn::es::TicketInfo commonTicketInfoList[ImportTicketCount];
    nn::es::TicketInfo personalizedTicketInfoList[ImportTicketCount];

    virtual void SetUp()
    {
        DeleteAllTicketForTest();

        ImportCommonTicketForTest(commonTicketData, commonTicketInfoList, ImportTicketCount);
        ImportPersonalizedTicketForTest(personalizedTicketData, personalizedTicketInfoList, ImportTicketCount);

        CheckCommonTicketCount(ImportTicketCount);
        CheckPersonalizedTicketCount(ImportTicketCount);
        CheckPrepurchaseRecordCount(0);
    }
};

class PrepurchaseRecord : public TicketTestBase
{
protected:
    virtual void SetUp()
    {
        DeleteAllTicketForTest();

        ImportPrepurchaseRecordForTest(ImportPrepurchaseRecordCount);

        CheckCommonTicketCount(0);
        CheckPersonalizedTicketCount(0);
        CheckPrepurchaseRecordCount(ImportPrepurchaseRecordCount);
    }
};

class TicketAndPrepurchaseRecord : public TicketTestBase
{
protected:
    char commonTicketData[ImportTicketCount][nn::es::MaxTicketSize];
    char personalizedTicketData[ImportTicketCount][nn::es::MaxTicketSize];

    nn::es::TicketInfo commonTicketInfoList[ImportTicketCount];
    nn::es::TicketInfo personalizedTicketInfoList[ImportTicketCount];

    virtual void SetUp()
    {
        DeleteAllTicketForTest();

        ImportCommonTicketForTest(commonTicketData, commonTicketInfoList, ImportTicketCount);
        ImportPersonalizedTicketForTest(personalizedTicketData, personalizedTicketInfoList, ImportTicketCount);
        ImportPrepurchaseRecordForTest(ImportPrepurchaseRecordCount);

        CheckCommonTicketCount(ImportTicketCount);
        CheckPersonalizedTicketCount(ImportTicketCount);
        CheckPrepurchaseRecordCount(ImportPrepurchaseRecordCount);
    }
};

class DuplicatedTicket : public TicketTestBase
{
protected:
    virtual void SetUp()
    {
        DeleteAllTicketForTest();

        CheckCommonTicketCount(0);
        CheckPersonalizedTicketCount(0);
        CheckPrepurchaseRecordCount(0);
    }

    virtual void TearDown()
    {
    }
};

TEST_F(Ticket, DeleteTicket)
{
    // チケットを削除する数
    const int DeleteTicketCount = 1;

    // 削除するチケットの番号
    const int DeleteTicketNumber = 1;

    nn::es::RightsIdIncludingKeyId deleteRightsIdList[DeleteTicketCount];
    std::memcpy(&deleteRightsIdList[0], &commonTicketInfoList[DeleteTicketNumber].rightsId, sizeof(nn::es::RightsIdIncludingKeyId));
    nn::es::DeleteTicket(deleteRightsIdList, DeleteTicketCount);

    // チケットの枚数が正しいかチェック
    CheckCommonTicketCount(ImportTicketCount - DeleteTicketCount);

    nn::es::RightsIdIncludingKeyId rightsIdList[MaxTicketCount];
    nn::es::ListCommonTicketRightsIds(rightsIdList, MaxTicketCount);

    // チケットが正しく削除されているかチェック
    for (int i = 0; i < ImportTicketCount; i++)
    {
        bool isExist = false;

        for (int j = 0; j < ImportTicketCount; j++)
        {
            if (commonTicketInfoList[i].rightsId == rightsIdList[j])
            {
                isExist = true;
                break;
            }
        }

        if (i == DeleteTicketNumber)
        {
            EXPECT_FALSE(isExist);
        }
        else
        {
            EXPECT_TRUE(isExist);
        }
    }
}

TEST_F(Ticket, DeletePersonalizedTicket)
{
    nn::es::AccountId deleteAccountId = personalizedTicketInfoList[0].accountId;
    nn::es::DeletePersonalizedTicket(deleteAccountId);

    int remainingPersonalizedTicketCount = 0;
    for (int i = 0; i < ImportTicketCount; i++)
    {
        if (personalizedTicketInfoList[i].accountId != deleteAccountId)
        {
            remainingPersonalizedTicketCount++;
        }
    }

    // パーソナライズドチケットが正しく削除されているかチェック
    CheckPersonalizedTicketCount(remainingPersonalizedTicketCount);

    // コモンチケットに影響がないことをチェック
    CheckCommonTicketCount(ImportTicketCount);
}

TEST_F(Ticket, DeleteAllCommonTicket)
{
    nn::es::DeleteAllCommonTicket();

    // コモンチケットが全て削除されているかチェック
    CheckCommonTicketCount(0);

    // パーソナライズドチケットに影響がないことをチェック
    CheckPersonalizedTicketCount(ImportTicketCount);
}

TEST_F(Ticket, DeleteAllPersonalizedTicket)
{
    nn::es::DeleteAllPersonalizedTicket();

    // パーソナライズドチケットが全て削除されているかチェック
    CheckPersonalizedTicketCount(0);

    // コモンチケットに影響がないことをチェック
    CheckCommonTicketCount(ImportTicketCount);
}

TEST_F(Ticket, DeleteAllPersonalizedTicketExcludingList)
{
    // 削除から除外するチケットの数
    const int TicketExcludingCount = 1;

    // 削除から除外するチケットの番号
    const int TicketExcludingNumber = 0;

    nn::es::TicketId ticketIdList[TicketExcludingCount];
    ticketIdList[0] = personalizedTicketInfoList[TicketExcludingNumber].ticketId;

    nn::es::DeleteAllPersonalizedTicket(ticketIdList, TicketExcludingCount);

    // チケットの数が正しいかチェック
    CheckPersonalizedTicketCount(TicketExcludingCount);

    // 残ったチケットが正しいかチェック
    nn::es::RightsIdIncludingKeyId rightsIdList[MaxTicketCount];
    nn::es::ListPersonalizedTicketRightsIds(rightsIdList, MaxTicketCount);
    EXPECT_TRUE(personalizedTicketInfoList[TicketExcludingNumber].rightsId == rightsIdList[0]);
}

TEST_F(Ticket, ListCommonTicketRightsIds)
{
    nn::es::RightsIdIncludingKeyId rightsIdList[MaxTicketCount];
    int count = nn::es::ListCommonTicketRightsIds(rightsIdList, MaxTicketCount);

    // チケットの数が正しいかチェック
    EXPECT_EQ(ImportTicketCount, count);

    // チケットのリストが正しいかチェック
    for (int i = 0; i < ImportTicketCount; i++)
    {
        EXPECT_TRUE(rightsIdList[i] == commonTicketInfoList[i].rightsId);
    }
}

TEST_F(Ticket, ListPersonalizedTicketRightsIds)
{
    nn::es::RightsIdIncludingKeyId rightsIdList[MaxTicketCount];
    int count = nn::es::ListPersonalizedTicketRightsIds(rightsIdList, MaxTicketCount);

    // チケットの数が正しいかチェック
    EXPECT_EQ(ImportTicketCount, count);

    // チケットのリストが正しいかチェック
    for (int i = 0; i < ImportTicketCount; i++)
    {
        EXPECT_TRUE(rightsIdList[i] == personalizedTicketInfoList[i].rightsId);
    }
}

TEST_F(Ticket, ListMissingPersonalizedTicket)
{
    // インポートしていないチケットの TicketId
    const nn::es::TicketId NotImportTicketId = 0xAAAAAAAAAAAAAAAA;

    // インポートした ImportTicketNumber 枚のチケットの TicketId と インポートしていない1枚のチケットの TicketId 格納する配列
    const int TotalTicketIdCount = ImportTicketCount + 1;
    nn::es::TicketId outTicketIdList[TotalTicketIdCount];
    nn::es::TicketId ticketIdList[TotalTicketIdCount];

    // インポートした チケットの TicketId を格納する
    for (int i = 0; i < ImportTicketCount; i++) {
        ticketIdList[i] = personalizedTicketInfoList[i].ticketId;
    }

    // インポートしていないチケットの TicketId を格納する
    ticketIdList[TotalTicketIdCount - 1] = NotImportTicketId;

    // 出力用の配列を初期化
    for (int i = 0; i < TotalTicketIdCount; i++) {
        outTicketIdList[i] = 0;
    }

    int count = nn::es::ListMissingPersonalizedTicket(outTicketIdList, TotalTicketIdCount, ticketIdList, TotalTicketIdCount);

    // 持っていないチケットの数が正しいかチェック
    EXPECT_EQ(1, count);

    // 持っていないチケットの TicketId が正しいかチェック
    EXPECT_EQ(NotImportTicketId, outTicketIdList[0]);
}

TEST_F(Ticket, GetCommonTicketSize)
{
    for (int i = 0; i < ImportTicketCount; i++)
    {
        size_t outSize;
        ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::GetCommonTicketSize(&outSize, commonTicketInfoList[i].rightsId));

        // チケットのサイズが正しいか比較
        EXPECT_EQ(commonTicketInfoList[i].ticketSize, outSize);
    }
}

TEST_F(Ticket, GetCommonTicketData)
{
    for (int i = 0; i < ImportTicketCount; i++)
    {
        char outTicketBuffer[nn::es::MaxTicketSize];
        size_t outSize;
        ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::GetCommonTicketData(&outSize, outTicketBuffer, sizeof(outTicketBuffer), commonTicketInfoList[i].rightsId));

        // チケットのサイズが正しいか比較
        EXPECT_EQ(commonTicketInfoList[i].ticketSize, outSize);

        // チケットのバイナリが正しいか比較
        EXPECT_TRUE(memcmp(commonTicketData[i], outTicketBuffer, outSize) == 0);
    }
}

TEST_F(Ticket, GetEncryptedTicketSize)
{
    for (int i = 0; i < ImportTicketCount; i++)
    {
        size_t outSize;

        // 暗号化されたチケットが取得できることだけを確認する
        ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::GetEncryptedTicketSize(&outSize, commonTicketInfoList[i].rightsId));
    }

    for (int i = 0; i < ImportTicketCount; i++)
    {
        size_t outSize;

        // 暗号化されたチケットが取得できることだけを確認する
        ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::GetEncryptedTicketSize(&outSize, personalizedTicketInfoList[i].rightsId));
    }
}

TEST_F(Ticket, GetEncryptedTicketData)
{
    for (int i = 0; i < ImportTicketCount; i++)
    {
        char outEncryptedTicketBuffer[nn::es::MaxEncryptedTicketSize];
        char outKeyBuffer[256];
        size_t outSize;
        nn::es::TicketId ticketId;

        // 暗号化されたチケットが取得できることだけを確認する
        ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::GetEncryptedTicketData(&ticketId, &outSize, outEncryptedTicketBuffer, sizeof(outEncryptedTicketBuffer), outKeyBuffer, sizeof(outKeyBuffer), commonTicketInfoList[i].rightsId));
    }

    for (int i = 0; i < ImportTicketCount; i++)
    {
        char outEncryptedTicketBuffer[nn::es::MaxEncryptedTicketSize];
        char outKeyBuffer[256];
        size_t outSize;
        nn::es::TicketId ticketId;

        // 暗号化されたチケットが取得できることだけを確認する
        ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::GetEncryptedTicketData(&ticketId, &outSize, outEncryptedTicketBuffer, sizeof(outEncryptedTicketBuffer), outKeyBuffer, sizeof(outKeyBuffer), personalizedTicketInfoList[i].rightsId));
    }
}

TEST_F(Ticket, OwnTicket)
{
    bool outOwnTicketList[MaxTicketCount];
    nn::es::RightsIdIncludingKeyId rightsIdList[MaxTicketCount];

    // インポートしていないチケットの RightsId
    const nn::es::RightsIdIncludingKeyId NotImportRightsId = {
        { 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA }
    };

    for (int i = 0; i < MaxTicketCount; i++)
    {
        if (i < ImportTicketCount)
        {
            rightsIdList[i] = commonTicketInfoList[i].rightsId;
        }
        else
        {
            rightsIdList[i] = NotImportRightsId;
        }
    }

    nn::es::OwnTicket(outOwnTicketList, rightsIdList, MaxTicketCount);

    // 所有しているチケットが正しく取得できているかチェック
    for (int i = 0; i < MaxTicketCount; i++)
    {
        if (i < ImportTicketCount)
        {
            EXPECT_TRUE(outOwnTicketList[i]);
        }
        else
        {
            EXPECT_FALSE(outOwnTicketList[i]);
        }
    }
}

TEST_F(Ticket, GetTicketInfo)
{
    nn::es::TicketInfo outTicketInfoList[MaxTicketCount];
    nn::es::RightsIdIncludingKeyId rightsIdList[MaxTicketCount];

    // インポートしていないチケットの RightsId
    const nn::es::RightsIdIncludingKeyId NotImportRightsId = {
        { 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA }
    };

    for (int i = 0; i < MaxTicketCount; i++)
    {
        if (i < ImportTicketCount)
        {
            rightsIdList[i] = commonTicketInfoList[i].rightsId;
        }
        else
        {
            rightsIdList[i] = NotImportRightsId;
        }
    }

    int count = nn::es::GetTicketInfo(outTicketInfoList, MaxTicketCount, rightsIdList, MaxTicketCount);

    // チケットの情報を取得した数が正しいかチェック
    EXPECT_EQ(2 * ImportTicketCount, count);

    // 情報を取得したチケットが本当に存在しているかチェック
    for (int i = 0; i < count; i++)
    {
        bool isExist = false;

        for (size_t j = 0; j < ImportTicketCount; j++)
        {
            if (nnt::es::operator==(outTicketInfoList[i], commonTicketInfoList[j]) || nnt::es::operator==(outTicketInfoList[i], personalizedTicketInfoList[j]))
            {
                isExist = true;
                break;
            }
        }

        EXPECT_TRUE(isExist);
    }
}

TEST_F(Ticket, ListLightTicketInfo)
{
    nn::es::LightTicketInfo outTicketInfoList[MaxTicketCount];

    // インポートしていないチケットの RightsId
    const nn::es::RightsIdIncludingKeyId NotImportRightsId = {
        { 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA }
    };

    int count = nn::es::ListLightTicketInfo(outTicketInfoList, MaxTicketCount, NotImportRightsId);

    // チケットの情報を取得した数が正しいかチェック
    EXPECT_EQ(0, count);

    for (int i = 0; i < ImportTicketCount; i++)
    {
        nn::es::RightsIdIncludingKeyId rightsId = commonTicketInfoList[i].rightsId;

        count = nn::es::ListLightTicketInfo(outTicketInfoList, MaxTicketCount, rightsId);

        // チケットの情報を取得した数が正しいかチェック
        EXPECT_EQ(2, count);
    }
}

TEST_F(Ticket, Sign)
{
    nn::es::Sign sign;
    nn::es::Certificate certificate;

    // テスト用チャレンジデータ
    int data = 8000;

    nn::es::SignData(&sign, &certificate, &data, sizeof(data));
}

TEST_F(Ticket, GetCommonTicketAndCertificateSize)
{
    for (int i = 0; i < ImportTicketCount; i++)
    {
        size_t outTicketSize;
        size_t outCertificateSize;
        ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::GetCommonTicketAndCertificateSize(&outTicketSize, &outCertificateSize, commonTicketInfoList[i].rightsId));

        // チケットのサイズが正しいか比較
        EXPECT_EQ(commonTicketInfoList[i].ticketSize, outTicketSize);

        // 証明書のサイズが正しいか比較
        EXPECT_EQ(nnt::es::CertificateSize, outCertificateSize);
    }
}

TEST_F(Ticket, GetCommonTicketAndCertificateData)
{
    for (int i = 0; i < ImportTicketCount; i++)
    {
        char outTicketBuffer[nn::es::MaxTicketSize];
        char outCertificateBuffer[nnt::es::CertificateSize];
        size_t outTicketSize;
        size_t outCertificateSize;
        ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::GetCommonTicketAndCertificateData(&outTicketSize, &outCertificateSize, outTicketBuffer, outCertificateBuffer, sizeof(outTicketBuffer), sizeof(outCertificateBuffer), commonTicketInfoList[i].rightsId));

        // チケットのサイズが正しいか比較
        EXPECT_EQ(commonTicketInfoList[i].ticketSize, outTicketSize);

        // チケットのバイナリが正しいか比較
        EXPECT_TRUE(memcmp(commonTicketData[i], outTicketBuffer, outTicketSize) == 0);

        // 証明書のサイズが正しいか比較
        EXPECT_EQ(nnt::es::CertificateSize, outCertificateSize);

        // 証明書のバイナリが正しいか比較
        EXPECT_TRUE(memcmp(nnt::es::Certificate, outCertificateBuffer, outCertificateSize) == 0);
    }
}

TEST_F(Ticket, CountDesignatedTicket)
{
    // インポートしていないチケットの RightsId
    const nn::es::RightsIdIncludingKeyId NotImportRightsId = {
        { 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA }
    };

    int count = nn::es::CountTicket(NotImportRightsId.GetRightsId());

    // チケットの情報を取得した数が正しいかチェック
    EXPECT_EQ(0, count);

    for (int i = 0; i < ImportTicketCount; i++)
    {
        nn::es::RightsIdIncludingKeyId rightsId = commonTicketInfoList[i].rightsId;

        count = nn::es::CountTicket(rightsId.GetRightsId());

        // チケットの情報を取得した数が正しいかチェック
        EXPECT_EQ(2, count);
    }
}

TEST_F(Ticket, ListDesignatedTicketRightsIds)
{
    nn::es::RightsIdIncludingKeyId rightsIdList[MaxTicketCount];

    // インポートしていないチケットの RightsId
    const nn::es::RightsIdIncludingKeyId NotImportRightsId = {
        { 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA }
    };

    int count = nn::es::ListTicketRightsIds(rightsIdList, MaxTicketCount, NotImportRightsId.GetRightsId());

    // チケットの情報を取得した数が正しいかチェック
    EXPECT_EQ(0, count);

    for (int i = 0; i < ImportTicketCount; i++)
    {
        nn::es::RightsIdIncludingKeyId rightsId = commonTicketInfoList[i].rightsId;

        count = nn::es::ListTicketRightsIds(rightsIdList, MaxTicketCount, rightsId.GetRightsId());

        // チケットの情報を取得した数が正しいかチェック
        EXPECT_EQ(1, count);
    }
}

TEST_F(PrepurchaseRecord, DeletePrepurchaseRecord)
{
    nn::es::DeletePrepurchaseRecord(TestPrepurchaseRecord[0]);

    // 予約購入記録が正しく削除されているかチェック
    CheckPrepurchaseRecordCount(ImportPrepurchaseRecordCount - 1);
}

TEST_F(PrepurchaseRecord, DeleteAllPrepurchaseRecord)
{
    nn::es::DeleteAllPrepurchaseRecord();

    // 予約購入記録が正しく削除されているかチェック
    CheckPrepurchaseRecordCount(0);
}

TEST_F(PrepurchaseRecord, ListPrepurchaseRecordRightsIds)
{
    nn::es::RightsIdIncludingKeyId rightsIdList[MaxTicketCount];
    int count = nn::es::ListPrepurchaseRecordRightsIds(rightsIdList, MaxTicketCount);

    EXPECT_EQ(ImportPrepurchaseRecordCount, count);

    // 予約購入記録のリストが正しいかチェック
    for (int i = 0; i < ImportPrepurchaseRecordCount; i++)
    {
        EXPECT_TRUE(rightsIdList[i] == TestPrepurchaseRecord[i].rightsId);
    }
}

TEST_F(PrepurchaseRecord, ListPrepurchaseRecordInfo)
{
    nn::es::PrepurchaseRecord prepurchaseRecordList[MaxTicketCount];

    // 予約購入記録が正しいかチェック
    for (int i = 0; i < ImportPrepurchaseRecordCount; i++)
    {
        int count = nn::es::ListPrepurchaseRecordInfo(prepurchaseRecordList, MaxTicketCount, TestPrepurchaseRecord[i].rightsId);

        EXPECT_EQ(1, count);

        EXPECT_TRUE(nnt::es::operator==(TestPrepurchaseRecord[i], prepurchaseRecordList[0]));
    }
}

TEST_F(PrepurchaseRecord, CountDesignatedPrepurchaseRecord)
{
    // インポートしていないチケットの RightsId
    const nn::es::RightsIdIncludingKeyId NotImportRightsId = {
        { 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA }
    };

    int count = nn::es::CountPrepurchaseRecord(NotImportRightsId.GetRightsId());

    // チケットの情報を取得した数が正しいかチェック
    EXPECT_EQ(0, count);

    for (int i = 0; i < ImportTicketCount; i++)
    {
        nn::es::RightsIdIncludingKeyId rightsId = TestPrepurchaseRecord[i].rightsId;

        count = nn::es::CountPrepurchaseRecord(rightsId.GetRightsId());

        // チケットの情報を取得した数が正しいかチェック
        EXPECT_EQ(1, count);
    }
}

TEST_F(PrepurchaseRecord, ListDesignatedPrepurchaseRecordRightsIds)
{
    nn::es::RightsIdIncludingKeyId rightsIdList[MaxTicketCount];

    // インポートしていないチケットの RightsId
    const nn::es::RightsIdIncludingKeyId NotImportRightsId = {
        { 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA }
    };

    int count = nn::es::ListPrepurchaseRecordRightsIds(rightsIdList, MaxTicketCount, NotImportRightsId.GetRightsId());

    // チケットの情報を取得した数が正しいかチェック
    EXPECT_EQ(0, count);

    for (int i = 0; i < ImportTicketCount; i++)
    {
        nn::es::RightsIdIncludingKeyId rightsId = TestPrepurchaseRecord[i].rightsId;

        count = nn::es::ListPrepurchaseRecordRightsIds(rightsIdList, MaxTicketCount, rightsId.GetRightsId());

        // チケットの情報を取得した数が正しいかチェック
        EXPECT_EQ(1, count);
    }
}

TEST_F(TicketAndPrepurchaseRecord, DeleteTicketAndPrepurchaseRecord)
{
    // コモンチケットを全て削除
    nn::es::DeleteAllCommonTicket();

    // コモンチケットが全て削除されているかチェック
    CheckCommonTicketCount(0);

    // 予約購入記録に影響がないことをチェック
    CheckPrepurchaseRecordCount(ImportPrepurchaseRecordCount);

    // パーソナライズドチケットを全て削除
    nn::es::DeleteAllPersonalizedTicket();

    // パーソナライズドチケットが全て削除されているかチェック
    CheckPersonalizedTicketCount(0);

    // 予約購入記録に影響がないことをチェック
    CheckPrepurchaseRecordCount(ImportPrepurchaseRecordCount);

    // 予約購入記録を全て削除
    nn::es::DeleteAllPrepurchaseRecord();

    // 予約購入記録が全て削除されているかチェック
    CheckPrepurchaseRecordCount(0);
}

TEST_F(TicketAndPrepurchaseRecord, ListTicketAndPrepurchaseRecord)
{
    // コモンチケットが正しくリストできるかを確認
    {
        nn::es::RightsIdIncludingKeyId rightsIdList[MaxTicketCount];
        int count = nn::es::ListCommonTicketRightsIds(rightsIdList, MaxTicketCount);

        // チケットの数が正しいかチェック
        EXPECT_EQ(ImportTicketCount, count);

        // チケットのリストが正しいかチェック
        for (int i = 0; i < ImportTicketCount; i++)
        {
            EXPECT_TRUE(rightsIdList[i] == commonTicketInfoList[i].rightsId);
        }
    }

    // パーソナライズドチケットが正しくリストできるかを確認
    {
        nn::es::RightsIdIncludingKeyId rightsIdList[MaxTicketCount];
        int count = nn::es::ListPersonalizedTicketRightsIds(rightsIdList, MaxTicketCount);

        // チケットの数が正しいかチェック
        EXPECT_EQ(ImportTicketCount, count);

        // チケットのリストが正しいかチェック
        for (int i = 0; i < ImportTicketCount; i++)
        {
            EXPECT_TRUE(rightsIdList[i] == personalizedTicketInfoList[i].rightsId);
        }
    }

    // 予約購入記録が正しくリストできるかを確認
    {
        nn::es::RightsIdIncludingKeyId rightsIdList[MaxTicketCount];
        int count = nn::es::ListPrepurchaseRecordRightsIds(rightsIdList, MaxTicketCount);

        EXPECT_EQ(ImportPrepurchaseRecordCount, count);

        // 予約購入記録のリストが正しいかチェック
        for (int i = 0; i < ImportPrepurchaseRecordCount; i++)
        {
            EXPECT_TRUE(rightsIdList[i] == TestPrepurchaseRecord[i].rightsId);
        }
    }
}

// DeletePersonalizedTicket(nn::es::AccountId accountId) は
// AccountId が一致しているパーソナライズドチケットと予約購入記録を削除するので、
// それらが意図通りに削除されているかを確認する
TEST_F(TicketAndPrepurchaseRecord, DeletePersonalizedTicket)
{
    nn::es::AccountId deleteAccountId = personalizedTicketInfoList[0].accountId;
    nn::es::DeletePersonalizedTicket(deleteAccountId);

    int remainingPersonalizedTicketCount = 0;
    for (int i = 0; i < ImportTicketCount; i++)
    {
        if (personalizedTicketInfoList[i].accountId != deleteAccountId)
        {
            remainingPersonalizedTicketCount++;
        }
    }

    // パーソナライズドチケットが正しく削除されているかチェック
    CheckPersonalizedTicketCount(remainingPersonalizedTicketCount);

    int remainingPrepurchaseRecordCount = 0;
    for (int i = 0; i < ImportPrepurchaseRecordCount; i++)
    {
        if (TestPrepurchaseRecord[i].accountId != deleteAccountId)
        {
            remainingPrepurchaseRecordCount++;
        }
    }

    // 予約購入記録が正しく削除されているかチェック
    CheckPrepurchaseRecordCount(remainingPrepurchaseRecordCount);

    // コモンチケットに影響がないことをチェック
    CheckCommonTicketCount(ImportTicketCount);
}

// DeleteAllPersonalizedTicket(const nn::es::TicketId ticketIdExclusionList[], int count) は
// 指定した TicketId 以外のパーソナライズドチケットと予約購入記録を全て削除するので、
// それらが意図通りに削除されているかを確認する
TEST_F(TicketAndPrepurchaseRecord, DeleteAllPersonalizedTicket)
{
    // 削除から除外するチケットの数
    const int TicketExcludingCount = 1;

    // 削除から除外するチケットの番号
    const int TicketExcludingNumber = 0;

    nn::es::TicketId ticketIdList[TicketExcludingCount];
    ticketIdList[0] = personalizedTicketInfoList[TicketExcludingNumber].ticketId;

    nn::es::DeleteAllPersonalizedTicket(ticketIdList, TicketExcludingCount);

    // チケットの数が正しいかチェック
    CheckPersonalizedTicketCount(TicketExcludingCount);

    // 残ったチケットが正しいかチェック
    {
        nn::es::RightsIdIncludingKeyId rightsIdList[MaxTicketCount];
        nn::es::ListPersonalizedTicketRightsIds(rightsIdList, MaxTicketCount);
        EXPECT_TRUE(personalizedTicketInfoList[TicketExcludingNumber].rightsId == rightsIdList[0]);
    }

    // 予約購入記録の数が正しいかチェック
    CheckPrepurchaseRecordCount(TicketExcludingCount);

    // 残った予約購入記録が正しいかチェック
    {
        nn::es::RightsIdIncludingKeyId rightsIdList[MaxTicketCount];
        nn::es::ListPrepurchaseRecordRightsIds(rightsIdList, MaxTicketCount);
        EXPECT_TRUE(TestPrepurchaseRecord[TicketExcludingNumber].rightsId == rightsIdList[0]);
    }
}

// ListMissingPersonalizedTicket(nn::es::TicketId outTicketIdList[], int OutTicketIdListCount, const nn::es::TicketId TicketIdList[], int ticketIdListCount) は
// 保有していない TicketId のパーソナライズドチケットを全てリストアップする(予約購入情報は持っていても保有しているとはみなされない)ので、
// それらが意図通りに動作しているかを確認する
TEST_F(TicketAndPrepurchaseRecord, ListMissingPersonalizedTicket)
{
    // インポートしていないチケットの TicketId
    const nn::es::TicketId NotImportTicketId = 0xAAAAAAAAAAAAAAAA;

    // 予約購入情報がインポートされており、パーソナライズドチケットはインポートされていない TicketId
    const nn::es::TicketId PrepurchaseRecordTicketId = TestPrepurchaseRecord[3].ticketId;

    // インポートした ImportTicketNumber 枚のチケットの TicketId と インポートしていない2枚のチケットの TicketId 格納する配列
    const int TotalTicketIdCount = ImportTicketCount + 2;
    nn::es::TicketId outTicketIdList[TotalTicketIdCount];
    nn::es::TicketId ticketIdList[TotalTicketIdCount];

    // インポートした チケットの TicketId を格納する
    for (int i = 0; i < ImportTicketCount; i++) {
        ticketIdList[i] = personalizedTicketInfoList[i].ticketId;
    }

    // インポートしていないチケットの TicketId を格納する
    ticketIdList[TotalTicketIdCount - 2] = NotImportTicketId;

    // 予約購入情報がインポートされており、パーソナライズドチケットはインポートされていない TicketId を格納する
    ticketIdList[TotalTicketIdCount - 1] = PrepurchaseRecordTicketId;

    // 出力用の配列を初期化
    for (int i = 0; i < TotalTicketIdCount; i++) {
        outTicketIdList[i] = 0;
    }

    int count = nn::es::ListMissingPersonalizedTicket(outTicketIdList, TotalTicketIdCount, ticketIdList, TotalTicketIdCount);

    // 持っていないチケットの数が正しいかチェック
    EXPECT_EQ(2, count);

    // 持っていないチケットの TicketId が正しいかチェック
    EXPECT_EQ(NotImportTicketId, outTicketIdList[0]);
    EXPECT_EQ(PrepurchaseRecordTicketId, outTicketIdList[1]);
}

TEST_F(DuplicatedTicket, ImportDuplicatedTicket)
{
    // ベースとなるチケットの情報
    nn::es::TicketInfo baseTicketInfo = {};
    baseTicketInfo.rightsId = { { 0x01, 0x00, 0x4b, 0x90, 0x00, 0x49, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } };
    baseTicketInfo.deviceId = 0x6000000000000000;
    baseTicketInfo.ticketId = 0x0100000000000000;
    baseTicketInfo.accountId = 16000000;
    baseTicketInfo.ticketVersion = 0;
    baseTicketInfo.ticketSize = 704;

    {
        uint64_t ticketBuffer[nn::es::MaxTicketSize];
        int ticketSize = nnt::es::PublishTicket(ticketBuffer, sizeof(ticketBuffer), {}, baseTicketInfo.deviceId, baseTicketInfo.ticketId, baseTicketInfo.rightsId, baseTicketInfo.accountId);
        EXPECT_EQ(baseTicketInfo.ticketSize, ticketSize);

        // チケットをインポートする
        ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::ImportTicket(ticketBuffer, ticketSize, nnt::es::Certificate, nnt::es::CertificateSize));

        // 各種取得できる情報が正しいかをチェック
        {
            // 所有しているパーソナライズドチケットは1つであるかチェック
            CheckPersonalizedTicketCount(1);

            // 権利を持っているパーソナライズドチケットの RightsId の数は1つであるかチェック
            nn::es::RightsIdIncludingKeyId list[MaxTicketCount];
            int count = nn::es::ListPersonalizedTicketRightsIds(list, MaxTicketCount);
            EXPECT_EQ(1, count);
            EXPECT_EQ(baseTicketInfo.rightsId, list[0]);

            // RightsId でクエリすると1つのチケットの情報を取得できるかチェック
            nn::es::TicketInfo info[MaxTicketCount];
            count = nn::es::GetTicketInfo(info, MaxTicketCount, &baseTicketInfo.rightsId, 1);
            EXPECT_EQ(1, count);
            EXPECT_TRUE(nnt::es::operator==(baseTicketInfo, info[0]));

            nn::es::LightTicketInfo lightInfo[nn::account::UserCountMax + 1 + 1];
            count = nn::es::ListLightTicketInfo(lightInfo, nn::account::UserCountMax + 1 + 1, baseTicketInfo.rightsId);
            EXPECT_EQ(1, count);
        }

        // 既にインポートされているチケットを再度インポートする
        // 成功するが、チケット DB の内容には変化がないかを確認する
        ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::ImportTicket(ticketBuffer, ticketSize, nnt::es::Certificate, nnt::es::CertificateSize));

        // 各種取得できる情報が前回と等しいかチェック
        {
            CheckPersonalizedTicketCount(1);

            nn::es::RightsIdIncludingKeyId list[MaxTicketCount];
            int count = nn::es::ListPersonalizedTicketRightsIds(list, MaxTicketCount);
            EXPECT_EQ(1, count);
            EXPECT_EQ(baseTicketInfo.rightsId, list[0]);

            nn::es::TicketInfo info[MaxTicketCount];
            count = nn::es::GetTicketInfo(info, MaxTicketCount, &baseTicketInfo.rightsId, 1);
            EXPECT_EQ(1, count);
            EXPECT_TRUE(nnt::es::operator==(baseTicketInfo, info[0]));

            nn::es::LightTicketInfo lightInfo[nn::account::UserCountMax + 1 + 1];
            count = nn::es::ListLightTicketInfo(lightInfo, nn::account::UserCountMax + 1 + 1, baseTicketInfo.rightsId);
            EXPECT_EQ(1, count);
        }
    }

    // ベースとなるチケットの情報と RightsId が同じで別のチケットであるチケットの情報
    nn::es::TicketInfo anotherTicketInfo = baseTicketInfo;
    anotherTicketInfo.deviceId++;
    anotherTicketInfo.ticketId++;
    anotherTicketInfo.accountId++;

    {
        uint64_t ticketBuffer[nn::es::MaxTicketSize];
        int ticketSize = nnt::es::PublishTicket(ticketBuffer, sizeof(ticketBuffer), {}, anotherTicketInfo.deviceId, anotherTicketInfo.ticketId, anotherTicketInfo.rightsId, anotherTicketInfo.accountId);
        EXPECT_EQ(anotherTicketInfo.ticketSize, ticketSize);

        // RightsId が同じで別のチケットであるチケットをインポートする
        ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::ImportTicket(ticketBuffer, ticketSize, nnt::es::Certificate, nnt::es::CertificateSize));

        // 各種取得できる情報が正しいかをチェック
        {
            // 所有しているパーソナライズドチケットは2つであるかチェック
            CheckPersonalizedTicketCount(2);

            // 権利を持っているパーソナライズドチケットの RightsId の数は1つであるかチェック
            nn::es::RightsIdIncludingKeyId list[MaxTicketCount];
            int count = nn::es::ListPersonalizedTicketRightsIds(list, MaxTicketCount);
            EXPECT_EQ(1, count);
            EXPECT_EQ(baseTicketInfo.rightsId, list[0]);

            // RightsId でクエリすると2つのチケットの情報を取得できるかチェック
            nn::es::TicketInfo info[MaxTicketCount];
            count = nn::es::GetTicketInfo(info, MaxTicketCount, &baseTicketInfo.rightsId, 1);
            EXPECT_EQ(2, count);
            EXPECT_TRUE(nnt::es::operator==(baseTicketInfo, info[0]));
            EXPECT_TRUE(nnt::es::operator==(anotherTicketInfo, info[1]));

            nn::es::LightTicketInfo lightInfo[nn::account::UserCountMax + 1 + 1];
            count = nn::es::ListLightTicketInfo(lightInfo, nn::account::UserCountMax + 1 + 1, baseTicketInfo.rightsId);
            EXPECT_EQ(2, count);
        }
    }
}

TEST_F(DuplicatedTicket, ImportDuplicatedPrepurchaseRecord)
{
    // ベースとなる予約購入記録
    nn::es::PrepurchaseRecord baseRecord = {};
    baseRecord.rightsId = { { 0x01, 0x00, 0x4b, 0x90, 0x00, 0x49, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } };
    baseRecord.ticketId = 0x0100000000000000;
    baseRecord.accountId = 16000000;
    baseRecord.deliveryScheduledTime = { 1505952222 };

    // 予約購入記録をインポートする
    ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::ImportPrepurchaseRecord(baseRecord));

    // 各種取得できる情報が正しいかをチェック
    {
        // 所有している予約購入記録は1つであるかチェック
        CheckPrepurchaseRecordCount(1);

        // 権利を持っている予約購入記録の RightsId の数は1つであるかチェック
        nn::es::RightsIdIncludingKeyId list[MaxTicketCount];
        int count = nn::es::ListPrepurchaseRecordRightsIds(list, MaxTicketCount);
        EXPECT_EQ(1, count);
        EXPECT_EQ(baseRecord.rightsId, list[0]);

        // RightsId でクエリすると1つの予約購入記録を取得できるかチェック
        nn::es::PrepurchaseRecord gotRecords[nn::account::UserCountMax + 1 + 1];
        count = nn::es::ListPrepurchaseRecordInfo(gotRecords, nn::account::UserCountMax + 1 + 1, baseRecord.rightsId);
        EXPECT_EQ(1, count);
        EXPECT_TRUE(nnt::es::operator==(baseRecord, gotRecords[0]));
    }

    // 既にインポートされている予約購入記録を再度インポートする
    // 成功するが、チケット DB の内容には変化がないかを確認する
    ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::ImportPrepurchaseRecord(baseRecord));

    // 各種取得できる情報が前回と等しいかチェック
    {
        CheckPrepurchaseRecordCount(1);

        nn::es::RightsIdIncludingKeyId list[MaxTicketCount];
        int count = nn::es::ListPrepurchaseRecordRightsIds(list, MaxTicketCount);
        EXPECT_EQ(1, count);
        EXPECT_EQ(baseRecord.rightsId, list[0]);

        nn::es::PrepurchaseRecord gotRecords[nn::account::UserCountMax + 1 + 1];
        count = nn::es::ListPrepurchaseRecordInfo(gotRecords, nn::account::UserCountMax + 1 + 1, baseRecord.rightsId);
        EXPECT_EQ(1, count);
        EXPECT_TRUE(nnt::es::operator==(baseRecord, gotRecords[0]));
    }

    // 非キー値のみがベースと異なる予約購入記録
    nn::es::PrepurchaseRecord modifiedRecord = baseRecord;
    modifiedRecord.deliveryScheduledTime.value++;

    // 変更された予約購入記録をインポートする
    // 変更された予約購入記録で元々の記録が上書きされたかを確認する
    ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::ImportPrepurchaseRecord(modifiedRecord));

    // 各種取得できる情報が正しいかをチェック
    {
        // 所有している予約購入記録は1つであるかチェック
        CheckPrepurchaseRecordCount(1);

        // 権利を持っている予約購入記録の RightsId の数は1つであるかチェック
        nn::es::RightsIdIncludingKeyId list[MaxTicketCount];
        int count = nn::es::ListPrepurchaseRecordRightsIds(list, MaxTicketCount);
        EXPECT_EQ(1, count);
        EXPECT_EQ(baseRecord.rightsId, list[0]);

        // RightsId でクエリすると1つの予約購入記録を取得できるかチェック
        // 取得された予約購入記録は変更されているかチェック
        nn::es::PrepurchaseRecord gotRecords[nn::account::UserCountMax + 1 + 1];
        count = nn::es::ListPrepurchaseRecordInfo(gotRecords, nn::account::UserCountMax + 1 + 1, baseRecord.rightsId);
        EXPECT_EQ(1, count);
        EXPECT_TRUE(nnt::es::operator==(modifiedRecord, gotRecords[0]));
    }

    // RightsId が同じで別の予約購入記録である予約購入記録
    nn::es::PrepurchaseRecord anotherRecord = baseRecord;
    anotherRecord.ticketId++;
    anotherRecord.accountId++;

    // RightsId が同じで別の予約購入記録である予約購入記録をインポートする
    ES_TEST_FAILURE_UNLESS_RESULT_SUCCESS(nn::es::ImportPrepurchaseRecord(anotherRecord));

    // 各種取得できる情報が正しいかをチェック
    {
        // 所有している予約購入記録は2つであるかチェック
        CheckPrepurchaseRecordCount(2);

        // 権利を持っている予約購入記録の RightsId の数は1つであるかチェック
        nn::es::RightsIdIncludingKeyId list[MaxTicketCount];
        int count = nn::es::ListPrepurchaseRecordRightsIds(list, MaxTicketCount);
        EXPECT_EQ(1, count);
        EXPECT_EQ(baseRecord.rightsId, list[0]);

        // RightsId でクエリすると2つの予約購入記録を取得できるかチェック
        nn::es::PrepurchaseRecord gotRecords[nn::account::UserCountMax + 1 + 1];
        count = nn::es::ListPrepurchaseRecordInfo(gotRecords, nn::account::UserCountMax + 1 + 1, baseRecord.rightsId);
        EXPECT_EQ(2, count);
        EXPECT_TRUE(nnt::es::operator==(modifiedRecord, gotRecords[0]));
        EXPECT_TRUE(nnt::es::operator==(anotherRecord, gotRecords[1]));
    }
}
