﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/es.h>
#include <nn/es/es_RightsApi.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>

TEST(ActiveRightsContext, DeviceSharedELicense)
{
    nn::es::Initialize();

    // ActiveRightsContext の作成とハンドルの取得
    auto handle = nn::es::CreateActiveRightsContext();

    // デバイス共有でアカウント情報を登録
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::SetUsersOfAccountRestrictedRights(handle, nullptr, 0, true));
    }

    {
        // RightsId の登録
        static const nn::es::RightsId rightsIdList1[] =
        {
            // 適当な値
            0xfedcba9876540000ull,
            0xfedcba9876540001ull,
            0xfedcba9876540000ull,  // あえて重複を入れておく
            0xfedcba9876540002ull,
        };
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::RegisterRightsIdList(handle, rightsIdList1, sizeof(rightsIdList1) / sizeof(rightsIdList1[0])));

        // RightsId の追加登録
        static const nn::es::RightsId rightsIdList2[] =
        {
            // 適当な値
            0xfedcba9876540004ull,
            0xfedcba9876540001ull,  // rightsIdList1 との重複
            0xfedcba9876540003ull,
        };
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::RegisterRightsIdList(handle, rightsIdList2, sizeof(rightsIdList2) / sizeof(rightsIdList2[0])));
    }

    {
        // 登録した RightsId の取得（重複分は削除されている）
        const int length = 10;
        nn::es::RightsId rightsIdList[length];
        auto count = nn::es::ListUsingRightsIds(rightsIdList, length, handle);
        NN_LOG("nn::es::ListUsingRightsIds() count = %d\n", count);

        static const nn::es::RightsId expectedRightsIdList[] =
        {
            0xfedcba9876540000ull,
            0xfedcba9876540001ull,
            0xfedcba9876540002ull,
            0xfedcba9876540004ull,
            0xfedcba9876540003ull,
        };

        EXPECT_EQ(5, count);
        EXPECT_TRUE(std::equal(std::cbegin(rightsIdList), std::cend(rightsIdList), std::cbegin(expectedRightsIdList)));

        // 登録した RightsId の取得（isTemporaryOnly = true）
        count = nn::es::ListUsingRightsIds(rightsIdList, length, handle, true);
        NN_LOG("nn::es::ListUsingRightsIds() count = %d\n", count);

        // チェック前は、権利なしと同じ状態なので全 RightsId が返る
        EXPECT_EQ(5, count);
        EXPECT_TRUE(std::equal(std::cbegin(rightsIdList), std::cend(rightsIdList), std::cbegin(expectedRightsIdList)));
    }

    {
        // eLicense のチェックと不要 RightsId の削除
        nn::es::CheckRightsIdListValidity(handle);
    }

    {
        const int length = 8;
        nn::account::NintendoAccountId naIdList[length] = {};
        int count = 0;
        auto isAccountRestricted = nn::es::GetUsersOfAccountRestrictedRights(&count, naIdList, length, handle);
        NN_LOG("nn::es::GetUsersOfAccountRestrictedRights() ret = %s\n", isAccountRestricted ? "true" : "false");
        EXPECT_EQ(false, isAccountRestricted);
    }

    {
        const int length = 10;
        nn::es::ELicenseId eLicenseId[length];
        auto count = nn::es::ListUsingELicenseIds(eLicenseId, length, handle, true, {0x123456789abcde0});
        NN_LOG("nn::es::ListUsingELicenseIds() count = %d\n", count);
        EXPECT_EQ(0, count);
    }

    {
        const int length = 10;
        nn::es::ELicenseId eLicenseId[length];
        auto count = nn::es::ListUsingELicenseIds(eLicenseId, length, handle, false, {0x123456789abcde0});
        NN_LOG("nn::es::ListUsingELicenseIds() count = %d\n", count);
        EXPECT_EQ(0, count);
    }

    {
        const int length = 10;
        nn::account::NintendoAccountId naId[length];
        auto count = nn::es::ListUsingELicenseOwnerIds(naId, length, handle);
        NN_LOG("nn::es::ListUsingELicenseOwnerIds() count = %d\n", count);
        EXPECT_EQ(0, count);
    }

    {
        // 対応する eLicense がないので、権利リストは 0 個になる
        nn::es::RemoveUnavailableRightsIds(handle);
    }

    {
        const int length = 10;
        nn::es::ELicenseId eLicenseId[length];
        auto count = nn::es::ListUsingELicenseIds(eLicenseId, length, handle, true, {0x123456789abcde0});
        NN_LOG("nn::es::ListUsingELicenseIds() count = %d\n", count);
        EXPECT_EQ(0, count);
    }


    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::BeginUsingActiveRightsContext(handle));
    }

#if 0
    {
        nn::es::RightsIdIncludingKeyId rightsId = {};
        int keyGeneration = {};

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::RegisterTitleKey(&rightsId, &keyGeneration, 1));
        nn::es::UnregisterAllTitleKey();
    }
#endif

    {
        // 登録した RightsId の取得
        // 対応する eLicense がないので RemoveInvalidRightsIds() で 0 になる
        const int length = 10;
        nn::es::RightsId rightsIdList[length];
        auto count = nn::es::ListUsingRightsIds(rightsIdList, length, handle);
        NN_LOG("nn::es::ListUsingRightsIds() count = %d\n", count);
        EXPECT_EQ(0, count);
    }

    {
        auto status = nn::es::GetActiveRightsContextStatus(handle);
        NN_LOG("Context's status = %d\n", status);
    }

    {
        auto expiedTick = nn::es::GetActiveRightsContextExpiredTime(handle);
        NN_LOG("nn::es::GetActiveRightsContextExpiredTime() leftTime=%lld(sec)\n", (expiedTick - nn::os::GetSystemTick()).ToTimeSpan().GetSeconds());
    }

    {
        nn::es::EndUsingActiveRightsContext(handle);
    }

    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::ForceActivateRightsContextForExit(handle));
    }

    // ハンドルのクローズ
    nn::es::CloseActiveRightsContextHandle(handle);

    nn::es::Finalize();
}   // NOLINT(impl/function_size)

TEST(ActiveRightsContext, RestrictedAccountELicense)
{
    nn::es::Initialize();

    // ActiveRightsContext の作成とハンドルの取得
    auto handle = nn::es::CreateActiveRightsContext();

    // アカウント限定権利の利用者アカウントを登録
    {
        const int count = 1;
        nn::account::NintendoAccountId naIdList[count] = { {0x123456789abcde0} };
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::SetUsersOfAccountRestrictedRights(handle, naIdList, count, false));
    }

    {
        const int length = 10;
        nn::es::RightsId rightsIdList[length] = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::RegisterRightsIdList(handle, rightsIdList, length));
    }

    {
        // eLicense のチェックと不要 RightsId の削除
        nn::es::CheckRightsIdListValidity(handle);
    }

    {
        const int length = 10;
        nn::es::ELicenseId eLicenseId[length];
        auto count = nn::es::ListUsingELicenseIds(eLicenseId, length, handle, true, {0x123456789abcde0});
        NN_LOG("nn::es::ListUsingELicenseIds() count = %d\n", count);
        EXPECT_EQ(0, count);
    }

    {
        const int length = 10;
        nn::es::ELicenseId eLicenseId[length];
        auto count = nn::es::ListUsingELicenseIds(eLicenseId, length, handle, false, {0x123456789abcde0});
        NN_LOG("nn::es::ListUsingELicenseIds() count = %d\n", count);
        EXPECT_EQ(0, count);
    }

    {
        const int length = 10;
        nn::account::NintendoAccountId naId[length];
        auto count = nn::es::ListUsingELicenseOwnerIds(naId, length, handle);
        NN_LOG("nn::es::ListUsingELicenseOwnerIds() count = %d\n", count);
        EXPECT_EQ(0, count);
    }

    {
        // 対応する eLicense がないので、権利リストは 0 個になる
        nn::es::RemoveUnavailableRightsIds(handle);
    }

    {
        const int length = 8;
        nn::account::NintendoAccountId naIdList[length] = {};
        int count = 0;
        auto isAccountRestricted = nn::es::GetUsersOfAccountRestrictedRights(&count, naIdList, length, handle);
        NN_LOG("nn::es::GetUsersOfAccountRestrictedRights() ret = %s\n", isAccountRestricted ? "true" : "false");
        EXPECT_EQ(true, isAccountRestricted);
        NN_LOG("nn::es::GetUsersOfAccountRestrictedRights() count = %d\n", count);
        EXPECT_EQ(1, count);
        EXPECT_EQ(0x123456789abcde0ull, naIdList[0].id);
    }

    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::BeginUsingActiveRightsContext(handle));
    }

#if 0
    {
        nn::es::RightsIdIncludingKeyId rightsId = {};
        int keyGeneration = {};

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::RegisterTitleKey(&rightsId, &keyGeneration, 1));
        nn::es::UnregisterAllTitleKey();
    }
#endif

    {
        const int length = 10;
        nn::es::RightsId rightsIdList[length];
        auto count = nn::es::ListUsingRightsIds(rightsIdList, length, handle);
        NN_LOG("nn::es::ListUsingRightsIds() count = %d\n", count);
    }

    {
        auto status = nn::es::GetActiveRightsContextStatus(handle);
        NN_UNUSED(status);
    }

    {
        auto expiedTick = nn::es::GetActiveRightsContextExpiredTime(handle);
        NN_LOG("nn::es::GetActiveRightsContextExpiredTime() leftTime=%lld(sec)\n", (expiedTick - nn::os::GetSystemTick()).ToTimeSpan().GetSeconds());
    }

    {
        nn::es::EndUsingActiveRightsContext(handle);
    }

    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::ForceActivateRightsContextForExit(handle));
    }

    // ハンドルのクローズ
    nn::es::CloseActiveRightsContextHandle(handle);

    nn::es::Finalize();
}   // NOLINT(impl/function_size)

#if 0
// マニュアル操作が必要なため、とりあえず無効化しておく
TEST(ActiveRightsContext, GetActiveRightsContextExpiredTimeChangedEvent)
{
    nn::es::Initialize();

    // ActiveRightsContext の作成とハンドルの取得
    auto handle = nn::es::CreateActiveRightsContext();

    // 適当な RightsId を登録
    static const nn::es::RightsId rightsIdList[] = { 0xfedcba9876540000 };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::RegisterRightsIdList(handle, rightsIdList, 1));
    nn::es::CheckRightsIdListValidity(handle);
    nn::es::RemoveUnavailableRightsIds(handle);

    {
        auto sfHandle = nn::es::GetActiveRightsContextExpiredTimeChangedEvent(handle);
        nn::os::SystemEvent event;
        event.AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), nn::os::EventClearMode_ManualClear);
        sfHandle.Detach();

        event.Clear();
        NN_ABORT_UNLESS(event.TryWait() == false);

        NN_LOG("===========================================================================\n");
        NN_LOG("Please execute the following command.\n");
        NN_LOG("But it's required to register any NintendoAccount beforehand.\n");
        NN_LOG("$> RunOnTarget.exe DevMenuCommandSystem.nsp -- ticket import-elicense 0\n");
        NN_LOG("===========================================================================\n");

        // インポートが完了するまで待機する
        NN_LOG("Now waiting ...\n\n");
        event.Wait();
        event.Clear();
        NN_LOG("Detected completing nn::es::EndImportELicenseArchive().\n\n");
        NN_ABORT_UNLESS(event.TryWait() == false);
    }

    // ハンドルのクローズ
    nn::es::CloseActiveRightsContextHandle(handle);

    nn::es::Finalize();
}
#endif
