﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/err/err_ErrorContext.h>
#include <nn/err/err_LibraryApi.h>
#include <nn/err/err_ShowApplicationErrorApi.h>
#include <nn/err/err_ShowErrorApi.h>
#include <nn/err/err_ShowErrorApiForSystem.h>
#include <nn/err/err_ShowUnacceptableAddOnContentVersionErrorApi.h>
#include <nn/err/err_ShowUnacceptableApplicationVersionErrorApi.h>
#include <nn/err/err_SystemApi.h>
#include <nn/err/err_SystemErrorArg.h>

#include <nn/nn_Log.h>
#include <nn/fs/fs_Result.h>

#include <nn/oe.h>
#include <nn/os.h>
#include <nn/pctl/pctl_ResultSystem.h>
#include <nn/time.h>
#include <nn/time/time_StandardUserSystemClock.h>

using namespace nn;
using namespace nn::err;

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
const int ThreadStackSize = 8192;
NN_ALIGNAS(4096) char g_OeMessageHandleThreadStack[ThreadStackSize];
nn::os::ThreadType g_OeMessageHandleThread;

void OeMessageHandleThreadFunc(void* arg)
{
    NN_UNUSED(arg);

    for( ;;)
    {
        auto message = nn::oe::PopNotificationMessage();
        switch( message )
        {
        case nn::oe::MessageFocusStateChanged:
            break;
        case nn::oe::MessageExitRequest:
            return;
        default:
            break;
        }
    }
}
#endif

class ShowErrorTest : public testing::Test
{
protected:
    static void SetUpTestCase()
    {
        nn::time::Initialize();
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        nn::oe::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_OeMessageHandleThread, OeMessageHandleThreadFunc, NULL, g_OeMessageHandleThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&g_OeMessageHandleThread);
#endif
    }
    static void TearDownTestCase()
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
#endif
        nn::time::Finalize();
    }
};

nn::Result GetResultAllocationMemoryFailed()
{
    return nn::fs::ResultAllocationMemoryFailed();
}

TEST_F(ShowErrorTest, ErrorCode)
{
    nn::err::ShowError(nn::err::MakeErrorCode(2001, 1));
}

TEST_F(ShowErrorTest, Result)
{
    nn::err::ShowError(nn::fs::ResultAllocationMemoryFailed());
}

TEST_F(ShowErrorTest, ErrorResultVariantResult)
{
    ErrorResultVariant variant = GetResultAllocationMemoryFailed();
    ASSERT_EQ(ErrorResultVariantState_HasResult, variant.GetState());
    ShowError(variant);
}

TEST_F(ShowErrorTest, ErrorResultVariantErrorCode)
{
    ErrorResultVariant variant = MakeErrorCode(2110, 2001);
    ASSERT_EQ(ErrorResultVariantState_HasErrorCode, variant.GetState());
    ShowError(variant);
}

TEST_F(ShowErrorTest, ErrorResultVariantAssignment)
{
    ErrorResultVariant variant;
    ASSERT_EQ(ErrorResultVariantState_HasNone, variant.GetState());

    {
        auto errorCode = MakeErrorCode(2110, 2001);
        variant = errorCode;
        ASSERT_EQ(ErrorResultVariantState_HasErrorCode, variant.GetState());
        ASSERT_EQ(errorCode.category, static_cast<ErrorCode>(variant).category);
        ASSERT_EQ(errorCode.number, static_cast<ErrorCode>(variant).number);
        ShowError(variant);
    }

    {
        auto result = GetResultAllocationMemoryFailed();
        variant = result;
        ASSERT_EQ(ErrorResultVariantState_HasResult, variant.GetState());
        ASSERT_EQ(result.GetInnerValueForDebug(), static_cast<Result>(variant).GetInnerValueForDebug());
        ShowError(variant);
    }

    variant = ErrorResultVariant();
    EXPECT_EQ(ErrorResultVariantState_HasNone, variant.GetState());
}

TEST_F(ShowErrorTest, ParentalControl)
{
    nn::err::ShowError(nn::pctl::ResultRestrictedByRating());
    nn::err::ShowError(nn::pctl::ResultSnsPostRestricted());
    nn::err::ShowError(nn::pctl::ResultFreeCommunicationRestricted());
}

NN_ALIGNAS(ErrorViewerStartupParamAddressAlignment) char s_WorkBuffer[ErrorViewerStartupParamSizeMax];

TEST_F(ShowErrorTest, ErrorRecord)
{
    size_t actualSize;
    nn::time::PosixTime posixTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime));
    {
        // SystemData
        CreateErrorViewerStartupParamForRecordedError(s_WorkBuffer, &actualSize, ErrorViewerStartupParamSizeMax, "2001-0001", nullptr, posixTime);
        ShowErrorRecord(s_WorkBuffer, actualSize);
    }
    {
        // SystemError
        CreateErrorViewerStartupParamForRecordedError(s_WorkBuffer, &actualSize, ErrorViewerStartupParamSizeMax, "2001-0001", "RecordedMessage for SystemError", posixTime);
        ShowErrorRecord(s_WorkBuffer, actualSize);
    }
    {
        // ApplicationError
        CreateErrorViewerStartupParamForRecordedError(s_WorkBuffer, &actualSize, ErrorViewerStartupParamSizeMax, "2-ABCDE-0001", "ApplicationError for ApplicationError", posixTime);
        ShowErrorRecord(s_WorkBuffer, actualSize);
    }
}

TEST_F(ShowErrorTest, Eula)
{
    auto regions = {
        nn::settings::system::RegionCode_Japan,
        nn::settings::system::RegionCode_Usa,
        nn::settings::system::RegionCode_Europe,
        nn::settings::system::RegionCode_Australia,
        nn::settings::system::RegionCode_China,
        nn::settings::system::RegionCode_Korea,
        nn::settings::system::RegionCode_Taiwan
    };

    for( auto r : regions )
    {
        nn::err::ShowEula(r);
    }
}

const char LongMessage[] = "" \
"0000000000111111111122222222223333333333444444444455555555556666666666777777777788888888889999999999" \
"0000000000111111111122222222223333333333444444444455555555556666666666777777777788888888889999999999" \
"0000000000111111111122222222223333333333444444444455555555556666666666777777777788888888889999999999" \
"0000000000111111111122222222223333333333444444444455555555556666666666777777777788888888889999999999" \
"0000000000111111111122222222223333333333444444444455555555556666666666777777777788888888889999999999" \
"0000000000111111111122222222223333333333444444444455555555556666666666777777777788888888889999999999" \
"0000000000111111111122222222223333333333444444444455555555556666666666777777777788888888889999999999" \
"0000000000111111111122222222223333333333444444444455555555556666666666777777777788888888889999999999" \
"0000000000111111111122222222223333333333444444444455555555556666666666777777777788888888889999999999" \
"0000000000111111111122222222223333333333444444444455555555556666666666777777777788888888889999999999";


TEST_F(ShowErrorTest, ApplicationError)
{
    nn::err::ApplicationErrorArg arg(1234, "ApplicationError:Dialog", "ApplicationError:FullScreen", nn::settings::LanguageCode::Make(nn::settings::Language_Japanese));
    nn::err::ShowApplicationError(arg);
}

TEST_F(ShowErrorTest, ApplicationErrorDialog)
{
    nn::err::ApplicationErrorArg arg(1234, "ApplicationError:Dialog", "", nn::settings::LanguageCode::Make(nn::settings::Language_Japanese));
    nn::err::ShowApplicationError(arg);
}

TEST_F(ShowErrorTest, ApplicationErrorFullScreen)
{
    nn::err::ApplicationErrorArg arg(1234, "", "ApplicationError:FullScreen", nn::settings::LanguageCode::Make(nn::settings::Language_Japanese));
    nn::err::ShowApplicationError(arg);
}

TEST_F(ShowErrorTest, ShowSystemError)
{
    auto errorCode = err::MakeErrorCode(2345, 6789);
    nn::err::SystemErrorArg arg(errorCode, "SystemError:Dialog", LongMessage, nn::settings::LanguageCode::Make(nn::settings::Language_Japanese));
    nn::err::ShowSystemError(arg);
}

TEST_F(ShowErrorTest, ShowSystemErrorWithErrorContext)
{
    auto errorCode = err::MakeErrorCode(2345, 6789);
    nn::err::SystemErrorArg arg(errorCode, LongMessage, LongMessage, nn::settings::LanguageCode::Make(nn::settings::Language_Japanese));

    ErrorContext errorContext;
    errorContext.type = ErrorContextType::Http;
    std::strncpy(errorContext.http.fqdn, "nintendo.co.jp", sizeof(errorContext.http.fqdn));
    std::strncpy(errorContext.http.ip, "255.255.255.255", sizeof(errorContext.http.ip));

    arg.AttachErrorContext(&errorContext);

    nn::err::ShowSystemError(arg);
}

NN_ALIGNAS(os::MemoryPageSize) err::EulaData eulaData;

#if defined(NN_BUILD_CONFIG_OS_WIN)
// 実機の場合 Ocean が定義している正しいデータフォーマットを渡さないとエラービューアが Fatal を起こすのでここでは実施しない。
TEST_F(ShowErrorTest, SystemUpdateEula)
{
    eulaData.dataCount = 4;
    eulaData.data[0].language = settings::Language::Language_AmericanEnglish;
    eulaData.data[0].size = 1;
    eulaData.data[0].body[0] = 0u;
    eulaData.data[1].language = settings::Language::Language_BritishEnglish;
    eulaData.data[1].size = 2;
    eulaData.data[1].body[0] = 1u;
    eulaData.data[2].language = settings::Language::Language_Japanese;
    eulaData.data[2].size = 3;
    eulaData.data[2].body[0] = 2u;
    eulaData.data[3].language = settings::Language::Language_Dutch;
    eulaData.data[3].size = 4;
    eulaData.data[3].body[0] = 3u;

    nn::err::ShowSystemUpdateEula(nn::settings::system::RegionCode_China, eulaData);
}
#endif

TEST_F(ShowErrorTest, ResultWithBacktrace)
{
    nn::Result results[ResultBacktrace::CountMax];
    results[0] = fs::ResultMountNameAlreadyExists();
    for( int i = 1; i < ResultBacktrace::CountMax - 1; i++ )
    {
        results[i] = fs::ResultAllocationMemoryFailed();
    }
    results[ResultBacktrace::CountMax - 1] = fs::ResultMountNameAlreadyExists();

    ResultBacktrace resultBacktrace;
    ResultBacktrace::Make(&resultBacktrace, results, ResultBacktrace::CountMax);
    err::ShowError(nn::fs::ResultDataCorrupted(), resultBacktrace);
}

TEST_F(ShowErrorTest, ResultWithHttpErrorContext)
{
    ErrorContext errorContext;
    errorContext.type = ErrorContextType::Http;
    std::strncpy(errorContext.http.fqdn, "nintendo.co.jp", sizeof(errorContext.http.fqdn));
    std::strncpy(errorContext.http.ip, "255.255.255.255", sizeof(errorContext.http.ip));
    err::ShowError(nn::fs::ResultDataCorrupted(), errorContext);
}

TEST_F(ShowErrorTest, ResultWithFileSystemContext)
{
    ErrorContext errorContext;
    errorContext.type = ErrorContextType::FileSystem;
    std::strncpy(errorContext.fileSystem.path, "mount:/directory/subdirectory/subsubdirectory/file", sizeof(errorContext.fileSystem.path));
    errorContext.fileSystem.fsResultValue = fs::ResultPathNotFound::InnerValue;
    err::ShowError(nn::fs::ResultDataCorrupted(), errorContext);
}

TEST_F(ShowErrorTest, ResultWithWebMediaPlayerContext)
{
    ErrorContext errorContext;
    errorContext.type = ErrorContextType::WebMediaPlayer;
    std::strncpy(errorContext.webMediaPlayer.openUrl, "https://somewhere/where/webmedia/opened?param0=x&param1=y", sizeof(errorContext.webMediaPlayer.openUrl));
    for( int i = 0; i < NN_ARRAY_SIZE(errorContext.webMediaPlayer.lastSocketErrors); i++ )
    {
        errorContext.webMediaPlayer.lastSocketErrors[i] = i;
    }
    err::ShowError(nn::fs::ResultDataCorrupted(), errorContext);
}

TEST_F(ShowErrorTest, ResultWithLocalContentShareContext)
{
    ErrorContext errorContext;
    errorContext.type = ErrorContextType::LocalContentShare;
    errorContext.localContentShare.applicationId = nn::ncm::ApplicationId{ 0x1111222233334444 };
    errorContext.localContentShare.resultInnerValue = 0x123;
    errorContext.localContentShare.numKey = 2;
    for( int i = 0; i < 2; i++ )
    {
        errorContext.localContentShare.keyList[i].id = static_cast<uint64_t>(i);
        errorContext.localContentShare.keyList[i].version = static_cast<uint32_t>(i);
        errorContext.localContentShare.keyList[i].type = static_cast<ncm::ContentMetaType>(i);
        errorContext.localContentShare.keyList[i].installType = static_cast<ncm::ContentInstallType>(i);
    }
    errorContext.localContentShare.isSender = true;
    errorContext.localContentShare.isApplicationRequest = true;
    errorContext.localContentShare.hasExFatDriver = true;
    errorContext.localContentShare.ip = 1;
    err::ShowError(nn::fs::ResultDataCorrupted(), errorContext);
}

TEST_F(ShowErrorTest, ErrorCodeWithHttpErrorContext)
{
    ErrorCode errorCode = { 2900, 0000 };
    ErrorContext errorContext;
    errorContext.type = ErrorContextType::Http;
    std::strncpy(errorContext.http.fqdn, "nintendo.co.jp", sizeof(errorContext.http.fqdn));
    std::strncpy(errorContext.http.ip, "255.255.255.255", sizeof(errorContext.http.ip));
    err::ShowError(errorCode, errorContext);
}

TEST_F(ShowErrorTest, ErrorCodeFileSystemContext)
{
    ErrorCode errorCode = { 2900, 0000 };
    ErrorContext errorContext;
    errorContext.type = ErrorContextType::FileSystem;
    std::strncpy(errorContext.fileSystem.path, "mount:/directory/subdirectory/subsubdirectory/file", sizeof(errorContext.fileSystem.path));
    errorContext.fileSystem.fsResultValue = fs::ResultPathNotFound::InnerValue;
    err::ShowError(errorCode, errorContext);
}

TEST_F(ShowErrorTest, ErrorCodeWithWebMediaPlayerContext)
{
    ErrorCode errorCode = { 2900, 0000 };
    ErrorContext errorContext;
    errorContext.type = ErrorContextType::WebMediaPlayer;
    std::strncpy(errorContext.webMediaPlayer.openUrl, "https://somewhere/where/webmedia/opened?param0=x&param1=y", sizeof(errorContext.webMediaPlayer.openUrl));
    for( int i = 0; i < NN_ARRAY_SIZE(errorContext.webMediaPlayer.lastSocketErrors); i++ )
    {
        errorContext.webMediaPlayer.lastSocketErrors[i] = i;
    }
    err::ShowError(errorCode, errorContext);
}

TEST_F(ShowErrorTest, ErrorCodeWithLocalContentShareContext)
{
    ErrorCode errorCode = { 2900, 0000 };
    ErrorContext errorContext;
    errorContext.type = ErrorContextType::LocalContentShare;
    errorContext.localContentShare.applicationId = nn::ncm::ApplicationId{ 0x1111222233334444 };
    errorContext.localContentShare.resultInnerValue = 0x123;
    errorContext.localContentShare.numKey = 2;
    for( int i = 0; i < 2; i++ )
    {
        errorContext.localContentShare.keyList[i].id = static_cast<uint64_t>(i);
        errorContext.localContentShare.keyList[i].version = static_cast<uint32_t>(i);
        errorContext.localContentShare.keyList[i].type = static_cast<ncm::ContentMetaType>(i);
        errorContext.localContentShare.keyList[i].installType = static_cast<ncm::ContentInstallType>(i);
    }
    errorContext.localContentShare.isSender = true;
    errorContext.localContentShare.isApplicationRequest = true;
    errorContext.localContentShare.hasExFatDriver = true;
    errorContext.localContentShare.ip = 1;
    err::ShowError(errorCode, errorContext);
}

TEST_F(ShowErrorTest, ShowUnacceptableAddOnContentVersionError)
{
    nn::err::ShowUnacceptableAddOnContentVersionError();
}

TEST_F(ShowErrorTest, ShowUnacceptableApplicationVersionError)
{
    nn::err::ShowUnacceptableApplicationVersionError();
}

TEST(ShowErrorDeathTest, ApplicationErrorNoMessageDeathTest)
{
    err::ApplicationErrorArg arg;
    arg.SetLanguageCode(nn::settings::LanguageCode::Make(nn::settings::Language_Japanese));
    ASSERT_DEATH_IF_SUPPORTED(err::ShowApplicationError(arg), "");
}

TEST(ShowErrorDeathTest, ApplicationErrorNoLanguageDeathTest)
{
    err::ApplicationErrorArg arg;
    arg.SetDialogMessage("Hello, Dialog!");
    ASSERT_DEATH_IF_SUPPORTED(err::ShowApplicationError(arg), "");
}

TEST(ShowErrorDeathTest, SystemErrorNoMessageDeathTest)
{
    err::SystemErrorArg arg;
    arg.SetLanguageCode(nn::settings::LanguageCode::Make(nn::settings::Language_Japanese));
    ASSERT_DEATH_IF_SUPPORTED(err::ShowSystemError(arg), "");
}

TEST(ShowErrorDeathTest, SystemErrorNoLanguageDeathTest)
{
    err::SystemErrorArg arg;
    arg.SetDialogMessage("Hello, Dialog!");
    ASSERT_DEATH_IF_SUPPORTED(err::ShowSystemError(arg), "");
}
