﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/erpt.h>
#include <nn/erpt/erpt_Manager.h>
#include <nn/erpt/erpt_Report.h>
#include <nn/err/err_ApiForApplet.h>
#include <nn/err/err_ApiForErrorViewer.h>
#include <nn/err/err_ErrorViewerAppletParam.h>
#include <nn/err/err_LibraryApi.h>
#include <nn/err/err_Result.h>
#include <nn/err/err_SystemTypes.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/pctl/pctl_ResultSystem.h>
#include <nn/settings.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/time.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include "err_ErptUtil.h"
#include "err_ErptUtilForErrorViewer.h"
#include "err_StringUtil.h"

#include "testErr_Util.h"

using namespace nn;

namespace
{
    NN_ALIGNAS(8) char g_StartupParamBuffer[err::ErrorViewerStartupParamSizeMax];

    void AssertArray0Cleared(const Bit8* reserved, size_t size)
    {
        for( size_t i = 0; i < size; i++ )
        {
            ASSERT_EQ(0, reserved[i]);
        }
    }
}

TEST(StringUtil, MakeErrorCodeString)
{
    err::ErrorCode errorCode;
    errorCode.category = 2345u;
    errorCode.number = 6789u;
    char errorCodeStr[err::ErrorCode::StringLengthMax];
    err::detail::MakeErrorCodeString(errorCodeStr, err::ErrorCode::StringLengthMax, errorCode);
    EXPECT_STREQ("2345-6789", errorCodeStr);
}

TEST(StringUtil, MakeApplicationErrorCodeString)
{
    ns::ApplicationErrorCodeCategory category;
    util::Strlcpy(category.value, "ABCDE", sizeof(category.value));
    err::ApplicationErrorCodeNumber number = 1234u;
    char errorCodeStr[err::ErrorCode::StringLengthMax];
    err::detail::MakeApplicationErrorCodeString(errorCodeStr, err::ErrorCode::StringLengthMax, category, number);
    EXPECT_STREQ("2-ABCDE-1234", errorCodeStr);
}

TEST(StringUtil, ParseErrorCodeString)
{
    err::ErrorCode errorCode;
    err::detail::ParseErrorCodeString(&errorCode, "2345-6789");
    EXPECT_EQ(2345u, errorCode.category);
    EXPECT_EQ(6789u, errorCode.number);
}

TEST(StringUtil, ParseApplicationErrorCodeString)
{
    ns::ApplicationErrorCodeCategory category;
    err::ApplicationErrorCodeNumber number;
    err::detail::ParseApplicationErrorCodeString(&category, &number, "2-ABCDE-1234");
    EXPECT_STREQ("ABCDE", category.value);
    EXPECT_EQ(1234u, number);
}

TEST(ErrorViewerParam, Common)
{
    nn::err::ErrorViewerStartupParamCommon param(nn::err::ErrorType::SystemData);

    param.isJumpEnabled = false;
    EXPECT_FALSE(nn::err::IsJumpEnabled(&param));

    param.isJumpEnabled = true;
    EXPECT_TRUE(nn::err::IsJumpEnabled(&param));

    AssertArray0Cleared(param.reserved, sizeof(param.reserved));
}

TEST(ErrorViewerParam, SystemData)
{
    nn::err::ErrorViewerStartupParamForSystemData param;

    EXPECT_EQ(nn::err::ErrorType::SystemData, nn::err::GetErrorType(&param));
    EXPECT_FALSE(nn::err::IsErrorRecordDisplayRequested(&param));
    EXPECT_TRUE(nn::err::IsJumpEnabled(&param));
    EXPECT_FALSE(param.hasResultBacktrace);
    EXPECT_FALSE(param.hasErrorContext);
}

TEST(ErrorViewerParam, SystemError)
{
    nn::err::ErrorViewerStartupParamForSystemError param;
    auto ja = settings::LanguageCode::Make(settings::Language::Language_Japanese);
    util::Strlcpy(param.languageCode.string, ja.string, sizeof(param.languageCode.string));

    EXPECT_EQ(nn::err::ErrorType::SystemError, nn::err::GetErrorType(&param));
    EXPECT_FALSE(nn::err::IsErrorRecordDisplayRequested(&param));
    EXPECT_TRUE(nn::err::IsJumpEnabled(&param));
    EXPECT_STREQ(nn::err::GetMessageLanguageCode(&param).string, ja.string);
    AssertArray0Cleared(param.reserved, sizeof(param.reserved));
}

TEST(ErrorViewerParam, ApplicationError)
{
    nn::err::ErrorViewerStartupParamForApplicationError param;
    auto ja = settings::LanguageCode::Make(settings::Language::Language_Japanese);
    util::Strlcpy(param.languageCode.string, ja.string, sizeof(param.languageCode.string));

    EXPECT_EQ(nn::err::ErrorType::ApplicationError, nn::err::GetErrorType(&param));
    EXPECT_FALSE(nn::err::IsErrorRecordDisplayRequested(&param));
    EXPECT_TRUE(nn::err::IsJumpEnabled(&param));
    EXPECT_STREQ(nn::err::GetMessageLanguageCode(&param).string, ja.string);
    AssertArray0Cleared(param.reserved, sizeof(param.reserved));
}

TEST(ErrorViewerParam, Eula)
{
    auto regions = {
        nn::settings::system::RegionCode_Australia,
        nn::settings::system::RegionCode_China,
        nn::settings::system::RegionCode_Europe,
        nn::settings::system::RegionCode_Japan,
        nn::settings::system::RegionCode_Korea,
        nn::settings::system::RegionCode_Taiwan,
        nn::settings::system::RegionCode_Usa,
    };

    for( auto r : regions )
    {
        nn::err::ErrorViewerStartupParamForEula param;
        param.regionCode = r;

        EXPECT_EQ(nn::err::ErrorType::Eula, nn::err::GetErrorType(&param));
        EXPECT_FALSE(nn::err::IsErrorRecordDisplayRequested(&param));
        EXPECT_TRUE(nn::err::IsJumpEnabled(&param));
        EXPECT_EQ(r, nn::err::GetEulaRegionCode(&param));
        AssertArray0Cleared(param.reserved, sizeof(param.reserved));
    }
}

template<typename R>
void ParentalControlTest(R result)
{
    nn::err::ErrorViewerStartupParamForParentalControl param;
    param.pctlResultRestricted = result;

    EXPECT_EQ(nn::err::ErrorType::ParentalControl, nn::err::GetErrorType(&param));
    EXPECT_FALSE(nn::err::IsErrorRecordDisplayRequested(&param));
    EXPECT_TRUE(nn::err::IsJumpEnabled(&param));
    EXPECT_EQ(result.GetInnerValueForDebug(), nn::err::GetParentalControlResultRestricted(&param).GetInnerValueForDebug());
    AssertArray0Cleared(param.reserved, sizeof(param.reserved));
}

TEST(ErrorViewerParam, ParentalControl)
{
    ParentalControlTest(nn::pctl::ResultFreeCommunicationRestricted());
    ParentalControlTest(nn::pctl::ResultSnsPostRestricted());
    ParentalControlTest(nn::pctl::ResultRestrictedByRating());
}

TEST(ErrorViewerParam, RecordedSystemData)
{
    nn::time::Initialize();
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };
    auto time = nn::time::PosixTime();
    nn::time::StandardUserSystemClock::GetCurrentTime(&time);

    nn::err::ErrorViewerStartupParamForRecordedSystemData param;
    param.timeOfOccurrence = time;

    EXPECT_EQ(nn::err::ErrorType::RecordedSystemData, nn::err::GetErrorType(&param));
    EXPECT_TRUE(nn::err::IsErrorRecordDisplayRequested(&param));
    EXPECT_TRUE(nn::err::IsJumpEnabled(&param));
    EXPECT_EQ(time.value, nn::err::GetTimeOfOccurrence(&param).value);
    AssertArray0Cleared(param.reserved, sizeof(param.reserved));
}

TEST(ErrorViewerParam, RecordedSystemError)
{
    nn::time::Initialize();
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };
    auto time = nn::time::PosixTime();
    nn::time::StandardUserSystemClock::GetCurrentTime(&time);

    nn::err::ErrorViewerStartupParamForRecordedSystemError param;
    param.timeOfOccurrence = time;

    EXPECT_EQ(nn::err::ErrorType::RecordedSystemError, nn::err::GetErrorType(&param));
    EXPECT_TRUE(nn::err::IsErrorRecordDisplayRequested(&param));
    EXPECT_TRUE(nn::err::IsJumpEnabled(&param));
    EXPECT_EQ(time.value, nn::err::GetTimeOfOccurrence(&param).value);
    AssertArray0Cleared(param.reserved, sizeof(param.reserved));

    auto timeOfOccurrence = nn::time::PosixTime();
    timeOfOccurrence.value = 1234;
    size_t actualParamSize;
    memset(g_StartupParamBuffer, 0, sizeof(g_StartupParamBuffer));
    ASSERT_TRUE(err::CreateErrorViewerStartupParamForRecordedError(g_StartupParamBuffer, &actualParamSize, sizeof(g_StartupParamBuffer),
        "2345-6789", "SystemErrorDescription", timeOfOccurrence));
    err::ErrorInfo errorInfo(g_StartupParamBuffer);
    char16_t errorCodeString[err::ErrorCode::StringLengthMax];
    errorInfo.GetErrorCodeString(errorCodeString, err::ErrorCode::StringLengthMax);
    EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16("2345-6789", errorCodeString));
    EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16("SystemErrorDescription", errorInfo.GetFullScreenViewUiSettings().message));
}

TEST(ErrorViewerParam, RecordedApplicationError)
{
    nn::time::Initialize();
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };
    auto time = nn::time::PosixTime();
    nn::time::StandardUserSystemClock::GetCurrentTime(&time);

    nn::err::ErrorViewerStartupParamForRecordedApplicationError param;
    param.timeOfOccurrence = time;

    EXPECT_EQ(nn::err::ErrorType::RecordedApplicationError, nn::err::GetErrorType(&param));
    EXPECT_TRUE(nn::err::IsErrorRecordDisplayRequested(&param));
    EXPECT_TRUE(nn::err::IsJumpEnabled(&param));
    EXPECT_EQ(time.value, nn::err::GetTimeOfOccurrence(&param).value);
    AssertArray0Cleared(param.reserved0, sizeof(param.reserved0));
    AssertArray0Cleared(param.reserved1, sizeof(param.reserved1));

    auto timeOfOccurrence = nn::time::PosixTime();
    timeOfOccurrence.value = 1234;
    size_t actualParamSize;
    memset(g_StartupParamBuffer, 0, sizeof(g_StartupParamBuffer));
    ASSERT_TRUE(err::CreateErrorViewerStartupParamForRecordedError(g_StartupParamBuffer, &actualParamSize, sizeof(g_StartupParamBuffer),
        "2-ABCDE-6789", "ApplicationErrorDescription", timeOfOccurrence));
    err::ErrorInfo errorInfo(g_StartupParamBuffer);
    char16_t errorCodeString[err::ErrorCode::StringLengthMax];
    errorInfo.GetErrorCodeString(errorCodeString, err::ErrorCode::StringLengthMax);
    EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16("2-ABCDE-6789", errorCodeString));
    EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16("ApplicationErrorDescription", errorInfo.GetFullScreenViewUiSettings().message));
}

NN_ALIGNAS(4096) err::EulaData eulaData;
Bit8 eulaDataBuffer[err::EulaDataSizeMax];

TEST(ErrorViewerParam, SystemUpdateEula)
{
    eulaData.dataCount = err::EulaDataCountMax;
    for( int i = 0; i < eulaData.dataCount; i++ )
    {
        eulaData.data[i].language = static_cast<settings::Language>(i);
        eulaData.data[i].size = 1 + i * 10;
        for( size_t j = 0; j < eulaData.data[i].size; j++ )
        {
            eulaData.data[i].body[j] = static_cast<Bit8>(i);
        }
        AssertArray0Cleared(eulaData.data[i].reserved, sizeof(eulaData.data[i].reserved));
    }
    AssertArray0Cleared(eulaData.reserved, sizeof(eulaData.reserved));

    size_t size;
    for( int i = 0; i < eulaData.dataCount; i++ )
    {
        ASSERT_TRUE(err::GetEulaData(&size, eulaDataBuffer, sizeof(eulaDataBuffer), &eulaData, static_cast<settings::Language>(i)));
        ASSERT_EQ(static_cast<size_t>(1 + i * 10), size);
        for( size_t j = 0; j < size; j++ )
        {
            ASSERT_EQ(static_cast<Bit8>(i), eulaDataBuffer[j]);
        }
    }

}

class ErrorRecordTest : public testing::Test
{
protected:
    virtual void SetUp()
    {
        m_ErptManager.Initialize();
    }
    virtual void TearDown()
    {
        m_ErptManager.Finalize();
    }

    static void SetUpTestCase()
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        ns::Initialize();
#endif
    }

    static void TearDownTestCase()
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        ns::Finalize();
#endif
    }
private:
    static const uint32_t ReportBufferSize = 48 * 1024;
    erpt::Manager m_ErptManager;
    char m_ReportBuffer[ReportBufferSize];
public:
    char m_WorkBuffer[err::RequiredBufferSizeForRecordError];
    const char* Find(const char* haystack, size_t haystackLength, const char* needle, size_t needleLength) NN_NOEXCEPT
    {
        if( needleLength > haystackLength )
        {
            return nullptr;
        }
        for( size_t i = 0; i <= haystackLength - needleLength; i++ )
        {
            if( memcmp(haystack + i, needle, needleLength) == 0 )
            {
                return haystack + i;
            }
        }
        return nullptr;
    }
    void OpenAndReadFirstReport(uint32_t* outReadCount) NN_NOEXCEPT
    {
        erpt::ReportList reportList;
        NNT_ASSERT_RESULT_SUCCESS(m_ErptManager.GetReportList(reportList));
        ASSERT_TRUE(reportList.reportCount >= 1u);
        // 直前に記録したエラーは [0] に入る。
        erpt::Report report;
        NNT_ASSERT_RESULT_SUCCESS(report.Open(reportList.Report[0].reportId));
        int64_t reportSize;
        NNT_ASSERT_RESULT_SUCCESS(report.GetSize(&reportSize));
        ASSERT_TRUE(reportSize < ReportBufferSize);
        NNT_ASSERT_RESULT_SUCCESS(report.Read(outReadCount, reinterpret_cast<uint8_t*>(m_ReportBuffer), ErrorRecordTest::ReportBufferSize));
        m_ReportBuffer[*outReadCount] = '\0';

    }
    void Check(const char* key, const char* value) NN_NOEXCEPT
    {
        uint32_t readCount;
        OpenAndReadFirstReport(&readCount);

        // TORIAEZU : key の後ろに value がある、という感じで探す。
        auto pKey = Find(m_ReportBuffer, readCount, key, strlen(key));
        ASSERT_TRUE(pKey != nullptr) << key << " not found.";
        auto pValue = Find(pKey, readCount - (pKey - m_ReportBuffer), value, strlen(value));
        ASSERT_TRUE(pValue != nullptr) << value << " not found";
        // value の長さが 256 文字を超えた場合、+1 + (strlen(value)/32) は成り立たなくなるのでテストケース更新時に注意。
        ASSERT_EQ(pKey + strlen(key) + 1 + (strlen(value) / 32), pValue);
    }
    // Key の項目があることだけを確認する。
    bool HasKey(const char* key) NN_NOEXCEPT
    {
        uint32_t readCount;
        OpenAndReadFirstReport(&readCount);

        auto pKey = Find(m_ReportBuffer, readCount, key, strlen(key));
        return pKey != nullptr;
    }

    void CreateMinimalErrorReport()
    {
        auto context = erpt::Context(erpt::CategoryId::ErrorInfo);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::FieldId::ErrorCode, "2345-6789", static_cast<uint32_t>(strlen("2345-6789"))));
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateReport(erpt::ReportType::ReportType_Visible));
    }
};

// TORIAEZU : ErrorViewer 向けの API ではないが、ここでテスト。
TEST_F(ErrorRecordTest, RecordErrorForTelemetry)
{
    {
        // ResultBacktrace なし。
        err::RecordErrorForTelemetry(err::ResultSystemProgramAbort(), ncm::ApplicationId{ 0x0123456701234567 }, nullptr, nullptr, 0);
        Check("ApplicationID", "0123456701234567");
    }

    {
        // ResultBacktrace あり。
        static char workBuffer[3 * sizeof(uint32_t)];
        err::ResultBacktrace resultBacktrace;
        resultBacktrace.count = 3;
        resultBacktrace.results[0] = err::ResultApplicationAbort();
        resultBacktrace.results[1] = err::ResultUnacceptableApplicationVersion();
        resultBacktrace.results[2] = err::ResultServiceStatusParseError();
        err::RecordErrorForTelemetry(err::ResultSystemProgramAbort(), ncm::ApplicationId{ 0x0123456701234567 }, resultBacktrace, workBuffer, sizeof(workBuffer));
        Check("ApplicationID", "0123456701234567");
        EXPECT_TRUE(HasKey("ResultBacktrace"));
    }
}


// HOS版 RecordError 内ではアプレット環境でないと動かない機能を使っているのでひとまずテストを動かさないようにしておく。
#if defined( NN_BUILD_CONFIG_OS_WIN )

TEST_F(ErrorRecordTest, RecordSystemData)
{
    nn::err::ErrorViewerStartupParamForSystemData param;
    param.SetErrorCode(err::MakeErrorCode(2345, 6789));

    err::RecordError(&param, m_WorkBuffer, err::RequiredBufferSizeForRecordError);

    Check("ErrorCode", "2345-6789");
}

TEST_F(ErrorRecordTest, RecordSystemErrorBothMessage)
{
    err::ErrorViewerStartupParamForSystemError param;
    param.errorCode.category = 2468;
    param.errorCode.number = 1357;
    util::Strlcpy(param.dialogMessage, "DialogMessage", sizeof(param.dialogMessage));
    util::Strlcpy(param.fullScreenMessage, "FullScreenMessage", sizeof(param.fullScreenMessage));

    err::RecordError(&param, m_WorkBuffer, err::RequiredBufferSizeForRecordError);

    Check("ErrorCode", "2468-1357");
    Check("ErrorDescription", "FullScreenMessage");
}

TEST_F(ErrorRecordTest, RecordSystemErrorDialogMessage)
{
    err::ErrorViewerStartupParamForSystemError param;
    param.errorCode.category = 2345;
    param.errorCode.number = 6789;
    util::Strlcpy(param.dialogMessage, "DialogMessage", sizeof(param.dialogMessage));
    util::Strlcpy(param.fullScreenMessage, "", sizeof(param.fullScreenMessage));

    err::RecordError(&param, m_WorkBuffer, err::RequiredBufferSizeForRecordError);

    Check("ErrorCode", "2345-6789");
    Check("ErrorDescription", "DialogMessage");
}

TEST_F(ErrorRecordTest, RecordSystemErrorFullScreenMessage)
{
    err::ErrorViewerStartupParamForSystemError param;
    param.errorCode.category = 2345;
    param.errorCode.number = 6789;
    util::Strlcpy(param.dialogMessage, "", sizeof(param.dialogMessage));
    util::Strlcpy(param.fullScreenMessage, "FullScreenMessage", sizeof(param.fullScreenMessage));

    err::RecordError(&param, m_WorkBuffer, err::RequiredBufferSizeForRecordError);

    Check("ErrorCode", "2345-6789");
    Check("ErrorDescription", "FullScreenMessage");
}

TEST_F(ErrorRecordTest, RecordApplicationError)
{
    err::ErrorViewerStartupParamForApplicationError param;
    param.applicationErrorCodeNumber = 1;
    util::Strlcpy(param.dialogMessage, "DialogMessage", sizeof(param.dialogMessage));
    util::Strlcpy(param.fullScreenMessage, "ApplicationError:FullScreenMessage", sizeof(param.fullScreenMessage));

    err::RecordError(&param, m_WorkBuffer, err::RequiredBufferSizeForRecordError);

    Check("ErrorCode", "2-DUMMY-0001");
    Check("ErrorDescription", "ApplicationError:FullScreenMessage");
}

TEST_F(ErrorRecordTest, ErptUtilApplicationInfo)
{
    err::detail::SubmitApplicationInfo(m_WorkBuffer, err::RequiredBufferSizeForRecordError);
    CreateMinimalErrorReport();
    EXPECT_TRUE(HasKey("ApplicationID"));
    EXPECT_TRUE(HasKey("ApplicationTitle"));
    EXPECT_TRUE(HasKey("ApplicationVersion"));
    // EXPECT_TRUE(HasKey("ApplicationStorageLocation")); // 記録未対応。
    err::detail::ClearApplicationInfo();
    CreateMinimalErrorReport();
    EXPECT_FALSE(HasKey("ApplicationID"));
    EXPECT_FALSE(HasKey("ApplicationTitle"));
    EXPECT_FALSE(HasKey("ApplicationVersion"));
    EXPECT_FALSE(HasKey("ApplicationStorageLocation"));
}

TEST_F(ErrorRecordTest, ErptUtilStorageFreeSpaceInfo)
{
    err::detail::SubmitFileSystemInfo();
    CreateMinimalErrorReport();
    EXPECT_TRUE(HasKey("NANDFreeSpace"));
    EXPECT_TRUE(HasKey("NANDSpeedMode"));
    EXPECT_TRUE(HasKey("NANDCID"));
    EXPECT_TRUE(HasKey("NANDNumActivationFailures"));
    EXPECT_TRUE(HasKey("NANDNumActivationErrorCorrections"));
    EXPECT_TRUE(HasKey("NANDNumReadWriteFailures"));
    EXPECT_TRUE(HasKey("NANDNumReadWriteErrorCorrections"));
//    EXPECT_TRUE(HasKey("NANDErrorLog"));// WIN 版では存在しない。実機ではエラーログが合った場合のみ存在する。


    EXPECT_TRUE(HasKey("SDCardFreeSpace")); // 実機で実行するようにした場合、SDCardFreeSpace は SDカードがささっていないと存在しない項目になる。
    EXPECT_TRUE(HasKey("MicroSDSpeedMode"));
    EXPECT_TRUE(HasKey("MicroSDCID"));
    EXPECT_TRUE(HasKey("SdCardUserAreaSize"));
    EXPECT_TRUE(HasKey("SdCardProtectedAreaSize"));
    EXPECT_TRUE(HasKey("SdCardNumActivationFailures"));
    EXPECT_TRUE(HasKey("SdCardNumActivationErrorCorrections"));
    EXPECT_TRUE(HasKey("SdCardNumReadWriteFailures"));
    EXPECT_TRUE(HasKey("SdCardNumReadWriteErrorCorrections"));
    EXPECT_TRUE(HasKey("SdCardMountStatus"));
//    EXPECT_TRUE(HasKey("SdCardErrorLog"));// WIN 版では存在しない。実機ではエラーログが合った場合のみ存在する。

    EXPECT_TRUE(HasKey("GameCardCID"));
    EXPECT_TRUE(HasKey("GameCardDeviceId"));
    EXPECT_TRUE(HasKey("GameCardCrcErrorCount"));
    EXPECT_TRUE(HasKey("GameCardAsicCrcErrorCount"));
    EXPECT_TRUE(HasKey("GameCardRefreshCount"));
    EXPECT_TRUE(HasKey("GameCardReadRetryCount"));
    EXPECT_TRUE(HasKey("GameCardTimeoutRetryErrorCount"));
    EXPECT_TRUE(HasKey("GameCardInsertionCount"));
    EXPECT_TRUE(HasKey("GameCardRemovalCount"));
    EXPECT_TRUE(HasKey("GameCardAsicInitializeCount"));
    EXPECT_TRUE(HasKey("GameCardAsicReinitializeCount"));
    EXPECT_TRUE(HasKey("GameCardAsicReinitializeFailureCount"));
    EXPECT_TRUE(HasKey("GameCardAsicReinitializeFailureDetail"));
    EXPECT_TRUE(HasKey("GameCardRefreshSuccessCount"));
    EXPECT_TRUE(HasKey("GameCardAwakenCount"));
    EXPECT_TRUE(HasKey("GameCardAwakenFailureCount"));
    EXPECT_TRUE(HasKey("GameCardReadCountFromInsert"));
    EXPECT_TRUE(HasKey("GameCardReadCountFromAwaken"));
    EXPECT_TRUE(HasKey("GameCardLastReadErrorPageAddress"));
    EXPECT_TRUE(HasKey("GameCardLastReadErrorPageCount"));

    EXPECT_TRUE(HasKey("FsRemountForDataCorruptCount"));
    EXPECT_TRUE(HasKey("FsRemountForDataCorruptRetryOutCount"));

    EXPECT_TRUE(HasKey("FsPooledBufferPeakFreeSize"));
    EXPECT_TRUE(HasKey("FsPooledBufferRetriedCount"));
    EXPECT_TRUE(HasKey("FsPooledBufferReduceAllocationCount"));
    EXPECT_TRUE(HasKey("FsBufferManagerPeakFreeSize"));
    EXPECT_TRUE(HasKey("FsBufferManagerRetriedCount"));
    EXPECT_TRUE(HasKey("FsExpHeapPeakFreeSize"));
    EXPECT_TRUE(HasKey("FsBufferPoolPeakFreeSize"));
    EXPECT_TRUE(HasKey("FsPatrolReadAllocateBufferSuccessCount"));
    EXPECT_TRUE(HasKey("FsPatrolReadAllocateBufferFailureCount"));

    err::detail::ClearFileSystemInfo();
    CreateMinimalErrorReport();
    EXPECT_FALSE(HasKey("NANDFreeSpace"));
    EXPECT_FALSE(HasKey("NANDSpeedMode"));
    EXPECT_FALSE(HasKey("NANDCID"));
    EXPECT_FALSE(HasKey("NANDNumActivationFailures"));
    EXPECT_FALSE(HasKey("NANDNumActivationErrorCorrections"));
    EXPECT_FALSE(HasKey("NANDNumReadWriteFailures"));
    EXPECT_FALSE(HasKey("NANDNumReadWriteErrorCorrections"));

    EXPECT_FALSE(HasKey("SDCardFreeSpace"));
    EXPECT_FALSE(HasKey("MicroSDSpeedMode"));
    EXPECT_FALSE(HasKey("MicroSDCID"));
    EXPECT_FALSE(HasKey("SdCardUserAreaSize"));
    EXPECT_FALSE(HasKey("SdCardProtectedAreaSize"));
    EXPECT_FALSE(HasKey("SdCardNumActivationFailures"));
    EXPECT_FALSE(HasKey("SdCardNumActivationErrorCorrections"));
    EXPECT_FALSE(HasKey("SdCardNumReadWriteFailures"));
    EXPECT_FALSE(HasKey("SdCardNumReadWriteErrorCorrections"));
    EXPECT_FALSE(HasKey("SdCardMountInfo"));

    EXPECT_FALSE(HasKey("GameCardCID"));
    EXPECT_FALSE(HasKey("GameCardCrcErrorCount"));
    EXPECT_FALSE(HasKey("GameCardAsicCrcErrorCount"));
    EXPECT_FALSE(HasKey("GameCardRefreshCount"));
    EXPECT_FALSE(HasKey("GameCardReadRetryCount"));
    EXPECT_FALSE(HasKey("GameCardTimeoutRetryErrorCount"));
    EXPECT_FALSE(HasKey("GameCardInsertionCount"));
    EXPECT_FALSE(HasKey("GameCardRemovalCount"));
    EXPECT_FALSE(HasKey("GameCardAsicInitializeCount"));

    EXPECT_FALSE(HasKey("FsRemountForDataCorruptCount"));
    EXPECT_FALSE(HasKey("FsRemountForDataCorruptRetryOutCount"));

    EXPECT_FALSE(HasKey("FsPooledBufferPeakFreeSize"));
    EXPECT_FALSE(HasKey("FsPooledBufferRetriedCount"));
    EXPECT_FALSE(HasKey("FsPooledBufferReduceAllocationCount"));
    EXPECT_FALSE(HasKey("FsBufferManagerPeakFreeSize"));
    EXPECT_FALSE(HasKey("FsBufferManagerRetriedCount"));
    EXPECT_FALSE(HasKey("FsExpHeapPeakFreeSize"));
    EXPECT_FALSE(HasKey("FsBufferPoolPeakFreeSize"));
    EXPECT_FALSE(HasKey("FsPatrolReadAllocateBufferSuccessCount"));
    EXPECT_FALSE(HasKey("FsPatrolReadAllocateBufferFailureCount"));
}

#endif

