﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testEns_MessagePackFileInputStream.h"

namespace nnt { namespace ens {

MessagePackFileInputStream::MessagePackFileInputStream() NN_NOEXCEPT
    : m_IsOpened(false)
    , m_LastError(nn::ResultSuccess())
    , m_FilePosition(0)
    , m_pCacheBuffer(nullptr)
    , m_CacheBufferSize(0)
    , m_ReadSize(0)
    , m_ReadPosition(0)
{
}

MessagePackFileInputStream::MessagePackFileInputStream(void* pBuffer, size_t size) NN_NOEXCEPT
    : m_IsOpened(false)
    , m_LastError(nn::ResultSuccess())
    , m_FilePosition(0)
    , m_pCacheBuffer(reinterpret_cast<nn::Bit8*>(pBuffer))
    , m_CacheBufferSize(size)
    , m_ReadSize(0)
    , m_ReadPosition(0)
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER(size, 0u);
}

MessagePackFileInputStream::~MessagePackFileInputStream() NN_NOEXCEPT
{
    Close();
}

nn::Result MessagePackFileInputStream::Open(const char* pPath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPath);

    NN_SDK_ASSERT(!m_IsOpened);

    NN_RESULT_DO(nn::fs::OpenFile(&m_Handle, pPath, nn::fs::OpenMode_Read));
    m_IsOpened = true;
    m_LastError = nn::ResultSuccess();

    m_FilePosition = 0;

    m_ReadSize = 0;
    m_ReadPosition = 0;

    NN_RESULT_SUCCESS;
}

void MessagePackFileInputStream::Close() NN_NOEXCEPT
{
    if (m_IsOpened)
    {
        nn::fs::CloseFile(m_Handle);
        m_IsOpened = false;
    }
}

nn::Result MessagePackFileInputStream::GetLastReadError() const NN_NOEXCEPT
{
    return m_LastError;
}

bool MessagePackFileInputStream::Read(void* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER(size, 0u);

    NN_SDK_ASSERT(m_IsOpened);

    if (m_CacheBufferSize > 0)
    {
        return ReadWithMemoryCache(pBuffer, size);
    }
    else
    {
        return ReadWithoutMemoryCache(pBuffer, size);
    }
}

bool MessagePackFileInputStream::GetDataPointerFromMemoryCache(const void** ppOutData, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutData);
    NN_SDK_REQUIRES_GREATER(size, 0u);

    if (m_CacheBufferSize == 0)
    {
        return false;
    }

    size_t remainSize = m_ReadSize - m_ReadPosition;

    if (remainSize < size)
    {
        return false;
    }

    *ppOutData = &m_pCacheBuffer[m_ReadPosition];
    m_ReadPosition += size;

    return true;
}

bool MessagePackFileInputStream::ReadWithMemoryCache(void* pBuffer, size_t size) NN_NOEXCEPT
{
    size_t readSize = ReadFromMemoryCache(pBuffer, size);

    // キャッシュからすべて読み込んだ。
    if (readSize == size)
    {
        return true;
    }

    NN_SDK_ASSERT_EQUAL(m_ReadPosition, m_ReadSize);

    pBuffer = reinterpret_cast<nn::Bit8*>(pBuffer) + readSize;
    size -= readSize;

    // 直接読み込んだ方が速い場合は直接読み込む。
    if (m_CacheBufferSize <= size)
    {
        if (ReadFromFile(pBuffer, size) != size)
        {
            return false;
        }

        return true;
    }

    // 読み込み処理が 1 回は行われており、読み込んだサイズがキャッシュバッファサイズ未満ならすべて読み込んだと判定できるはず。
    if (m_ReadSize > 0 && m_ReadSize < m_CacheBufferSize)
    {
        return false;
    }

    m_ReadSize = ReadFromFile(m_pCacheBuffer, m_CacheBufferSize);

    if (m_ReadSize == 0)
    {
        return false;
    }

    m_ReadPosition = 0;

    readSize = ReadFromMemoryCache(pBuffer, size);

    NN_SDK_ASSERT_EQUAL(readSize, size);

    return true;
}

bool MessagePackFileInputStream::ReadWithoutMemoryCache(void* pBuffer, size_t size) NN_NOEXCEPT
{
    if (ReadFromFile(pBuffer, size) != size)
    {
        return false;
    }

    return true;
}

size_t MessagePackFileInputStream::ReadFromFile(void* pBuffer, size_t size) NN_NOEXCEPT
{
    size_t readSize;

    m_LastError = nn::fs::ReadFile(&readSize, m_Handle, m_FilePosition, pBuffer, size);

    if (m_LastError.IsFailure())
    {
        return 0;
    }

    m_FilePosition += static_cast<int64_t>(readSize);

    return readSize;
}

size_t MessagePackFileInputStream::ReadFromMemoryCache(void* pBuffer, size_t size) NN_NOEXCEPT
{
    size_t copySize = std::min(size, m_ReadSize - m_ReadPosition);

    if (copySize > 0)
    {
        std::memcpy(pBuffer, &m_pCacheBuffer[m_ReadPosition], copySize);
        m_ReadPosition += copySize;
    }

    return copySize;
}

}}
