﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testEns_Common.h"

#include <nn/ens/detail/core/ens_AuthTokenCacheManager.h>

namespace
{
    template <size_t N>
    void DoSimpleTest2() NN_NOEXCEPT
    {
        nn::ens::detail::core::AuthTokenCacheManager<N> manager;

        const size_t N2 = N * 2;

        nn::ens::UserId cachedUserIds[N2] = {};

        for (size_t n = 0; n < N2; n++)
        {
            cachedUserIds[n] = nn::ens::InvalidUserId;
        }

        for (uint64_t i = 1; i <= 1000; i++)
        {
            nn::ens::UserId userId = {i};

            nn::ens::detail::AuthToken token = {};
            nn::util::SNPrintf(token.value, sizeof (token.value), "userId_%llu", userId.value);

            manager.Set(userId, token);

            for (size_t n = N2 - 1; n > 0; n--)
            {
                cachedUserIds[n] = cachedUserIds[n - 1];
            }

            cachedUserIds[0] = userId;

            for (size_t n = 0; n < N; n++)
            {
                if (cachedUserIds[n] != nn::ens::InvalidUserId)
                {
                    nn::ens::detail::AuthToken token1 = {};
                    nn::ens::detail::AuthToken token2 = {};

                    EXPECT_TRUE(manager.Get(&token1, cachedUserIds[n]));
                    nn::util::SNPrintf(token2.value, sizeof (token2.value), "userId_%llu", cachedUserIds[n].value);

                    EXPECT_STREQ(token1.value, token2.value);
                }
            }
            // 後半はキャッシュから消えているはず。
            for (size_t n = N; n < N2; n++)
            {
                if (cachedUserIds[n] != nn::ens::InvalidUserId)
                {
                    EXPECT_FALSE(manager.Get(&token, cachedUserIds[n]));
                }
            }
        }
    }
}

TEST(EnsAuthTokenCacheManager, Simple1)
{
    const size_t N = 3;

    nn::ens::detail::core::AuthTokenCacheManager<N> manager;

    // クリア後にもう一度実行するため、2 回繰り返す。
    for (int c = 0; c < 2; c++)
    {
        for (uint64_t i = 1; i <= N + 1; i++)
        {
            nn::ens::UserId userId = {i};

            nn::ens::detail::AuthToken token = {};
            nn::util::SNPrintf(token.value, sizeof (token.value), "userId_%llu", userId.value);

            manager.Set(userId, token);
        }

        // 最初にセットしたものは消えている。
        {
            nn::ens::UserId userId = {1ull};
            nn::ens::detail::AuthToken token = {};

            EXPECT_FALSE(manager.Get(&token, userId));
        }
        for (uint64_t i = 2; i <= N; i++)
        {
            nn::ens::UserId userId = {i};
            nn::ens::detail::AuthToken token = {};

            EXPECT_TRUE(manager.Get(&token, userId));
        }

        manager.Clear();

        for (uint64_t i = 1; i <= N; i++)
        {
            nn::ens::UserId userId = {i};
            nn::ens::detail::AuthToken token = {};

            EXPECT_FALSE(manager.Get(&token, userId));
        }
    }
}

TEST(EnsAuthTokenCacheManager, Simple2)
{
    DoSimpleTest2<1>();
    DoSimpleTest2<2>();
    DoSimpleTest2<3>();
    DoSimpleTest2<4>();
    DoSimpleTest2<8>();
}

TEST(EnsAuthTokenCacheManager, Overwrite)
{
    nn::ens::detail::core::AuthTokenCacheManager<2> manager;

    // インデックス = 0 の時の上書き。
    {
        nn::ens::UserId userId = {1ull};
        nn::ens::detail::AuthToken token = {"12345"};

        manager.Set(userId, token);
        manager.Set(userId, token);

        EXPECT_TRUE(manager.Get(&token, userId));
    }

    manager.Clear();

    // インデックス > 0 の時の上書き。
    {
        nn::ens::UserId userId1 = {1ull};
        nn::ens::UserId userId2 = {2ull};
        nn::ens::detail::AuthToken token1 = {"1"};
        nn::ens::detail::AuthToken token2 = {"2"};

        manager.Set(userId1, token1);
        manager.Set(userId2, token2);
        manager.Set(userId1, token1);

        EXPECT_TRUE(manager.Get(&token1, userId1));
        EXPECT_TRUE(manager.Get(&token2, userId2));

        EXPECT_STREQ(token1.value, "1");
        EXPECT_STREQ(token2.value, "2");
    }
}
