﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testEns_Common.h"
#include "../../Common/testEns_AccountUtility.h"
#include "../../Common/testEns_NetworkUtility.h"

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStack[64 * 1024];

    NN_ALIGNAS(4096) nn::Bit8 g_ServiceWorkMemory[nn::ens::RequiredMemorySizeMin + 4 * 1024 * 1024];
}

namespace
{
    void WorkerThread(void*) NN_NOEXCEPT
    {
        nn::ens::StartServiceLoop("acbaa", g_ServiceWorkMemory, sizeof (g_ServiceWorkMemory));
    }

    void GenerateCredential(const char* pToken) NN_NOEXCEPT
    {
        nn::ens::Credential credential = {};
        nn::ens::AsyncContext context;

        nn::ens::GenerateCredential(&context, &credential, pToken);

        context.GetEvent().Wait();

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

#else

        if (context.GetResult().IsFailure() &&
            !nn::ens::ResultHttpErrorOperationTimedout::Includes(context.GetResult()))
        {
            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }

#endif
    }
}

class EnsConnectionReuse : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfig));

        nnt::ens::ReserveSockets();
        nnt::ens::InitializeLibcurl();
        nnt::ens::EnableCommunicationLogDump();
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
        nnt::ens::FinalizeLibcurl();
        nnt::ens::CancelSocketsReservation();

        nn::socket::Finalize();
    }

    virtual void SetUp() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
            g_ThreadStack, sizeof (g_ThreadStack), nn::os::DefaultThreadPriority + 1));

        nn::os::StartThread(&g_Thread);

        nn::nifm::SubmitNetworkRequestAndWait();
    }

    virtual void TearDown() NN_NOEXCEPT
    {
        nn::nifm::CancelNetworkRequest();

        nn::ens::StopServiceLoop();

        nn::os::DestroyThread(&g_Thread);
    }
};

TEST_F(EnsConnectionReuse, Reuse)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    nn::ens::detail::LibrarySettings::SetHttpKeepAliveTimeout(nn::TimeSpan::FromSeconds(1));

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    NN_LOG("###################################################################### (1)\n");

    EXPECT_NO_FATAL_FAILURE(GenerateCredential(token.Get()));

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1500));

    NN_LOG("###################################################################### (2: reuse = no)\n");

    // ここで「Re-using existing connection! (#0) with host api.hac.td1.acbaa.srv.nintendo.net」が出ないことを目視する。
    EXPECT_NO_FATAL_FAILURE(GenerateCredential(token.Get()));

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

    NN_LOG("###################################################################### (3: reuse = yes)\n");

    // ここで「Re-using existing connection! (#0) with host api.hac.td1.acbaa.srv.nintendo.net」が出ることを目視する。
    EXPECT_NO_FATAL_FAILURE(GenerateCredential(token.Get()));
}

#if 0 // 時間が掛かるので調べたい時だけ有効化する。

TEST_F(EnsConnectionReuse, KeepAlive)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    // サーバー側の HTTP Keep-Alive タイムアウト（約 1 分）以内に定期的に通信を行う。
    const int WaitSeconds = 40;
    const int Count = 3;

    // ライブラリの自動 cleanup を無効化する。
    nn::ens::detail::LibrarySettings::SetHttpKeepAliveTimeout(nn::TimeSpan::FromSeconds(WaitSeconds + 10));

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    for (int n = 0; n < Count; n++)
    {
        // n > 0 の時、ここで「Re-using existing connection! (#0) with host api.hac.td1.acbaa.srv.nintendo.net」が出ることを目視する。
        EXPECT_NO_FATAL_FAILURE(GenerateCredential(token.Get()));

        if (n != Count - 1)
        {
            for (int i = 0; i < WaitSeconds; i++)
            {
                if ((i % 10) == 0)
                {
                    NN_LOG("(%d) Sleep... (remain = %d seconds)\n", n + 1, WaitSeconds - i);
                }

                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }
        }
    }
}

TEST_F(EnsConnectionReuse, KeepAliveTimeout)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    // サーバー側の HTTP Keep-Alive タイムアウト（約 1 分）以上待機する。
    const int WaitSeconds = 80;

    // ライブラリの自動 cleanup を無効化する。
    nn::ens::detail::LibrarySettings::SetHttpKeepAliveTimeout(nn::TimeSpan::FromSeconds(WaitSeconds + 10));

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    EXPECT_NO_FATAL_FAILURE(GenerateCredential(token.Get()));

    for (int i = 0; i < WaitSeconds; i++)
    {
        if ((i % 10) == 0)
        {
            NN_LOG("Sleep... (remain = %d seconds)\n", WaitSeconds - i);
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    // ここで「Connection 0 seems to be dead!」が出ることを目視する。
    EXPECT_NO_FATAL_FAILURE(GenerateCredential(token.Get()));
}

#endif
